"use strict";
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.PPLWorkerProxyService = void 0;
/**
 * Simple worker proxy that communicates with PPL worker
 */
class PPLWorkerProxyService {
    worker;
    messageId = 0;
    pendingMessages = new Map();
    /**
     * Set up the worker
     */
    setup(workerSrc) {
        if (this.worker) {
            return; // Already set up
        }
        // Create worker from source
        const blob = new Blob([workerSrc], { type: 'application/javascript' });
        this.worker = new Worker(URL.createObjectURL(blob));
        // Handle messages from worker
        this.worker.onmessage = (e) => {
            const { id, result, error } = e.data;
            const pending = this.pendingMessages.get(id);
            if (pending) {
                this.pendingMessages.delete(id);
                if (error) {
                    pending.reject(new Error(error));
                }
                else {
                    pending.resolve(result);
                }
            }
        };
        // Handle worker errors
        this.worker.onerror = (error) => {
            // eslint-disable-next-line no-console
            console.error('PPL Worker error:', error);
        };
    }
    /**
     * Tokenize PPL content
     */
    async tokenize(content) {
        if (!this.worker) {
            throw new Error('PPL Worker Proxy Service has not been setup!');
        }
        return this.sendMessage('tokenize', [content]);
    }
    /**
     * Validate PPL content and get error markers
     */
    async validate(content) {
        if (!this.worker) {
            throw new Error('PPL Worker Proxy Service has not been setup!');
        }
        return this.sendMessage('validate', [content]);
    }
    /**
     * Stop the worker
     */
    stop() {
        if (this.worker) {
            this.worker.terminate();
            this.worker = undefined;
        }
        this.pendingMessages.clear();
    }
    /**
     * Send a message to the worker and wait for response
     */
    sendMessage(method, args) {
        return new Promise((resolve, reject) => {
            if (!this.worker) {
                reject(new Error('Worker not available'));
                return;
            }
            const id = ++this.messageId;
            this.pendingMessages.set(id, { resolve, reject });
            this.worker.postMessage({
                id,
                method,
                args,
            });
            // Set a timeout to prevent hanging
            setTimeout(() => {
                const pending = this.pendingMessages.get(id);
                if (pending) {
                    this.pendingMessages.delete(id);
                    reject(new Error('Worker timeout'));
                }
            }, 5000);
        });
    }
}
exports.PPLWorkerProxyService = PPLWorkerProxyService;
//# sourceMappingURL=worker_proxy_service.js.map