/*
 * Decompiled with CFR 0.152.
 */
package ghidra.server;

import ghidra.framework.store.local.LocalFileSystem;
import ghidra.server.RepositoryManager;
import ghidra.util.HashUtilities;
import ghidra.util.NumericUtilities;
import ghidra.util.exception.DuplicateNameException;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.regex.Pattern;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.x500.X500Principal;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class UserManager {
    static final Logger log = LogManager.getLogger(UserManager.class);
    public static final String X500_NAME_FORMAT = "RFC2253";
    public static final String ANONYMOUS_USERNAME = "-anonymous-";
    public static final String USER_PASSWORD_FILE = "users";
    public static final String DN_LOG_FILE = "UnknownDN.log";
    private static final String SSH_KEY_FOLDER = LocalFileSystem.HIDDEN_DIR_PREFIX + "ssh";
    private static final String SSH_PUBKEY_EXT = ".pub";
    private static final char[] DEFAULT_PASSWORD = "changeme".toCharArray();
    private static final int DEFAULT_PASSWORD_TIMEOUT_DAYS = 1;
    private static final int NO_EXPIRATION = -1;
    private RepositoryManager repositoryMgr;
    private final File userFile;
    private final File sshDir;
    private boolean enableLocalPasswords;
    private long defaultPasswordExpirationMS;
    private PrintWriter dnLogOut;
    private LinkedHashMap<String, UserEntry> userList = new LinkedHashMap();
    private HashMap<X500Principal, UserEntry> dnLookupMap = new HashMap();
    private long lastUserListChange;
    private static final Pattern VALID_USERNAME_REGEX = Pattern.compile("[a-zA-Z0-9][a-zA-Z0-9.\\-_/\\\\]*");

    UserManager(RepositoryManager repositoryMgr, boolean enableLocalPasswords, int defaultPasswordExpirationDays) {
        this.repositoryMgr = repositoryMgr;
        this.enableLocalPasswords = enableLocalPasswords;
        if (defaultPasswordExpirationDays < 0) {
            defaultPasswordExpirationDays = 1;
        }
        this.defaultPasswordExpirationMS = (long)defaultPasswordExpirationDays * 24L * 3600L * 1000L;
        log.info("Instantiating User Manager " + (enableLocalPasswords ? "(w/password management)" : ""));
        this.userFile = new File(repositoryMgr.getRootDir(), USER_PASSWORD_FILE);
        try {
            this.readUserListIfNeeded();
            this.clearExpiredPasswords();
            log.info("User file contains " + this.userList.size() + " entries");
        }
        catch (FileNotFoundException e) {
            log.error("Existing User file not found.");
        }
        catch (IOException e) {
            log.error((Object)e);
        }
        log.info("Known Users:");
        for (String name : this.userList.keySet()) {
            X500Principal x500User;
            Object dnStr = "";
            UserEntry entry = this.userList.get(name);
            if (entry != null && (x500User = entry.x500User) != null) {
                dnStr = " DN={" + x500User.getName() + "}";
            }
            log.info("   " + name + (String)dnStr);
        }
        this.sshDir = new File(repositoryMgr.getRootDir(), SSH_KEY_FOLDER);
        this.initSSH();
    }

    private void initSSH() {
        if (!this.sshDir.exists()) {
            this.sshDir.mkdir();
            return;
        }
        String[] list = this.sshDir.list((dir, name) -> name.endsWith(SSH_PUBKEY_EXT));
        if (list.length == 0) {
            return;
        }
        log.info("Users with stored SSH public key:");
        for (String fname : list) {
            String user = fname.substring(0, fname.length() - SSH_PUBKEY_EXT.length());
            if (!this.userList.containsKey(user)) continue;
            log.info("   " + user);
        }
    }

    public File getSSHPubKeyFile(String username) {
        if (!this.userList.containsKey(username)) {
            return null;
        }
        File f = new File(this.sshDir, username + SSH_PUBKEY_EXT);
        if (f.isFile()) {
            return f;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addUser(String username, char[] passwordHash, X500Principal x500User) throws DuplicateNameException, IOException {
        if (username == null) {
            throw new IllegalArgumentException();
        }
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            if (this.userList.containsKey(username)) {
                throw new DuplicateNameException("User " + username + " already exists");
            }
            UserEntry entry = new UserEntry();
            entry.username = username;
            entry.passwordHash = passwordHash;
            entry.passwordTime = new Date().getTime();
            entry.x500User = x500User;
            this.userList.put(username, entry);
            if (x500User != null) {
                this.dnLookupMap.put(x500User, entry);
            }
            this.writeUserList();
            log.info("User '" + username + "' added");
        }
    }

    public void addUser(String username) throws DuplicateNameException, IOException {
        this.addUser(username, (char[])null);
    }

    void addUser(String username, char[] saltedPasswordHash) throws DuplicateNameException, IOException {
        if (saltedPasswordHash == null && this.enableLocalPasswords) {
            saltedPasswordHash = this.getDefaultPasswordHash();
        }
        this.addUser(username, saltedPasswordHash, null);
    }

    public void addUser(String username, X500Principal x500User) throws DuplicateNameException, IOException {
        char[] passwordHash = this.enableLocalPasswords ? this.getDefaultPasswordHash() : null;
        this.addUser(username, passwordHash, x500User);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public X500Principal getDistinguishedName(String username) {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            UserEntry entry = this.userList.get(username);
            if (entry != null) {
                return entry.x500User;
            }
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String getUserByDistinguishedName(X500Principal x500User) {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            UserEntry entry = this.dnLookupMap.get(x500User);
            return entry != null ? entry.username : null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean setDistinguishedName(String username, X500Principal x500User) throws IOException {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            UserEntry oldEntry = (UserEntry)this.userList.remove(username);
            if (oldEntry != null) {
                if (oldEntry.x500User != null) {
                    this.dnLookupMap.remove(oldEntry.x500User);
                }
                UserEntry entry = new UserEntry();
                entry.username = username;
                entry.passwordHash = oldEntry.passwordHash;
                entry.x500User = x500User;
                this.userList.put(username, entry);
                if (x500User != null) {
                    this.dnLookupMap.put(x500User, entry);
                }
                this.writeUserList();
                return true;
            }
            return false;
        }
    }

    private void checkValidPasswordHash(char[] saltedPasswordHash) throws IOException {
        int i;
        if (saltedPasswordHash == null || saltedPasswordHash.length != 68) {
            throw new IOException("Invalid password hash");
        }
        for (i = 0; i < 4; ++i) {
            if (this.isLetterOrDigit(saltedPasswordHash[i])) continue;
            throw new IOException("Password set failed due invalid salt: " + new String(saltedPasswordHash) + " (" + i + "," + saltedPasswordHash[i] + ")");
        }
        for (i = 4; i < saltedPasswordHash.length; ++i) {
            if (this.isLowercaseHexDigit(saltedPasswordHash[i])) continue;
            throw new IOException("Password set failed due to invalid hash: " + new String(saltedPasswordHash) + " (" + i + "," + saltedPasswordHash[i] + ")");
        }
    }

    private boolean isLetterOrDigit(char c) {
        if (c < '0') {
            return false;
        }
        if (c > '9' && c < 'A') {
            return false;
        }
        if (c > 'Z' && c < 'a') {
            return false;
        }
        return c <= 'z';
    }

    private boolean isLowercaseHexDigit(char c) {
        if (c < '0') {
            return false;
        }
        if (c > '9' && c < 'a') {
            return false;
        }
        return c <= 'f';
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean setPassword(String username, char[] saltedSHA256PasswordHash, boolean isTemporary) throws IOException {
        if (!this.enableLocalPasswords) {
            throw new IOException("Local passwords are not used");
        }
        this.checkValidPasswordHash(saltedSHA256PasswordHash);
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            UserEntry oldEntry = (UserEntry)this.userList.remove(username);
            if (oldEntry != null) {
                UserEntry entry = new UserEntry();
                entry.username = username;
                entry.passwordHash = saltedSHA256PasswordHash;
                entry.passwordTime = isTemporary ? new Date().getTime() : -1L;
                entry.x500User = oldEntry.x500User;
                this.userList.put(username, entry);
                if (entry.x500User != null) {
                    this.dnLookupMap.put(entry.x500User, entry);
                }
                this.writeUserList();
                return true;
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean canSetPassword(String username) {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            UserEntry userEntry = this.userList.get(username);
            return this.enableLocalPasswords && userEntry != null && userEntry.passwordHash != null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getPasswordExpiration(String username) {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            UserEntry userEntry = this.userList.get(username);
            if (userEntry != null && userEntry.passwordHash != null && userEntry.passwordHash.length != 68) {
                return 0L;
            }
            return this.getPasswordExpiration(userEntry);
        }
    }

    private long getPasswordExpiration(UserEntry user) {
        long timeRemaining = 0L;
        if (user != null) {
            if (this.defaultPasswordExpirationMS == 0L || user.passwordTime == -1L) {
                return -1L;
            }
            if (user.passwordTime != 0L && (timeRemaining = this.defaultPasswordExpirationMS - (new Date().getTime() - user.passwordTime)) <= 0L) {
                timeRemaining = 0L;
            }
        }
        return timeRemaining;
    }

    public boolean resetPassword(String username, char[] saltedPasswordHash) throws IOException {
        if (!this.enableLocalPasswords) {
            return false;
        }
        return this.setPassword(username, saltedPasswordHash != null ? saltedPasswordHash : this.getDefaultPasswordHash(), true);
    }

    private char[] getDefaultPasswordHash() {
        return HashUtilities.getSaltedHash((String)HashUtilities.SHA256_ALGORITHM, (char[])DEFAULT_PASSWORD);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean removeUser(String username) throws IOException {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            UserEntry oldEntry = (UserEntry)this.userList.remove(username);
            if (oldEntry != null) {
                if (oldEntry.x500User != null) {
                    this.dnLookupMap.remove(oldEntry.x500User);
                }
                this.writeUserList();
                this.repositoryMgr.userRemoved(username);
                log.info("User removed from server: " + username);
                return true;
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String[] getUsers() {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            String[] names = new String[this.userList.size()];
            Iterator<String> iter = this.userList.keySet().iterator();
            int i = 0;
            while (iter.hasNext()) {
                names[i++] = iter.next();
            }
            return names;
        }
    }

    void clearExpiredPasswords() throws IOException {
        if (this.defaultPasswordExpirationMS == 0L) {
            return;
        }
        boolean dataChanged = false;
        for (UserEntry entry : this.userList.values()) {
            if (entry.passwordHash == null || !this.enableLocalPasswords || this.getPasswordExpiration(entry) != 0L) continue;
            entry.passwordHash = null;
            entry.passwordTime = 0L;
            dataChanged = true;
            log.warn("Default password expired for user '" + entry.username + "'");
        }
        if (dataChanged) {
            this.writeUserList();
        }
    }

    void readUserListIfNeeded() throws IOException {
        long lastMod = this.userFile.lastModified();
        if (this.lastUserListChange == lastMod) {
            if (lastMod == 0L) {
                this.writeUserList();
            }
            return;
        }
        LinkedHashMap<String, UserEntry> list = new LinkedHashMap<String, UserEntry>();
        HashMap<X500Principal, UserEntry> lookupMap = new HashMap<X500Principal, UserEntry>();
        UserManager.readUserList(this.userFile, list, lookupMap);
        this.userList = list;
        this.dnLookupMap = lookupMap;
        this.lastUserListChange = lastMod;
    }

    static void listUsers(File repositoriesRootDir) {
        File userFile = new File(repositoriesRootDir, USER_PASSWORD_FILE);
        LinkedHashMap<String, UserEntry> list = new LinkedHashMap<String, UserEntry>();
        HashMap<X500Principal, UserEntry> lookupMap = new HashMap<X500Principal, UserEntry>();
        try {
            UserManager.readUserList(userFile, list, lookupMap);
            System.out.println("\nRepository Server Users:");
            if (list.isEmpty()) {
                System.out.println("   <No users have been added>");
            } else {
                for (String name : list.keySet()) {
                    System.out.println("  " + name);
                }
            }
        }
        catch (IOException e) {
            System.out.println("\nFailed to read user file: " + e.getMessage());
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static void readUserList(File file, Map<String, UserEntry> usersIndexByName, Map<X500Principal, UserEntry> x500LookupMap) throws IOException {
        try (BufferedReader br = new BufferedReader(new FileReader(file));){
            String line;
            while ((line = br.readLine()) != null) {
                if (line.startsWith("#")) continue;
                try {
                    StringTokenizer st = new StringTokenizer(line, ":");
                    UserEntry entry = new UserEntry();
                    entry.username = st.nextToken();
                    if (!UserManager.isValidUserName(entry.username)) {
                        log.error("Invalid user name, skipping: " + entry.username);
                        continue;
                    }
                    if (st.hasMoreTokens()) {
                        entry.passwordHash = st.nextToken().toCharArray();
                        if (st.hasMoreTokens()) {
                            String dn;
                            try {
                                String timeStr = st.nextToken();
                                entry.passwordTime = "*".equals(timeStr) ? -1L : NumericUtilities.parseHexLong((String)timeStr);
                            }
                            catch (NumberFormatException e) {
                                log.error("Invalid password time - forced expiration: " + entry.username);
                                entry.passwordTime = 0L;
                            }
                            if (st.hasMoreTokens() && (dn = st.nextToken()).length() > 0) {
                                entry.x500User = new X500Principal(dn);
                            }
                        }
                    }
                    usersIndexByName.put(entry.username, entry);
                    if (entry.x500User == null) continue;
                    x500LookupMap.put(entry.x500User, entry);
                }
                catch (NoSuchElementException noSuchElementException) {}
            }
            return;
        }
    }

    private void writeUserList() throws IOException {
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(this.userFile));){
            for (UserEntry entry : this.userList.values()) {
                bw.write(entry.username);
                bw.write(":");
                if (entry.passwordHash != null) {
                    bw.write(entry.passwordHash);
                    bw.write(58);
                    if (entry.passwordTime == -1L) {
                        bw.write(42);
                    } else {
                        bw.write(Long.toHexString(entry.passwordTime));
                    }
                } else {
                    bw.write("*:*");
                }
                if (entry.x500User != null) {
                    bw.write(":");
                    bw.write(entry.x500User.getName());
                }
                bw.newLine();
            }
        }
        this.lastUserListChange = this.userFile.lastModified();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isValidUser(String username) {
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            return this.userList.containsKey(username);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void authenticateUser(String username, char[] password) throws IOException, FailedLoginException {
        if (username == null || password == null) {
            throw new FailedLoginException("Invalid authentication data");
        }
        RepositoryManager repositoryManager = this.repositoryMgr;
        synchronized (repositoryManager) {
            this.clearExpiredPasswords();
            UserEntry entry = this.userList.get(username);
            if (entry == null) {
                throw new FailedLoginException("Unknown user: " + username);
            }
            if (entry.passwordHash == null || entry.passwordHash.length < 32) {
                throw new FailedLoginException("User password not set, must be reset");
            }
            if (entry.passwordHash.length == 32 && Arrays.equals(HashUtilities.getHash((String)HashUtilities.MD5_ALGORITHM, (char[])password), entry.passwordHash)) {
                return;
            }
            char[] salt = new char[4];
            System.arraycopy(entry.passwordHash, 0, salt, 0, 4);
            if (entry.passwordHash.length == 36) {
                if (!Arrays.equals(HashUtilities.getSaltedHash((String)HashUtilities.MD5_ALGORITHM, (char[])salt, (char[])password), entry.passwordHash)) {
                    throw new FailedLoginException("Incorrect password");
                }
            } else if (entry.passwordHash.length == 68) {
                if (!Arrays.equals(HashUtilities.getSaltedHash((String)HashUtilities.SHA256_ALGORITHM, (char[])salt, (char[])password), entry.passwordHash)) {
                    throw new FailedLoginException("Incorrect password");
                }
            } else {
                throw new FailedLoginException("User password not set, must be reset");
            }
        }
    }

    private PrintWriter getDNLog() throws IOException {
        if (this.dnLogOut == null) {
            File dnLog = new File(this.userFile.getParentFile(), DN_LOG_FILE);
            this.dnLogOut = new PrintWriter(new FileOutputStream(dnLog, true), true);
        }
        return this.dnLogOut;
    }

    public void logUnknownDN(String username, X500Principal principal) {
        try {
            this.getDNLog().println(username + "; " + String.valueOf(principal));
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static boolean isValidUserName(String s) {
        return VALID_USERNAME_REGEX.matcher(s).matches() && s.length() <= 60;
    }

    private static class UserEntry {
        private String username;
        private X500Principal x500User;
        private char[] passwordHash;
        private long passwordTime;

        private UserEntry() {
        }
    }
}

