// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "Game_manager.h"
#include <iostream>
#include <cstdlib>




// *******************
// *** CONSTRUCTOR ***
// *******************
FS::Game_manager::Game_manager() : p_model(&timer),                          
                                   my_settings_screen(&my_settings)
{
    current_state = INTRO;
    p_model.init_sprite_manager( &snd_engine );
    snd_engine.turn_on( my_settings.is_sound() );
    
} // FS::Game_manager::Game_manager

// ******************
// *** DESTRUCTOR ***
// ******************
FS::Game_manager::~Game_manager()
{
    // ...
    
} // FS::Game_manager::~Game_manager




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void FS::Game_manager::start_loop()
{
    SDL_Event event;
    loop_running = true;
    // Stores return val from GetReadyDisplay::redraw_level_select.
    int start_level = 0; 
    bool done;  // used for HI_SCORES case...
    
    // Before we enter the loop, display the title screen
    my_title_screen.draw();

    while (loop_running)
    {
        while ( SDL_PollEvent(&event) )
        {
            if ( current_state == ENTERING_NAME )
            {
                if ( event.type == SDL_KEYDOWN )
                {
                    input_ptr->add_keydown_event(event);
                } // if
            }
            else
            {
                switch (event.type)
                {
                    case SDL_KEYDOWN:
                    case SDL_KEYUP:
                    case SDL_JOYAXISMOTION:
                    case SDL_JOYBUTTONDOWN:
                        input_ptr->add_event( event, my_settings.is_joystick_selected() );
                        break;
                    case SDL_QUIT:
                        loop_running = false;
                        break;        
                } // switch
            } // if ... else
        } // while
        
        switch (current_state)
        {
            case INTRO:
                handle_selection( my_title_screen.update() );
                break;
           
            case SELECTING_START_LEVEL:
                start_level = my_gr_screen.update_level_select();
                if ( start_level )
                {
                    current_state = GET_READY;
                    p_model.new_game(start_level);
                    my_gr_screen.init_countdown( timer.get_time(), start_level );
                } // if
                break;
                     
            case GET_READY:
                if ( my_gr_screen.update_countdown(timer.get_time(), p_model.get_level()) )
                {
                    current_state = PLAY;
                    p_model.new_level(timer.get_time());
                } // if
                break;
                
            case PLAY:
                switch ( p_model.update(timer.get_time()) )
                {
                    case PR_OK:
                        break;
                    case PR_LEVEL_COMPLETED:
                        if (current_mode == MODE_TUTORIAL)
                        {
                            current_state = INTRO;
                            my_title_screen.draw();
                        }
                        else
                        {    
                            current_state = GET_READY;
                            my_gr_screen.init_countdown( timer.get_time(), p_model.get_level() );
                        } // if ... else
                        break;
                    case PR_GAME_OVER:
                        my_settings.set_furthest_level(p_model.get_level());
                        if ( my_hs_screen.check_candidate(p_model.get_score()) )
                        {
                            current_state = ENTERING_NAME;
                            my_hs_screen.init_input(p_model.get_score());
                        }
                        else
                        {
                            current_state = INTRO;
                            my_title_screen.draw();
                        } // if ... else
                        break;
                } // switch
                break;
               
            case SETTINGS:
                if ( my_settings_screen.update() )
                {
                    snd_engine.turn_on( my_settings.is_sound() );
                    current_state = INTRO;
                    my_title_screen.draw();
                } // if        
                break;
                
            case HI_SCORES:
                User_input::Keyboard_Event ke;
                done = false;
                
                // Make sure all events are gobbled up.
                while ( input_ptr->next_event(ke) )
                {
                    if ( ke == User_input::ESCAPE_PRESSED 
                            || ke == User_input::JUMP_KEY_PRESSED )
                    {
                        done = true;
                    } // if
                } // while
                
                if (done)
                {
                    current_state = INTRO;
                    my_title_screen.draw(); 
                } // if
                
                break;

            case ENTERING_NAME:
                if ( my_hs_screen.input() )
                {
                    current_state = HI_SCORES;
                    my_hs_screen.show_high_scores();
                } // if
                break;

        } // switch

        SDL_Delay(10);  // 100 fps
        
    } // while

    my_settings.write_config_file();
    my_hs_screen.write_high_scores();
    
} // FS::Game_manager::start_loop

// **************************************************

void FS::Game_manager::handle_selection(Title_screen::Selection sel)
{
    switch (sel)
    {
        case Title_screen::none:
            break;
    	case Title_screen::play:
            current_mode = MODE_GAME;
            if (my_settings.get_furthest_level() > 1)
            {
                current_state = SELECTING_START_LEVEL;
                my_gr_screen.init_level_select( my_settings.get_furthest_level() );
            }
            else
            {
                current_state = GET_READY;
                p_model.new_game();
                my_gr_screen.init_countdown( timer.get_time(), p_model.get_level() );
            } // if ... else
            break;
        case Title_screen::tutorial:
            current_mode = MODE_TUTORIAL;
            current_state = PLAY;   // FIXME: or GET_READY?
            p_model.new_game(1); // FIXME: in future, select a tutorial level
            p_model.new_level(timer.get_time());
            break;
        case Title_screen::settings:
            current_state = SETTINGS;
            my_settings_screen.reset();
            my_settings_screen.draw();
            break;
        case Title_screen::hi_scores:
            current_state = HI_SCORES;
            my_hs_screen.show_high_scores();
            break;
        case Title_screen::quit:
            loop_running = false;
            break;
    } // switch
    
} // FS::Game_manager::handle_selection

// **************************************************
// **************************************************
// **************************************************
// **************************************************
// **************************************************
