// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "High_scores_display.h"




extern char* fs_dir;




// *******************
// *** CONSTRUCTOR ***
// *******************
FS::High_scores_display::High_scores_display()
{
    read_high_scores();
    
    //font_height = my_gfx->get_font(GfxStore::LARGE_FONT).getHeight();
    
} // FS::High_scores_display::High_scores_display()

// ******************
// *** DESTRUCTOR ***
// ******************
FS::High_scores_display::~High_scores_display()
{
    // nothing
    
} // FS::High_scores_display::~High_scores_display()




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************

void FS::High_scores_display::print_table()
{
    std::vector<High_score_entry>::iterator my_iter;
    
    my_iter = my_entries.begin();
    do
    {
        std::cout << my_iter->name << " ... " << my_iter->score << "\n";
        ++my_iter;
    } while (my_iter != my_entries.end());
            
} // FS::High_scores_display::print_table()

// **************************************************

bool FS::High_scores_display::check_candidate(int sc) const 
{
    std::vector<High_score_entry>::const_reverse_iterator my_iter;
    my_iter = my_entries.rbegin();   // this doesn't like method to be const
    
    do
    {
        if (sc > my_iter->score)
        {
            return true;
        } // if
        ++my_iter;
    } while (my_iter != my_entries.rend());
    
    return false;
    
} // FS::High_scores_display::check_candidate()

// **************************************************

void FS::High_scores_display::add_entry(const char* nm, int sc)
{
    High_score_entry hse;
    
    strncpy(hse.name, nm, NAME_MAX_LEN);
    hse.name[NAME_MAX_LEN] = '\0';
    sprintf(hse.score_str, "%06d", sc);
    hse.score = sc;
    
    std::vector<High_score_entry>::iterator my_iter;
    my_iter = my_entries.begin();
    // FIXME - we're assuming function is being used correctly
    // (i.e. after a call to check_candidate() has returned true).
    do
    {
        if (sc > my_iter->score)
        {
            break;
        } // if
        ++my_iter;
    } while (my_iter != my_entries.end());

    new_entry = my_entries.insert(my_iter, hse);
    my_entries.pop_back();
    
} // FS::High_scores_display::add_entry()
    
// **************************************************

void FS::High_scores_display::read_high_scores()
{
    // load table and populate my_entries list
    std::string my_file = fs_dir;
    my_file += "/table.txt";
    
    std::ifstream fin( my_file.c_str() );
    if ( !fin.is_open() )
    {
        std::cerr << "Can't open " << my_file << " for reading. Using defaults.\n";
        populate_default_table();
        return;
    } // if
    
    std::string my_line;
    std::string temp_name;
    int temp_score;
    int split_index;
    High_score_entry temp_entry;
    
    my_entries.clear(); // ?!
    
    for (int i = 0; i < NUM_ENTRIES; ++i)
    {
        getline(fin, my_line);
    
        split_index = my_line.find('=');
        
        temp_name = my_line.substr(0, split_index);
        // FIXME: no exception-checking.
        temp_score = boost::lexical_cast<int>( my_line.substr(split_index + 1) );
            
        strncpy(temp_entry.name, temp_name.c_str(), NAME_MAX_LEN);
        temp_entry.name[NAME_MAX_LEN] = '\0';
        sprintf(temp_entry.score_str, "%06d", temp_score);
        temp_entry.score = temp_score;
        
        my_entries.push_back(temp_entry);
        
    } // for
    
    fin.close(); 
    
} // FS::High_scores_display::read_high_scores()
   
// **************************************************

void FS::High_scores_display::write_high_scores()
{
    std::string my_file = fs_dir;
    my_file += "/table.txt";
    
    std::ofstream fout( my_file.c_str() );
    if ( !fout.is_open() )
    {
        std::cerr << "Can't open " << my_file << " for writing. " <<
                "Unable to save high scores.\n";
        return;
    } // if
    
    std::vector<High_score_entry>::iterator my_iter;
    my_iter = my_entries.begin();
    
    do
    {
        fout << my_iter->name << "=" << my_iter->score << std::endl;
        ++my_iter;
    } while (my_iter != my_entries.end());
    
    fout.close();
    
} // FS::High_scores_display::write_high_scores()

// **************************************************

void FS::High_scores_display::show_high_scores()
{
    gfx_ptr->draw_background(FS_gfx::CATS_WHISKERS);

    int i = 0;
    int y = TOP_Y;
    do
    {
        gfx_ptr->draw_text( FS_gfx::LARGE_FONT, NAME_X, y, my_entries[i].name );
        gfx_ptr->draw_text( FS_gfx::LARGE_FONT, SCORE_X, y, my_entries[i].score_str );
        ++i;
        y += Y_GAP;
    } while ( i < NUM_ENTRIES );

    SDL_GL_SwapBuffers();

} // FS::High_scores_display::show_high_scores

// **************************************************

void FS::High_scores_display::init_input(int score)
{
    add_entry("anon", score);   
    
    new_entry_name.clear();
    
    std::vector<High_score_entry>::iterator my_iter;
    my_iter = my_entries.begin();
    
    int y = TOP_Y;
    gfx_ptr->draw_background(FS_gfx::CATS_WHISKERS);
    
    do
    {
        if (my_iter == new_entry)
        {
            // Draw score for new entry in high score table.
            gfx_ptr->draw_text( FS_gfx::LARGE_FONT, SCORE_X, y, my_iter->score_str );
            new_entry_y = y;
        }
        else
        {
            gfx_ptr->draw_text( FS_gfx::LARGE_FONT_FAINT, NAME_X, y, my_iter->name );
            gfx_ptr->draw_text( FS_gfx::LARGE_FONT_FAINT, SCORE_X, y, my_iter->score_str );
        } // if ... else
        
        ++my_iter;
        y += Y_GAP;
        
    } while (my_iter != my_entries.end());
    
    // only erase_rect.w ever changes...
    dest_rect.x = NAME_X;
    dest_rect.y = new_entry_y;
    
    erase_rect.x = NAME_X;
    erase_rect.y = new_entry_y;
    erase_rect.h = font_height;
    
    SDL_GL_SwapBuffers();
    
} // FS::High_scores_display::init_input()

// **************************************************

bool FS::High_scores_display::input()
{
    bool must_redraw = false;
    bool must_erase = false;
    SDL_keysym my_keysym;
    
    while ( input_ptr->next_keydown_event(my_keysym) )
    {
        if ( ((my_keysym.sym >= SDLK_a) && (my_keysym.sym <= SDLK_z)) 
                || ((my_keysym.sym >= SDLK_0) && (my_keysym.sym <= SDLK_9)) )
        {
            // a letter/number was typed... is there still room?
            // FIXME: C-style casts!
            if ( new_entry_name.length() < ((unsigned int) NAME_MAX_LEN) )
            {
                if ( my_keysym.mod & KMOD_SHIFT )
                {
                    new_entry_name += toupper((char) my_keysym.sym);
                }
                else
                {
                    new_entry_name += ((char) my_keysym.sym);
                } // if ... else
                must_redraw = true;
            }
            else
            {
                // beep!
            } // if ... else
        } 
        else if ( my_keysym.sym == SDLK_BACKSPACE )
        {
            if ( new_entry_name.length() > 0 )
            {
                /*erase_rect.w = my_gfx->get_font(GfxStore::LARGE_FONT).getTextWidth(
                        new_entry_name.c_str());*/
                new_entry_name.erase(--new_entry_name.end());
                must_erase = true;
                must_redraw = true;
            }
            else
            {
                // beep!
            } // if ... else
        }
        else if ( my_keysym.sym == SDLK_RETURN )
        {
            // if string is empty, we'll stick with default name (= "anon")
            // otherwise copy new name 
            if ( !new_entry_name.empty() )
            {
                // there will always be enough room because of NAME_MAX_LEN limit
                strcpy(new_entry->name, new_entry_name.c_str());
            } // if
            // FIXME: empty 'keydown' list!
            return true;
        } // if ... else
    } // while
    
    if ( must_erase || must_redraw )
    {
        std::vector<High_score_entry>::iterator my_iter;
        my_iter = my_entries.begin();
    
        int y = TOP_Y;
        gfx_ptr->draw_background(FS_gfx::CATS_WHISKERS);
    
        do
        {
            if (my_iter == new_entry)
            {
                // Draw name (so far)  and score for new entry in high score table.
                gfx_ptr->draw_text( FS_gfx::LARGE_FONT, NAME_X, y, new_entry_name );
                gfx_ptr->draw_text( FS_gfx::LARGE_FONT, SCORE_X, y, my_iter->score_str );
            }
            else
            {
                gfx_ptr->draw_text( FS_gfx::LARGE_FONT_FAINT, NAME_X, y, my_iter->name );
                gfx_ptr->draw_text( FS_gfx::LARGE_FONT_FAINT, SCORE_X, y, my_iter->score_str );
            } // if ... else
        
            ++my_iter;
            y += Y_GAP;
        
        } while (my_iter != my_entries.end());
        SDL_GL_SwapBuffers();
    } // if
    
    return false;
    
} // FS::High_scores_display::input()

// **************************************************

void FS::High_scores_display::populate_default_table()
{
    High_score_entry temp_entry;
    
    //char * names[] = { "Bertie", "Agatha", "Oofy", "Pauline", "Spode" };
    std::string names[] = { "Bertie", "Agatha", "Oofy", "Pauline", "Spode" };
    int scores[] = { 4000, 3000, 2000, 1000, 500 };
    
    my_entries.clear(); // ?!
    
    for (int i = 0; i < NUM_ENTRIES; ++i)
    {
        strncpy(temp_entry.name, names[i].c_str(), NAME_MAX_LEN);
        temp_entry.name[NAME_MAX_LEN] = '\0';
        sprintf(temp_entry.score_str, "%06d", scores[i]);
        temp_entry.score = scores[i];
        
        my_entries.push_back(temp_entry);
    } // for    
    
} // FS::High_scores_display::populate_default_table()

// **************************************************
// **************************************************
 
 
    
