// $Id$

// Fish Supper
// Copyright 2006, 2007, 2009, 2010 Matthew Clarke <mafferyew@googlemail.com>
//
// This file is part of Fish Supper.
//
// Fish Supper is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Fish Supper is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Fish Supper.  If not, see <http://www.gnu.org/licenses/>.




#include "Sprite_manager.h"




// *******************
// *** CONSTRUCTOR ***
// *******************
FS::Sprite_manager::Sprite_manager() :
        alberts_row(LOWER_BANK)
{
    // FIXME - make these into constants?
    int temp_min_ys[] = { 87, 162, 237, 312, 387, 462 };
    int temp_max_ys[] = { 136, 211, 286, 361, 436, 511 };
     
    for (int i = 0; i < NUM_ROWS; ++i)
    {
        rows_min_ys[i] = temp_min_ys[i];
        rows_max_ys[i] = temp_max_ys[i];
    } // for
    
} // FS::Sprite_manager::Sprite_manager

// ******************
// *** DESTRUCTOR ***
// ******************
FS::Sprite_manager::~Sprite_manager() 
{
    delete_logs_and_collectables();
            
} // FS::Sprite_manager::~Sprite_manager 




// ************************
// *** MEMBER FUNCTIONS ***
// ************************

// **************************************************
 
void FS::Sprite_manager::reset()
{
    the_cat.reset();
    alberts_row = LOWER_BANK;
    the_fish.reset();
    is_first_pass = true;
    cat_crystal_collisions_disabled = false;
    disabled_crystal = 0;
    
    current_state = PLAYING_GAME;
    
    sound_engine_ptr->play_sound(Sound_engine::RIVER, -1, 30);
    
    redraw_crystals = redraw_fish = redraw_score = redraw_level = redraw_lives = true;
    pause_menu_needs_drawing = pause_menu_needs_erasing = false;
    
} // FS::Sprite_manager::reset
         
// **************************************************

void FS::Sprite_manager::restart_level()
{
    the_cat.reset();
    alberts_row = LOWER_BANK;
    
    // These two shouldn't need resetting, but won't do any
    // harm to include these lines anyway...
    cat_crystal_collisions_disabled = false;
    disabled_crystal = 0;
    
    logs_iter = my_logs.begin();
    while ( logs_iter != my_logs.end() )
    {
        (*logs_iter)->reset();
        ++logs_iter;
    } // while
    
    collectables_iter = my_collectables.begin();
    while ( collectables_iter != my_collectables.end() )
    {
        (*collectables_iter)->reset();
        ++collectables_iter;
    } // while
    
    redraw_crystals = true;   // forces crystals held to be re-drawn
    
    current_state = PLAYING_GAME;
    
} // FS::Sprite_manager::restart_level

// **************************************************

FS::Sprite_manager::State 
        FS::Sprite_manager::update( int t, const ScrollingData& sd, PauseType pause_t )
{
    gfx_ptr->draw_background(FS_gfx::BACKGROUND);

    // Record time in member variable for convenience 
    // - it's needed by several functions.
    time_passed = t;    
    
    // Update and draw logs and collectables.
    std::for_each( my_logs.begin(), my_logs.end(), boost::bind(&Log::update, _1, sd) );
    std::for_each( my_collectables.begin(), my_collectables.end(), 
            boost::bind(&Collectable_sprite::update, _1, sd) );

    bool is_running = (pause_t == PAUSE_NONE);
    the_cat.update(time_passed, is_running);
    
    switch (current_state)
    {
        case PLAYING_GAME:
            // If Albert is airborne or in water, we don't need
            // to check for any collisions.
            if ( the_cat.is_airborne() )
            {
                // FIXME: only need to reset these once.
                cat_crystal_collisions_disabled = false;
                disabled_crystal = 0;
                alberts_row = ROW_NONE;
                break;
            }
            else if ( the_cat.is_in_water_or_being_zapped() )
            {
                alberts_row = ROW_NONE;
                break;  // don't want collisions checking! 
            }
            else if ( the_cat.is_splashed_zapped_out() )
            {
                alberts_row = ROW_NONE;
                player_data_ptr->dec_num_lives();
                redraw_lives = true;
                if ( player_data_ptr->get_num_lives() == 0 )
                {
                    the_cat.start_leave_in_a_huff(time_passed);
                    current_state = RUNNING_GAME_OVER_ROUTINE;
                    sound_engine_ptr->play_sound(Sound_engine::MIAOW);
                    break;
                } 
                else
                {
                    current_state = REQUEST_LEVEL_RESTART;
                    break;
                } // if ... else
            } // if ... else 
            
            check_collisions();
            
            break;
        
        // FIXME: condense following 2 case statements into 1!
        case RUNNING_LEVEL_COMPLETED_ROUTINE:
            if ( the_cat.has_left() )
            {
                current_state = LEVEL_COMPLETED;
                sound_engine_ptr->stop_all_sounds();
            } // if
            break;
            
        case RUNNING_GAME_OVER_ROUTINE:
            if ( the_cat.has_left() )
            {
                current_state = GAME_OVER;
                sound_engine_ptr->stop_all_sounds();
            } // if
            break;
        
        case REQUEST_LEVEL_RESTART:    
        case LEVEL_COMPLETED:
        case GAME_OVER:
            break;
            
    } // switch
    
    // FIXME: why not make this one function call?
    player_data_ptr->draw_score();
    player_data_ptr->draw_lives();
    player_data_ptr->draw_level();
    the_fish.draw();
    player_data_ptr->draw_crystals_held();
    // Draw cat last of all.
    the_cat.schedule_draw();

    if ( pause_menu_needs_drawing )
    {
        pause_menu_ptr->draw();
    } // if
/*    
    redraw_crystals = redraw_fish = redraw_score = redraw_level = redraw_lives = false;
    
    if ( pause_menu_needs_drawing )
    {
        pause_menu_ptr->draw();
    }
    else if ( pause_menu_needs_erasing )
    {
        pause_menu_needs_erasing = false;
        pause_menu_ptr->erase();
    } // if ... else
        
    //redraw_screen();
*/    
    return current_state;
    //return PLAYING_GAME;
    
} // FS::Sprite_manager::update
    
// **************************************************

void FS::Sprite_manager::check_collisions()
{ 
    if ( the_cat.is_on_log() )
    {
        check_cat_collectable_collision( the_cat.get_host_log_row() );
    }
    else
    {
        RowDescription row_desc = calc_alberts_row();
        
        switch (row_desc)
        {
            case ROW_0:
            case ROW_1:
            case ROW_2:
            case ROW_3:
            case ROW_4:
            case ROW_5: 
                // check logs on given row
                // N.B. Check for crystals first and Albert would be able
                // to grab a crystal just before he landed on a log - might be
                // the crystal that'd keep him on that log!
                switch ( check_cat_log_collision((int) row_desc) )
                {
                    case ALLOWED:
                        alberts_row = row_desc;
                        break;
                    case DISALLOWED:
                        alberts_row = ROW_NONE;
                        the_cat.start_zap(time_passed);
                        break;
                    case MISSED:
                        alberts_row = ROW_NONE;
                        the_cat.start_splash(time_passed);
                        sound_engine_ptr->play_sound(Sound_engine::SPLASH);
                        break;
                } // switch
                break;
                
            case UPPER_BANK:
                alberts_row = row_desc;
                // check for collision with fish
                if ( Collisions::intersects(the_cat.get_coll_box(), the_fish.get_coll_box()) )
                {
                    player_data_ptr->inc_score(PTS_FISH);
                    redraw_score = true;
                    the_cat.start_leave_with_fish(time_passed);
                    the_fish.remove_fish();
                    redraw_fish = true;
                    current_state = RUNNING_LEVEL_COMPLETED_ROUTINE;
                } // if
                break;
                
            case LOWER_BANK:
                alberts_row = row_desc;
                break;
                
            case ROW_NONE:
                alberts_row = row_desc;
                // SPLASH!!!
                the_cat.start_splash(time_passed);
                sound_engine_ptr->play_sound(Sound_engine::SPLASH);
                break;
                
        } // switch
        
    } // if ... else
      
} // FS::Sprite_manager::check_collisions 
        
// **************************************************

FS::RowDescription FS::Sprite_manager::calc_alberts_row()
{
    if ( the_cat.upper_y() >= LOWER_BANK_MIN_Y )
    {
        return LOWER_BANK;
    }
    else if ( the_cat.lower_y() <= UPPER_BANK_MAX_Y )
    {
        return UPPER_BANK;
    }
    else
    {
        for (int i = 0; i < NUM_ROWS; ++i)
        {
            if ( (the_cat.upper_y() >= rows_min_ys[i]) &&
                    (the_cat.lower_y() <= rows_max_ys[i]) )
            {
                return (RowDescription) i;
            } // if
        } // for
    } // if ... else
    
    return ROW_NONE;
    
} // FS::Sprite_manager::calc_alberts_row
            
// **************************************************

FS::Sprite_manager::CatLogCollisionType FS::Sprite_manager::check_cat_log_collision(int row)
{
    logs_iter = my_logs.begin();
    while ( logs_iter != my_logs.end() )
    {
        if ( ((*logs_iter)->get_row() == row) && 
                Collisions::contains((*logs_iter)->get_coll_box(), the_cat.get_coll_box()) )
        {
            the_cat.set_new_host_log(*logs_iter);
            
            if ( player_data_ptr->is_allowed_on_log((*logs_iter)->get_color()/*, 
                    my_play_display*/) )
            {
                switch ( (*logs_iter)->get_color() )
                {
                    case L_RED:
                    case L_YELLOW:
                    case L_BLUE:
                        player_data_ptr->inc_score(PTS_ONE_COLOR_LOG);
                        redraw_crystals = true;
                        break;
                    case L_ORANGE:
                    case L_GREEN:
                    case L_PURPLE:
                        player_data_ptr->inc_score(PTS_TWO_COLOR_LOG);
                        redraw_crystals = true;
                        break;
                    case L_BROWN:
                        player_data_ptr->inc_score(PTS_BROWN_LOG);
                        break;
                    case L_BACKGROUND:  // should never happen
                        break;
                } // switch
                redraw_score = true;
                return ALLOWED;
            }
            else
            {
                return DISALLOWED;
            } // if ... else
        } // if
        ++logs_iter;
    } // for
    
    return MISSED;   // i.e. SPLASH!!!
            
} // FS::Sprite_manager::check_cat_log_collision

// **************************************************

void FS::Sprite_manager::check_cat_collectable_collision(int row)
{
    if (cat_crystal_collisions_disabled)
    {
        // Is Albert still 'over' the disabled crystal?
        if ( Collisions::intersects(the_cat.get_coll_box(), disabled_crystal->get_coll_box()) )
        {
            return;
        }
        else
        {
            cat_crystal_collisions_disabled = false;
            disabled_crystal = 0;
        } // if ... else
    } // if
    
    collectables_iter = my_collectables.begin();
    while ( collectables_iter != my_collectables.end() )
    {
        if ( (*collectables_iter)->is_active() )
        {
            if ( Collisions::intersects(the_cat.get_coll_box(), 
                    (*collectables_iter)->get_coll_box()) )
            {
                // is this a crystal or a bonus?
                switch ( (*collectables_iter)->get_id() )
                {
                    case Collectable_sprite::CRYSTAL:
                        handle_crystal_collision( 
                                dynamic_cast<Crystal *> (*collectables_iter) );
                        break;
                    case Collectable_sprite::BONUS:
                        handle_bonus_collision( dynamic_cast<Bonus *> (*collectables_iter) );
                        break;
                } // switch
                break;  // only need to find one collision
            } // if
        } // if
        ++collectables_iter;
    } // while
    
} // FS::Sprite_manager::check_cat_collectable_collision
 
// **************************************************
        
void FS::Sprite_manager::create_scrolling_sprites()
{
    delete_logs_and_collectables();
    create_logs();
    create_collectables();
    
} // FS::Sprite_manager::create_scrolling_sprites

// **************************************************
 
void FS::Sprite_manager::create_logs( /*const Cell * const * my_cells*/ )
{
    int start_index;
    int end_index;
    LogColor color = L_BACKGROUND;
    
    for ( int row = 0; row < NUM_ROWS; ++row )
    {
        // reset indices
        start_index = end_index = -1;
        const Scroller::Cell* my_cells = scroller_ptr->get_row_of_cells(row);
        
        for ( int col = 0; col < MAX_COLS; ++col )
        {
            if ( my_cells[col].lc != L_BACKGROUND )
            {
                if ( start_index < 0 )
                {
                    start_index = col;
                    color = my_cells[col].lc;
                } // if
            }
            else if (start_index >= 0)  // AND current block is background
            {
                end_index = col;
                
                my_logs.push_back(
                        new Log( start_index * CELL_WIDTH, rows_min_ys[row],
                                (end_index - start_index) * CELL_WIDTH, row, color ) );
                
                // Get ready for another log on same row.
                start_index = end_index = -1;
            } // if ... else
        } // for col
    } // for row
    
} // FS::Sprite_manager::create_logs

// **************************************************

void FS::Sprite_manager::create_collectables( /*const Cell * const * my_cells*/ )
{
    for ( int row = 0; row < NUM_ROWS; ++row )
    {
        const Scroller::Cell* my_cells = scroller_ptr->get_row_of_cells(row);
        for ( int col = 0; col < MAX_COLS; ++col )
        {
            // If there's a crystal here there won't be a bonus.
            if ( my_cells[col].cc != Crystal::color_none_c )   
            {
                my_collectables.push_back(
                        new Crystal( col * CELL_WIDTH, rows_min_ys[row], row,
                                my_cells[col].cc ) );
            } 
            else if ( my_cells[col].bt != B_NONE )
            {
                my_collectables.push_back(
                        new Bonus( col * CELL_WIDTH, rows_min_ys[row], row, 
                                my_cells[col].bt ) );
            } // if ... else    
        } // for col
    } // for row
    
} // FS::Sprite_manager::create_collectables

// **************************************************

void FS::Sprite_manager::handle_crystal_collision(Crystal* c)
{
    // update player data - i.e. crystals held
    Crystal::Color cc = player_data_ptr->add_crystal(c->get_color()/*, my_play_display*/);
    redraw_crystals = true;
    sound_engine_ptr->play_sound(Sound_engine::PING);
    if (cc == Crystal::color_none_c)
    {
        c->deactivate();
    }
    else
    {
        // Have to stop Albert picking up this crystal 
        // again straight afterwards...
        c->set_color(cc);
        cat_crystal_collisions_disabled = true;
        disabled_crystal = c;
    } // if ... else 
    
} // FS::Sprite_manager::handle_crystal_collision() 

// **************************************************

void FS::Sprite_manager::handle_bonus_collision(Bonus* b)
{
    // What type of bonus are we dealing with?
    switch ( b->get_bonus_type() )
    {
        case B_DOUBLE_JUMP:
            b->deactivate();
            // tell Albert class that next jump's gonna be a 'double'
            the_cat.set_next_jump_double();
            break;
        case B_LOGS_TO_BROWN:
            b->deactivate();
            set_logs_to_brown( the_cat.get_host_log_row() );
            break;
        case B_CHANGE_DIRECTION:
            b->deactivate();
            scroller_ptr->change_direction( the_cat.get_host_log_row() );
            break;
        case B_SLOW_DOWN:
            b->deactivate();
            scroller_ptr->halve_velocity( the_cat.get_host_log_row() );
            break;
        case B_SPEED_UP:
            b->deactivate();
            scroller_ptr->double_velocity( the_cat.get_host_log_row() );
            break;
        case B_NONE:
            // should never happen - keeps compiler happy
            break;
    } // switch
    
} // FS::Sprite_manager::handle_bonus_collision

// **************************************************

void FS::Sprite_manager::set_logs_to_brown(int row)
{
    logs_iter = my_logs.begin();
    while ( logs_iter != my_logs.end() )
    {
        if ( (*logs_iter)->get_row() == row )
        {
            (*logs_iter)->set_color(L_BROWN);
        } // if
        ++logs_iter;
    } // while
    
} // FS::Sprite_manager::set_logs_to_brown
        
// **************************************************

void FS::Sprite_manager::delete_logs_and_collectables()
{
    if ( !my_logs.empty() )
    {
        logs_iter = my_logs.begin();
        while ( logs_iter != my_logs.end() )
        { 
            delete (* logs_iter);
            ++logs_iter;
        } // while
    } // if
    // Gets rid of pointers after pointed-to memory has been freed.
    my_logs.clear();    
    
    if ( !my_collectables.empty() )
    {
        collectables_iter = my_collectables.begin();
        while ( collectables_iter != my_collectables.end() )
        {
            delete (* collectables_iter);
            ++collectables_iter;
        } // while
    } // if
    my_collectables.clear();
    
} // FS::Sprite_manager::delete_logs_and_collectables
        
// ************************************************** 

void FS::Sprite_manager::deactivate_pause_menu()
{
    pause_menu_needs_drawing = false;
    pause_menu_needs_erasing = true;
    sound_engine_ptr->resume_all_sounds();
    
} // FS::Sprite_manager::deactivate_pause_menu

// **************************************************

void FS::Sprite_manager::activate_pause_menu() 
{ 
    pause_menu_needs_drawing = true; 
    sound_engine_ptr->pause_all_sounds();

} // FS::Sprite_manager::activate_pause_menu

// **************************************************

void FS::Sprite_manager::redraw_screen()
{
    /*
    if ( is_first_pass )
    {
        my_blitter->blit_and_update( GfxStore::BACKGROUND );
        is_first_pass = false;
    }
    else
    {
        my_blitter->blit_and_update();
    } // if ... else
    */
    SDL_GL_SwapBuffers();

} // FS::Sprite_manager::redraw_screen
    
// **************************************************

const FS::LogColor FS::Sprite_manager::get_host_log_color() const
{
    return the_cat.get_host_log()->get_color();
    
} // FS::Sprite_manager::get_host_log_color

// **************************************************
     
