<?php
declare( strict_types = 1 );

namespace Wikimedia\Parsoid\NodeData;

use Psr\Container\ContainerInterface;
use Wikimedia\JsonCodec\Hint;
use Wikimedia\JsonCodec\JsonClassCodec;
use Wikimedia\JsonCodec\JsonCodecable;
use Wikimedia\JsonCodec\JsonCodecInterface;
use Wikimedia\Parsoid\DOM\DocumentFragment;
use Wikimedia\Parsoid\Tokens\SourceRange;
use Wikimedia\Parsoid\Tokens\Token;
use Wikimedia\Parsoid\Utils\DOMDataUtils;
use Wikimedia\Parsoid\Utils\Utils;

/**
 * Editing data for a DOM node.  Managed by DOMDataUtils::get/setDataMw().
 *
 * To reduce memory usage, most of the properties need to be dynamic, but
 * we use the property declarations below to allow type checking.
 *
 * @property list<string|TemplateInfo> $parts
 * @property string $name
 * @property string $extPrefix
 * @property string $extSuffix
 * @property list<DataMwAttrib> $attribs Complex attributes of an HTML tag
 * @property string $src
 * @property DocumentFragment $caption
 * @property string $thumb
 * @property bool $autoGenerated
 * @property list<DataMwError> $errors
 * @property DataMwBody $body
 * @property DocumentFragment $html
 * @property float $scale
 * @property string $starttime
 * @property string $endtime
 * @property string $thumbtime
 * @property string $page
 * == Annotations ==
 * @property string $rangeId
 * @property SourceRange $wtOffsets
 * @property bool $extendedRange
 * @property DataMwExtAttribs $extAttribs Attributes for an extension tag or annotation
 */
#[\AllowDynamicProperties]
class DataMw implements JsonCodecable {

	public function __construct( array $initialVals = [] ) {
		foreach ( $initialVals as $k => $v ) {
			switch ( $k ) {
				case 'attrs':
					// T367616: facilitate renaming.
					if ( $v instanceof DataMwExtAttribs ) {
						$this->extAttribs = $v;
					} else {
						$this->extAttribs = new DataMwExtAttribs( $v );
					}
					break;
				// Add cases here for components which should be instantiated
				// as proper classes.
				default:
					$this->$k = $v;
					break;
			}
		}
	}

	/** Returns true iff there are no dynamic properties of this object. */
	public function isEmpty(): bool {
		$result = (array)$this;
		return $result === [];
	}

	/**
	 * Helper method to facilitate renaming the 'attrs' property to
	 * 'extAttribs' (T367616).
	 * @note that numeric key values will be converted from string
	 *   to int by PHP when they are used as array keys
	 * @return ?array<string|int,string|array<Token|string>>
	 */
	public function getExtAttribs(): ?array {
		if ( isset( $this->extAttribs ) ) {
			return $this->extAttribs->getValues();
		}
		return null;
	}

	/**
	 * Helper method to facilitate renaming the 'attrs' property to
	 * 'extAttribs' (T367616).
	 * @param string $name
	 * @return string|array<Token|string>|null
	 */
	public function getExtAttrib( string $name ) {
		if ( isset( $this->extAttribs ) ) {
			return $this->extAttribs->get( $name );
		}
		return null;
	}

	/**
	 * Helper method to facilitate renaming the 'attrs' property to
	 * 'extAttribs' (T367616).
	 * @param string $name
	 * @param string|array<Token|string>|null $value
	 *  Setting to null will unset it from the array.
	 */
	public function setExtAttrib( string $name, $value ): void {
		if ( !isset( $this->extAttribs ) ) {
			$this->extAttribs = new DataMwExtAttribs;
		}
		$this->extAttribs->set( $name, $value );
	}

	public function __clone() {
		// Deep clone non-primitive properties

		// 1. Properties which are lists of cloneable objects
		foreach ( [ 'parts', 'attribs', 'errors' ] as $prop ) {
			if ( isset( $this->$prop ) ) {
				$this->$prop = Utils::cloneArray( $this->$prop );
			}
		}
		// 2. Properties which are cloneable objects
		foreach ( [ 'body', 'wtOffsets', 'extAttribs' ] as $prop ) {
			if ( isset( $this->$prop ) ) {
				$this->$prop = clone $this->$prop;
			}
		}
		// 3. Properties which are DocumentFragments
		foreach ( [ 'caption', 'html' ] as $field ) {
			if ( isset( $this->$field ) ) {
				$this->$field = DOMDataUtils::cloneDocumentFragment( $this->$field );
			}
		}
	}

	/** @inheritDoc */
	public static function jsonClassHintFor( string $keyname ) {
		static $hints = null;
		if ( $hints === null ) {
			$hints = [
				'attribs' => Hint::build( DataMwAttrib::class, Hint::USE_SQUARE, Hint::LIST ),
				// T367616: 'attrs' should be renamed to 'extAttribs' in
				// a future revision of the MediaWiki DOM Spec
				'attrs' => Hint::build( DataMwExtAttribs::class, Hint::ALLOW_OBJECT ),
				'body' => Hint::build( DataMwBody::class, Hint::ALLOW_OBJECT ),
				'wtOffsets' => Hint::build( SourceRange::class, Hint::USE_SQUARE ),
				'parts' => Hint::build( TemplateInfo::class, Hint::STDCLASS, Hint::LIST ),
				'errors' => Hint::build( DataMwError::class, Hint::LIST ),
				// 'caption' and 'html' are not hinted as DocumentFragment because
				// we manually encode/decode them for MW Dom Spec 2.8.0 compat.
			];
		}
		return $hints[$keyname] ?? null;
	}

	/** @inheritDoc */
	public function toJsonArray( JsonCodecInterface $codec ): array {
		$result = (array)$this;
		$order = array_flip( array_keys( $result ) );
		// T367616: 'attrs' should be renamed to 'extAttribs' in
		// a future revision of the MediaWiki DOM Spec
		if ( isset( $result['extAttribs'] ) ) {
			$order['attrs'] = $order['extAttribs'];
			$result['attrs'] = $result['extAttribs'];
			unset( $result['extAttribs'] );
		}
		// T367141: Third party clients (eg Cite) create arrays instead of
		// error objects.  We should convert them to proper DataMwError
		// objects once those exist.
		if ( isset( $result['errors'] ) ) {
			$result['errors'] = array_map(
				static fn ( $e ) => is_array( $e ) ? DataMwError::newFromJsonArray( $e ) :
					( $e instanceof DataMwError ? $e : DataMwError::newFromJsonArray( (array)$e ) ),
				$result['errors']
			);
		}
		// Legacy encoding of parts.
		if ( isset( $result['parts'] ) ) {
			$result['parts'] = array_map( static function ( $p ) {
				if ( $p instanceof TemplateInfo ) {
					$type = $p->type ?? 'template';
					if ( $type === 'old-parserfunction' ) {
						$type = 'template';
					}
					$pp = (object)[];
					$pp->$type = $p;
					return $pp;
				}
				return $p;
			}, $result['parts'] );
		}
		// caption/html compatibility with MediaWiki DOM Spec 2.8.0
		// See [[mw:Parsoid/MediaWiki DOM spec/Rich Attributes]] Phase 3
		// for discussion about alternate _h/_t marking for DocumentFragments
		foreach ( [ 'caption', 'html' ] as $field ) {
			if ( isset( $result[$field] ) ) {
				$c = $codec->toJsonArray( $result[$field], DocumentFragment::class );
				if ( is_string( $c['_h'] ?? null ) ) {
					$result[$field] = $c['_h'];
				} else {
					$result[$field] = $c;
				}
			}
		}
		uksort( $result, static fn ( $a, $b )=>( $order[$a] ?? -1 ) - ( $order[$b] ?? -1 ) );
		return $result;
	}

	/** @inheritDoc */
	public static function newFromJsonArray( JsonCodecInterface $codec, array $json ): DataMw {
		$order = array_flip( array_keys( $json ) );
		// Decode legacy encoding of parts.
		if ( isset( $json['parts'] ) ) {
			$json['parts'] = array_map( static function ( $p ) {
				if ( is_object( $p ) ) {
					if ( isset( $p->templatearg ) ) {
						$type = 'templatearg';
					} elseif ( isset( $p->parserfunction ) ) {
						$type = 'parserfunction';
					} else {
						$type = 'template';
					}
					$p = $p->$type;
					/** @var TemplateInfo $p */
					if ( $type === 'template' && isset( $p->func ) ) {
						$type = 'old-parserfunction';
					}
					$p->type = $type;
				}
				return $p;
			}, $json['parts'] );
		}
		// Usually '_h' or '_t' is used as a marker for caption/html, but
		// allow a bare string as well.
		foreach ( [ 'caption', 'html' ] as $field ) {
			$c = $json[$field] ?? null;
			$c = is_string( $c ) ? [ '_h' => $c ] : $c;
			if ( $c !== null ) {
				$json[$field] =
					$codec->newFromJsonArray( $c, DocumentFragment::class );
			}
		}
		// T367616: 'attrs' should be renamed to 'extAttribs' in
		// a future revision of the MediaWiki DOM Spec
		if ( isset( $json['attrs'] ) ) {
			$order['extAttribs'] = $order['attrs'];
			$json['extAttribs'] = $json['attrs'];
			unset( $json['attrs'] );
		}
		uksort( $json, static fn ( $a, $b )=>( $order[$a] ?? -1 ) - ( $order[$b] ?? -1 ) );
		return new DataMw( $json );
	}

	/**
	 * Custom JsonClassCodec for DataMw.
	 *
	 * Because the 'caption' and 'html' fields have embedded DocumentFragments
	 * that /don't/ use the standard encoding, we need to use a custom
	 * class codec which allows us to manually encode
	 * the DocumentFragment (by passing the codec itself to the
	 * serialization/deserialization methods).
	 */
	public static function jsonClassCodec(
		JsonCodecInterface $codec, ContainerInterface $serviceContainer
	): JsonClassCodec {
		return new class( $codec ) implements JsonClassCodec {
			private JsonCodecInterface $codec;

			public function __construct( JsonCodecInterface $codec ) {
				$this->codec = $codec;
			}

			/** @inheritDoc */
			public function toJsonArray( $obj ): array {
				return $obj->toJsonArray( $this->codec );
			}

			/** @inheritDoc */
			public function newFromJsonArray( string $className, array $json ) {
				return $className::newFromJsonArray( $this->codec, $json );
			}

			/** @inheritDoc */
			public function jsonClassHintFor( string $className, string $keyName ) {
				return $className::jsonClassHintFor( $keyName );
			}
		};
	}
}
