/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package errs

import (
	"errors"
	"testing"

	"github.com/google/go-cmp/cmp"
)

type sampleErrorT struct {
	data string
}

func (e *sampleErrorT) Error() string {
	return e.data
}

func TestNew(t *testing.T) {
	t.Parallel()

	sampleErr := New("fail")

	tests := []struct {
		name     string
		caseFunc func(t *testing.T)
	}{
		{
			"+valid.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Fail.",
					New("fail").Error(),
				); diff != "" {
					t.Fatalf("New().Error() missmatch (-want +got):\n%s", diff)
				}
			},
		},
		{
			"+wrapped.errors.Is()",
			func(t *testing.T) {
				if !errors.Is(
					Wrap(sampleErr, "message"),
					sampleErr,
				) {
					t.Fatalf("New() errors.Is didn't match underlying error")
				}
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			tt.caseFunc(t)
		})
	}
}

func TestErrorf(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name     string
		caseFunc func(t *testing.T)
	}{
		{
			"+valid.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Fail died of old age 69.",
					Errorf("fail %s %d", "died of old age", 69).Error(),
				); diff != "" {
					t.Fatalf(
						"Errorf().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		// no need to test more cases as Errorf wraps New
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			tt.caseFunc(t)
		})
	}
}

func TestWrap(t *testing.T) {
	t.Parallel()

	sampleErr := errors.New("fail")
	sampleErrT := &sampleErrorT{"fail"}

	tests := []struct {
		name     string
		caseFunc func(t *testing.T)
	}{
		{
			"+valid.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Message: fail.",
					Wrap(sampleErr, "message").Error(),
				); diff != "" {
					t.Fatalf("Wrap().Error() missmatch (-want +got):\n%s", diff)
				}
			},
		},
		{
			"+doubleWrap.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Message2: message1: fail.",
					Wrap(Wrap(sampleErr, "message1"), "message2").Error(),
				); diff != "" {
					t.Fatalf("Wrap().Error() missmatch (-want +got):\n%s", diff)
				}
			},
		},
		{
			"+tripleWrap.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Message3: message2: message1: fail.",
					Wrap(
						Wrap(
							Wrap(sampleErr, "message1"),
							"message2",
						),
						"message3",
					).Error(),
				); diff != "" {
					t.Fatalf("Wrap().Error() missmatch (-want +got):\n%s", diff)
				}
			},
		},
		{
			"+errors.Is",
			func(t *testing.T) {
				if !errors.Is(
					Wrap(sampleErr, "message"),
					sampleErr,
				) {
					t.Fatalf("Wrap() errors.Is didn't match underlying error")
				}
			},
		},
		{
			"+doubleWrapErrors.Is",
			func(t *testing.T) {
				if !errors.Is(
					Wrap(Wrap(sampleErr, "message1"), "message2"),
					sampleErr,
				) {
					t.Fatalf("Wrap() errors.Is didn't match underlying error")
				}
			},
		},
		{
			"+errors.As",
			func(t *testing.T) {
				dest := &sampleErrorT{}

				if !errors.As(
					Wrap(sampleErrT, "message"),
					&dest,
				) {
					t.Fatalf("Wrap() errors.As didn't match underlying error")
				}

				if diff := cmp.Diff(
					&sampleErrorT{"fail"},
					dest,
					cmp.AllowUnexported(sampleErrorT{}),
				); diff != "" {
					t.Fatalf(
						"Wrap() errors.As result didn't match underlying "+
							"error (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"+doubleWrapErrors.As",
			func(t *testing.T) {
				dest := &sampleErrorT{}

				if !errors.As(
					Wrap(Wrap(sampleErrT, "message1"), "message2"),
					&dest,
				) {
					t.Fatalf("Wrap() errors.As didn't match underlying error")
				}

				if diff := cmp.Diff(
					&sampleErrorT{"fail"},
					dest,
					cmp.AllowUnexported(sampleErrorT{}),
				); diff != "" {
					t.Fatalf(
						"Wrap() errors.As result didn't match underlying "+
							"error (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"-nilErr.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Message2: nil error.",
					Wrap(nil, "message2").Error(),
				); diff != "" {
					t.Fatalf("Wrap().Error() missmatch (-want +got):\n%s", diff)
				}
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			tt.caseFunc(t)
		})
	}
}

func TestWrapf(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name     string
		caseFunc func(t *testing.T)
	}{
		{
			"+valid.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Message foo: fail.",
					Wrapf(errors.New("fail"), "message %s", "foo").Error(),
				); diff != "" {
					t.Fatalf(
						"Wrapf().Error() missmatch (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		// Wrapf uses Wrap underlying, so it's not necessary to test all cases.
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			tt.caseFunc(t)
		})
	}
}

func TestWrapConst(t *testing.T) {
	t.Parallel()

	sampleErr := errors.New("fail")
	sampleConstErr := errors.New("constant err")
	sampleMessageWrappedErr := Wrap(
		errors.New("fail message wrapped"), "message",
	)
	sampleConstWrappedErr := WrapConst(
		errors.New("ups"), errors.New("constant"),
	)
	sampleErrT := &sampleErrorT{"fail"}

	tests := []struct {
		name     string
		caseFunc func(t *testing.T)
	}{
		{
			"+valid.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Constant err: fail.",
					WrapConst(sampleErr, sampleConstErr).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		{
			"+withMessageWrapped.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Constant err: message: fail message wrapped.",
					WrapConst(sampleMessageWrappedErr, sampleConstErr).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		{
			"+doubleWrap.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Constant err: constant 1: fail.",
					WrapConst(
						WrapConst(sampleErr, errors.New("constant 1")),
						sampleConstErr,
					).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		{
			"+crazyWrapping.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Message 6: fail: message 5: fail: message 4: fail: "+
						"message 3: fail: message 2: fail: message 1: fail.",
					WrapConst(
						WrapConst(
							WrapConst(
								Wrap(sampleErr, "message 1"),
								Wrap(sampleErr, "message 2"),
							),
							Wrap(sampleErr, "message 3"),
						),
						WrapConst(
							WrapConst(
								Wrap(sampleErr, "message 4"),
								Wrap(sampleErr, "message 5"),
							),
							Wrap(sampleErr, "message 6"),
						),
					).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		{
			"+constWrappedErrAsConstant.Error()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Constant: ups: fail.",
					WrapConst(
						sampleErr,
						sampleConstWrappedErr,
					).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		{
			"+errErrors.Is",
			func(t *testing.T) {
				if !errors.Is(
					WrapConst(
						sampleErr,
						sampleConstErr,
					),
					sampleErr,
				) {
					t.Fatalf(
						"WrapConst() errors.Is didn't match underlying error",
					)
				}
			},
		},
		{
			"+constantErrors.Is",
			func(t *testing.T) {
				if !errors.Is(
					WrapConst(
						sampleErr,
						sampleConstErr,
					),
					sampleConstErr,
				) {
					t.Fatalf(
						"WrapConst() errors.Is didn't match underlying error",
					)
				}
			},
		},
		{
			"+nestedErrors.Is",
			func(t *testing.T) {
				if !errors.Is(
					WrapConst(
						WrapConst(
							WrapConst(sampleErr, sampleErr),
							WrapConst(sampleErr, sampleErr),
						),
						WrapConst(
							WrapConst(sampleErr, sampleErr),
							WrapConst(sampleErr, sampleConstErr),
						),
					),
					sampleConstErr,
				) {
					t.Fatalf(
						"WrapConst() errors.Is didn't match underlying error",
					)
				}
			},
		},
		{
			"+errErrors.As",
			func(t *testing.T) {
				dest := &sampleErrorT{}

				if !errors.As(
					WrapConst(sampleErrT, sampleConstErr),
					&dest,
				) {
					t.Fatalf(
						"WrapConst() errors.As didn't match underlying error",
					)
				}

				if diff := cmp.Diff(
					&sampleErrorT{"fail"},
					dest,
					cmp.AllowUnexported(sampleErrorT{}),
				); diff != "" {
					t.Fatalf(
						"WrapConst() errors.As result didn't match underlying "+
							"error (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"+constantErrors.As",
			func(t *testing.T) {
				dest := &sampleErrorT{}

				if !errors.As(
					WrapConst(sampleErr, sampleErrT),
					&dest,
				) {
					t.Fatalf(
						"WrapConst() errors.As didn't match underlying error",
					)
				}

				if diff := cmp.Diff(
					&sampleErrorT{"fail"},
					dest,
					cmp.AllowUnexported(sampleErrorT{}),
				); diff != "" {
					t.Fatalf(
						"WrapConst() errors.As result didn't match underlying "+
							"error (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"+nestedErrors.As",
			func(t *testing.T) {
				dest := &sampleErrorT{}

				if !errors.As(
					WrapConst(
						WrapConst(
							WrapConst(sampleErr, sampleErr),
							WrapConst(sampleErr, sampleErr),
						),
						WrapConst(
							WrapConst(sampleErr, sampleErr),
							WrapConst(sampleErr, sampleErrT),
						),
					),
					&dest,
				) {
					t.Fatalf(
						"WrapConst() errors.As didn't match underlying error",
					)
				}

				if diff := cmp.Diff(
					&sampleErrorT{"fail"},
					dest,
					cmp.AllowUnexported(sampleErrorT{}),
				); diff != "" {
					t.Fatalf(
						"WrapConst() errors.As result didn't match underlying "+
							"error (-want +got):\n%s",
						diff,
					)
				}
			},
		},
		{
			"-nilErr.Error()()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Constant err: nil error.",
					WrapConst(
						nil,
						sampleConstErr,
					).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		{
			"-nilConstant.Error()()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Nil error: fail.",
					WrapConst(
						sampleErr,
						nil,
					).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
		{
			"-allNil.Error()()",
			func(t *testing.T) {
				if diff := cmp.Diff(
					"Nil error: nil error.",
					WrapConst(nil, nil).Error(),
				); diff != "" {
					t.Fatalf(
						"WrapConst().Error() missmatch (-want +got):\n%s", diff,
					)
				}
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()
			tt.caseFunc(t)
		})
	}
}

func Test_stringError_rawError(t *testing.T) {
	t.Parallel()

	type fields struct {
		msg string
	}

	tests := []struct {
		name   string
		fields fields
		want   string
	}{
		{
			"+valid",
			fields{"foo"},
			"foo",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			e := &stringError{
				msg: tt.fields.msg,
			}
			got := e.rawError()
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("stringError.rawError() = %s", diff)
			}
		})
	}
}

func Test_messageWrappedError_rawError(t *testing.T) {
	t.Parallel()

	type fields struct {
		err error
		msg string
	}
	tests := []struct {
		name   string
		fields fields
		want   string
	}{
		{
			"+valid",
			fields{errors.New("foo"), "bar"},
			"bar: foo",
		},
		{
			"+rawErrorImpl",
			fields{New("foo"), "bar"},
			"bar: foo",
		},
		{
			"-nil",
			fields{nil, "bar"},
			"bar: nil error",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			e := &messageWrappedError{
				err: tt.fields.err,
				msg: tt.fields.msg,
			}
			got := e.rawError()
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("messageWrappedError.rawError() = %s", diff)
			}
		})
	}
}

func Test_constantWrappedError_rawError(t *testing.T) {
	t.Parallel()

	type fields struct {
		err      error
		constant error
	}

	tests := []struct {
		name   string
		fields fields
		want   string
	}{
		{
			"+valid",
			fields{errors.New("foo"), errors.New("bar")},
			"bar: foo",
		},
		{
			"+rawErrorErrImpl",
			fields{New("foo"), errors.New("bar")},
			"bar: foo",
		},
		{
			"+rawErrorConstImpl",
			fields{errors.New("foo"), New("bar")},
			"bar: foo",
		},
		{
			"-nil",
			fields{},
			"nil error: nil error",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			e := &constantWrappedError{
				err:      tt.fields.err,
				constant: tt.fields.constant,
			}
			got := e.rawError()
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("constantWrappedError.rawError() = %s", diff)
			}
		})
	}
}

func Test_zabbixErrorString(t *testing.T) {
	t.Parallel()

	type args struct {
		s string
	}

	tests := []struct {
		name string
		args args
		want string
	}{
		{
			"+valid",
			args{"foo"},
			"Foo.",
		},
		{
			"+longer",
			args{"foo bar baz quux"},
			"Foo bar baz quux.",
		},
		{
			"+withTrailingDot",
			args{"foo bar baz quux."},
			"Foo bar baz quux.",
		},
		{
			"+withSpaces",
			args{"   foo bar baz quux.   "},
			"Foo bar baz quux.",
		},
		{
			"-empty",
			args{""},
			"",
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := zabbixErrorString(tt.args.s)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("zabbixErrorString() = %s", diff)
			}
		})
	}
}
