// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
//
// Returns a singleton instance of an InitialNetworkPresenter
//
define(['hp/core/EventDispatcher',
        'fs/services/settings/NetworkSetupService',
        'fs/services/status/StatusService',
        'fs/services/settings/CertificateService',
        'hp/core/Localizer',  
        'hp/lib/date'],
function(EventDispatcher, networkSetupService, statusService, certificateService, localizer) { "use strict";

    var NetworkSetupPresenter = (function() {

        // Total timeout / length of progress bar when adding a new node.  The longest time we've seen in testing was about 105
        // seconds.
        var ADD_NODE_DELAY = 120000; // msec

        // Total timeout / length of progress bar under other circumstances.  The longest time we've seen in testing was about
        // 22 seconds.
        var NORMAL_DELAY = 30000; // msec
        
        // for redirect we wait another 15 seconds to ensure that the new certificate is generated and configured if needed
        var REDIRECT_DELAY = 60000;
        // How long to wait, after a getTaskStatus request times out, before trying again.
        var TASK_RESULT_DELAY = 3000; // msec
        
        // How long to wait, after a checkReachable request times out, before trying again.
        var REACHABLE_DELAY = 1500; // msec
         
        // How long to wait, after achieving contact with the new hostname, before giving up on achieving contact with the old
        // hostname.  Should be large enough for a couple of attempts at the old hostname.
        //var REACHABLE_OFFSET = 3000; // msec
        
        // Should we do a reload window after 10s timeout to deal with the dynamic certificates 
        var shouldReload = false;
        var manualToDhcp = false; 
        
        var MAX_TRY=10;
        /**
         * @constructor Singleton class containing presentation
         *              logic for the initial network settings page
         * @type {InitialNetworkPresenter}
         */
        function NetworkSetupPresenter() {
            
            this.STATIC = "STATIC";
            this.UNCONFIGURE = "UNCONFIGURE";
            this.DHCP = "DHCP";
            
            this.MANUAL_TO_DHCP_ERROR = "MANUAL_TO_DHCP";
            this.REACHABLE_ERROR = "REACHABLE_ERROR";
            
            var STATIC = "STATIC";
            var UNCONFIGURE = "UNCONFIGURE";
            var DHCP = "DHCP";
            
            var dispatcher = new EventDispatcher();

            // The network configuration, as retrieved from the server.
            var originalConfig;

            // Total amount of time to wait for node to be accessible.
            var setConfigTimeout;

            // Timestamp when setNetworkConfiguration was started.
            var setConfigStartTime;

            // Flag indicating that we are currently polling the server for reachability.
            var checkingReachable;

            // Timers on reachable hosts.  This is an object keyed by hostname (with '' for the current/old host).
            var hostReachableTimers = {};

            // Flag whether we should be trying to talk to the pre-change hostname.
            var originalHostAllowed;

            // Flag whether we should redirect the browser when the network configuration finishes.
            var shouldRedirect;
            
        
            // Hostname we want to be connected to when we're done.
            var newHostname;

            // URI for the setNetworkConfiguration asynchronous task.
            var taskURI;

            // Forward function references.
            var checkTaskResult;
            var handleTaskStatus;

            // For unit testing, allow a stub of the timers created and cleared.
            var timerHost = window;

            var taskWarning = false ;
            
            var retryCount = 0;
            
            var oldIP, newIP;
            
            
            

            /**
             * Stop the interval timers pinging the hosts for when they become reachable.
             */
            function clearTimers() {
                for (var host in hostReachableTimers) {
                    if (hostReachableTimers[host]) {
                        timerHost.clearTimeout(hostReachableTimers[host]);
                        hostReachableTimers[host] = null;
                    }
                }
            }

            /**
             * Report a failure of the configuration change to the view.
             *
             * @param {Object} errorInfo Error information
             * @param {Object} handlers Handlers for the setNetworkConfiguration results.
             */
            function onSetConfigFailed(errorInfo, handlers) {
                handlers.error(errorInfo);
            }

            /**
             * Handler for when a host becomes reachable.
             *
             * @param {string} host Hostname of the host that became reachable, '' for current.
             * @param {Object} handlers Handlers for the setNetworkConfiguration results.
             */
            function onReachable(host, handlers) {
                if (!host) {
                    // Old host is reachable again; it must have been a transient outage due to network reconfiguration.
                    // Go back to polling on the task status.
                    clearTimers();
                    checkTaskResult(host, handlers);
                } else {
                    // We got to the new node.
                    checkingReachable = false;
                    clearTimers();

                    // Let's see if we can get task status from the new host.  It's unlikely but worth trying.
                    networkSetupService.getTaskStatus(host, taskURI, {
                        success: function(status) {
                            // Yes, so poll on the task status as needed.
                            handleTaskStatus(host, status, handlers);
                        },
                        error: function(errorInfo) {
                            // Can't get to the task status.  This is probably because of the self-signed certificate at the new
                            // address.  We'll call it good.
                            handlers.success(host);
                        }
                    });
                }
            }
            /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether this network configuration uses DHCP.
             */
            function isDhcpIPv4Used(data) {
                return data.ipv4Type == DHCP ;
            }
            
             /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether this network configuration uses DHCP.
             */
           function isDhcpIPv6Used (data) {
                return  data.ipv6Type == DHCP;
            }
           /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether this network configuration uses IPv6.
             */            
            function isIpv6Used (data) {
                return data.ipv6Type != UNCONFIGURE;
            }
            /**
             * Check whether the server is reachable, on either the old or the new address.
             *
             * @param {string} host Hostname of the server to check, '' for current.
             * @param {Object} handlers Handlers for the setNetworkConfiguration results.
             */
            function checkReachable(host, handlers) {
                hostReachableTimers[host] = null;
                networkSetupService.checkReachable(host, {
                    success: function() {
                        onReachable(host, handlers);
                    },
                    error: function() {
                        // If we've already flagged that the host is reachable by some other means,
                      //we can ignore this error.
                        // This flag is also cleared when the overall operation times out.
                        if (!checkingReachable) {
                            return;
                        }

                        // If we're checking our target node, and it isn't reachable after the end of the
                        //total timeout period,
                        // we'll call this a timeout.
                        if (host || !shouldRedirect) {                            
                            if (Date.now() - setConfigStartTime > setConfigTimeout) {                                 
                                clearTimers();                                
                                // in certain cases such as redirecting hostname->IP and vice verse
                                // IE treats this as switching between security zones
                                // sometimes it causes failure in "checkReachable" even though the new
                                // appliance
                                // is probably reachable
                                // At this point appliance address has been changes, so we need to 
                                // redirect a user
                                // regardless of whether reachability check was successful.
                                // In FF and Chrome the reachability test succeeds.
                                // Treat timeout as a success and redirect to the new appliance address                                
                                onReachable(host, handlers); 
                                return;                                 
                            }
                        }

                        // Wait a while and check reachability again.
                        hostReachableTimers[host] = timerHost.setTimeout(function() {
                            checkReachable(host, handlers);
                        }, REACHABLE_DELAY);
                    }
                });
            }

            /**
             * Wait until the server is reachable, on either the old or the new hostname.
             *
             * @param {Object} handlers Handlers for the setNetworkConfiguration results.
             */
            function waitForReachable(handlers) {
                checkingReachable = true;
                
                 if (shouldRedirect) {
                    hostReachableTimers[newHostname] = timerHost.setTimeout(function() {
                        checkReachable(newHostname, handlers);
                    }, REACHABLE_DELAY);
                    return;
                } 
                if (originalHostAllowed) {
                    hostReachableTimers[''] = timerHost.setTimeout(function() {
                            checkReachable('', handlers);
                    }, REACHABLE_DELAY);
                }                 


   
            }

            /**
             * Handle the task status returned from the asynchronous setNetworkConfiguration call.Incase of TaskT
             * there are only 3 states 'Running', 'Completed' and 'Error'.So the Default has been pointed to Error
             *
             * @param {string} host Hostname from which task status was requested, '' for current.
             * @param {TaskResource} status Task resource with status.
             * @param {Object} handlers Handlers for the setNetworkConfiguration results.
             */
            handleTaskStatus = function(host, status, handlers) {
                switch (status.taskState) {
                case 'Running':
                    // Job is still running, we'll poll again after an interval.
                    timerHost.setTimeout(function() {
                        checkTaskResult(host, handlers);
                    }, TASK_RESULT_DELAY);
                    break;

                case 'Completed':
                    // Job is finished successfully.  Since we were able to get task status from our old address, there's no
                    // need to redirect except in the case where there was a hostname change and we need to force page reload
  
                    handlers.success(host);
                    taskWarning = false;
                    break;

                default: 
                    // Job failed.
                  // Backend errors are classified as warnings and errors. The warnings will be populated as part of
                  // "taskErrors" which comes as array in the task response .Rest all the other errors are pouplated as part 
                   // of the taskStatus .Hence the special handling needed. 
                  if (status.taskStatus && status.taskErrors !== undefined   && status.taskErrors.length > 0  ) {
                        onSetConfigFailed(status.taskErrors, handlers);
                        taskWarning = true;
                    }
                  else if (status.taskStatus) {// other backend errors
                        onSetConfigFailed({message: status.taskStatus}, handlers);
                    } else {
                        onSetConfigFailed({
                            message: localizer.getString('fs.settings.network.error.set.unknown.msg')
                        }, handlers);
                    }
                    break;
                }
            };

            /**
             * Method called periodically to check the results of the running setNetworkConfiguration asynchronous task.
             *
             * @param {string} host Hostname of the host to request task status from, '' for current.
             * @param {Object} handlers Handlers for the setNetworkConfiguration results.
             */
            checkTaskResult = function(host, handlers) {
                checkingReachable = false;
                networkSetupService.getTaskStatus(host, taskURI, {
                    success: function(status) {
                    handleTaskStatus(host, status, handlers);
                    },
                     error: function(errorInfo) {
                        // check for max 10 times if any errors are present before proceeding,
                        //this is to make sure their are no network errors in case of IP change .
                         
                       if(retryCount < MAX_TRY){
                           timerHost.setTimeout(function() {
                               checkTaskResult(oldIP, handlers);
                           }, TASK_RESULT_DELAY);
                           retryCount++;                             
                           }else{
                        if (errorInfo.message.substr(0, 8) == 'timeout:') {
                            waitForReachable(handlers);
                        } else if (errorInfo.errorCode == 'ERR_USE_FLOATING_ADDR' && shouldRedirect) {
                            // This implies that the network change has been at least partially successful.
                            originalHostAllowed = false;
                            waitForReachable(handlers);
                        } else {
                            
                            // the problem may be caused by a newly generated certificate that a user can only accept in a browser
                            // try to continue ieven if not redirect
                            if (shouldRedirect) {
                                originalHostAllowed = false;
                                clearTimers();
                               //handlers.success(newHostname);
                               waitForReachable(handlers); 
                              
                            } else {
                                // ignore the error for Beta
                                originalHostAllowed = true;
                                clearTimers();
                                waitForReachable(handlers);
                                /**
                                onSetConfigFailed({
                                   errorMessage: localizer.getString('fs.settings.network.error.set.unknown.msg')
                                }, handlers);
                                **/
                           }
                           
                        }
                    }
                  }
                });
               
            };

            /**
             * The appliance setNetworkConfiguration call succeeded, which means the asynchronous process of changing the network
             * configuration (and possibly adding a node) has begun.  Periodically monitor it.
             *
             * @param {Object} result Result of the setNetworkConfiguration call.
             * @param {Object} handlers Handlers for the setNetworkConfiguration results.
             */
            function onSetConfigSuccess(result, handlers) {
                taskURI = result.uri;
                handlers.running(setConfigTimeout);
                checkTaskResult('', handlers);
            }

            /**
             * Figure out whether we should redirect the browser when the change is complete.
             *
             * @param {Object} networkObject Network configuration object
             * @returns {String} address to which we should redirect (or current address if no redirect wanted)
             */
            
            function redirectWanted(networkObject) {
                var addrs;
                manualToDhcp = false;
                if (networkObject.confOneNode) {
                    addrs = networkObject.addrs['node' + networkObject.localNodeID];
                } else {
                    addrs = networkObject.addrs.cluster;
                }
               var current = statusService.getState().usingAddr;
               
               oldIP = current;             
                // handle DHCP case

               // minimize switching between IP and hostname on redirect as it does not work in IE
               // when checking reachability
               // try to stay on the current windonw.control.host as long as possible
               
               // DHCP ->DHCP 
                if (isDhcpIPv4Used(networkObject) && isDhcpIPv4Used(originalConfig)) 
                {
                    
                    if (networkObject.hostname == originalConfig.hostname) {
                        return timerHost.location.host;
                    } else  { 
                        return current;
                    }
                }
               // manual to DHCP -> redirect to hostname
                
                if (isDhcpIPv4Used(networkObject) && !isDhcpIPv4Used(originalConfig))  {
                    manualToDhcp = true;
                    return networkObject.hostname;
                }                 
   
                if (current == addrs.ipv4 || current == addrs.ipv6) {
                    return timerHost.location.host;
                } else if (current.indexOf(':') >= 0 && addrs.ipv6) {
                   newIP = addrs.ipv6;
                    return addrs.ipv6;
                } else {
                    newIP = addrs.ipv4;
                    return addrs.ipv4;
                }
            }
            

            /**
             * Set the desired network configuration.
             *
             * @param {Object} networkObject Network configuration object
             * @param {{success:function(redirectHost):void, error:function(ErrorInfo):void}} handlers
             *     Handlers for success and error return.  If the parameter to the success handler is set, it names the hostname
             *     or IP address to which the browser should be redirected.
             * @return {int} number of seconds that the UI should show in the progress bar
             */
            this.setNetworkConfiguration = function(networkObject, handlers) {
                // How long are we going to wait before giving up?
 
                originalHostAllowed = true;
                // reset to zero for the retry logic 
                retryCount=0;

                // Figure out our new hostname.
                newHostname = redirectWanted(networkObject);
              
               shouldRedirect = ((networkObject.hostname != originalConfig.hostname) ||
                                 (newHostname != timerHost.location.host) ) &&
                                (timerHost.location.protocol == 'https:');
                                 
              
               if (originalConfig.confOneNode && !networkObject.confOneNode) {
                   setConfigTimeout = ADD_NODE_DELAY;
               } else if (shouldRedirect) {
                   setConfigTimeout = REDIRECT_DELAY;                  
               } else {
                   setConfigTimeout = NORMAL_DELAY;
               }
               setConfigStartTime = Date.now();
     
                // in addition to the redirect  we want to be more defensive for going DHCp to MANUAL and vice verse                
               shouldReload = ((networkObject.hostname != originalConfig.hostname) || 
                   (newHostname != timerHost.location.host) ) &&
                       (timerHost.location.protocol == 'https:');
               
                 // Start the actual configuration change.
                networkSetupService.setNetworkConfiguration(networkObject, {

                    success: function(result) {
                        onSetConfigSuccess(result, handlers);
                    },
                    error: function(errorInfo) {
                        onSetConfigFailed(errorInfo, handlers);
                    }
                });
            };

            /**
             * Validate the proposed network configuration.
             *
             * @param {Object} networkObject Network configuration object
             * @param {{success:function(unused):void, error:function(ErrorInfo):void}} handlers
             *     Handlers for success and error return
             */
            this.validateNetworkConfiguration = function(networkObject, handlers) {
                networkSetupService.validateNetworkConfiguration(networkObject, handlers);
            };

            /**
             * Retrieve the current network configuration.
             *
             * @param {{success:function(NetworkObject):void, error:function(ErrorInfo):void}} handlers
             *     Handlers for success and error return
             */
            this.getNetworkConfiguration = function(handlers) {
                networkSetupService.getNetworkConfiguration({
                    success: function(networkObject) {
                        originalConfig = networkObject;
                        handlers.success(originalConfig);
                    },
                    error: handlers.error
                });
            };

            /**
             * Return whether the UI should show two nodes (vs. one).
             *
             * @param {Object} networkObject Network data from the server.
             * @param {function(boolean):void} callback Function called with the resulting two-node flag.
             */
            this.showTwoNode = function(callback) {
                if (!originalConfig.confOneNode) {
                    callback(true);
                } else if (originalConfig.twoNodeData) {
                    networkSetupService.validatePeerNode({
                        success: function() {
                            callback(true);
                        },
                        error: function(errorInfo) {
                            callback(false, errorInfo.message);
                        }
                    });
                } else {
                    callback(false);
                }
            };

            /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether a fully qualified domain name is required in this network configuration.
             */
            this.fqdnRequired = function(data) {
                return data.priDns || data.altDns;
            };

            /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether IPv4 addresses are needed in this network configuration.
             */
            this.ipv4DataRequired = function(data) {
                return data.ipv4Type == STATIC;
            };

            /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether IPv6 addresses are needed in this network configuration.
             */
            this.ipv6DataRequired = function(data) {
                return data.ipv6Type == STATIC;
            };

            /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether IPv6 is in use in this network configuration.
             */
            this.ipv6Used = function(data) {
                return isIpv6Used(data);
            };

            /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether this network configuration needs to accommodate IPv6 DNS server addresses.
             */
            this.ipv6DNS = function(data) {
                return this.ipv6Used(data) || data.priDns.indexOf(':') >= 0 || data.altDns.indexOf(':') >= 0;
            };

            /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether this network configuration uses DHCP.
             */
            this.dhcpUsed = function(data) {
                 return data.ipv4Type == this.DHCP || data.ipv6Type == this.DHCP;
            };
             /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether this network configuration uses DHCP.
             */
            this.dhcpIPv4Used = function(data) {
                return isDhcpIPv4Used(data);
            };
            
             /**
             * @param {Object} data Network data from the view.
             * @return {boolean} Whether this network configuration uses DHCP.
             */
            this.dhcpIPv6Used = function(data) {
                return isDhcpIPv6Used(data);
            };
             /**      
             * @return {boolean} Whether IPv6 is disabled
             */
            this.iPv6Disabled = function(data) {
                return  networkSetupService.getIPv6Disabled();
            };    
            
            /**
             * @return {boolean} Whether the second IP is used (Altair)
             */  
            this.isSecondIPUsed = function() {
              return ("ipv4_2" in originalConfig.addrs.node1);
            };
            /**
             * Retrieve the certificate information
             *
             * @param {Object} handlers The success and error handler methods.
             */
            this.getCertificate = function(handlers) {
                certificateService.getCertificate({
                    success: handlers.success,
                    error: handlers.error 
                });
            };
            
            /**
             * @param {String} priDns from the view.
             * @param {String} altDns from the view.
             * @return {boolean} Whether at least one DNS name server is specified in the view.
             */
            this.dnsServerSpecified = function(priDns, altDns) {
                if (priDns != null && priDns != '') {
                    return true;
                }
                if (altDns != null && altDns != '') {
                    return true;
                }
                return false;
            };

            /**
             * @param {String} hostname from the view.
             * @param {String} priDns from the view.
             * @param {String} altDns from the view.
             * @return {String} domain if the fully qualified hostname is specified, otherwise null.
             */
            this.getDomainName = function(hostname, priDns, altDns) {
                if (hostname == null || !this.dnsServerSpecified(priDns, altDns)) {
                     return null;
                }
                var i = hostname.indexOf('.');
                if ( i == -1) {
                    return null;
                } else {
                    return (hostname.substring (i+1, hostname.length));
                }
            };
            this.shouldReloadWindow = function() {
                return shouldReload;
            };
            
            this.shouldRedirect = function() {
                return shouldRedirect;
            };
           /**
             * Add a listener for a specified event.
             *
             * @public
             * @param {string} eventName The name of the event.
             * @param {function(...)} callback
             */
            this.on = function(eventName, callback) {
                dispatcher.on(eventName, callback); 
            };

            /**
             * Remove a listener for the specified event.
             *
             * @public
             * @param {string} eventName The name of the event.
             * @param {function(...)} callback
             */
            this.off = function(eventName, callback) {
                dispatcher.off(eventName, callback);
            };
            /**
             * @private For unit testing only...
             */
            this._setTimerHost = function(th) {
                timerHost = th;
            };

            /**
             * Flag indicating their are Backend Task Errors 
             
             * @public 
             * ! 
             */
            this.hasTaskWarnings= function() {
              return taskWarning;
            };

        }

        return new NetworkSetupPresenter();

    }());

    return NetworkSetupPresenter;

});
