// (C) Copyright 2012 Hewlett-Packard Development Company, L.P.

define(['hp/core/Localizer',
    'hp/core/Style',
    'text!hpPages/core/hp_select_sort_directions.html',
    'jquery'],
function(localizer, style, sortDirectionsHtml) {
"use strict";

    (function($) {
        // jQuery plugin definition

        // You can describe the control completely in HTML or
        // by passing in configuration options here, for example:
        // {label: 'Label',
        //  selected: 'Value 1',
        //  options:['Value 1', 'Value 2'],
        //  multiSelect: false,
        //  sorting: true}
        $.fn.hpSelect = function(action, args) {

            var BODY = 'body';
            var LABEL = '> label';
            var VALUE = '> .hp-value';
            var OPTIONS = '> .hp-options';
            var OPTION = OPTIONS + ' li';
            var SELECT_OPTION = OPTIONS +
                ' li:not([data-sort-direction]):not(.hp-sub-options)';
            var SORT_DIRECTION_OPTION = OPTIONS + ' li[data-sort-direction]';
            var ALL_OPTION = OPTION + '[data-id=all], ' + OPTION + '[data-all=true]';
            var ACTIVE = 'hp-active';
            var SELECTED = 'hp-selected';
            var VALUE_SEPARATOR = ', ';
            var SPACE = 32;
            var TAB = 9;
            var ESCAPE = 27;
            var ENTER = 13;
            var UP_ARROW = 38;
            var DOWN_ARROW = 40;

            function hpSelect(elem) {

                var container;
                var multiSelect = false;
                var sorting = false;

                // extract the id for an option, what we send in a change event
                function optionId(option) {
                    if ($(option).data('id')) {
                        // option has a 'data-id' attribute
                        return $(option).data('id');
                    } else if ($('> a', option).length > 0) {
                        // special case anchor options
                        // get the text of the anchor
                        return $.trim($('> a', option).text());
                    } else {
                        return $.trim(option.text());
                    }
                }

                // extract the visible value for an option, what we show the user
                function optionValue(option) {
                    if ($('> a', option).length > 0) {
                        // special case anchor options
                        // get the text of the anchor
                        return $.trim($('> a', option).text());
                    } else {
                        return $.trim(option.text());
                    }
                }

                function setFromSelection() {
                    var oldValue = $(VALUE, container).text();
                    var oldDirection = null;
                    var values = [];
                    var ids = [];
                    var value;
                    var direction = null;

                    $(SELECT_OPTION + '.' + SELECTED, container).each(
                        function (index, option) {
                            values.push(optionValue($(option)));
                            ids.push(optionId($(option)));
                        });
                    value = values.join(VALUE_SEPARATOR);

                    if (sorting) {
                        $(SORT_DIRECTION_OPTION, container).each(function (index, option) {
                            var optionDirection = $(option).data('sort-direction');
                            if (container.hasClass('hp-sort-' + optionDirection)) {
                                oldDirection = optionDirection;
                                container.removeClass('hp-sort-' + optionDirection);
                                return false;
                            }
                        });
                        direction = $(SORT_DIRECTION_OPTION + '.' + SELECTED, container).
                            data('sort-direction');
                        $(container).addClass('hp-sort-' + direction);
                    }

                    if (value !== oldValue || direction !== oldDirection) {
                        $(VALUE, container).text(value);
                        if (sorting) {
                            container.trigger('change', {id : ids[0], direction: direction});
                        } else {
                            container.trigger('change',
                                (ids.length === 1 ? [ids[0]] : [ids]));
                        }
                    }
                }

                function initialActive() {
                    if ($(OPTION + '.' + ACTIVE, container).length === 0) {
                        $(OPTION + '.' + SELECTED, container).first().
                            addClass(ACTIVE);
                    }
                }

                function nextActive() {
                    var options = $(OPTION, container).not('.hp-sub-options');
                    initialActive();
                    options.each(function (index, elem) {
                        if ($(elem).hasClass(ACTIVE) &&
                            options.length > (index + 1)) {
                            $(elem).removeClass(ACTIVE);
                            $(options[index+1]).addClass(ACTIVE);
                            return false;
                        }
                    });
                }

                function previousActive() {
                    var options = $(OPTION, container).not('.hp-sub-options');
                    initialActive();
                    options.each(function (index, elem) {
                        if ($(elem).hasClass(ACTIVE) && index > 0) {
                            $(elem).removeClass(ACTIVE);
                            $(options[index-1]).addClass(ACTIVE);
                            return false;
                        }
                    });
                }

                function onMousedown(ev) {
                    var option = $(ev.target).closest('li');
                    if (sorting && option.data('sort-direction')) {
                        $(SORT_DIRECTION_OPTION, container).removeClass(SELECTED);
                        option.addClass(SELECTED);
                    } else if (multiSelect) {
                        if (option.attr('data-id') === 'all' || option.attr('data-all')) {
                            // user clicked 'all' item, clear everything else
                            $(OPTION, container).not('.hp-sub-options').removeClass(SELECTED);
                            option.addClass(SELECTED);
                        } else {
                            option.toggleClass(SELECTED);
                            if (option.hasClass(SELECTED)) {
                                // clear 'all' item, if any
                                $(ALL_OPTION, container).removeClass(SELECTED);
                            } else {
                                // if nothing is selected, select the 'all' item, if any
                                if ($(OPTION + '.' + SELECTED, container).length === 0) {
                                    $(ALL_OPTION, container).addClass(SELECTED);
                                }
                            }
                        }
                    } else {
                        $(SELECT_OPTION, container).removeClass(SELECTED);
                        option.addClass(SELECTED);
                    }
                    setFromSelection();
                }

                function onMousemove() {
                    $(OPTION, container).removeClass(ACTIVE);
                }

                function onKeypress(ev) {
                    var keyCode = (ev.which ? ev.which : ev.keyCode);
                    if (keyCode == ENTER) {
                        $(OPTION, container).removeClass(SELECTED);
                        $(OPTION + '.' + ACTIVE, container).removeClass(ACTIVE).
                            addClass(SELECTED);
                        setFromSelection();

                        // follow anchor, if any
                        if (! multiSelect && ! sorting) {
                            var anchor = $(OPTION + '.' + SELECTED + ' a',
                                container);
                            if (anchor.length > 0) {
                                anchor.trigger('click');
                            }
                        }

                        deactivate();
                    } else if (keyCode == ESCAPE || keyCode == TAB) {
                        deactivate();
                    } else if (keyCode == DOWN_ARROW) {
                        nextActive();
                        ev.preventDefault();
                    } else if (keyCode == UP_ARROW) {
                        previousActive();
                        ev.preventDefault();
                    }
                }

                function deactivate(ev) {
                    var proceed = true;
                    if (ev && multiSelect) {
                        // for multi-select, don't deactivate if we click within
                        // the container
                        proceed =
                            ($(ev.target).parents('.hp-options').
                                filter(function (index) {
                                    return ($(this).parent()[0] === container[0]);
                                }).length === 0);
                    }
                    if (proceed) {
                        $(BODY).off('click.hpSelect', deactivate);
                        if (!multiSelect) {
                            $(document).off('keydown.hpSelect', onKeypress);
                        }
                        $(OPTIONS, container).
                            css({width: '', 'padding-top': '', left: '', right: ''});
                        $(LABEL, container).css('width', '');
                        $(VALUE, container).css('width', '');
                        container.css('width', '');
                        if (! container.hasClass('hp-pinned')) {
                            container.removeClass(ACTIVE);
                        }
                        $(OPTION, container).removeClass(ACTIVE);
                    }
                }

                function activate() {
                    var options = $(OPTIONS, container);
                    var value = $(VALUE, container);
                    var label = $(LABEL, container);

                    container.addClass(ACTIVE);

                    var numOptions = $(OPTION, container).not('.hp-sub-options').length;
                    var height = 30;
                    if (numOptions > 0) {
                        height = $(OPTION, container).not('.hp-sub-options').first().outerHeight() *
                            (numOptions <= 8 ? numOptions : 8);
                    }

                    options.css({
                        width: Math.max(options.width(), value.outerWidth(), label.outerWidth()) + 12,
                        height: height,
                        'padding-top': label.outerHeight(),
                        overflow: 'auto'
                    });
                    label.css({width: options.width() - style.scrollBarWidth()});
                    value.css({width: value.width()}); // freeze value size

                    var boundary = options.
                        parents('.hp-menu-boundary, body').first();
                    if ((options.offset().left + options.outerWidth()) >
                        (boundary.offset().left + boundary.innerWidth()) &&
                        options.offset().left > options.outerWidth()) {

                        options.css({left: 'auto', right: '0px'});
                    }

                    if (!multiSelect) {
                        $(document).on('keydown.hpSelect', onKeypress);
                        $(container).on('mousemove.hpSelect', onMousemove);
                    }
                    // avoid bouncing
                    setTimeout(function () {
                        $(BODY).on('click.hpSelect', deactivate);
                    }, 50);
                }
                
                function populate(args){
                    var sortDirectionOptions;
                    
                    if (args.hasOwnProperty('label')) {
                        $(LABEL, container).text(args.label);
                    }

                    if (args.hasOwnProperty('selected')) {
                        $(VALUE, container).text(args.selected.ids);
                    }

                    if (args.hasOwnProperty('options')) {
                        $(OPTION, container).remove();
                        $.each(args.options, function(index, value) {
                            if ('object' === typeof(value)) {
                                $(OPTIONS, container).
                                    append($('<li data-id="' + value.id +
                                        '">' + value.name + '</li>'));
                            } else {
                                $(OPTIONS, container).
                                    append($('<li>' + value + '</li>'));
                            }
                        });
                    }

                    if (args.hasOwnProperty('multiSelect') &&
                        args.multiSelect) {
                        container.addClass('hp-select-multi');
                    }

                    if (args.hasOwnProperty('sorting') && args.sorting) {
                        $('.hp-options', container).removeClass('hp-options').
                            wrap('<div class="hp-options"></div>');
                        sortDirectionOptions = $(sortDirectionsHtml);
                        localizer.localizeDom(sortDirectionOptions);
                        $('.hp-options', container).append(sortDirectionOptions);
                        container.addClass('hp-sort');
                    }
                }

                function build() {                    

                    container.addClass('hp-select').attr('tabindex', 0);

                    // set up required DOM elements if they aren't already there.

                    if ($(LABEL, container).length === 0) {
                        container.prepend('<label></label>');
                    }

                    if ($(VALUE, container).length === 0) {
                        $(LABEL, container).after('<div class="hp-value"></div>');
                    }

                    if ($(OPTIONS, container).length === 0) {
                        container.append('<ol class="hp-options"></ol>');
                    }

                    // populate with any supplied options

                    if (typeof args === 'object') {
                        populate(args);                        
                    }
                }

                function parseSelection(arg) {
                    var selection = {};
                    if (typeof arg === 'string') {
                        // initialize from args if it's a string
                        selection.ids = [$.trim(arg)];
                    } else if ($.isArray(arg)) {
                        // initialize from args if it's an array
                        selection.ids = arg;
                    } else if (arg) {
                        selection.ids = arg.ids;
                        if (sorting) {
                            selection.direction = arg.direction;
                        }
                    }
                    return selection;
                }

                function getInitialSelection() {
                    var selection = {};

                    if ($(VALUE, container).text().length > 0) {
                        // element value
                        if (multiSelect) {
                            selection.ids =
                                $(VALUE, container).text().split(VALUE_SEPARATOR);
                        } else {
                            selection.ids = [$(VALUE, container).text()];
                        }
                    } else if ($(SELECT_OPTION + '.' + SELECTED, container).length > 0) {
                        // selected option
                        if (multiSelect) {
                            selection.ids = [];
                            $(SELECT_OPTION + '.' + SELECTED, container).each(
                                function (index, option) {
                                    selection.ids.push(optionId($(option)));
                                });
                        } else {
                            selection.ids =
                                [optionId($(OPTION + '.' + SELECTED, container).first())];
                        }
                    } else {
                        // first option
                        selection.ids = [optionId($(SELECT_OPTION, container).first())];
                    }

                    if (sorting) {
                        if ($(SORT_DIRECTION_OPTION + '.hp-selected', container).length > 0) {
                            selection.direction =
                                $(SORT_DIRECTION_OPTION + '.hp-selected', container).
                                    data('sort-direction');
                        } else {
                            // first direction
                            selection.direction =
                                $(SORT_DIRECTION_OPTION, container).first().
                                    data('sort-direction');
                        }
                    }

                    return selection;
                }

                function alignSelection(selection) {

                    // set selected option(s)
                    $(OPTION, container).removeClass(SELECTED);
                    $(SELECT_OPTION, container).
                        each(function (index, option) {
                            if ($.inArray(optionId($(option)), selection.ids) !== -1) {
                                $(option).addClass(SELECTED);
                            }
                        });

                    if (sorting) {
                        $(SORT_DIRECTION_OPTION, container).removeClass(SELECTED);
                        $(OPTION + '[data-sort-direction="' + selection.direction + '"]',
                            container).addClass(SELECTED);
                    }

                    setFromSelection();
                }

                function onClick() {
                    if (!container.hasClass(ACTIVE)) {
                        activate();
                    }
                }

                function onKeyUp(ev) {
                    var keyCode = (ev.which ? ev.which : ev.keyCode);
                    if (keyCode == SPACE) {
                        if (!container.hasClass(ACTIVE)) {
                            activate();
                        }
                    }
                }

                function getSelection() {
                    return {
                        ids: $('li:not([data-sort-direction]):not(.hp-sub-options).hp-selected', container).data('id'),
                        direction: $('li[data-sort-direction].hp-selected', container).data('sort-direction')
                    }
                };

                this.initialize = function () {
                    var selection; // {ids: ..., direction: ...}
                    var result = $(elem);
                    
                    container = $(elem);

                    if ('destroy' !== action) {
                        if (! action || 'object' === typeof(action)) {
                            args = action;
                            action = 'initialize';
                        }

                        if ('initialize' === action) {
                            build();
                        } else if ('get' === action) {
                            result = getSelection();
                        }

                        multiSelect = container.hasClass('hp-select-multi');
                        sorting = container.hasClass('hp-sort');

                        if ('initialize' === action) {
                            if (args && args.selection) {
                                selection = parseSelection(args.selection);
                            } else {
                                selection = getInitialSelection();
                            }
                        } else if ('set' === action) {
                            selection = parseSelection(args);
                        }

                        alignSelection(selection);

                        if ('initialize' === action) {
                            container.on('click.hpSelect', onClick);
                            container.on('keyup.hpSelect', onKeyUp);
                            // unbind the onMousedownEvent
                            $(OPTION, container).not('.hp-sub-options').
                                off('.hpSelect');
                            $(OPTIONS, container).
                                on('mousedown.hpSelect', "li:not('.hp-sub-options')", onMousedown);
                        }
                    } else {
                        container.off('.hpSelect');
                        $(OPTIONS, container).off('.hpSelect');
                        deactivate();
                    }

                    return result;
                }

            }

            // pluginify
            var ret;
            var instance = null;
            this.each(function() {
                var $elem = $(this);
                instance = new hpSelect($elem[0]);
                ret = instance.initialize();
                if (action === 'get') {
                    // "get" only returns the first elements value, similar to jquery.val()
                    return false;
                }
                ret = ret ? ret.add(ret) : ret;
            });
            return ret;
            
        };
    }(jQuery));
});
