// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
define(['hp/core/EventDispatcher',
    'hp/core/Router',
    'hp/core/UrlFragment',
    'hp/core/LinkTargetBuilder',
    'hp/services/IndexService',
    'hp/services/IndexFilter',
    'hp/model/DevelopmentSettings',
    'hp/model/Session',
    'hp/core/Banner',
    'hp/core/Localizer',
    'hp/services/Log'],
function(EventDispatcher, router, urlFragment, linkTargetBuilder, indexService, IndexFilter,
    settings, session, banner, localizer, log) { "use strict";

    /*
     * Manages a Resource, its index results, selection, and filter.
     *
     * Takes three types of input:
     * 1) view changes, such as selecting something or changing a filter
     * 2) browser location changes via the Router
     * 3) Resource changes, such as arrival of index results
     * The typical flow is 1-2-3: user makes a change -> update location,
     * react to location change by adjusting the Resource.
     * But the browser location can change at any time, so sometimes it's 2-1-3.
     * And, for managing the selection, especially multi-select, it can be 1-3-2,
     * because the Resource controls the multi-select id.
     *
     * We prioritize whatever the browser URL indicates.
     * If there's a resource URI that doesn't exist, we leave it there so we
     * can indicate that to the user.
     * If the user changes a filter or searches, we start by clearing any
     * selection in the browser URL. However, if, after the index results return,
     * we find that the prior selected URI matches, we restore the selection.
     * Otherwise, we select a new resource.
     *
     * Some interesting scenarios covered:
     *  #/view/r/resource ->
     *  #/view/r/non-existing-resource -> show error
     *  #/view/r/resource?filter ->
     *  #/view/r/unmatching-resource?filter -> show unmatched
     *  #/view/r/non-existing-resource?filter -> show error
     *  #/view/r/resource (empty) -> show error and empty
     *  #/view/r/unmatching-resource?filter (empty) -> show empty
     *  #/view/r/non-existing-resource?filter (empty) -> show error and empty
     */

    var MasterPanePresenter = (function() {

        var ROUTE_SUFFIX = '(/.*|$)';

        /**
         * @constructor
         * @type {MasterPanePresenter}
         */
        function MasterPanePresenter() {
            var dispatcher = new EventDispatcher();
            var timer = null;
            var resource = null;
            var routePrefix = null;
            var authCategory = null;
            var location = null;
            var awaitingResults = false;
            // The following flag is set to true when the location does not specify selection.
            var selectFromResults = false;
            var initialSort = null;
            var notifyFilterChange = true;
            var headerTitle = '';
            // The following flag is set to true when the selected item can not be located in the
            // current index result, which will trigger showMore() to load more index result.
            var pageToSelection = false;
            // The following flag is set to true when the RHS displays the "item not found" message.
            // If set, getItem() is called whenever new index result arrives.
            var selectedItemNotFound = false;
            // reload index result is needed
            var reloadIndexNeeded = false;
            // Flag indicating this pane presenter is the global view
            var globalView = true;

            function startsWith(string, pattern) {
                return (pattern === string.substr(0, pattern.length));
            }

            // Prototyping method to make sonar happy (like in C), since restartTimer() and getIndexResults() reference each other
            var getIndexResults;

            /**
             * @private
             * Start polling the index service for results.
             */
            function restartTimer(always) {
                var doStart = always;
                if (settings.getRefreshInterval()) {
                    if (timer) {
                        clearTimeout(timer);
                        doStart = true;
                    }
                    if (doStart) {
                        timer = setTimeout(getIndexResults,
                                settings.getRefreshInterval());
                    }
                }
            }

            // Method prototyped above to make sonar happy
            getIndexResults = function() {
                if (! awaitingResults) {
                    awaitingResults = true;
                    resource.getIndexResults({
                        error: function () {
                            awaitingResults = false;
                            restartTimer();
                        },
                        success: function () {
                            awaitingResults = false;
                            restartTimer();
                        }
                    });
                }
            };

            function showMore() {
                var newFilter = new IndexFilter(resource.getIndexFilter());
                newFilter.bumpCount();
                if (resource.setIndexFilter(newFilter)) {
                    getIndexResults();
                }
            }

            function setSelectionFromResults(indexResults) {
                var uris, multiSelectId, selection;
                if (0 === indexResults.total) {
                    resource.clearSelectedUris();
                } else {
                    // if we have any already selected, use them
                    selection = resource.getSelection();
                    uris = selection.uris;
                    multiSelectId = selection.multiSelectId;
                    if (uris.length === 0) {
                        // attempt to restore to previous selection
                        // This can happen when the selection is cleared
                        // when we 'add' or when we change a filter but
                        // the prior selection is still available.
                        selection = resource.getPriorSelection();
                        uris = selection.uris;
                        multiSelectId = selection.multiSelectId;
                    }

                    if (uris.length > 0) {
                        // Look for the uris in the index result.
                        var foundUris = $.grep(uris, function (uri) {
                            return resource.getIndexResultForUri(uri);
                        });

                        // If we know about these uris already, use them if
                        // 1 - all selected items are found in the loaded index result, or
                        // 2 - the user changes filter, then use whatever is found (including none)
                        if ((uris.length === foundUris.length) ||
                            (resource.getIndexFilter().getUserQuery() && !resource.haveMore())) {
                            uris = foundUris;
                        }
                    }

                    if (uris.length === 0) {
                        // we don't have any existing selection, select the first one
                        uris = [indexResults.members[0].uri];
                        multiSelectId = null;
                    }

                    router.replaceWith(urlFragment.replaceUris(location, uris,
                            multiSelectId), 'master pane select uris');
                }
            }

            /**
             * Preprocess the new route to determine:
             * - if the route is triggered by a "reset all" filter clicked.
             *   If so, the selections are removed from the location and reroute.
             * - if the old sort should be preserved
             * - if the old query should be preserved. If so, then the old
             *   filter is installed, and then reroute.
             */
            function preprocessRoute(newFilter, location, noSelectionInRoute) {
                var oldFilter = resource.getIndexFilter();
                var continueProcessRoute = true;

                // if the location has a sort, keep it
                if (! newFilter.getSort()) {
                    // if we have a sort already, use it
                    if (oldFilter && oldFilter.getSort()) {
                        newFilter.setSort(oldFilter.getSort());
                    } else if (initialSort) {
                        newFilter.setSort(initialSort);
                    }
                }

                // if we have paged, keep the pages
                newFilter.useCount(oldFilter);

                // Determine if this route change comes from the user
                // selecting one of the "reset all" filters. If so, remove
                // the FILTER_RESET flag and the uris in the location and re-route.
                if (newFilter.hasResetFlag()) {
                    newFilter.unsetResetFlag();
                    location = newFilter.updateLocation(urlFragment.replaceUris(location, []));
                    reloadIndexNeeded = resource.setIndexFilter(newFilter);
                    router.replaceWith(location,
                            "master filter reset");
                    continueProcessRoute = false;
                } else if (! newFilter.getUserQuery() && noSelectionInRoute &&
                    oldFilter && oldFilter.getUserQuery()) {
                    // Determine if we need to preserve old user query. We preserve the old
                    // query if the route contains no selection, and the route
                    // does not specify query.
                    newFilter.setUserQuery(oldFilter.getUserQuery());
                    location = newFilter.updateLocation(location);
                    router.replaceWith(location, "master use old query");
                    continueProcessRoute = false;
                }

                // if there is any user query, make sure the filter change is fired
                // eventually so that other views (e.g. SearchBoxView) sync up.
                if (newFilter.getUserQuery()) {
                    notifyFilterChange = true;
                }
                return continueProcessRoute;
            }

            function setDocumentTitle() {
                var context = '',
                    view = urlFragment.getView(location),
                    friendlyView,
                    uris = resource.getSelection().uris,
                    indexResult;
                if (! startsWith(view, 'show')) {
                    friendlyView = view.replace(/\//g, '//').replace(/[^\w\-]./, ' ');
                    context = friendlyView.charAt(0).toUpperCase() +
                        friendlyView.slice(1) + ' ';
                }
                if (uris && uris.length > 0) {
                    if (uris.length > 1) {
                        context += uris.length;
                    } else {
                        indexResult = resource.getIndexResultForUri(uris[0]);
                        if (indexResult && indexResult.name) {
                            context += indexResult.name;
                        }
                    }
                }
                banner.setDocumentTitle(context);
            }

            function onItemChange(item) {
                dispatcher.fire("itemChange", item);
                // Do we have this item in the index results?
                if (item) {
                    if(! resource.getIndexResultForUri(item.uri)) {
                        // We don't have an index result.
                        // This is likely due to the item being
                        // filtered out due to paging. Increase the page
                        // count until we find it.
                        pageToSelection = true;
                        // Either the index results or the item can
                        // return first. If the index results, have already arrived,
                        // get the next page.
                        if (! awaitingResults && resource.haveSome()) {
                            showMore();
                        }
                    } else {
                        // reset the flag if the item is located.
                        selectedItemNotFound = false;
                    }
                }
            }

            function onItemError(errorInfo) {
                dispatcher.fire("itemError", errorInfo);
                pageToSelection = false;
            }

            function onIndexResultsChanged(indexResults) {
                var selection,
                    haveSelection,
                    beingRemoved = false;

                if (routePrefix) {
                    setDocumentTitle();
                }
                dispatcher.fire("indexResultsChange", indexResults);

                if (selectFromResults) {
                    setSelectionFromResults(indexResults);
                    // we might not have selected any because none exist yet.
                    if (resource.getSelection().uris.length > 0) {
                        selectFromResults = false;
                    }
                } else if (pageToSelection || selectedItemNotFound) {
                    selection = resource.getSelection();
                    haveSelection = false;
                    $.each(selection.uris, function (index, uri) {
                        if (resource.getIndexResultForUri(uri)) {
                            haveSelection = true;
                            resource.getItem(uri, null, true);
                            return false;
                        }
                        else if (resource.isBeingRemoved(uri)) {
                            beingRemoved = true;
                        }
                    });
                    if (selection.uris.length > 0 && ! haveSelection) {
                        if (indexResults.total > indexResults.count) {
                            showMore();
                        } else {
                            // couldn't find it, report no item if not being removed
                            if (!beingRemoved) {
                                dispatcher.fire("itemError", {errorMessage : 'Not found'});
                            }
                            pageToSelection = false;
                            // Flag that RHS is displaying "item not found" error so that
                            // when the next index result arrives, this item should be
                            // retrieved again using getItem() call.
                            selectedItemNotFound = true;
                        }
                    }
                }
            }

            function onIndexResultsError(errorInfo) {
                dispatcher.fire("indexResultsError", errorInfo);
            }

            function onSelectionChanged(selection) {
                if (routePrefix) {
                    setDocumentTitle();
                }
                dispatcher.fire("selectionChange", selection);
                if (location) {
                    if (selection.uris.length > 0) {
                        router.go(urlFragment.replaceUris(location, selection.uris,
                            selection.multiSelectId),
                            "master selection change");
                    } else {
                        // remove any uris from location
                        router.replaceWith(urlFragment.replaceUris(location,
                            selection.uris),
                            "master selection empty change");
                    }
                }
            }

            function getReferrer(uri) {
                indexService.getIndexForResource(uri, {
                    success: function (resources) {
                        if (resources && resources.members && resources.members[0]) {
                            dispatcher.fire("referrerChange", resources.members[0]);
                        }
                    }
                });
            }

            function onFilterChange(filter) {
                var noFilter = true;
                var referrerUri = filter.getReferrerUri();
                if (referrerUri) {
                    getReferrer(referrerUri);
                    headerTitle =
                        localizer.getString('core.master.header.referenced');
                    noFilter = false;
                } else {
                    dispatcher.fire("referrerChange", undefined);
                }

                if (noFilter && filter.isCustom()) {
                    headerTitle =
                        localizer.getString('core.master.header.filtered');
                    noFilter = false;
                }

                if (noFilter) {
                    headerTitle =
                        localizer.getString('core.master.header.all');
                } else if (filter.name) {
                    headerTitle = filter.name;
                }
                notifyFilterChange = false;
                dispatcher.fire("filterChange", filter);
            }

            /**
             * When we show or edit, take care of managing the selection.
             * If the location has a uri, set the selection to it.
             * If it doesn't, choose the first result.
             */
            function onRouteChange(locationArg) {
                //log.log('!!! MasterPanePresenter('+routePrefix+'.onRouteChange - ' + locationArg);
                location = locationArg;
                var view = urlFragment.getView(location);
                var uris = urlFragment.getUris(location);
                var multiSelectId = parseInt(urlFragment.getMultiSelectId(location), 10);
                var noSelectionInRoute = ! (uris && uris.length > 0) && (! multiSelectId);
                var newFilter = new IndexFilter(location);
                var reload = false;
                // determine if we need to trigger showMore()
                var loadNextPage = false;
                selectFromResults = false;

                if (! preprocessRoute(newFilter, location, noSelectionInRoute)) {
                    return;
                }

                reload = resource.setIndexFilter(newFilter) || reloadIndexNeeded;
                reloadIndexNeeded = false;

                if (uris && uris.length > 0) {
                    // this is what the route location wants
                    resource.setSelectedUris(uris);
                    if (! reload) {
                        // see if we know about the selection already
                        var cached = $.grep(uris, function (uri) {
                            return resource.getIndexResultForUri(uri);
                        });
                        if (cached.length === 0) {
                            // Not able to find any item in index result. If there is not
                            // already an outstanding request to load the next page,
                            // trigger it here.
                            if ((! awaitingResults) && (resource.haveMore())) {
                                loadNextPage = true;
                            } else {
                                reload = true;
                            }
                        }
                    }
                }
                else if (multiSelectId) {
                    resource.selectMultiSelectId(multiSelectId);
                    reload = true;
                } else {
                    // no selection in location, load and select
                    resource.clearSelectedUris();
                    if ('add' !== view.split('/')[0]) {
                        selectFromResults = true;
                    }
                    reload = true;
                }

                setDocumentTitle();

                if (notifyFilterChange) {
                    // notify views of initial filter
                    notifyFilterChange = false;
                    dispatcher.fire("filterChange", newFilter);
                }

                if (loadNextPage) {
                    // can not find selection in the loaded index result; load more
                    pageToSelection = true;
                    showMore();
                } else if (reload) {
                    getIndexResults();
                }
            }

            this.init = function(resourceArg, routePrefixArg, authCategoryArg, contextSensitiveView) {
                routePrefix = routePrefixArg;

                // initialize resource
                resource = resourceArg;

                authCategory = authCategoryArg;

                headerTitle =
                    localizer.getString('core.master.header.all');

                if (routePrefix) {
                    // initialize route watching
                    router.watch(routePrefix + ' master route change',
                        '^' + routePrefix + ROUTE_SUFFIX, {change: onRouteChange});
                }

                globalView = !contextSensitiveView;
            };

            this.getSelection = function () {
                return resource.getSelection();
            };

            this.selectIndexResult = function (indexResult) {
                if (indexResult) {
                    resource.setSelectedUris([indexResult.uri]);
                }
            };

            this.toggleSelectIndexResult = function (indexResult) {
                var uris;
                if (indexResult) {
                    // remove if already there
                    uris = $.grep(resource.getSelection().uris, function (uri) {
                        return (uri !== indexResult.uri);
                    });
                    // if we didn't remove anything, add it
                    if (uris.length === resource.getSelection().uris.length) {
                        uris.push(indexResult.uri);
                    }
                    resource.setSelectedUris(uris);
                }
            };

            this.selectContiguousIndexResults = function (indexResult) {
                resource.selectContiguousToUri(indexResult.uri);
            };

            this.showMore = showMore;

            this.getFilterValue = function (name) {
                var filter = resource.getIndexFilter();
                var result = 'all';
                if (filter) {
                    var value = filter.getProperty(name);
                    if (value) {
                        result = value;
                    }
                }
                return result;
            };

            this.setFilterValue = function (name, value) {
                var filter = new IndexFilter(resource.getIndexFilter());
                if (value && 'all' !== value) {
                    if (filter.setProperty(name, value)) {
                        if (globalView) {
                            // clear uris on filter change
                            if(location)
                            	router.go(urlFragment.replaceUris(
                                	filter.updateLocation(location), []),
                                	"master " + name + " filter change");
                        } else if (resource.setIndexFilter(filter)) {
                            resource.getIndexResults();
                        }
                    }
                } else {
                    if (filter.unsetProperty(name)) {
                        if (globalView) {
                            filter.setResetFlag();
                            router.go(filter.updateLocation(location),
                                "master " + name + " filter change");
                        } else if (resource.setIndexFilter(filter)) {
                            resource.getIndexResults();
                        }
                    }
                }
            };

            this.resetFilters = function () {
                var filter = new IndexFilter(resource.getIndexFilter());
                filter.setResetFlag();

                if (globalView) {
                    filter.setProperties({});
                    router.go(filter.updateLocation(location), "master filter reset");
                } else if (resource.setIndexFilter(filter)) {
                    resource.getIndexResults();
                }
            };

            this.setSort = function (propertyName, direction) {
                var filter;
                if (location) {
                    filter = new IndexFilter(resource.getIndexFilter());
                    filter.setSort(propertyName, direction);
                    router.go(filter.updateLocation(location),
                        "master sort asc change");
                } else if (routePrefix) {
                    initialSort = {name: propertyName, direction: direction};
                } else {
                    filter = new IndexFilter(resource.getIndexFilter());
                    filter.setSort(propertyName, direction);
                    if (resource.setIndexFilter(filter)) {
                        getIndexResults();
                    }
                }
            };

            this.getSort = function () {
                var filter = resource.getIndexFilter();
                return (filter ? filter.getSort() : null);
            };

            this.getReferrerClearLocation = function () {
                return routePrefix + '/show';
            };

            this.getSearchText = function () {
                var filter = resource.getIndexFilter();
                return (filter ? filter.getUserQuery() : '');
            };

            this.getHeaderTitle = function () {
                return headerTitle;
            };

            this.getEmptyMessage = function () {
                var categoryLabel = linkTargetBuilder.
                    categoryLabel(resource.category).toLowerCase();
                if (resource.haveContacted()) {
                    return localizer.getString('core.master.noItems', [categoryLabel]);
                } else {
                    return localizer.getString('core.master.loadingItems', [categoryLabel]);
                }
            };

            /**
             * @public
             * Stop the timer polling of the index service.
             */
            this.pause = function () {
                timer = clearTimeout(timer);
                timer = null;
                location = null;
                resource.off('indexResultsChange', onIndexResultsChanged);
                resource.off('indexResultsError', onIndexResultsError);
                resource.off('itemChange', onItemChange);
                resource.off('itemError', onItemError);
                resource.off('selectionChange', onSelectionChanged);
                resource.off("filterChange", onFilterChange);
            };

            /**
             * @public
             * Resume the timer polling of the index service.
             */
            this.resume = function () {
                resource.on('indexResultsChange', onIndexResultsChanged);
                resource.on('indexResultsError', onIndexResultsError);
                resource.on('itemChange', onItemChange);
                resource.on('itemError', onItemError);
                resource.on('selectionChange', onSelectionChanged);
                resource.on("filterChange", onFilterChange);

                // reset filter count
                var filter = resource.getIndexFilter();
                if (filter) {
                    var newFilter = new IndexFilter(filter);
                    if (newFilter.resetCount()) {
                        resource.setIndexFilter(newFilter);
                        resource.renotify();
                    }
                }

                if (! routePrefix) {
                    getIndexResults();
                }

                restartTimer(true);
            };

            /**
             * Use by MasterPaneView to trigger renotifying views when
             * the view is changed.
             */
            this.renotify = function (sendResults) {
                resource.renotify(sendResults);
            };

            this.haveContacted = function () {
                return resource.haveContacted();
            };

            this.haveSome = function () {
                return resource.haveSome();
            };

            this.haveMore = function () {
                return resource.haveMore();
            };

            this.disableUpdates = function() {
                if(timer) {
                    clearTimeout(timer);
                    timer = null;
                }
            };

            this.enableUpdates = function() {
                if(!timer) {
                    restartTimer(true);
                }
            };

            /**
             * Add a listener for a specified event.
             * @param {string} eventName The name of the event.
             * @param {function(...)}
             */
            this.on = function (eventName, callback) {
                dispatcher.on(eventName, callback);
            };

            this.off = function (eventName, callback) {
                dispatcher.off(eventName, callback);
            };
        }

        return MasterPanePresenter;
    }());

    return MasterPanePresenter;
});
