// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
/**
 * @type {MasterPaneView}
 */
define(['hp/presenter/MasterPanePresenter',
    'hp/view/MasterTableView',
    'hp/view/MasterGridView',
    'hp/view/SearchBoxView',
    'hp/view/MasterErrorView',
    'hp/core/LinkTargetBuilder',
    'hp/core/Localizer',
    'hp/core/Style',
    'text!hpPages/core/master_sub_nav.html',
    'text!hpPages/core/master_header.html',
    'text!hpPages/core/no_details.html',
    'text!hpPages/core/no_items.html',
    'jquery',
    'hp/lib/jquery.hpSelect',
    'hp/lib/jquery.hpEllipsis',
    'hp/lib/jquery.hpCollapsible'],
function (MasterPanePresenter, MasterTableView, MasterGridView, searchBoxView, errorView,
    linkTargetBuilder, localizer, style,
    subnavHtml, headerHtml, noDetailsHtml, noItemsHtml) {
"use strict";

    var MasterPaneView = (function () {

        // consts can go here
        var MASTER_PANE = '.hp-master-pane';
        var DETAILS_PANE = '.hp-details-pane';
        var TABLE_WRAPPER = '.dataTables_wrapper';
        var TABLE = '.hp-master-table';
        var GRID = '.hp-master-grid';
        var TITLE = '.hp-master-header .hp-master-title';
        var HEADER = MASTER_PANE + ' .hp-master-header';
        var COUNT = '.hp-page-item-count';
        var FILTERBAR_CONTROL = '.hp-filterbar-control';
        var RESET_FILTERS = '.hp-filter-reset';
        var FILTERBAR = '.hp-master-filter';
        var FILTER = '.hp-filter[data-filter-property]';
        var SORTER = '#hp-master-sort';
        var EMPTY_MESSAGE = DETAILS_PANE + ' .hp-empty-message';
        var ACTIVE = 'hp-active';
        var FILTERBAR_OPEN = 'hp-filterbar-open';

        /**
         * @constructor
         */
        function MasterPaneView() {

            //
            // private class instance variables
            //
            var presenter = new MasterPanePresenter();
            var page = null;
            var globalView = true;
            var table = null;
            var grid = null;
            var active = false;
            var activeView = false;
            var itemOk = true;
            var noDetails = $(noDetailsHtml);
            var noItems = $(noItemsHtml);
            var relayoutTimer;

            function fireRelayout() {
                // we delay so any CSS animation can finish
                setTimeout(function () {
                    $(DETAILS_PANE, page).css('left',
                        $(MASTER_PANE, page).outerWidth());
                    $(page).trigger('relayout');
                }, style.animationDelay());
            }

            function resizeFilterbar() {
                if ($(page).hasClass(FILTERBAR_OPEN)) {
                    // adjust filter bar width based on contents
                    var filterWidth = 200;
                    $(FILTERBAR + ' > *', page).each(function (index, element) {
                        filterWidth = Math.max(filterWidth,
                            $(element).outerWidth(true));
                    });
                    $(FILTERBAR, page).css('width', filterWidth);
                    $(FILTERBAR, page).next().css('left', filterWidth + 15);
                    fireRelayout();
                }
            }

            function closeFilterbar() {
                $('.hp-sub-nav .hp-page-label', page).after(
                    $(FILTERBAR + ' .hp-filter', page).removeClass('hp-active hp-pinned'));
                $(FILTERBAR, page).css('width', '');
                $(FILTERBAR, page).next().css('left', '');
                $('.hp-sub-nav .hp-filter-reset').show();
                $(page).removeClass(FILTERBAR_OPEN);
                fireRelayout();
            }

            function openFilterbar() {
                $(page).addClass(FILTERBAR_OPEN);
                $('.hp-sub-nav .hp-filter-reset').hide();
                $(FILTERBAR, page).append(
                    $('.hp-sub-nav .hp-filter', page).addClass('hp-active hp-pinned'));
                resizeFilterbar();
            }

            function toggleFilterbar() {
                if ($(FILTERBAR, page).length > 0 &&
                    ! $(page).hasClass(FILTERBAR_OPEN)) {
                    openFilterbar();
                } else {
                    closeFilterbar();
                }
            }

            function resetFilters() {
                presenter.resetFilters();
            }

            function onReferrerChange(resource) {
                $('.hp-master-referrer', page).remove();
                if (resource) {
                    var elem = $('<div></div>').addClass('hp-master-referrer');
                    elem.append(
                        $(linkTargetBuilder.makeLink(resource.name, resource.uri)));
                    elem.append($('<a></a>').addClass('hp-close').
                        attr('href', '#' + presenter.getReferrerClearLocation()));
                    if (resource.name.length > 20) {
                        $('a:first-child', elem).hpEllipsis();
                    }
                    $('.hp-sub-nav', page).append(elem);
                }
            }

            function onPropertyFilterChange(ev, value) {
                var name = $(this).attr('data-filter-property');
                if (name) {
                    presenter.setFilterValue(name, value);
                }
            }

            function onSortChange(ev, value) {
                presenter.setSort(value, 'asc');
            }

            /**
             * @private
             * @param {Object} data the IndexResults
             */
            function onIndexResultsChange(indexResults) {
                if (page) {
                    $(MASTER_PANE + ' #hp-no-items', page).remove();
                    $(TITLE, page).text(presenter.getHeaderTitle());
                    if (indexResults) {
                        $(COUNT, page).text(indexResults.total);
                        $(EMPTY_MESSAGE, page).remove();
                        if (indexResults.total === 0) {
                            $(DETAILS_PANE, page).addClass('hp-empty').
                                append('<div class="hp-empty-message">' +
                                    presenter.getEmptyMessage() + '</div>');
                            $(MASTER_PANE, page).addClass('hp-empty');
                        } else {
                            if (itemOk) {
                                $(DETAILS_PANE, page).removeClass('hp-empty');
                                $(EMPTY_MESSAGE, page).remove();
                            }
                            $(MASTER_PANE, page).removeClass('hp-empty');
                        }
                    }
                }
            }

            function onIndexResultsError(errorInfo) {
                if (page) {
                    $(MASTER_PANE + ' #hp-no-items', page).remove();
                    $(MASTER_PANE, page).append(noItems);
                }
            }


            function onItemChange(item) {
                itemOk = true;
                if (page) {
                    $(DETAILS_PANE, page).removeClass('hp-empty');
                    $(EMPTY_MESSAGE, page).remove();
                    $(DETAILS_PANE + ' #hp-no-details', page).remove();
                }
            }

            function onItemError(errorInfo) {
                itemOk = false;
                if (page) {
                    $(DETAILS_PANE + ' #hp-no-details', page).remove();
                    $(DETAILS_PANE, page).addClass('hp-empty').append(noDetails);
                    if (errorInfo) {
                        $('.hp-collapsible', noDetails).show();
                        $('.hp-collapsible', noDetails).hpCollapsible();
                       // clear fields as we are reusing Dom elements
                        if (errorInfo.message) {
                            $('#nc-error-msg', noDetails).text(errorInfo.message);
                        } else {
                            $('#nc-error-msg', noDetails).text('');
                        }

                        if (errorInfo.details) {
                            $('#nc-error-details', noDetails).text(errorInfo.details);
                        } else {
                            $('#nc-error-details', noDetails).text('');
                        }
                        if (errorInfo.resolution) {
                            $('#nc-error-resolution', noDetails).text(errorInfo.recommendedActions.join('\n'));
                        } else {
                            $('#nc-error-resolution', noDetails).text('');
                        }
                    } else {
                        $('.hp-collapsible', noDetails).hide();
                    }
                }
            }


            function onSelectionChange(selection) {
                if (page && globalView) {
                    searchBoxView.setLocalSearchText(presenter.getSearchText());
                }
            }

            function onFilterChange(filter) {
                if (page) {
                    $(FILTER, page).each(function (index, elem) {
                        var name = $(elem).attr('data-filter-property');
                        var value = presenter.getFilterValue(name);
                        $(elem).hpSelect('set', value);
                    });

                    if (globalView) {
                        searchBoxView.setLocalSearchText(presenter.getSearchText());
                    }
                }
            }

            function onClickTable() {
                if (table !== activeView) {
                    if (grid === activeView) {
                        grid.pause();
                        $(GRID, page).removeClass(ACTIVE);
                        $('.hp-master-grid-view', page).removeClass(ACTIVE);
                    }
                    $('.hp-master-table-view', page).addClass(ACTIVE);
                    $(TABLE_WRAPPER, page).addClass(ACTIVE);
                    if (active) {
                        table.resume();
                    }
                    activeView = table;
                    presenter.renotify(true);
                }
            }

            function onClickGrid() {
                if (grid !== activeView) {
                    if (table === activeView) {
                        table.pause();
                        $(TABLE_WRAPPER, page).removeClass(ACTIVE);
                        $('.hp-master-table-view', page).removeClass(ACTIVE);
                    }
                    $('.hp-master-grid-view', page).addClass(ACTIVE);
                    $(GRID, page).addClass(ACTIVE);
                    if (active) {
                        grid.resume();
                    }
                    activeView = grid;
                    presenter.renotify(true);
                }
            }

            function onResizeDrag(event, ui) {
                var masterPane = $(MASTER_PANE, page);
                // what column sizes should we align to?
                var quanta = Math.floor($(window).width() / 12);
                var newWidth = Math.max(200, ui.size.width);
                newWidth = quanta * Math.floor((newWidth + (quanta / 2)) / quanta);

                masterPane.addClass('hp-resized');
                // adjust width since master pane is box sized for content-box
                // clear height that ui-resizable set
                masterPane.css({'width': newWidth  -
                    (masterPane.outerWidth() - masterPane.width()), 'height': ''});
                $(DETAILS_PANE, page).css('left', newWidth);
                // we've done the panes, re-layout the content when animation finishes
                clearTimeout(relayoutTimer);
                relayoutTimer = setTimeout(function () {
                    $(page).trigger('relayout');
                }, style.animationDelay());
            }

            this.enableUpdates = function () {
                presenter.enableUpdates();
            };

            this.disableUpdates = function () {
                presenter.disableUpdates();
            };

            /**
             * @public
             * Stop the timer polling of the index service.
             */
            this.pause = function () {
                if (active) {
                    active = false;
                    presenter.pause();
                    presenter.off("indexResultsChange", onIndexResultsChange);
                    presenter.off("indexResultsError", onIndexResultsError);
                    presenter.off("itemChange", onItemChange);
                    presenter.off("itemError", onItemError);
                    presenter.off("selectionChange", onSelectionChange);
                    presenter.off("referrerChange", onReferrerChange);
                    presenter.off("filterChange", onFilterChange);
                    if (activeView) {
                        activeView.pause();
                    }
                    errorView.pause();
                }
            };

            /**
             * @public
             * Resume the timer polling of the index service.
             */
            this.resume = function () {
                if (! active) {
                    active = true;
                    presenter.on("indexResultsChange", onIndexResultsChange);
                    presenter.on("indexResultsError", onIndexResultsError);
                    presenter.on("itemChange", onItemChange);
                    presenter.on("itemError", onItemError);
                    presenter.on("selectionChange", onSelectionChange);
                    presenter.on("referrerChange", onReferrerChange);
                    presenter.on("filterChange", onFilterChange);
                    if (activeView) {
                        activeView.resume();
                    }
                    // Resume the presenter after everyone is listening again.
                    // This ensures they hear about the latest state.
                    presenter.resume();
                    errorView.resume();

                    fireRelayout(); // in case window was resized while we were away
                    setTimeout(resizeFilterbar, 400);
                }
            };

            /**
             * @public
             * Intialize the view.
             */
            this.init = function (args) {
                var categoryLabel;

                page = args.page;

                localizer.localizeDom(noDetails);
                localizer.localizeDom(noItems);

                categoryLabel = linkTargetBuilder.categoryLabel(args.resource.category).
                    toLowerCase();
                $('h1', noItems).
                    text(localizer.getString('core.noItems.header', [categoryLabel]));

                presenter.init(args.resource, args.routePrefix, args.authCategory, args.contextSensitive);
                errorView.init(presenter);

                if (page) {
                    globalView = !(args.contextSensitive);
                    if (globalView && $('.hp-sub-nav', page).length === 0) {
                        $(page).prepend($(subnavHtml));
                    }

                    $('.hp-sub-nav .hp-select.hp-filter', page).hpSelect();
                    $('.hp-master-filter .hp-select.hp-filter', page).hpSelect();
                    $(FILTERBAR_CONTROL, page).on('click', toggleFilterbar);
                    $(RESET_FILTERS, page).on('click', resetFilters);

                    if (! $('.hp-master-filter header ' + RESET_FILTERS, page).hasClass('hp-button')) {
                        // convert master header reset to button
                        $('.hp-master-filter header ' + RESET_FILTERS, page).addClass('hp-button');
                    }
                    $(FILTER, page).change(onPropertyFilterChange);

                    $(SORTER, page).hpSelect();
                    $(SORTER, page).change(onSortChange);
                    $(DETAILS_PANE, page).addClass('hp-empty');

                    if (globalView && $(HEADER, page).length === 0) {
                        $('.hp-master-pane', page).prepend($(headerHtml));
                    }
                    if (args.addLink) {
                        $(HEADER + ' .hp-master-add', page).
                            append($(args.addLink).addClass('hp-button hp-secondary'));
                    }

                    // Allow resizing the master pane if not full
                    if (globalView && $('.hp-master-full', page).length === 0) {
                        $('.hp-master-pane', page).resizable({handles : 'e',
                            resize: onResizeDrag});
                    }
                }

                if (args.dataTableOptions &&
                    (args.table || (page && $(TABLE, page).length > 0))) {
                    table = new MasterTableView();
                    table.init(presenter, args);
                }

                if (args.grid || (page && $(GRID, page).length > 0)) {
                    grid = new MasterGridView();
                    grid.init(presenter, args);
                }

                if (page && table && grid) {
                    $('.hp-master-view', page).addClass(ACTIVE);
                    $('.hp-master-table-view', page).click(onClickTable);
                    $('.hp-master-grid-view', page).click(onClickGrid);
                    onClickTable();
                } else if (table) {
                    onClickTable();
                } else if (grid) {
                    onClickGrid();
                }

                this.resume();
            };
        }

        return MasterPaneView;

    }());

    return MasterPaneView;
});
