// (C) Copyright 2012 Hewlett-Packard Development Company, L.P.
/**
 * @constructor
 * @type {NotificationsView}
 */
define(['hp/core/Notifications',
    'hp/presenter/NotificationsPresenter',
    'hp/core/Style',
    'text!hpPages/core/details_notifications.html',
    'jquery',
    'hp/lib/date',
    'hp/lib/jquery.hpStatus',
    'hp/lib/jquery.hpProgressBar'],
function(bannerNotifications, NotificationsPresenter, style, notificationsHtml) {"use strict";

    var NotificationsView = (function() {

        var CONTROLS = '.hp-controls';
        var BODY = '#hp-body-div';
        var AVAILABLE = 'hp-available';
        var ACTIVE = 'hp-active';
        var OPEN = 'hp-notify-open';
        var UP = 38;
        var DOWN = 40;
        var SPACE = 32;
        var TAB = 9;
        var ESCAPE = 27;
        var STATUSES = ['error', 'warning', 'ok'];

        /**
         * Constructor
         */
        function NotificationsView() {

            var presenter = null;
            var container = null;
            var template = null;
            var subTemplate = null;
            var notifications = [];
            var selectedIndex = -1;
            var selecting = false;
            var layoutTimer;
            var summaryLayoutTimer = null;
            
            function fireRelayout() {
                // we don't delay here because there's no animation involved.
                container.parents('.hp-page').trigger('relayout');
            }
            
            function onMouseMove() {
                selecting = true;
            }
            
            function onMouseUp() {
                $(BODY).off('mousemove', onMouseMove);
                $(BODY).off('mouseup', onMouseUp);
            }
            
            function onMouseDown() {
                $(BODY).on('mousemove', onMouseMove);
                $(BODY).on('mouseup', onMouseUp);
            }
            
            var onKeypress = function() {};

            function deactivate(event) {
                if (! selecting) {
                    if (! event || $(event.target).parents('.hp-selector').length === 0) {
                        $(BODY).off('click', deactivate);
                        $(container).off('mousedown', onMouseDown);
                        $(document).off('keydown', onKeypress);
                        $(container).removeClass(ACTIVE);
                    }
                } else {
                    selecting = false;
                }
            }

            function activate() {
                if (! $(container).hasClass(ACTIVE)) {
                    $(container).addClass(ACTIVE);
                    $(document).on('keydown', onKeypress);
                    // delay to avoid flickering
                    setTimeout(function () {
                        $(container).on('mousedown', onMouseDown);
                        $(BODY).on('click', deactivate);}, 50);
                }
            }
            
            function layout() {
                var parent = $('.hp-notification', container).parent();
                var notificationWidth = parent.innerWidth() - $(CONTROLS, container).outerWidth();
                var message = $('.hp-notification .hp-notification-summary .hp-message', container);

                // set notification width to avoid overlap with controls
                $('.hp-notification', container).css('width', notificationWidth - 1);
                // set message max-width so ellipsis works, 10 is for padding
                message.css('max-width',  $(CONTROLS, container).offset().left - message.offset().left - 10);
                // set parent min-width to avoid getting too small, 50/50
                parent.css('min-width', $(CONTROLS, container).outerWidth() * 2);
            }
            
            function onResize(event) {
                if (event.target == window) {
                    clearTimeout(layoutTimer);
                    layoutTimer = setTimeout(layout, 50);
                }
            }
            
            function hide() {
                $(container.removeClass(AVAILABLE));
                $(container).parent().removeClass(OPEN);
                fireRelayout();
            }
            
            function show() {
                $(container.addClass(AVAILABLE));
                $(container).parent().addClass(OPEN);
                fireRelayout();
            }
            
            function updateSummary() {
                var counts = {};
                $.each(STATUSES, function (index, status) {
                    counts[status] = 0;
                });
                $.each(notifications, function (index, notif) {
                    counts[notif.status] += 1;
                });
                $.each(STATUSES, function (index, status) {
                    $('.hp-summary-' + status + ' .hp-count', container).
                        text(counts[status]);
                    if (counts[status] > 0) {
                        $('.hp-summary-' + status + ' > .hp-status', container).
                            removeClass('hp-unset');
                    } else {
                        $('.hp-summary-' + status + ' > .hp-status', container).
                            addClass('hp-unset');
                    }
                });
                $(CONTROLS, container).toggle(notifications.length > 1);
                if (!summaryLayoutTimer) {
                    summaryLayoutTimer = setTimeout(function() {
                        summaryLayoutTimer = null;
                        layout();
                    }, 50);
                }
            }
            
            function resetSelector() {
                $('.hp-selector li').remove();
                $.each(notifications, function (index, notif) {
                    var selectorItem = template.clone();
                    if (notif.status) {
                        $('.hp-status', selectorItem).hpStatus(notif.status);
                    }
                    if (notif.timestamp) {
                        $('.hp-timestamp', selectorItem).
                            text(style.beautifyTimestamp(notif.timestamp));
                    }
                    $('.hp-message', selectorItem).text(notif.summary);
                    $('.hp-selector', container).append(selectorItem);
                });
                if (selectedIndex >= 0) {
                    $('.hp-selector li:nth-child(' + (selectedIndex + 1) + ')').
                        addClass('hp-selected');
                }
            }
            
            function updateItemSummary(notif, summary) {
                if (notif.status) {
                    $('> .hp-status', summary).
                        hpStatus(notif.status, (notif.changing && !notif.progress));
                } else {
                    $('> .hp-status', summary).
                        attr('class', 'hp-status').text('');
                }
                // Don't replace if it hasn't changed,
                // allows user selection to persist.
                if ($('.hp-message', summary).text() !== notif.summary) {
                    $('.hp-message', summary).text(notif.summary);
                }
                if (notif.timestamp) {
                    $('.hp-timestamp', summary).show().
                        text(style.beautifyTimestamp(notif.timestamp));
                } else {
                    $('.hp-timestamp', summary).text('').hide();
                }
                if (false && notif.sourceName) {
                    if (notif.sourceUri) {
                        $('.hp-source', summary).show().
                            attr('href', notif.sourceUri).
                            text(notif.sourceName);
                    } else {
                        $('.hp-source-name', summary).show().
                            text(notif.sourceName);
                    }
                } else {
                    $('.hp-source', summary).text('').hide();
                }
                if (notif.progress) {
                    $('.hp-progress', summary).
                        hpProgressBar(notif.progress).show();
                } else {
                    $('.hp-progress', summary).hide();
                }
                if (notif.step) {
                    $('.hp-step', summary).show().text(notif.step);
                } else {
                    $('.hp-step', summary).text('').hide();
                }
            }

            function updateItem() {
                var notif = notifications[selectedIndex];
                var item = $('.hp-notification', container);
                // don't change while the user is selecting
                if (! selecting && notif) {
                  
                    $('.hp-selector li', container).removeClass('hp-selected');
                    $('.hp-selector li:nth-child(' + (selectedIndex + 1) + ')').
                        addClass('hp-selected');
                    
                    updateItemSummary(notif, $('header.hp-notification-summary', item));
                    
                    $('.hp-sub-notifications').empty();
                    if (notif.children) {
                        $.each(notif.children, function (index, child) {
                            var childItem = subTemplate.clone();
                            updateItemSummary(child, childItem);
                            $('.hp-sub-notifications').append(childItem);
                        });
                        $('.hp-sub-notifications', container).show();
                    } else {
                        $('.hp-sub-notifications', container).hide();
                    }
                    if (notif.details) {
                        // Don't replace if it hasn't changed,
                        // allows user selection to persist.
                        if ($('.hp-details', item).html() !==
                            notif.details) {
                            $('.hp-details', container).
                                html(notif.details).removeClass('hp-unset');
                        }
                    }
                    if (notif.resolution) {
                        if ($('.hp-resolution', item).text() !== notif.resolution) {
                            $('.hp-resolution', item).text(notif.resolution);
                        }
                        $('.hp-resolution-container', item).show();
                    } else {
                        $('.hp-resolution-container', item).hide();
                    }
                    $('.hp-actions', item).empty();
                    if (notif.actions) {
                        $.each(notif.actions, function (index, action) {
                            $('.hp-actions', item).
                                append($(action).wrap('<li></li>').parent());
                        });
                    }
                    
                    if (! notif.details && ! notif.resolution &&
                        (! notif.actions || notif.actions.length === 0)) {
                        $('.hp-details', item).
                            text('No details').addClass('hp-unset');
                    }
                    
                    if (notif.dismiss) {
                        $('.hp-close', item).addClass(ACTIVE);
                    } else {
                        $('.hp-close', item).removeClass(ACTIVE);
                    }
                }
            }
            
            onKeypress = function(ev) {
                var keyCode = (ev.which ? ev.which : ev.keyCode);
                if (keyCode == SPACE) {
                    activate();
                } else if (keyCode == ESCAPE || keyCode == TAB) {
                    deactivate();
                } else if (keyCode == UP) {
                    selectedIndex = Math.max(0, selectedIndex - 1);
                    updateItem();
                    ev.preventDefault();
                } else if (keyCode == DOWN) {
                    selectedIndex = Math.min(notifications.length - 1, selectedIndex + 1);
                    updateItem();
                    ev.preventDefault();
                }
            };
            
            function onDismiss(ev) {
                var notif = notifications[selectedIndex];
                if (notif.dismiss) {
                    notifications.splice(selectedIndex, 1);
                    updateItem(true);
                    notif.dismiss(notif);
                }
                ev.stopPropagation();
            }
            
            function onSelect(ev) {
                var item = $(ev.currentTarget);
                selectedIndex = item.index();
                updateItem();
            }

            function sortNotifications(a, b) {
                // sort first by severity, then by timestamp
                // TODO: sort tasks correctly
                if (true || a.status === b.status) {
                    return ((a.timestamp > b.timestamp) ? -1 :
                        ((a.timestamp < b.timestamp) ? 1 : 0));
                } /* else {
                    if (a.status === 'error') return -1;
                    if (b.status === 'error') return 1;
                    if (a.status === 'warning') return -1;
                    if (b.status === 'warning') return 1;
                } */
            }
            
            function add(notif, expand) {
                if (! notif.hasOwnProperty('timestamp')) {
                    notif.timestamp = (new Date()).toISOString();
                }
                // replace if we already have it
                var existingIndex = -1;
                $.each(notifications, function (index, notif2) {
                    if (notif2.uri === notif.uri) {
                        existingIndex = index;
                        return false;
                    }
                });
                
                if (existingIndex >= 0) {
                    notifications[existingIndex] = notif;
                } else {
                    notifications.push(notif);
                }
                notifications.sort(sortNotifications);

                if (selectedIndex < 0) {
                    selectedIndex = 0;
                }
                updateItem();
                updateSummary();
                resetSelector();
                
                if (! container.hasClass(AVAILABLE)) {
                    show();
                }
                
                if (expand) {
                    activate();
                }
            }
            
            function remove(notif) {
                var existingIndex = -1;
                $.each(notifications, function (index, notif2) {
                    if (notif2.uri === notif.uri ||
                        (! notif.uri && ! notif2.uri &&
                        notif.timestamp === notif2.timestamp &&
                        notif.summary === notif2.summary)) {
                        
                        existingIndex = index;
                        return false;
                    }
                });
                if (existingIndex >= 0) {
                    notifications.splice(existingIndex, 1);
                    updateItem();
                    updateSummary();
                    resetSelector();
                }
            }
            
            function onNotificationsChange(notifications) {
                $.each(notifications, function (index, notif) {
                    add(notif, false);
                });
            }

            /**
             * @public
             * Add a notification message.
             * @param {notif} Object with properties as follows:
             *
             *    status: one of 'ok', 'error', 'warning', 'info'
             *    summary: minimal text summary, should start with an action if possible
             *    uri: identifier for this alert, task, etc.
             *          non-server-side notifications can use anything they want,
             *          this is used to allow updating progress of an existing
             *          notification.
             *    timestamp: timestamp in ISO 8601 string format
             *    sourceName: user visible name for the source
             *    sourceUri: uri for the source
             *    changing: true|false, whether the status should be anotated as changing
             *    progress: 0-100, a number indication what % of a task has been completed
             *    step: a short text description of the current progress step
             *    details: longer description that can have line breaks and links,
             *              will show up in a tooltip/flyout
             *    resolution: multi-line recommendation on how the user should proceed
             *    actions: Array of action links
             *    dismiss: Function called when the user dismisses
             *      function (notif){}
             *    children: child notif objects. Typically used for sub-tasks
             *
             *    The only property that is required is the summary.
             *
             * @param expand Boolean expand any details right away
             * @param addActivity Boolean add to banner
             * @param flashBanner Boolean flash in banner
             *
             */
            this.add = function (notif, expand, addActivity, flashBanner) {

                add(notif, expand);
                
                if (addActivity) {
                    bannerNotifications.add(notif, flashBanner);
                }
            };

            /**
             * @public
             * Remove a notification message
             * @param {notif} Object with similar fields to the add().
             *    These fields are used in this order to find one to match:
             *      uri, timestamp + summary
             */
            this.remove = function (notif) {
                remove(notif);
            };

            this.clear = function () {
                // avoid bouncing when changing selected resource
                setTimeout(function () {
                    if (selectedIndex < 0) {
                        hide();
                    }
                }, 200);
                notifications = [];
                selectedIndex = -1;
                updateSummary();
            };

            /**
             * @public
             */
            this.init = function (context, allLocation) {
                var options;
                if (context.hasOwnProperty('length')) {
                    options = {context : context, allLocation: allLocation};
                } else {
                    options = context;
                }
                
                if (options.hasOwnProperty('resource')) {
                    presenter = new NotificationsPresenter();
                    presenter.init(options.resource);
                }
                
                container = $(notificationsHtml);

                // insert HTML in context
                if ($('.hp-details-header', options.context).length > 0) {
                    $('.hp-details-header', options.context).after(container);
                } else if ($('.hp-master-header', options.context).length > 0) {
                    $('.hp-master-header', options.context).after(container);
                } else {
                    $('> header', options.context).after(container);
                }
                template = $('.hp-selector .hp-template', container).remove().
                    removeClass('hp-template');
                subTemplate = $('.hp-sub-notifications .hp-template', container).remove().
                    removeClass('hp-template');

                $(container).on('click', activate);
                if (options.allLocation) {
                    $('.hp-view-all', container).attr('href', options.allLocation);
                } else {
                    $('.hp-view-all', container).hide();
                }
                $(container).on('click', '.hp-close', onDismiss);
                $(container).on('click', '.hp-selector li', onSelect);
            };
            
            this.pause = function () {
                $('.hp-page').off('relayout', layout);
                $(window).off('resize', onResize);
                if (presenter) {
                    presenter.pause();
                    presenter.off('notificationsChange', onNotificationsChange);
                }
            };
            
            this.resume = function () {
                $('.hp-page').on('relayout', layout);
                $(window).on('resize', onResize);
                if (presenter) {
                    presenter.on('notificationsChange', onNotificationsChange);
                    presenter.resume();
                }
            };
        }

        return NotificationsView;
    }());

    return NotificationsView;
});
