// (C) Copyright 2011-2012 Hewlett-Packard Development Company, L.P.
define(['hp/presenter/SearchBoxPresenter',
        'hp/services/Log',
        'hp/core/Localizer',
        'jquery'], 
function(presenter, log, localizer) {"use strict";

    var SearchBoxView = ( function() {

        var CONTROL = '#hp-search-control';
        var MENU = '#hp-search-menu';
        var HEADER = MENU + ' > header';
        var INPUT = '#hp-search-input input';
        var CLEAR = HEADER + ' #hp-search-clear';
        var SEARCH_SCOPE = '#hp-search-scope';
        var CHOICES = '#hp-search-choices';
        var SUGGESTIONS = MENU + ' #hp-search-suggestions';
        var SUGGESTION = SUGGESTIONS + ' li';
        var RECENTS = MENU + ' #hp-search-recent';
        var RECENT = RECENTS + ' li';
        var TEMPLATE = MENU + ' #hp-search-item-template';
        var BODY = '#hp-body-div';
        var ACTIVE = "hp-active";
        var SELECTED = "hp-selected";
        var ENTER = 13;
        var ESCAPE = 27;
        var UP_ARROW = 38;
        var DOWN_ARROW = 40;
        
        /**
         * @constructor
         * @type {SearchBoxView}
         */
        function SearchBoxView() {

            var suggestionList = [];
            var suggestionIndex = -1;
            var banner = null;
            var template;
            
            function clearSuggestions() {
                $(SUGGESTIONS, banner).empty();
                suggestionIndex = -1;
                suggestionList = [];
                $(CHOICES, banner).hide();
            }
            
            function chooseRecent(ev) {
                var query = $(this).text();
                $(INPUT).val(query);
                doSearch(query);
                hideMenu();
            }
            
            function addRecents() {
                var recents = presenter.recents();
                var item;
                $(RECENTS, banner).empty();
                $.each(recents, function(index, recent) {
                    item = template.clone();
                    item.attr('id', '').text(recent);
                    item.show();
                    $(RECENTS, banner).append(item);
                });
                $(CHOICES, banner).show();
                $(RECENT, banner).click(chooseRecent);
            }
            
            function hideMenu(ev) {
                var value;
                // don't hide when header is clicked
                if (! ev || $(ev.target).parents(MENU).length === 0) {
                    $(BODY).unbind('click', hideMenu);
                    $(INPUT, banner).blur();
                    $(MENU, banner).removeClass(ACTIVE);
                    clearSuggestions();
                    
                    value = $(INPUT).val();
                    if (value && value.length > 0) {
                        $(CONTROL, banner).text(value);
                    } else {
                        $(CONTROL, banner).text(
                            localizer.getString('search.title'));
                    }
                }
            }
            
            function showMenu(selectText) {
                $(MENU, banner).addClass(ACTIVE);
                $(INPUT, banner).focus();
                if (selectText) {
                    $(INPUT, banner).select();
                }
                $(CLEAR, banner).toggle($(INPUT).val().length > 0);
                clearSuggestions();
                addRecents();
                // delay to avoid flickering
                setTimeout(function () {$(BODY).bind('click', hideMenu);}, 50);
            }
            
            function doSearch(query) {
                presenter.doSearch(query);
            }
            
            function updateFromSuggestion() {
                $(SUGGESTION).removeClass(SELECTED);
                if (suggestionIndex >= 0) {
                    $(INPUT).val(suggestionList[suggestionIndex]);
                    // highlight selected one
                    $(SUGGESTION + ':eq(' + suggestionIndex + ')').
                        addClass(SELECTED);
                }
            }
            
            function chooseSuggestion(ev) {
                var query = $(this).text();
                $(INPUT).val(query);
                doSearch(query);
                hideMenu();
            }
            
            function addSuggestions(data) {
                var item;
                suggestionList = data.suggestions;
                $.each(data.suggestions, function(index, suggestion) {
                    item = template.clone();
                    item.attr('id', '');
                    item.text(suggestion);
                    item.show();
                    $(SUGGESTIONS, banner).append(item);
                });
                $(CHOICES, banner).show();
                $(SUGGESTION, banner).click(chooseSuggestion);
            }
            
            function onSearchSuggestionChange(data) {
                $(SUGGESTIONS, banner).empty();
                addSuggestions(data);
                suggestionIndex = -1;
                updateFromSuggestion();
            }
            
            function onKeyDown(ev) {
                var query = $(INPUT).val();
                if (ENTER === ev.which &&
                    (query.length > 0 || ! presenter.isGlobal())) {
                    hideMenu();
                    doSearch(query);
                }
            }
            
            function onKeyUp(e) {
                var keyCode = (e.which ? e.which : e.keyCode);
                if (keyCode == ENTER) {
                    // enter key handled by keydown
                } else if (keyCode == ESCAPE) {
                    hideMenu();
                    //$(INPUT, banner).select();
                } else if (keyCode == DOWN_ARROW) {
                    suggestionIndex = suggestionIndex + 1;
                    if (suggestionIndex >= suggestionList.length) {
                        suggestionIndex = suggestionList.length - 1;
                    }
                    updateFromSuggestion();
                } else if (keyCode == UP_ARROW) {
                    suggestionIndex = suggestionIndex - 1;
                    if (suggestionIndex < -1) {
                        suggestionIndex = -1;
                    }
                    updateFromSuggestion();
                } else {
                    var query = $(INPUT).val();
                    if (query.length > 0) {
                        presenter.suggestions(query, 0, 10);
                    } else {
                        clearSuggestions();
                    }
                    $(CLEAR, banner).toggle(query.length > 0);
                }
            }
            
            function onClear() {
                $(INPUT, banner).val('');
                clearSuggestions();
                doSearch($(INPUT).val());
                $(CLEAR, banner).hide();
                $(INPUT, banner).focus();
            }

            /**
             * @public
             */
            this.init = function(bannerArg) {
                banner = bannerArg;
                template = $(TEMPLATE, banner);
                template.removeAttr('id').detach().removeClass('hp-template');
                
                presenter.init();
                presenter.on('searchSuggestionChange', onSearchSuggestionChange);
                
                $('#hp-search-scope-local', banner).attr('checked', true);
                
                $(SEARCH_SCOPE + ' input', banner).click(function (ev) {
                    presenter.setGlobal($(this).attr('id') === 'hp-search-scope-global');
                    var query = $(INPUT).val();
                    if (query.length > 0) {
                        presenter.suggestions(query, 0, 10);
                    }
                    addRecents();
                    $(INPUT, banner).focus();
                });
                
                $(CONTROL, banner).click(function (ev) {
                    // adjust banner search width on right depending upon the icons
                    $(MENU, banner).css('right', $('#hp-main-banner .hp-header-secondary').outerWidth());
                    if (! $(MENU, banner).hasClass(ACTIVE)) {
                        showMenu();
                    }
                });
                $(CLEAR, banner).click(onClear);
                
                var input = $(INPUT, banner);
                input.keyup(onKeyUp);
                input.keydown(onKeyDown);
                //input.focusout(hideMenu);
            };
            
            // used by SearchView to focus on resume()
            this.activate = function (selectText) {
                showMenu(selectText);
            };
            
            this.setCategory = function (category, label) {
                if (presenter.setCategory(category)) {
                    // setCategory() returns true if the category changed.
                    // we only want to reset the search if we change pages
                    
                    $(CONTROL, banner).text(
                        localizer.getString('search.title'));
                    $(INPUT, banner).val('');
                    
                    if (category) {
                        $('label[for="hp-search-scope-local"]', banner).text(label);
                        $(SEARCH_SCOPE, banner).show();
                        presenter.setGlobal(false);
                    } else {
                        $(SEARCH_SCOPE, banner).hide();
                        presenter.setGlobal(true);
                    }
                }
            };
            
            this.setLocal = function (localLabel) {
                if (localLabel) {
                    $("label[for='hp-search-scope-local']", banner).text(localLabel);
                    $(SEARCH_SCOPE, banner).show();
                    
                    presenter.setGlobal(false);
                    $(CONTROL, banner).text(
                        localizer.getString('search.title'));
                    $(INPUT, banner).val('');
                }
            };
            
            this.setLocalSearchText = function (query) {
                presenter.setGlobal(false);
                $('#hp-search-scope-local', banner).attr('checked', true);
                if (query) {
                    $(CONTROL, banner).text(query);
                    $(INPUT, banner).val(query);
                } else {
                    $(CONTROL, banner).text(
                        localizer.getString('search.title'));
                    $(INPUT, banner).val('');
                }
            };
            
            this.setGlobalSearchText = function (query) {
                if (query) {
                    presenter.setGlobal(true);
                    $('#hp-search-scope-global', banner).attr('checked', true);
                    $(CONTROL, banner).text(query);
                    $(INPUT, banner).val(query);
                }
            };
        }

        return new SearchBoxView();
    }());

    return SearchBoxView;
});
