#!/bin/bash

# (C) 2020 Intel Corporation. All rights reserved.
# Your use of Intel Corporation's design tools, logic functions and other
# software and tools, and its AMPP partner logic functions, and any output
# files any of the foregoing (including device programming or simulation
# files), and any associated documentation or information are expressly subject
# to the terms and conditions of the Intel Program License Subscription
# Agreement, Intel MegaCore Function License Agreement, or other applicable
# license agreement, including, without limitation, that your use is for the
# sole purpose of programming logic devices manufactured by Intel and sold by
# Intel or its authorized distributors.  Please refer to the applicable
# agreement for further details.


# This script configure host OS to allow OpenCL workload to run successful on
# the host.  There are 3 settings that need to be changed from the default
# PAC configuration.
#
# 1) Enable users to lock pages in memory:
#
# 2) Change intel-fpga device permissions:
#
# 3) Change number of 2M huge pages:

################################################################################
# Utility function for checking if operation was successful
################################################################################
check_res()
{
  ret_code="$1"
  msg="$2"
  if [ "$ret_code" -ne 0 ]; then
    echo "Error: $msg"
    exit "$ret_code"
  fi
}

################################################################################
# Configure memlock settings to allow any user to lock pages in memory
#
# By default in Linux ordinary users can only lock 64 KB of virtual memory
# in the host to prevent OS from paging the memory to disk.  The OPAE
# software stack requires locking pages in memory that are shared between
# the host and FPGA.  The default configuration set in this script allows
# users to lock ulimited number of pages in memory (up-to amount of physical)
# memory in host.
################################################################################
set_memlock_limits()
{
  echo "Configuring locked memory setting"
  sudo sh -c "cat > /etc/security/limits.d/90-intel-fpga-opencl-limits.conf" << EOF
*	soft	memlock	unlimited
*	hard	memlock	unlimited
EOF

  check_res $? "Error: unable to enable memlock settings"

  lock_limit=$(ulimit -l)
  if [ $lock_limit != "unlimited" ]; then
    echo "** NOTE:changes to max locked memory setting only take effet with new login session"
  fi
}

################################################################################
# Set device permissions to allow all users to perform PR
#
# By default the intel-fpga driver restricts management functions to require
# root privilege.  However, OpenCL expects that ordinary users can perform
# partial reconfiguration (PR) to load a bitstream when the application starts
# executing.  This changes the device permissions to allow any user to access
# management features that are needed to perform PR.
################################################################################
set_dev_permission()
{
  echo "Configuring udev rules for intel-fpga device permission"
  sudo sh -c "cat > /etc/udev/rules.d/90-intel-fpga-opencl.rules" << EOF
# Set device permission to allow regular user to perform PR, required for OpenCL

# A10 and S10 FME permisison
KERNEL=="intel-fpga-fme.[0-9]*", ACTION=="add|change", GROUP="root", MODE="0666"

# A10 port permission
KERNEL=="intel-fpga-port.[0-9]*" ATTRS{vendor}=="0x8086" ATTRS{device}=="0x09c4", ACTION=="add|change", GROUP="root", MODE="0666", RUN+="/bin/bash -c 'chmod 0666 %S%p/errors/clear %S%p/userclk_freqcntrcmd %S%p/userclk_freqcmd'"

# S10 port permission
KERNEL=="intel-fpga-port.[0-9]*" ATTRS{vendor}=="0x8086" ATTRS{device}=="0x0b2b", ACTION=="add|change", GROUP="root", MODE="0666", RUN+="/bin/bash -c 'chmod 0666 %S%p/errors/clear'"
KERNEL=="intel-pac-iopll.[0-9]*.auto" ATTRS{vendor}=="0x8086" ATTRS{device}=="0x0b2b", ACTION=="add|change", RUN+="/bin/bash -c 'chmod 0666 %S%p/userclk/frequency"
EOF

  check_res $? "Error: unable to configure intel-fpga device permission"

  sudo udevadm control --reload-rules
  sudo udevadm trigger --name /dev/intel-fpga-fme.*
  sudo udevadm trigger --name /dev/intel-fpga-port.*

}

################################################################################
# Configure 2M huge pages
#
# Hugepages are used to improve performance of the DMA operations between host
# and FPGA.  The PAC BSP requires a minimum of 4 2M hugepages per FPGA.
# However, there is additional optimization that allows user to pre-pin memory
# using hugepages.  The default setting in this script configures 2048 2M
# hugepages.  This means 4G of host memory is allocated for hugepages.
# This can be changed by configuring PAC_BSP_ENV_NUM_HUGEPAGES before running
# this script.
################################################################################
set_hugepages()
{
  NUM_HUGE_PAGES=${PAC_BSP_ENV_NUM_HUGEPAGES:-2048}
  echo "Configuring system with $NUM_HUGE_PAGES 2M hugepages"
  sudo sh -c "cat > /etc/sysctl.d/intel-fpga-opencl-sysctl.conf" << EOF
vm.nr_hugepages = $NUM_HUGE_PAGES
EOF

  check_res $? "Error: unable to configure hugepages"
  sudo sh -c "echo $NUM_HUGE_PAGES > /sys/kernel/mm/hugepages/hugepages-2048kB/nr_hugepages"
}


set_memlock_limits
set_dev_permission
set_hugepages

echo "Finished setup_permissions.sh script. All configuration settings are persistent."
