package require -exact qsys 17.0

# module properties
set_module_property DESCRIPTION "Pipelined clock crossing splitter"
set_module_property NAME mem_splitter
set_module_property VERSION 21.3
set_module_property GROUP "ACL Internal Components"
set_module_property AUTHOR "Altera OpenCL"
set_module_property DISPLAY_NAME "ACL Bank Splitter w. Reorder"
set_module_property INSTANTIATE_IN_SYSTEM_MODULE true
set_module_property EDITABLE false
set_module_property ELABORATION_CALLBACK elaborate
set_module_property VALIDATION_CALLBACK validate

# files
add_fileset          synth QUARTUS_SYNTH
set_fileset_property synth TOP_LEVEL mem_splitter
add_fileset_file     mem_splitter.sv SYSTEM_VERILOG PATH mem_splitter.sv
add_fileset_file     mem_router_reorder.sv SYSTEM_VERILOG PATH mem_router_reorder.sv
add_fileset_file     acl_iface_ll_fifo.sv SYSTEM_VERILOG PATH acl_iface_ll_fifo.sv
add_fileset_file     acl_iface_address_to_bankaddress.sv SYSTEM_VERILOG PATH acl_iface_address_to_bankaddress.sv

add_fileset          sim SIM_VERILOG
set_fileset_property sim TOP_LEVEL mem_splitter
add_fileset_file     mem_splitter.sv SYSTEM_VERILOG PATH mem_splitter.sv
add_fileset_file     mem_router_reorder.sv SYSTEM_VERILOG PATH mem_router_reorder.sv
add_fileset_file     acl_iface_ll_fifo.sv SYSTEM_VERILOG PATH acl_iface_ll_fifo.sv
add_fileset_file     acl_iface_address_to_bankaddress.sv SYSTEM_VERILOG PATH acl_iface_address_to_bankaddress.sv

add_fileset          simvhdl SIM_VHDL
set_fileset_property simvhdl TOP_LEVEL mem_splitter
add_fileset_file     mem_splitter.sv SYSTEM_VERILOG PATH mem_splitter.sv
add_fileset_file     mem_router_reorder.sv SYSTEM_VERILOG PATH mem_router_reorder.sv
add_fileset_file     acl_iface_ll_fifo.sv SYSTEM_VERILOG PATH acl_iface_ll_fifo.sv
add_fileset_file     acl_iface_address_to_bankaddress.sv SYSTEM_VERILOG PATH acl_iface_address_to_bankaddress.sv

# +-----------------------------------
# | parameters
# | 
add_parameter NUM_BANKS INTEGER 2
set_parameter_property NUM_BANKS DEFAULT_VALUE 2
set_parameter_property NUM_BANKS ALLOWED_RANGES {1 2 4 8}
set_parameter_property NUM_BANKS DISPLAY_NAME "Number of banks"
set_parameter_property NUM_BANKS AFFECTS_ELABORATION true
set_parameter_property NUM_BANKS HDL_PARAMETER true

add_parameter WIDTH_D INTEGER 256
set_parameter_property WIDTH_D DEFAULT_VALUE 256
set_parameter_property WIDTH_D DISPLAY_NAME "Data Width"
set_parameter_property WIDTH_D UNITS "bits" 
set_parameter_property WIDTH_D AFFECTS_ELABORATION true
set_parameter_property WIDTH_D HDL_PARAMETER true

add_parameter M_WIDTH_A INTEGER 31
set_parameter_property M_WIDTH_A DEFAULT_VALUE 31
set_parameter_property M_WIDTH_A DISPLAY_NAME "Host Address Width"
set_parameter_property M_WIDTH_A UNITS "bits"
set_parameter_property M_WIDTH_A AFFECTS_ELABORATION true
set_parameter_property M_WIDTH_A HDL_PARAMETER true

add_parameter S_WIDTH_A INTEGER 26
set_parameter_property S_WIDTH_A DEFAULT_VALUE 26
set_parameter_property S_WIDTH_A DISPLAY_NAME "Agent Address Width"
set_parameter_property S_WIDTH_A UNITS "bits"
set_parameter_property S_WIDTH_A AFFECTS_ELABORATION true
set_parameter_property S_WIDTH_A HDL_PARAMETER true
set_parameter_property S_WIDTH_A DERIVED true


add_parameter BURSTCOUNT_WIDTH INTEGER 6
set_parameter_property BURSTCOUNT_WIDTH DEFAULT_VALUE 6
set_parameter_property BURSTCOUNT_WIDTH DISPLAY_NAME "Burstcount Width"
set_parameter_property BURSTCOUNT_WIDTH UNITS ""
set_parameter_property BURSTCOUNT_WIDTH AFFECTS_ELABORATION true
set_parameter_property BURSTCOUNT_WIDTH HDL_PARAMETER true

add_parameter BYTEENABLE_WIDTH INTEGER 29
set_parameter_property BYTEENABLE_WIDTH DEFAULT_VALUE 29
set_parameter_property BYTEENABLE_WIDTH DISPLAY_NAME "Byteenable Width"
set_parameter_property BYTEENABLE_WIDTH TYPE INTEGER
set_parameter_property BYTEENABLE_WIDTH UNITS None
set_parameter_property BYTEENABLE_WIDTH AFFECTS_GENERATION false
set_parameter_property BYTEENABLE_WIDTH HDL_PARAMETER true
set_parameter_property BYTEENABLE_WIDTH DERIVED true

add_parameter MAX_PENDING_READS INTEGER 64
set_parameter_property MAX_PENDING_READS DEFAULT_VALUE 64
set_parameter_property MAX_PENDING_READS DISPLAY_NAME "Maximum Pending Reads"
set_parameter_property MAX_PENDING_READS UNITS ""
set_parameter_property MAX_PENDING_READS AFFECTS_ELABORATION true
set_parameter_property MAX_PENDING_READS HDL_PARAMETER true

add_parameter ASYNC_RESET INTEGER 1
set_parameter_property ASYNC_RESET DEFAULT_VALUE 1
set_parameter_property ASYNC_RESET DISPLAY_NAME "ASYNC_RESET"
set_parameter_property ASYNC_RESET UNITS ""
set_parameter_property ASYNC_RESET AFFECTS_ELABORATION true
set_parameter_property ASYNC_RESET HDL_PARAMETER true

add_parameter SYNCHRONIZE_RESET INTEGER 0
set_parameter_property SYNCHRONIZE_RESET DEFAULT_VALUE 0
set_parameter_property SYNCHRONIZE_RESET DISPLAY_NAME "SYNCHRONIZE_RESET"
set_parameter_property SYNCHRONIZE_RESET UNITS ""
set_parameter_property SYNCHRONIZE_RESET AFFECTS_ELABORATION true
set_parameter_property SYNCHRONIZE_RESET HDL_PARAMETER true
# | 
# +-----------------------------------

# +-----------------------------------
# | connection point mode
# | 
add_interface mode conduit end
set_interface_property mode ENABLED true
add_interface_port mode mode mode Input 2
# | 
# +-----------------------------------

proc validate {} {
    set num_banks [ get_parameter_value NUM_BANKS ]
    set width_d [ get_parameter_value WIDTH_D ]
    set m_width_a [ get_parameter_value M_WIDTH_A ]

    set byteenable_width [ expr $width_d / 8 ]
    set s_width_a [ expr int ( $m_width_a - (ceil(log($width_d / 8)/log(2))) + (ceil(log($num_banks)/log(2))) ) ]

    set_parameter_value S_WIDTH_A $s_width_a
    set_parameter_value BYTEENABLE_WIDTH $byteenable_width
}

proc elaborate {} {
    set num_banks [ get_parameter_value NUM_BANKS ]
    set width_d [ get_parameter_value WIDTH_D ]
    set m_width_a [ get_parameter_value M_WIDTH_A ]
    set s_width_a [ get_parameter_value S_WIDTH_A ]
    set burstcount_width [ get_parameter_value BURSTCOUNT_WIDTH ]
    set byteenable_width [ get_parameter_value BYTEENABLE_WIDTH ]
    set max_pending_reads [ get_parameter_value MAX_PENDING_READS ]

    # +-----------------------------------
    # | connection point clk
    # | 
    add_interface clk clock end
    set_interface_property clk ENABLED true
    add_interface_port clk clk clk Input 1
    # | 
    # +-----------------------------------
    
    # +-----------------------------------
    # | connection point clk_reset
    # | 
    add_interface clk_reset reset end
    set_interface_property clk_reset associatedClock clk
    set_interface_property clk_reset synchronousEdges DEASSERT
    set_interface_property clk_reset ENABLED true
    add_interface_port clk_reset resetn reset_n Input 1
    # | 
    # +-----------------------------------

    # +-----------------------------------
    # | connection point s
    # | 
    add_interface s avalon end
    set_interface_property s addressAlignment DYNAMIC
    set_interface_property s addressUnits WORDS
    set_interface_property s associatedClock clk
    set_interface_property s associatedReset clk_reset
    set_interface_property s burstOnBurstBoundariesOnly false
    set_interface_property s explicitAddressSpan 0
    set_interface_property s holdTime 0
    set_interface_property s isMemoryDevice false
    set_interface_property s isNonVolatileStorage false
    set_interface_property s linewrapBursts false
    set_interface_property s maximumPendingReadTransactions $max_pending_reads
    set_interface_property s printableDevice false
    set_interface_property s readLatency 0
    set_interface_property s readWaitTime 0
    set_interface_property s setupTime 0
    set_interface_property s timingUnits Cycles
    set_interface_property s writeWaitTime 0
    set_interface_property s ENABLED true

    add_interface_port s agent_address address Input $s_width_a
    add_interface_port s agent_writedata writedata Input $width_d
    add_interface_port s agent_read read Input 1
    add_interface_port s agent_write write Input 1
    add_interface_port s agent_burstcount burstcount Input $burstcount_width
    add_interface_port s agent_byteenable byteenable Input $byteenable_width
    add_interface_port s agent_waitrequest waitrequest Output 1
    add_interface_port s agent_readdata readdata Output $width_d
    add_interface_port s agent_readdatavalid readdatavalid Output 1
    # | 
    # +-----------------------------------

    # +-----------------------------------
    # | connection point bank1
    # | 
    add_interface bank1 avalon start
    set_interface_property bank1 addressUnits SYMBOLS
    set_interface_property bank1 associatedClock clk
    set_interface_property bank1 associatedReset clk_reset
    #In fact this host issues bursts only on burst boundaries, but we hide
    #this from qsys because it blows an extra 2000 registers.
    set_interface_property bank1 burstOnBurstBoundariesOnly false
    set_interface_property bank1 doStreamReads false
    set_interface_property bank1 doStreamWrites false
    set_interface_property bank1 linewrapBursts false
    set_interface_property bank1 readLatency 0
    set_interface_property bank1 ENABLED true

    add_interface_port bank1 bank1_address address Output $m_width_a
    add_interface_port bank1 bank1_writedata writedata Output $width_d
    add_interface_port bank1 bank1_read read Output 1
    add_interface_port bank1 bank1_write write Output 1
    add_interface_port bank1 bank1_burstcount burstcount Output $burstcount_width
    add_interface_port bank1 bank1_byteenable byteenable Output $byteenable_width
    add_interface_port bank1 bank1_waitrequest waitrequest Input 1
    add_interface_port bank1 bank1_readdata readdata Input $width_d
    add_interface_port bank1 bank1_readdatavalid readdatavalid Input 1
    # | 
    # +-----------------------------------

    if { $num_banks >= 2 } {
      # +-----------------------------------
      # | connection point bank2
      # | 
      add_interface bank2 avalon start
      set_interface_property bank2 addressUnits SYMBOLS
      set_interface_property bank2 associatedClock clk
      set_interface_property bank2 associatedReset clk_reset
      set_interface_property bank2 burstOnBurstBoundariesOnly false
      set_interface_property bank2 doStreamReads false
      set_interface_property bank2 doStreamWrites false
      set_interface_property bank2 linewrapBursts false
      set_interface_property bank2 readLatency 0
      set_interface_property bank2 ENABLED true
      add_interface_port bank2 bank2_address address Output $m_width_a
      add_interface_port bank2 bank2_writedata writedata Output $width_d
      add_interface_port bank2 bank2_read read Output 1
      add_interface_port bank2 bank2_write write Output 1
      add_interface_port bank2 bank2_burstcount burstcount Output $burstcount_width
      add_interface_port bank2 bank2_byteenable byteenable Output $byteenable_width
      add_interface_port bank2 bank2_waitrequest waitrequest Input 1
      add_interface_port bank2 bank2_readdata readdata Input $width_d
      add_interface_port bank2 bank2_readdatavalid readdatavalid Input 1
      # | 
      # +-----------------------------------
    }

    if { $num_banks >= 3 } {
      # +-----------------------------------
      # | connection point bank3
      # | 
      add_interface bank3 avalon start
      set_interface_property bank3 addressUnits SYMBOLS
      set_interface_property bank3 associatedClock clk
      set_interface_property bank3 associatedReset clk_reset
      set_interface_property bank3 burstOnBurstBoundariesOnly false
      set_interface_property bank3 doStreamReads false
      set_interface_property bank3 doStreamWrites false
      set_interface_property bank3 linewrapBursts false
      set_interface_property bank3 readLatency 0
      set_interface_property bank3 ENABLED true
      add_interface_port bank3 bank3_address address Output $m_width_a
      add_interface_port bank3 bank3_writedata writedata Output $width_d
      add_interface_port bank3 bank3_read read Output 1
      add_interface_port bank3 bank3_write write Output 1
      add_interface_port bank3 bank3_burstcount burstcount Output $burstcount_width
      add_interface_port bank3 bank3_byteenable byteenable Output $byteenable_width
      add_interface_port bank3 bank3_waitrequest waitrequest Input 1
      add_interface_port bank3 bank3_readdata readdata Input $width_d
      add_interface_port bank3 bank3_readdatavalid readdatavalid Input 1
      # | 
      # +-----------------------------------
    }

    if { $num_banks >= 4 } {
      # +-----------------------------------
      # | connection point bank4
      # | 
      add_interface bank4 avalon start
      set_interface_property bank4 addressUnits SYMBOLS
      set_interface_property bank4 associatedClock clk
      set_interface_property bank4 associatedReset clk_reset
      set_interface_property bank4 burstOnBurstBoundariesOnly false
      set_interface_property bank4 doStreamReads false
      set_interface_property bank4 doStreamWrites false
      set_interface_property bank4 linewrapBursts false
      set_interface_property bank4 readLatency 0
      set_interface_property bank4 ENABLED true
      add_interface_port bank4 bank4_address address Output $m_width_a
      add_interface_port bank4 bank4_writedata writedata Output $width_d
      add_interface_port bank4 bank4_read read Output 1
      add_interface_port bank4 bank4_write write Output 1
      add_interface_port bank4 bank4_burstcount burstcount Output $burstcount_width
      add_interface_port bank4 bank4_byteenable byteenable Output $byteenable_width
      add_interface_port bank4 bank4_waitrequest waitrequest Input 1
      add_interface_port bank4 bank4_readdata readdata Input $width_d
      add_interface_port bank4 bank4_readdatavalid readdatavalid Input 1
      # | 
      # +-----------------------------------
    }

    if { $num_banks >= 5 } {
      # +-----------------------------------
      # | connection point bank5
      # | 
      add_interface bank5 avalon start
      set_interface_property bank5 addressUnits SYMBOLS
      set_interface_property bank5 associatedClock clk
      set_interface_property bank5 associatedReset clk_reset
      set_interface_property bank5 burstOnBurstBoundariesOnly false
      set_interface_property bank5 doStreamReads false
      set_interface_property bank5 doStreamWrites false
      set_interface_property bank5 linewrapBursts false
      set_interface_property bank5 readLatency 0
      set_interface_property bank5 ENABLED true
      add_interface_port bank5 bank5_address address Output $m_width_a
      add_interface_port bank5 bank5_writedata writedata Output $width_d
      add_interface_port bank5 bank5_read read Output 1
      add_interface_port bank5 bank5_write write Output 1
      add_interface_port bank5 bank5_burstcount burstcount Output $burstcount_width
      add_interface_port bank5 bank5_byteenable byteenable Output $byteenable_width
      add_interface_port bank5 bank5_waitrequest waitrequest Input 1
      add_interface_port bank5 bank5_readdata readdata Input $width_d
      add_interface_port bank5 bank5_readdatavalid readdatavalid Input 1
      # | 
      # +-----------------------------------
    }

    if { $num_banks >= 6 } {
      # +-----------------------------------
      # | connection point bank6
      # | 
      add_interface bank6 avalon start
      set_interface_property bank6 addressUnits SYMBOLS
      set_interface_property bank6 associatedClock clk
      set_interface_property bank6 associatedReset clk_reset
      set_interface_property bank6 burstOnBurstBoundariesOnly false
      set_interface_property bank6 doStreamReads false
      set_interface_property bank6 doStreamWrites false
      set_interface_property bank6 linewrapBursts false
      set_interface_property bank6 readLatency 0
      set_interface_property bank6 ENABLED true
      add_interface_port bank6 bank6_address address Output $m_width_a
      add_interface_port bank6 bank6_writedata writedata Output $width_d
      add_interface_port bank6 bank6_read read Output 1
      add_interface_port bank6 bank6_write write Output 1
      add_interface_port bank6 bank6_burstcount burstcount Output $burstcount_width
      add_interface_port bank6 bank6_byteenable byteenable Output $byteenable_width
      add_interface_port bank6 bank6_waitrequest waitrequest Input 1
      add_interface_port bank6 bank6_readdata readdata Input $width_d
      add_interface_port bank6 bank6_readdatavalid readdatavalid Input 1
      # | 
      # +-----------------------------------
    }

    if { $num_banks >= 7 } {
      # +-----------------------------------
      # | connection point bank7
      # | 
      add_interface bank7 avalon start
      set_interface_property bank7 addressUnits SYMBOLS
      set_interface_property bank7 associatedClock clk
      set_interface_property bank7 associatedReset clk_reset
      set_interface_property bank7 burstOnBurstBoundariesOnly false
      set_interface_property bank7 doStreamReads false
      set_interface_property bank7 doStreamWrites false
      set_interface_property bank7 linewrapBursts false
      set_interface_property bank7 readLatency 0
      set_interface_property bank7 ENABLED true
      add_interface_port bank7 bank7_address address Output $m_width_a
      add_interface_port bank7 bank7_writedata writedata Output $width_d
      add_interface_port bank7 bank7_read read Output 1
      add_interface_port bank7 bank7_write write Output 1
      add_interface_port bank7 bank7_burstcount burstcount Output $burstcount_width
      add_interface_port bank7 bank7_byteenable byteenable Output $byteenable_width
      add_interface_port bank7 bank7_waitrequest waitrequest Input 1
      add_interface_port bank7 bank7_readdata readdata Input $width_d
      add_interface_port bank7 bank7_readdatavalid readdatavalid Input 1
      # | 
      # +-----------------------------------
    }

    if { $num_banks >= 8 } {
      # +-----------------------------------
      # | connection point bank8
      # | 
      add_interface bank8 avalon start
      set_interface_property bank8 addressUnits SYMBOLS
      set_interface_property bank8 associatedClock clk
      set_interface_property bank8 associatedReset clk_reset
      set_interface_property bank8 burstOnBurstBoundariesOnly false
      set_interface_property bank8 doStreamReads false
      set_interface_property bank8 doStreamWrites false
      set_interface_property bank8 linewrapBursts false
      set_interface_property bank8 readLatency 0
      set_interface_property bank8 ENABLED true
      add_interface_port bank8 bank8_address address Output $m_width_a
      add_interface_port bank8 bank8_writedata writedata Output $width_d
      add_interface_port bank8 bank8_read read Output 1
      add_interface_port bank8 bank8_write write Output 1
      add_interface_port bank8 bank8_burstcount burstcount Output $burstcount_width
      add_interface_port bank8 bank8_byteenable byteenable Output $byteenable_width
      add_interface_port bank8 bank8_waitrequest waitrequest Input 1
      add_interface_port bank8 bank8_readdata readdata Input $width_d
      add_interface_port bank8 bank8_readdatavalid readdatavalid Input 1
      # | 
      # +-----------------------------------
    }

}

