/*******************************************************************************
* Copyright 2018-2023 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
 *
 *  Content:
 *            Touch test for all available MKL VM math functions:
 *
 *            HA (High Accuracy), LA (Low Accuracy), EP (Enhanced Performance)
 *            single, double, complex precision function diffferent variants:
 *            basic, explicit mode, strided and strided with explicit mode
 *            are being called.
 *
 *******************************************************************************/

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <float.h>
#include <complex.h>
#if (defined __INTEL_COMPILER) || (defined __INTEL_LLVM_COMPILER)
  #include <mathimf.h>
#else
  #include <math.h>
#endif

#include "mkl.h"

/**
 * Common constants:
 */
/* MKL VM vector length - number of inputs to be evaluated */
static const int length = 1000;

/* Stride values for strided API tests */
static const int stride = 3;

/* Maximum printed number of errors for each function - to reduce log size */
static const int max_printed = 1;

/* Maximum allowed ulp's (+1.0 due to rounded reference double precision function) */
static const float  s_allowed_ulp[] = { 1.0f, 4.0f, 5000.0f };
static const float  c_allowed_ulp[] = { 1.0f, 4.0f, 9000.0f };
static const double d_allowed_ulp[] = { 1.0 + 1.0,   4.0 + 1.0,  7.0e7 };
static const double z_allowed_ulp[] = { 1.0 + 1.0,   4.0 + 1.0,  1.0e8 };

/* Fixed argument value */
static const double fixed = 3.14;

/* Enable errors printout */
static const int print_err = 1;

/* Enable warnings printout */
static const int print_warn = 0;

/**
 * Mapping to MKL VM accuracy mode constants:
 */
static const unsigned int vm_mode[] = {VML_HA, VML_LA, VML_EP};

/* Maximum function full name length */
#define NAME_LEN 64

/**
 * Available function API variants:
 */
enum VmApi
{
    kFunc = 0,   /* Basic */
    kMFunc,      /* Explicit mode */
    kFuncI,      /* Strided */
    kMFuncI,     /* Strided with explicit mode */
    kApiNum       /* Number of API variants */
};

/**
 * Available function accuracies:
 */
enum VmAccuracy
{
    kHA = 0,  /* HA */
    kLA,      /* LA */
    kEP,      /* EP */
    kAccNum    /* Number of accuracies */
};

/**
 * Available function precisions:
 */
enum VmPrecision
{
    kSP = 0,  /* Single precision */
    kDP,      /* Double precision */
    kCP,      /* Complex single precision */
    kZP,      /* Complex double precision */
};

/**
 * Available function argument layouts:
 */
enum VmLayout
{
    kVtoV = 0,     /* 1 vector argument  -> 1 vector result */
    kVVtoV,        /* 2 vector arguments -> 1 vector result */
    kVtoVV,        /* 1 vector argument  -> 2 vector results */
    kVXtoV,        /* 1 vector and 1 fixed arguments -> 1 vector result */
    kVVXtoV,       /* 2 vector and 4 fixed arguments -> 1 vector result */
    kVCtoVR,       /* 1 complex argument  -> 1 real result */
    kVRtoVC,       /* 1 real argument  -> 1 complex result */
};

/**
 * Input arguments:
 */
typedef struct
{
    float          *sarg1, *sarg2;
    double         *darg1, *darg2;
    float complex  *carg1, *carg2;
    double complex *zarg1, *zarg2;

    double *sref1, *sref2;
    double *dref1, *dref2;
    double complex *cref1, *cref2;
    double complex *zref1, *zref2;
} VmInputData;

/**
 * Output results:
 */
typedef struct
{
    float          *sres1[kApiNum], *sres2[kApiNum];
    double         *dres1[kApiNum], *dres2[kApiNum];
    float complex  *cres1[kApiNum], *cres2[kApiNum];
    double complex *zres1[kApiNum], *zres2[kApiNum];
} VmOutputData;

/**
 * Types for pointers to real reference functions :
 */
typedef double (*RefVtoV)(double);
typedef double (*RefVVtoV)(double, double);
typedef void   (*RefVtoVV)(double, double*, double* );
typedef double (*RefVVXtoV)(double, double, double, double, double, double );

/**
 * Types for pointers to complex reference functions :
 */
typedef double complex (*CRefVtoV)(double complex);
typedef double         (*CRefCtoR)(double complex);
typedef double complex (*CRefRtoC)(double);
typedef double complex (*CRefVVtoV)(double complex, double complex);

/**
 * Type for pointer to VM functions launcher:
 */
typedef void (*VmFunc)(int, VmInputData*, VmOutputData*);

/**
 * @brief VM functions family launchers
 *
 * Run all variants of VM function family
 *
 * @param[in] acc          Accuracy
 * @param[in] in           Input and reference resutl arrays
 * @param[out] out         Output arrays
 *
 */
 /**
 * Abs
 */
static void own_vm_abs (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAbs   (length, in->sarg1, out->sres1[kFunc]);
    vsAbsI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAbs   (length, in->darg1, out->dres1[kFunc]);
    vdAbsI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAbs   (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kFunc]);
    vcAbsI  (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kFuncI], stride);
    vzAbs   (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kFunc]);
    vzAbsI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kFuncI], stride);

    vmsAbs  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAbsI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAbs  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAbsI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAbs  (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kMFunc], mode);
    vmcAbsI (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kMFuncI], stride, mode);
    vmzAbs  (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kMFunc], mode);
    vmzAbsI (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_abs */

 /**
 * Arg
 */
static void own_vm_arg (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcArg   (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kFunc]);
    vcArgI  (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kFuncI], stride);
    vzArg   (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kFunc]);
    vzArgI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kFuncI], stride);

    vmcArg  (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kMFunc], mode);
    vmcArgI (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kMFuncI], stride, mode);
    vmzArg  (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kMFunc], mode);
    vmzArgI (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_arg */

 /**
 * CIS
 */
static void own_vm_cis (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcCIS   (length, in->sarg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcCISI  (length/stride, in->sarg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzCIS   (length, in->darg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzCISI  (length/stride, in->darg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmcCIS  (length, in->sarg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcCISI (length/stride, in->sarg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzCIS  (length, in->darg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzCISI (length/stride, in->darg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cis */

/**
 * Acos
 */
static void own_vm_acos (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAcos   (length, in->sarg1, out->sres1[kFunc]);
    vsAcosI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAcos   (length, in->darg1, out->dres1[kFunc]);
    vdAcosI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAcos   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAcosI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAcos   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAcosI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAcos  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAcosI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAcos  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAcosI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAcos  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAcosI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAcos  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAcosI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_acos */

/**
 * Atan
 */
static void own_vm_atan (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtan   (length, in->sarg1, out->sres1[kFunc]);
    vsAtanI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAtan   (length, in->darg1, out->dres1[kFunc]);
    vdAtanI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAtan   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAtanI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAtan   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAtanI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAtan  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAtanI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtan  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAtanI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAtan  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAtanI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAtan  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAtanI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atan */

/**
 * Asin
 */
static void own_vm_asin (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAsin   (length, in->sarg1, out->sres1[kFunc]);
    vsAsinI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAsin   (length, in->darg1, out->dres1[kFunc]);
    vdAsinI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAsin   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAsinI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAsin   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAsinI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAsin  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAsinI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAsin  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAsinI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAsin  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAsinI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAsin  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAsinI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_asin */

/**
 * Acosh
 */
static void own_vm_acosh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAcosh   (length, in->sarg1, out->sres1[kFunc]);
    vsAcoshI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAcosh   (length, in->darg1, out->dres1[kFunc]);
    vdAcoshI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAcosh   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAcoshI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAcosh   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAcoshI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAcosh  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAcoshI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAcosh  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAcoshI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAcosh  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAcoshI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAcosh  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAcoshI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_acosh */

/**
 * Asinh
 */
static void own_vm_asinh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAsinh   (length, in->sarg1, out->sres1[kFunc]);
    vsAsinhI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAsinh   (length, in->darg1, out->dres1[kFunc]);
    vdAsinhI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAsinh   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAsinhI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAsinh   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAsinhI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAsinh  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAsinhI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAsinh  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAsinhI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAsinh  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAsinhI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAsinh  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAsinhI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_asinh */

/**
 * Atanh
 */
static void own_vm_atanh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtanh   (length, in->sarg1, out->sres1[kFunc]);
    vsAtanhI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAtanh   (length, in->darg1, out->dres1[kFunc]);
    vdAtanhI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAtanh   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAtanhI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAtanh   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAtanhI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAtanh  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAtanhI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtanh  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAtanhI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAtanh  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAtanhI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAtanh  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAtanhI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atanh */

/**
 * Add
 */
static void own_vm_add (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAdd   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsAddI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdAdd   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdAddI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcAdd   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcAddI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAdd   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzAddI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAdd  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsAddI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdAdd  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdAddI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcAdd  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAddI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAdd  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAddI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_add */

/**
 * Sub
 */
static void own_vm_sub (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSub   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsSubI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdSub   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdSubI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcSub   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcSubI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSub   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzSubI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSub  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsSubI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdSub  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdSubI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcSub  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSubI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSub  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSubI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sub */

/**
 * Mul
 */
static void own_vm_mul (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsMul   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsMulI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdMul   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdMulI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcMul   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcMulI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzMul   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzMulI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsMul  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsMulI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdMul  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdMulI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcMul  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcMulI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzMul  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzMulI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_mul */

/**
 * Atan2
 */
static void own_vm_atan2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtan2   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsAtan2I  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdAtan2   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdAtan2I  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsAtan2  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsAtan2I (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtan2  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdAtan2I (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atan2 */

/**
 * Atan2pi
 */
static void own_vm_atan2pi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtan2pi   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsAtan2piI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdAtan2pi   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdAtan2piI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsAtan2pi  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsAtan2piI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtan2pi  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdAtan2piI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atan2pi */

/**
 * CopySign
 */
static void own_vm_copysign (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCopySign   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsCopySignI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdCopySign   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdCopySignI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsCopySign  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsCopySignI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdCopySign  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdCopySignI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_copysign */

/**
 * Fdim
 */
static void own_vm_fdim (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFdim   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFdimI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFdim   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFdimI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFdim  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFdimI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFdim  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFdimI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fdim */

/**
 * Fmax
 */
static void own_vm_fmax (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFmax   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFmaxI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFmax   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFmaxI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFmax  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFmaxI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFmax  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFmaxI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fmax */

/**
 * Fmin
 */
static void own_vm_fmin (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFmin   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFminI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFmin   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFminI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFmin  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFminI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFmin  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFminI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fmin */

/**
 * MaxMag
 */
static void own_vm_maxmag (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsMaxMag   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsMaxMagI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdMaxMag   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdMaxMagI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsMaxMag  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsMaxMagI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdMaxMag  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdMaxMagI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_maxmag */

/**
 * MinMag
 */
static void own_vm_minmag (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsMinMag   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsMinMagI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdMinMag   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdMinMagI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsMinMag  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsMinMagI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdMinMag  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdMinMagI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_minmag */

/**
 * Fmod
 */
static void own_vm_fmod (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFmod   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFmodI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFmod   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFmodI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFmod  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFmodI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFmod  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFmodI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fmod */

/**
 * Hypot
 */
static void own_vm_hypot (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsHypot   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsHypotI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdHypot   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdHypotI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsHypot  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsHypotI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdHypot  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdHypotI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_hypot */

/**
 * NextAfter
 */
static void own_vm_nextafter (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsNextAfter   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsNextAfterI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdNextAfter   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdNextAfterI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsNextAfter  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsNextAfterI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdNextAfter  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdNextAfterI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_nextafter */

/**
 * Powr
 */
static void own_vm_powr (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPowr   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsPowrI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdPowr   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdPowrI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsPowr  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsPowrI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdPowr  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdPowrI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_powr */

/**
 * Pow2o3
 */
static void own_vm_pow2o3 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPow2o3   (length, in->sarg1, out->sres1[kFunc]);
    vsPow2o3I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdPow2o3   (length, in->darg1, out->dres1[kFunc]);
    vdPow2o3I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsPow2o3  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsPow2o3I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdPow2o3  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdPow2o3I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_pow2o3 */

/**
 * Pow3o2
 */
static void own_vm_pow3o2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPow3o2   (length, in->sarg1, out->sres1[kFunc]);
    vsPow3o2I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdPow3o2   (length, in->darg1, out->dres1[kFunc]);
    vdPow3o2I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsPow3o2  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsPow3o2I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdPow3o2  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdPow3o2I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_pow3o2 */

/**
 * Cbrt
 */
static void own_vm_cbrt (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCbrt   (length, in->sarg1, out->sres1[kFunc]);
    vsCbrtI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCbrt   (length, in->darg1, out->dres1[kFunc]);
    vdCbrtI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCbrt  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCbrtI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCbrt  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCbrtI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cbrt */

/**
 * InvCbrt
 */
static void own_vm_invcbrt (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsInvCbrt   (length, in->sarg1, out->sres1[kFunc]);
    vsInvCbrtI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdInvCbrt   (length, in->darg1, out->dres1[kFunc]);
    vdInvCbrtI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsInvCbrt  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsInvCbrtI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdInvCbrt  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdInvCbrtI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_invcbrt */

/**
 * InvSqrt
 */
static void own_vm_invsqrt (int acc, VmInputData* in, VmOutputData* out)
{

    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsInvSqrt   (length, in->sarg1, out->sres1[kFunc]);
    vsInvSqrtI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdInvSqrt   (length, in->darg1, out->dres1[kFunc]);
    vdInvSqrtI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsInvSqrt  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsInvSqrtI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdInvSqrt  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdInvSqrtI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_invsqrt */

/**
 * Remainder
 */
static void own_vm_remainder (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsRemainder   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsRemainderI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdRemainder   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdRemainderI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsRemainder  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsRemainderI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdRemainder  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdRemainderI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_remainder */

/**
 * Div
 */
static void own_vm_div (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsDiv   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsDivI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdDiv   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdDivI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcDiv   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcDivI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzDiv   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzDivI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsDiv  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsDivI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdDiv  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdDivI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcDiv  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcDivI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzDiv  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzDivI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_div */

/**
 * Pow
 */
static void own_vm_pow (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPow   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsPowI  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdPow   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdPowI  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcPow   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcPowI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzPow   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzPowI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsPow  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsPowI (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdPow  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdPowI (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcPow  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcPowI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzPow  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzPowI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_div */

/**
 * Powx
 */
static void own_vm_powx (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];
    float  sarg2 = fixed;
    double darg2 = fixed;
    MKL_Complex8  carg2 = {fixed, fixed};
    MKL_Complex16 zarg2 = {fixed, fixed};


    vmlSetMode(mode);

    vsPowx   (length, in->sarg1, sarg2, out->sres1[kFunc]);
    vsPowxI  (length/stride, in->sarg1, stride, sarg2, out->sres1[kFuncI], stride);
    vdPowx   (length, in->darg1, darg2, out->dres1[kFunc]);
    vdPowxI  (length/stride, in->darg1, stride, darg2, out->dres1[kFuncI], stride);
    vcPowx   (length, (MKL_Complex8*)in->carg1, carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcPowxI  (length/stride, (MKL_Complex8*)in->carg1, stride, carg2, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzPowx   (length, (MKL_Complex16*)in->zarg1, zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzPowxI  (length/stride, (MKL_Complex16*)in->zarg1, stride, zarg2, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsPowx  (length, in->sarg1, sarg2, out->sres1[kMFunc], mode);
    vmsPowxI (length/stride, in->sarg1, stride, sarg2, out->sres1[kMFuncI], stride, mode);
    vmdPowx  (length, in->darg1, darg2, out->dres1[kMFunc], mode);
    vmdPowxI (length/stride, in->darg1, stride, darg2, out->dres1[kMFuncI], stride, mode);
    vmcPowx  (length, (MKL_Complex8*)in->carg1, carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcPowxI (length/stride, (MKL_Complex8*)in->carg1, stride, carg2, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzPowx  (length, (MKL_Complex16*)in->zarg1, zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzPowxI (length/stride, (MKL_Complex16*)in->zarg1, stride, zarg2, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_powx */

/**
 * LinearFrac
 */
static void own_vm_linearfrac (int acc, VmInputData* in, VmOutputData* out)
{
    float  sfixed = (float)fixed;
    double dfixed = (double)fixed;
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLinearFrac   (length, in->sarg1, in->sarg2, sfixed, sfixed, sfixed, sfixed, out->sres1[kFunc]);
    vsLinearFracI  (length/stride, in->sarg1, stride, in->sarg2, stride, sfixed, sfixed, sfixed, sfixed, out->sres1[kFuncI], stride);
    vdLinearFrac   (length, in->darg1, in->darg2, dfixed, dfixed, dfixed, dfixed, out->dres1[kFunc]);
    vdLinearFracI  (length/stride, in->darg1, stride, in->darg2, stride, dfixed, dfixed, dfixed, dfixed, out->dres1[kFuncI], stride);

    vmsLinearFrac  (length, in->sarg1, in->sarg2, sfixed, sfixed, sfixed, sfixed, out->sres1[kMFunc], mode);
    vmsLinearFracI (length/stride, in->sarg1, stride, in->sarg2, stride, sfixed, sfixed, sfixed, sfixed, out->sres1[kMFuncI], stride, mode);
    vmdLinearFrac  (length, in->darg1, in->darg2, dfixed, dfixed, dfixed, dfixed, out->dres1[kMFunc], mode);
    vmdLinearFracI (length/stride, in->darg1, stride, in->darg2, stride, dfixed, dfixed, dfixed, dfixed, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_linearfrac */

/**
 * Sqrt
 */
static void own_vm_sqrt (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSqrt   (length, in->sarg1, out->sres1[kFunc]);
    vsSqrtI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSqrt   (length, in->darg1, out->dres1[kFunc]);
    vdSqrtI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcSqrt   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcSqrtI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSqrt   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzSqrtI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSqrt  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSqrtI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSqrt  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSqrtI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcSqrt  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSqrtI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSqrt  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSqrtI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sqrt */

/**
 * Ln
 */
static void own_vm_ln (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLn   (length, in->sarg1, out->sres1[kFunc]);
    vsLnI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLn   (length, in->darg1, out->dres1[kFunc]);
    vdLnI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcLn   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcLnI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzLn   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzLnI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsLn  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLnI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLn  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLnI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcLn  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcLnI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzLn  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzLnI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_ln */

/**
 * Sind
 */
static void own_vm_sind (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSind   (length, in->sarg1, out->sres1[kFunc]);
    vsSindI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSind   (length, in->darg1, out->dres1[kFunc]);
    vdSindI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsSind  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSindI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSind  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSindI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sind */

/**
 * Cosd
 */
static void own_vm_cosd (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCosd   (length, in->sarg1, out->sres1[kFunc]);
    vsCosdI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCosd   (length, in->darg1, out->dres1[kFunc]);
    vdCosdI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCosd  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCosdI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCosd  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCosdI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cosd */

/**
 * Tand
 */
static void own_vm_tand (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTand   (length, in->sarg1, out->sres1[kFunc]);
    vsTandI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTand   (length, in->darg1, out->dres1[kFunc]);
    vdTandI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTand  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTandI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTand  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTandI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tand */

/**
 * Sinpi
 */
static void own_vm_sinpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSinpi   (length, in->sarg1, out->sres1[kFunc]);
    vsSinpiI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSinpi   (length, in->darg1, out->dres1[kFunc]);
    vdSinpiI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsSinpi  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSinpiI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSinpi  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSinpiI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sinpi */

/**
 * Cospi
 */
static void own_vm_cospi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCospi   (length, in->sarg1, out->sres1[kFunc]);
    vsCospiI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCospi   (length, in->darg1, out->dres1[kFunc]);
    vdCospiI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCospi  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCospiI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCospi  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCospiI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cospi */

/**
 * Tanpi
 */
static void own_vm_tanpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTanpi   (length, in->sarg1, out->sres1[kFunc]);
    vsTanpiI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTanpi   (length, in->darg1, out->dres1[kFunc]);
    vdTanpiI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTanpi  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTanpiI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTanpi  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTanpiI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tanpi */

/**
 * SinCos
 */
static void own_vm_sincos (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSinCos   (length, in->sarg1, out->sres1[kFunc], out->sres2[kFunc]);
    vsSinCosI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride, out->sres2[kFuncI], stride);
    vdSinCos   (length, in->darg1, out->dres1[kFunc], out->dres2[kFunc]);
    vdSinCosI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride, out->dres2[kFuncI], stride);

    vmsSinCos  (length, in->sarg1, out->sres1[kMFunc], out->sres2[kMFunc], mode);
    vmsSinCosI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, out->sres2[kMFuncI], stride, mode);
    vmdSinCos  (length, in->darg1, out->dres1[kMFunc], out->dres2[kMFunc], mode);
    vmdSinCosI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, out->dres2[kMFuncI], stride, mode);

    return;
} /* own_vm_sincos */

/**
 * Modf
 */
static void own_vm_modf (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsModf   (length, in->sarg1, out->sres1[kFunc], out->sres2[kFunc]);
    vsModfI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride, out->sres2[kFuncI], stride);
    vdModf   (length, in->darg1, out->dres1[kFunc], out->dres2[kFunc]);
    vdModfI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride, out->dres2[kFuncI], stride);

    vmsModf  (length, in->sarg1, out->sres1[kMFunc], out->sres2[kMFunc], mode);
    vmsModfI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, out->sres2[kMFuncI], stride, mode);
    vmdModf  (length, in->darg1, out->dres1[kMFunc], out->dres2[kMFunc], mode);
    vmdModfI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, out->dres2[kMFuncI], stride, mode);

    return;
} /* own_vm_modf */

/**
 * Conj
 */
static void own_vm_conj (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcConj   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcConjI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzConj   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzConjI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmcConj  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcConjI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzConj  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzConjI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_conj */

/**
 * MulByConj
 */
static void own_vm_mulbyconj (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcMulByConj   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcMulByConjI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzMulByConj   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzMulByConjI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmcMulByConj  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcMulByConjI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzMulByConj  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzMulByConjI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_mulbyconj */

/**
 * Sin
 */
static void own_vm_sin (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSin   (length, in->sarg1, out->sres1[kFunc]);
    vsSinI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSin   (length, in->darg1, out->dres1[kFunc]);
    vdSinI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcSin   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcSinI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSin   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzSinI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSin  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSinI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSin  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSinI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcSin  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSinI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSin  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSinI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sin */

/**
 * Cos
 */
static void own_vm_cos (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCos   (length, in->sarg1, out->sres1[kFunc]);
    vsCosI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCos   (length, in->darg1, out->dres1[kFunc]);
    vdCosI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcCos   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcCosI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzCos   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzCosI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsCos  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCosI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCos  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCosI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcCos  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcCosI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzCos  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzCosI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cos */

/**
 * Tan
 */
static void own_vm_tan (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTan   (length, in->sarg1, out->sres1[kFunc]);
    vsTanI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTan   (length, in->darg1, out->dres1[kFunc]);
    vdTanI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcTan   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcTanI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzTan   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzTanI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsTan  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTanI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTan  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTanI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcTan  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcTanI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzTan  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzTanI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tan */

/**
 * Inv
 */
static void own_vm_inv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsInv   (length, in->sarg1, out->sres1[kFunc]);
    vsInvI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdInv   (length, in->darg1, out->dres1[kFunc]);
    vdInvI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsInv  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsInvI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdInv  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdInvI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_inv */

/**
 * Sqr
 */
static void own_vm_sqr (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSqr   (length, in->sarg1, out->sres1[kFunc]);
    vsSqrI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSqr   (length, in->darg1, out->dres1[kFunc]);
    vdSqrI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsSqr  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSqrI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSqr  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSqrI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sqr */

/**
 * Frac
 */
static void own_vm_frac (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFrac   (length, in->sarg1, out->sres1[kFunc]);
    vsFracI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdFrac   (length, in->darg1, out->dres1[kFunc]);
    vdFracI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsFrac  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsFracI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdFrac  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdFracI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_frac */

/**
 * Ceil
 */
static void own_vm_ceil (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCeil   (length, in->sarg1, out->sres1[kFunc]);
    vsCeilI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCeil   (length, in->darg1, out->dres1[kFunc]);
    vdCeilI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCeil  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCeilI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCeil  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCeilI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_ceil */

/**
 * Floor
 */
static void own_vm_floor (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFloor   (length, in->sarg1, out->sres1[kFunc]);
    vsFloorI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdFloor   (length, in->darg1, out->dres1[kFunc]);
    vdFloorI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsFloor  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsFloorI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdFloor  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdFloorI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_floor */

/**
 * Round
 */
static void own_vm_round (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsRound   (length, in->sarg1, out->sres1[kFunc]);
    vsRoundI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdRound   (length, in->darg1, out->dres1[kFunc]);
    vdRoundI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsRound  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsRoundI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdRound  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdRoundI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_round */

/**
 * Trunc
 */
static void own_vm_trunc (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTrunc   (length, in->sarg1, out->sres1[kFunc]);
    vsTruncI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTrunc   (length, in->darg1, out->dres1[kFunc]);
    vdTruncI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTrunc  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTruncI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTrunc  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTruncI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_trunc */

/**
 * Rint
 */
static void own_vm_rint (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsRint   (length, in->sarg1, out->sres1[kFunc]);
    vsRintI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdRint   (length, in->darg1, out->dres1[kFunc]);
    vdRintI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsRint  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsRintI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdRint  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdRintI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_rint */

/**
 * NearbyInt
 */
static void own_vm_nearbyint (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsNearbyInt   (length, in->sarg1, out->sres1[kFunc]);
    vsNearbyIntI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdNearbyInt   (length, in->darg1, out->dres1[kFunc]);
    vdNearbyIntI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsNearbyInt  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsNearbyIntI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdNearbyInt  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdNearbyIntI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_nearbyint */

/**
 * Acospi
 */
static void own_vm_acospi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAcospi   (length, in->sarg1, out->sres1[kFunc]);
    vsAcospiI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAcospi   (length, in->darg1, out->dres1[kFunc]);
    vdAcospiI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsAcospi  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAcospiI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAcospi  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAcospiI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_acospi */

/**
 * Asinpi
 */
static void own_vm_asinpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAsinpi   (length, in->sarg1, out->sres1[kFunc]);
    vsAsinpiI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAsinpi   (length, in->darg1, out->dres1[kFunc]);
    vdAsinpiI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsAsinpi  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAsinpiI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAsinpi  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAsinpiI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_asinpi */

/**
 * Atanpi
 */
static void own_vm_atanpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtanpi   (length, in->sarg1, out->sres1[kFunc]);
    vsAtanpiI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAtanpi   (length, in->darg1, out->dres1[kFunc]);
    vdAtanpiI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsAtanpi  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAtanpiI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtanpi  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAtanpiI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atanpi */

/**
 * Sinh
 */
static void own_vm_sinh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSinh   (length, in->sarg1, out->sres1[kFunc]);
    vsSinhI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSinh   (length, in->darg1, out->dres1[kFunc]);
    vdSinhI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcSinh   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcSinhI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSinh   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzSinhI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSinh  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSinhI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSinh  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSinhI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcSinh  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSinhI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSinh  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSinhI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sinh */

/**
 * Cosh
 */
static void own_vm_cosh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCosh   (length, in->sarg1, out->sres1[kFunc]);
    vsCoshI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCosh   (length, in->darg1, out->dres1[kFunc]);
    vdCoshI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcCosh   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcCoshI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzCosh   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzCoshI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsCosh  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCoshI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCosh  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCoshI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcCosh  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcCoshI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzCosh  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzCoshI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cosh */

/**
 * Tanh
 */
static void own_vm_tanh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTanh   (length, in->sarg1, out->sres1[kFunc]);
    vsTanhI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTanh   (length, in->darg1, out->dres1[kFunc]);
    vdTanhI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcTanh   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcTanhI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzTanh   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzTanhI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsTanh  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTanhI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTanh  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTanhI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcTanh  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcTanhI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzTanh  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzTanhI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tanh */

/**
 * Exp
 */
static void own_vm_exp (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExp   (length, in->sarg1, out->sres1[kFunc]);
    vsExpI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExp   (length, in->darg1, out->dres1[kFunc]);
    vdExpI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcExp   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcExpI  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzExp   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzExpI  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsExp  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExpI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExp  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExpI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcExp  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcExpI (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzExp  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzExpI (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_exp */

/**
 * Exp2
 */
static void own_vm_exp2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExp2   (length, in->sarg1, out->sres1[kFunc]);
    vsExp2I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExp2   (length, in->darg1, out->dres1[kFunc]);
    vdExp2I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExp2  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExp2I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExp2  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExp2I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_exp2 */

/**
 * Exp10
 */
static void own_vm_exp10 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExp10   (length, in->sarg1, out->sres1[kFunc]);
    vsExp10I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExp10   (length, in->darg1, out->dres1[kFunc]);
    vdExp10I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExp10  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExp10I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExp10  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExp10I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_exp10 */

/**
 * Expm1
 */
static void own_vm_expm1 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExpm1   (length, in->sarg1, out->sres1[kFunc]);
    vsExpm1I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExpm1   (length, in->darg1, out->dres1[kFunc]);
    vdExpm1I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExpm1  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExpm1I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExpm1  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExpm1I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_expm1 */

/**
 * Log2
 */
static void own_vm_log2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLog2   (length, in->sarg1, out->sres1[kFunc]);
    vsLog2I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLog2   (length, in->darg1, out->dres1[kFunc]);
    vdLog2I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsLog2  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLog2I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLog2  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLog2I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_log2 */

/**
 * Log10
 */
static void own_vm_log10 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLog10   (length, in->sarg1, out->sres1[kFunc]);
    vsLog10I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLog10   (length, in->darg1, out->dres1[kFunc]);
    vdLog10I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcLog10   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcLog10I  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzLog10   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzLog10I  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsLog10  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLog10I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLog10  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLog10I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcLog10  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcLog10I (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzLog10  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzLog10I (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_log10 */

/**
 * Log1p
 */
static void own_vm_log1p (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLog1p   (length, in->sarg1, out->sres1[kFunc]);
    vsLog1pI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLog1p   (length, in->darg1, out->dres1[kFunc]);
    vdLog1pI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsLog1p  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLog1pI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLog1p  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLog1pI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_log1p */

/**
 * Erf
 */
static void own_vm_erf (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErf   (length, in->sarg1, out->sres1[kFunc]);
    vsErfI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErf   (length, in->darg1, out->dres1[kFunc]);
    vdErfI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErf  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErf  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erf */

/**
 * Erfc
 */
static void own_vm_erfc (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErfc   (length, in->sarg1, out->sres1[kFunc]);
    vsErfcI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErfc   (length, in->darg1, out->dres1[kFunc]);
    vdErfcI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErfc  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfcI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErfc  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfcI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erfc */

/**
 * ErfInv
 */
static void own_vm_erfinv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErfInv   (length, in->sarg1, out->sres1[kFunc]);
    vsErfInvI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErfInv   (length, in->darg1, out->dres1[kFunc]);
    vdErfInvI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErfInv  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfInvI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErfInv  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfInvI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erfinv */

/**
 * ErfcInv
 */
static void own_vm_erfcinv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErfcInv   (length, in->sarg1, out->sres1[kFunc]);
    vsErfcInvI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErfcInv   (length, in->darg1, out->dres1[kFunc]);
    vdErfcInvI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErfcInv  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfcInvI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErfcInv  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfcInvI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erfcinv */

/**
 * CdfNorm
 */
static void own_vm_cdfnorm (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCdfNorm   (length, in->sarg1, out->sres1[kFunc]);
    vsCdfNormI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCdfNorm   (length, in->darg1, out->dres1[kFunc]);
    vdCdfNormI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCdfNorm  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCdfNormI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCdfNorm  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCdfNormI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cdfnorm */

/**
 * CdfNormInv
 */
static void own_vm_cdfnorminv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCdfNormInv   (length, in->sarg1, out->sres1[kFunc]);
    vsCdfNormInvI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCdfNormInv   (length, in->darg1, out->dres1[kFunc]);
    vdCdfNormInvI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCdfNormInv  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCdfNormInvI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCdfNormInv  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCdfNormInvI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cdfnorminv */

/**
 * LGamma
 */
static void own_vm_lgamma (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLGamma   (length, in->sarg1, out->sres1[kFunc]);
    vsLGammaI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLGamma   (length, in->darg1, out->dres1[kFunc]);
    vdLGammaI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsLGamma  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLGammaI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLGamma  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLGammaI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_lgamma */

/**
 * TGamma
 */
static void own_vm_tgamma (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTGamma   (length, in->sarg1, out->sres1[kFunc]);
    vsTGammaI  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTGamma   (length, in->darg1, out->dres1[kFunc]);
    vdTGammaI  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTGamma  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTGammaI (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTGamma  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTGammaI (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tgamma */

/**
 * ExpInt1
 */
static void own_vm_expint1 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExpInt1   (length, in->sarg1, out->sres1[kFunc]);
    vsExpInt1I  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExpInt1   (length, in->darg1, out->dres1[kFunc]);
    vdExpInt1I  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExpInt1  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExpInt1I (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExpInt1  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExpInt1I (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_expint1 */

/**
 * Custom reference functions absent in LIBM:
 */
static double own_frac (double x) { return (x - trunc (x)); } /* signed fractional part */
static double own_invcbrt (double x) { return pow(x, -(1.0/3.0)); } /* inverse cube root */
static double own_pow2o3 (double x) { return pow(x, 2.0 / 3.0); } /* power x^(2/3) */
static double own_pow3o2 (double x) { return pow(x, 3.0 / 2.0); } /* power x^(3/2) */
static double own_inv (double x) { return 1.0 / x; } /* inverse 1/x */
static double own_sqr (double x) { return x * x; } /* square x*x */
static double own_add (double x, double y) { return x + y; } /* addition x+y */
static double own_sub (double x, double y) { return x - y; } /* subtraction x-y */
static double own_mul (double x, double y) { return x * y; } /* multiplication x-y */
static double own_div (double x, double y) { return x / y; } /* division x/y */
static double own_expint1 (double x) { double r; vmdExpInt1(1,&x, &r, VML_HA); return r; } /* exponential integral - exists in VM only */
static void   own_modf (double x, double* r1, double* r2) { *r2 = modf(x, r1); } /* truncated integer value |x| and the remaining fraction part x-|x| */
static double own_linearfrac (double x, double y, double a, double b, double c, double d) { return (x * a + b)/(y * c + d); } /* linear fraction (x*a + b)/(y*c + d)*/
static double own_minmag (double x, double y) { return (fabs(x) < fabs(y))?x:y; } /* min(|x|,|y|) */
static double own_maxmag (double x, double y) { return (fabs(x) > fabs(y))?x:y; } /* max(|x|,|y|) */
static void   own_sincos (double x, double* r1, double* r2) { *r1 = sin(x); *r2 = cos(x); return; } /* sin & cos */
static double own_invsqrt (double x) { return 1.0 / sqrt(x); } /* 1/sqrt(x) */

static double complex own_cadd (double complex x, double complex y) { return x + y; } /* complex x+y */
static double complex own_csub (double complex x, double complex y) { return x - y; } /* complex x+y */
static double complex own_cmul (double complex x, double complex y) { return x * y; } /* complex x*y */
static double complex own_cdiv (double complex x, double complex y) { return x / y; } /* complex x+y */
static double own_cabs (double complex x) { return hypot(creal(x),cimag(x)); } /* |x| */
static double own_carg (double complex x) { return atan2(cimag(x),creal(x)); } /* complex argument (atan2) */
static double complex own_cis (double x) { double r1, r2; own_sincos(x, &r2, &r1); double complex r = r1 + I * r2; return r; } /* complex CIS (sincos) */
static double complex own_cmulbyconj (double complex x, double complex y) { return x * conj(y); } /* complex x*conj(y) */

/**
 * @brief Safe malloc
 *
 * own_safe_malloc allocates memory and check resulted pointer.
 * Report error and exit application if unsuccessful.
 *
 * @param[in] size          Size in bytes
 * @return                  Pointer to allocated memory
 *
 */
static void* own_safe_malloc(int size)
{
    void* ptr = malloc (size);
    if (ptr == NULL)
    {
       fprintf (stderr, "\t\tERROR: %d bytes allocated unsuccessfully\n", size);
       exit(-1);
    }

    return ptr;
} /* own_safe_malloc */

/**
 * @brief Safe free
 *
 * own_safe_free deallocates memory.
 * Report error if NULL pointer passed.
 *
 * @param[in] ptr          Pointer to memory
 *
 */
static void own_safe_free(void *ptr)
{
    if (ptr != NULL) { free (ptr); }
    else
    {
       fprintf (stderr, "\t\tERROR: NULL pointer cannot be deallocated\n");
       exit(-1);
    }

    return;
} /* own_safe_free */

/**
 * @brief Allocate all input and reference result arrays
 *
 * Safe allocation of input and reference result arrays memory
 *
 * @param[in] len           Arrays length
 * @param[in, out] in       Input and reference resutl arrays
 * @param[in, out] out      Output arrays
 *
 */
static void own_allocate_data (int len, VmInputData* in, VmOutputData* out)
{
    in->sarg1    =  own_safe_malloc (len * sizeof (float));
    in->sarg2    =  own_safe_malloc (len * sizeof (float));
    in->darg1    =  own_safe_malloc (len * sizeof (double));
    in->darg2    =  own_safe_malloc (len * sizeof (double));
    in->carg1    =  own_safe_malloc (len * sizeof (float complex));
    in->carg2    =  own_safe_malloc (len * sizeof (float complex));
    in->zarg1    =  own_safe_malloc (len * sizeof (double complex));
    in->zarg2    =  own_safe_malloc (len * sizeof (double complex));
    in->sref1    =  own_safe_malloc (len * sizeof (double));
    in->sref2    =  own_safe_malloc (len * sizeof (double));
    in->dref1    =  own_safe_malloc (len * sizeof (double));
    in->dref2    =  own_safe_malloc (len * sizeof (double));
    in->cref1    =  own_safe_malloc (len * sizeof (double complex));
    in->cref2    =  own_safe_malloc (len * sizeof (double complex));
    in->zref1    =  own_safe_malloc (len * sizeof (double complex));
    in->zref2    =  own_safe_malloc (len * sizeof (double complex));

    for (int v = kFunc; v < kApiNum; v = v + 1)
    {
        out->sres1[v]   =  own_safe_malloc (len * sizeof (float));
        out->sres2[v]   =  own_safe_malloc (len * sizeof (float));
        out->dres1[v]   =  own_safe_malloc (len * sizeof (double));
        out->dres2[v]   =  own_safe_malloc (len * sizeof (double));
        out->cres1[v]   =  own_safe_malloc (len * sizeof (float complex));
        out->cres2[v]   =  own_safe_malloc (len * sizeof (float complex));
        out->zres1[v]   =  own_safe_malloc (len * sizeof (double complex));
        out->zres2[v]   =  own_safe_malloc (len * sizeof (double complex));
    }
} /* own_allocate_data */

/**
 * @brief Deallocate all input and reference result arrays
 *
 * Safe deallocation of input and reference result arrays memory
 *
 * @param[in, out] in       Input and reference resutl arrays
 * @param[in, out] out      Output arrays
 *
 */
static void own_deallocate_data (VmInputData* in, VmOutputData* out)
{
    own_safe_free (in->sarg1);
    own_safe_free (in->sarg2);
    own_safe_free (in->darg1);
    own_safe_free (in->darg2);
    own_safe_free (in->carg1);
    own_safe_free (in->carg2);
    own_safe_free (in->zarg1);
    own_safe_free (in->zarg2);
    own_safe_free (in->sref1);
    own_safe_free (in->sref2);
    own_safe_free (in->dref1);
    own_safe_free (in->dref2);
    own_safe_free (in->cref1);
    own_safe_free (in->cref2);
    own_safe_free (in->zref1);
    own_safe_free (in->zref2);

    for (int v = kFunc; v <= kMFuncI; v++)
    {
        own_safe_free (out->sres1[v]);
        own_safe_free (out->sres2[v]);
        own_safe_free (out->dres1[v]);
        own_safe_free (out->dres2[v]);
        own_safe_free (out->cres1[v]);
        own_safe_free (out->cres2[v]);
        own_safe_free (out->zres1[v]);
        own_safe_free (out->zres2[v]);
    }

    return;
} /* own_deallocate_data */

/**
 * @brief Single precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static float own_s_compute_ulp (float res, double ref )
{
    int ex       = 0;
    double den   = 1.0;
    float curulp = 0.0;
    if (isfinite (ref))
    {
        frexp (ref, &ex);                                     /* ex: integral power of two of ref */
        den = ldexp (1.0, ex - 24);                           /* den: ulp's denominator 2^(ex-p+1) */
        den = (den == 0.0)? 0x1.p-149 : den;                  /* if den=0 then replace by EPS to avoid divbyzero */
        curulp = (float)fabs ((((double)(res) - ref)) / den); /* |res-ref|/2^(ex-24) */
        curulp = isfinite (curulp)?curulp : FLT_MAX;          /* replace infinite ulp by big finite float number */
    }
    else
    {
        if (fpclassify (res) == (fpclassify (ref))) { curulp = 0; }
        else { curulp = FLT_MAX; }
    }

    return curulp;
} /* own_s_compute_ulp */

/**
 * @brief Complex single precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static float own_c_compute_ulp (float complex res, double complex ref )
{
    return fmaxf (own_s_compute_ulp(creal(res), creal(ref)), own_s_compute_ulp(cimag(res), cimag(ref)));
}

/**
 * @brief Double precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static double own_d_compute_ulp (double res, double ref )
{
    int ex        = 0;
    double den    = 1.0;
    double curulp = 0.0;
    if (isfinite (ref))
    {
        frexp (ref, &ex);                                      /* ex: integral power of two of ref */
        den = ldexp (1.0, ex - 53);                            /* den: ulp's denominator 2^(ex-p+1) */
        den = (den == 0.0)? 0x1.p-1074 : den;                  /* if den=0 then replace by EPS to avoid divbyzero */
        curulp = (double)fabs ((((double)(res) - ref)) / den); /* |res-ref|/2^(ex-53) */
        curulp = isfinite (curulp)?curulp : DBL_MAX;           /* replace infinite ulp by big finite double number */
    }
    else
    {
        if (fpclassify (res) == (fpclassify (ref))) { curulp = 0; }
        else { curulp = DBL_MAX; }
    }

    return curulp;
} /* own_d_compute_ulp */

/**
 * @brief Complex double precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static double own_z_compute_ulp (double complex res, double complex ref )
{
    return fmax (own_d_compute_ulp(creal(res), creal(ref)), own_d_compute_ulp(cimag(res), cimag(ref)));
}

/**
 * @brief Fill inputs
 *
 * Fills input vectors by random numbers
 *
 * @param[in] layout       Function arguments layout
 * @param[in] beg          Begin of input ranges
 * @param[in] end          End of input ranges
 * @param[out] in          Input arrays
 *
 */
static void own_fill_input (int layout, double beg, double end, VmInputData* in)
{
    srand(777);
    for (int i = 0; i < length; i = i + 1)
    {
        /**
         * Generate random numbers in [beg, end) range
         */
        double v1 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));
        double v2 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));
        double v3 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));
        double v4 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));

        in->darg1[i] = v1;
        in->darg2[i] = v2;
        in->sarg1[i] = (float)v1;
        in->sarg2[i] = (float)v2;

        in->zarg1[i] = v1 + I * v3;
        in->zarg2[i] = v2 + I * v4;
        in->carg1[i] = (float)v1 + I * (float)v3;
        in->carg2[i] = (float)v2 + I * (float)v4;
    } /* for (int i = 0; i < length; i++) */
} /* own_fill_input */

/**
 * @brief Fill reference vectors
 *
 * Compute reference results
 *
 * @param[in] layout       Function arguments layout
 * @param[in] reffunc      Real reference function
 * @param[in] creffunc     Complex reference function
 * @param[out] in          Input and reference results arrays
 *
 */
static void own_fill_reference (int layout, void* reffunc, void* creffunc, VmInputData* in)
{
    int has_real    = (reffunc != NULL)?1:0, has_complex = (creffunc != NULL)?1:0;
    for (int i = 0; i < length; i = i + 1)
    {
        if (layout == kVtoV)
        {
            if (has_real)
            {
                in->sref1[i] = ((RefVtoV)reffunc) (in->sarg1[i]);
                in->dref1[i] = ((RefVtoV)reffunc) (in->darg1[i]);
            }
            if (has_complex)
            {
                in->cref1[i] = ((CRefVtoV)creffunc) ((double complex) (in->carg1[i]));
                in->zref1[i] = ((CRefVtoV)creffunc) ( in->zarg1[i]);
            }
        }
        else if (layout == kVCtoVR)
        {
            if (has_real)
            {
                in->sref1[i] = ((RefVtoV)reffunc) (in->sarg1[i]);
                in->dref1[i] = ((RefVtoV)reffunc) (in->darg1[i]);
            }
            if (has_complex)
            {
                /* Use complex array containers to accept real results */
                ((double*)(in->cref1))[i] = ((CRefCtoR)creffunc) ((double complex) (in->carg1[i]));
                ((double*)(in->zref1))[i] = ((CRefCtoR)creffunc) (in->zarg1[i]);
            }
        }
        else if (layout == kVRtoVC)
        {
            if (has_real)
            {
                /* Use real arrays for kVRtoVC functions */
                in->sref1[i] = ((RefVtoV)reffunc) (in->sarg1[i]);
                in->dref1[i] = ((RefVtoV)reffunc) (in->darg1[i]);
            }
            if (has_complex)
            {
                in->cref1[i] = ((CRefRtoC)creffunc) (in->sarg1[i]);
                in->zref1[i] = ((CRefRtoC)creffunc) (in->darg1[i]);
            }
        }
        else if (layout == kVVtoV)
        {
            if (has_real)
            {
                in->sref1[i] = ((RefVVtoV)reffunc) (in->sarg1[i], in->sarg2[i]);
                in->dref1[i] = ((RefVVtoV)reffunc) (in->darg1[i], in->darg2[i]);
            }
            if (has_complex)
            {
                in->cref1[i] = ((CRefVVtoV)creffunc) ((double complex) (in->carg1[i]), (double complex) (in->carg2[i]));
                in->zref1[i] = ((CRefVVtoV)creffunc) (in->zarg1[i], in->zarg2[i]);
            }
        }
        else if (layout == kVXtoV)
        {
            float  sfixed = (float)fixed;
            double dfixed = (double)fixed;
            if (has_real)
            {
                in->sref1[i] = ((RefVVtoV)reffunc) (in->sarg1[i], sfixed);
                in->dref1[i] = ((RefVVtoV)reffunc) (in->darg1[i], dfixed);
            }
            if (has_complex)
            {
                double complex cfixed = sfixed + I * sfixed;
                double complex zfixed = dfixed + I * dfixed;
                in->cref1[i] = ((CRefVVtoV)creffunc) ((double complex) (in->carg1[i]), cfixed);
                in->zref1[i] = ((CRefVVtoV)creffunc) (in->zarg1[i], zfixed);
            }
        }
        else if (layout == kVtoVV)
        {
            ((RefVtoVV)reffunc) (in->sarg1[i], &(in->sref1[i]), &(in->sref2[i]));
            ((RefVtoVV)reffunc) (in->darg1[i], &(in->dref1[i]), &(in->dref2[i]));
        }
        else if (layout == kVVXtoV)
        {
            float  sfixed = (float)fixed;
            double dfixed = (double)fixed;
            in->sref1[i] = ((RefVVXtoV)reffunc) (in->sarg1[i], in->sarg2[i], sfixed, sfixed, sfixed, sfixed);
            in->dref1[i] = ((RefVVXtoV)reffunc) (in->darg1[i], in->darg2[i], dfixed, dfixed, dfixed, dfixed);
        }
    } /* for(int i = 0; i < length; i++) */
} /* own_fill_reference */

/**
 * @brief Full VM function name for printout
 *
 * Construct full VM function name with precision, api and accuracy suffices
 *
 * @param[out] buff        Pointer to output string buffer
 * @param[in] maxlen       Maximum string length
 * @param[in] fname        Base function name
 * @param[in] prec         Precision
 * @param[in] api          API variant
 * @param[in] acc          Accuracy
 * @return                 Pointer to constructed name
 *
 */
static char* own_full_name(char* buff, int maxlen, char* fname, int prec, int api, int acc)
{
    const char* api_prefix[]    = {"v", "vm", "v", "vm"};
    const char* api_suffix[]    = {"", "", "I", "I"};
    const char* prec_suffix[]   = {"s", "d", "c", "z"};
    const char* acc_suffix[]    = {"HA", "LA", "EP"};

    snprintf (buff, maxlen, "%s%s%s%s_%s ",
              api_prefix[api], prec_suffix[prec], fname, api_suffix[api], acc_suffix[acc]);

    return buff;
} /* own_full_name */

/**
 * @brief Printout ULP value
 *
 * Print arguments, results and ULP difference.
 *
 * @param[in] fname        Function name
 * @param[in] prec         Function precision
 * @param[in] layout       Function arguments layout
 * @param[in] acc          Function accuracy
 * @param[in] api          Function API variant
 * @param[in] idx          Vector index
 * @param[in] in           Input and reference resutl arrays
 * @param[out] out         Output arrays
 *
 */
static void own_print_ulp (char* fname, int prec, int layout, int acc, int api, int idx, double ulp, VmInputData* in, VmOutputData* out)
{
    char strbuff[NAME_LEN] = {0};

    if (prec == kSP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name (strbuff, NAME_LEN, fname, prec, api, acc), idx);
        fprintf (stderr, "%.2g {%a}", in->sarg1[idx], in->sarg1[idx]);
        if ((layout == kVVtoV) || (layout == kVVXtoV)) fprintf (stderr, ", %.2g {%a}", in->sarg2[idx], in->sarg2[idx]);
        fprintf (stderr, ") = %.2g {%a}", out->sres1[api][idx], out->sres1[api][idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.2g {%a}", out->sres2[api][idx], out->sres2[api][idx]);
        fprintf (stderr, ", expected = %.3lg {%la}", in->sref1[idx], in->sref1[idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.3lg {%la}", in->sref2[idx], in->sref2[idx]);
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
    }
    else if (prec == kDP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name (strbuff, NAME_LEN, fname, prec, api, acc), idx);
        fprintf (stderr, "%.3lg {%la}",in->darg1[idx],in->darg1[idx]);
        if ((layout == kVVtoV) || (layout == kVVXtoV)) fprintf (stderr, ", %.3lg {%la}", in->darg2[idx], in->darg2[idx]);
        fprintf (stderr, ") = %.3lg {%la}", out->dres1[api][idx], out->dres1[api][idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.3lg {%la}", out->dres2[api][idx], out->dres2[api][idx]);
        fprintf (stderr, ", expected = %.3lg {%la}", in->dref1[idx], in->dref1[idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.3lg {%la}", in->dref2[idx], in->dref2[idx]);
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
   }
    else if (prec == kCP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name(strbuff, NAME_LEN, fname, prec, api, acc), idx);
        if (layout == kVRtoVC) fprintf (stderr, "%.2g {%a}", in->sarg1[idx], in->sarg1[idx]);
        else                  fprintf (stderr, "%.2g+i*%.2g {%a+i*%a}",
                              creal(in->carg1[idx]), cimag(in->carg1[idx]), creal(in->carg1[idx]), cimag(in->carg1[idx]));
        if (layout == kVVtoV)  fprintf (stderr, ", %.2g+i*%.2g {%a+i*%a}",
                              creal(in->carg2[idx]), cimag(in->carg2[idx]), creal(in->carg2[idx]), cimag(in->carg2[idx]));
        if (layout == kVCtoVR) fprintf (stderr, ") = %.2g {%a}", ((float*)(out->cres1[api]))[idx], ((float*)(out->cres1[api]))[idx]);
        else                  fprintf (stderr, ") = %.2g+i*%.2g {%a+i*%a}",
                              creal(out->cres1[api][idx]), cimag(out->cres1[api][idx]), creal(out->cres1[api][idx]), cimag(out->cres1[api][idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.2g+i*%.2g {%a+i*%a}",
                              creal(out->cres2[api][idx]), cimag(out->cres2[api][idx]), creal(out->cres2[api][idx]), cimag(out->cres2[api][idx]));
        if (layout == kVCtoVR) fprintf (stderr, ", expected = %.3lg {%la}", ((double*)(in->cref1))[idx], ((double*)(in->cref1))[idx]);
        else                  fprintf (stderr, ", expected = %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->cref1[idx]), cimag(in->cref1[idx]), creal(in->cref1[idx]), cimag(in->cref1[idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.2g+i*%.2g {%la+i*%la}", creal(in->cref2[idx]),
                              cimag(in->cref2[idx]), creal(in->cref2[idx]), cimag(in->cref2[idx]));
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
    }
    else if (prec == kZP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name(strbuff, NAME_LEN, fname, prec, api, acc), idx);
        if (layout == kVRtoVC) fprintf (stderr, "%.3lg {%la}", in->darg1[idx], in->darg1[idx]);
        else                  fprintf (stderr, "%.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zarg1[idx]), cimag(in->zarg1[idx]), creal(in->zarg1[idx]), cimag(in->zarg1[idx]));
        if (layout == kVVtoV)  fprintf (stderr, ", %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zarg2[idx]), cimag(in->zarg2[idx]), creal(in->zarg2[idx]), cimag(in->zarg2[idx]));
        if (layout == kVCtoVR) fprintf (stderr, ") = %.3lg {%la}", ((double*)(out->zres1[api]))[idx], ((double*)(out->zres1[api]))[idx]);
        else                  fprintf (stderr, ") = %.3lg+i*%.3lg {%la+i*%la}",
                              creal(out->zres1[api][idx]), cimag(out->zres1[api][idx]), creal(out->zres1[api][idx]), cimag(out->zres1[api][idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.3lg+i*%.3lg {%la+i*%la}",
                              creal(out->zres2[api][idx]), cimag(out->zres2[api][idx]), creal(out->zres2[api][idx]), cimag(out->zres2[api][idx]));
        if (layout == kVCtoVR) fprintf (stderr, ", expected = %.3lg {%la}", ((double*)(in->zref1))[idx], ((double*)(in->zref1))[idx]);
        else                  fprintf (stderr, ", expected = %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zref1[idx]), cimag(in->zref1[idx]), creal(in->zref1[idx]), cimag(in->zref1[idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zref2[idx]), cimag(in->zref2[idx]), creal(in->zref2[idx]), cimag(in->zref2[idx]));
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
    }

    fflush (stderr);
    return;
} /* own_print_ulp */

/**
 * @brief Evaluation of one VM functions family
 *
 * Measure accuracy on VM functions family in comparison to reference scalar implementations.
 *
 * @param[in] fname        Function name
 * @param[in] beg          Begin of input ranges
 * @param[in] end          End of input ranges
 * @param[in] vmfunc       Pointer to VM functions launcher
 * @param[in] reffunc      Real reference function
 * @param[in] creffunc     Complex reference function
 * @param[in] layout       Function arguments layout
 * @param[in] in           Input and reference resutl arrays
 * @param[out] out         Output arrays
 * @return                 Total number of errors
 *
 */
static int own_evaluate_func (char* fname, double beg, double end,
                              VmFunc vmfunc, void* reffunc, void* creffunc,
                              int layout, VmInputData* in, VmOutputData* out)
{
    int printed = 0;
    int err = 0, warn = 0;
    int serr  = 0, derr  = 0, cerr  = 0, zerr  = 0,
        swarn = 0, dwarn = 0, cwarn = 0, zwarn = 0;
    float  sulp[kApiNum][kAccNum]    = {0}, culp[kApiNum][kAccNum]    = {0},
           smaxulp[kApiNum][kAccNum] = {0}, cmaxulp[kApiNum][kAccNum] = {0},
           sresulp[kAccNum]          = {0}, cresulp[kAccNum]          = {0};
    double dulp[kApiNum][kAccNum]    = {0}, zulp[kApiNum][kAccNum]    = {0},
           dmaxulp[kApiNum][kAccNum] = {0}, zmaxulp[kApiNum][kAccNum] = {0},
           dresulp[kAccNum]          = {0}, zresulp[kAccNum]          = {0};
    int has_real = (reffunc != NULL)?1:0, has_complex = (creffunc != NULL)?1:0;

    own_fill_input (layout, beg, end, in);
    own_fill_reference (layout, reffunc, creffunc, in);

    for (int a = kHA; a < kAccNum; a = a + 1)
    {
        /* Launch all API's of function family */
        vmfunc (a, in, out);

        for (int v = kFunc; v < kApiNum; v = v + 1)
        {
            printed = 0;

            if (has_real)
            {
                for (int i = 0; i < length; i = i + 1)
                {
                    warn = 0; err = 0;
                    /* Use stride increment for evaluating strided functions */
                    if (((v == kFuncI) || (v == kMFuncI)) && ((i % stride) || (i >= length/stride))) continue;

                    /* Compute ULP */
                    sulp[v][a] = own_s_compute_ulp (out->sres1[v][i], in->sref1[i]);
                    dulp[v][a] = own_d_compute_ulp (out->dres1[v][i], in->dref1[i]);
                    if (layout == kVtoVV)
                    {
                        sulp[v][a] = fmaxf (sulp[v][a], own_s_compute_ulp (out->sres2[v][i], in->sref2[i]));
                        dulp[v][a] = fmax  (dulp[v][a], own_d_compute_ulp (out->dres1[v][i], in->dref1[i]));
                    }

                    /* Check if ULP is greater than allowed */
                    if (sulp[v][a] > s_allowed_ulp[a])
                    {
                        /* Allows HA/LA linearfrac functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if ((layout == kVVXtoV) && (sulp[v][a] < s_allowed_ulp[kEP])) warn = 1;
                        else err = 1;

                        swarn += warn; serr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kSP, layout, a, v, i, sulp[v][a], in, out);
                            printed++;
                        }
                    }

                    /* Check if ULP is greater than allowed */
                    if (dulp[v][a] > d_allowed_ulp[a])
                    {
                        /* Allows HA/LA linearfrac functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if ((layout == kVVXtoV) && (dulp[v][a] < d_allowed_ulp[kEP])) warn = 1;
                        else err = 1;

                        dwarn += warn; derr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kDP, layout, a, v, i, dulp[v][a], in, out);
                            printed++;
                        }
                    }
                    smaxulp[v][a] = fmaxf (smaxulp[v][a], sulp[v][a]);
                    dmaxulp[v][a] = fmax  (dmaxulp[v][a], dulp[v][a]);
                } /* for (int i = 0; i < length; i++) */
                sresulp[a] = fmaxf (sresulp[a], smaxulp[v][a]);
                dresulp[a] = fmax  (dresulp[a], dmaxulp[v][a]);
            } /* if (has_real) */

            if (has_complex)
            {
                for (int i = 0; i < length; i++)
                {
                    /* Use stride increment for evaluating strided functions */
                    if (((v == kFuncI) || (v == kMFuncI)) && ((i % stride) || (i >= length/stride))) continue;

                    if (layout == kVCtoVR)
                    {
                        float  *sres1 = (float*)(out->cres1[v]);
                        double *sref1 = (double*)(in->cref1);
                        double *dres1 = (double*)(out->zres1[v]), *dref1 = (double*)(in->zref1);
                        culp[v][a] = own_s_compute_ulp (sres1[i], sref1[i]);
                        zulp[v][a] = own_d_compute_ulp (dres1[i], dref1[i]);
                    }
                    else
                    {
                        culp[v][a] = own_c_compute_ulp (out->cres1[v][i], in->cref1[i]);
                        zulp[v][a] = own_z_compute_ulp (out->zres1[v][i], in->zref1[i]);
                    }

                    if (culp[v][a] > c_allowed_ulp[a])
                    {
                        /* Allows HA/LA complex functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if (culp[v][a] < c_allowed_ulp[kEP]) warn = 1;
                        else err = 1;

                        cwarn += warn; cerr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kCP, layout, a, v, i, culp[v][a], in, out);
                            printed++;
                        }
                    }
                    if (zulp[v][a] > z_allowed_ulp[a])
                    {
                        /* Allows HA/LA complex functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if (zulp[v][a] < z_allowed_ulp[kEP]) warn = 1;
                        else err = 1;

                        zwarn += warn; zerr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kZP, layout, a, v, i, zulp[v][a], in, out);
                            printed++;
                        }
                    }
                    cmaxulp[v][a] = fmaxf (cmaxulp[v][a], culp[v][a]);
                    zmaxulp[v][a] = fmax  (zmaxulp[v][a], zulp[v][a]);
                } /* for (int i = 0; i < length; i++) */
                cresulp[a] = fmaxf (cresulp[a], cmaxulp[v][a]);
                zresulp[a] = fmax  (zresulp[a], zmaxulp[v][a]);
            } /* if (has_complex) */
        } /* for (int v = kFunc; v <= kMFuncI; v++) */
    } /* for (int a = kHA; a <= kEP; a++) */

    if (has_real)
    {
        fprintf (stdout, "\ts%-11s, ha:,%7.2g, la:, %7.2g, ep:, %7.2g, %s\n",
                 fname, sresulp[kHA], sresulp[kLA], sresulp[kEP], (serr)?"OVER":(swarn)?"WARN":"NORM");
        fprintf (stdout, "\td%-11s, ha:,%7.2lg, la:, %7.2lg, ep:, %7.2lg, %s\n",
                 fname, dresulp[kHA], dresulp[kLA], dresulp[kEP], (derr)?"OVER":(dwarn)?"WARN":"NORM");
    }
    if (has_complex)
    {
        fprintf (stdout, "\tc%-11s, ha:,%7.2g, la:, %7.2g, ep:, %7.2g, %s\n",
                 fname, cresulp[kHA], cresulp[kLA], cresulp[kEP], (cerr)?"OVER":(cwarn)?"WARN":"NORM");
        fprintf (stdout, "\tz%-11s, ha:,%7.2lg, la:, %7.2lg, ep:, %7.2lg, %s\n",
                 fname, zresulp[kHA], zresulp[kLA], zresulp[kEP], (zerr)?"OVER":(zwarn)?"WARN":"NORM");
    }

    fflush (stdout);
    // Return total number of errors
    return (serr + derr + cerr + zerr);
} /* own_evaluate_func */

/**
 * @brief Provide string description of VML status code
 *
 * vml_status_string provides string description of VML status code st
 *
 * @param[in] st  VML status code
 * @return         const char* with text of corresponding code
 *
 */
static const char* vml_status_string(int st) {

    switch (st) {
        case VML_STATUS_OK: return "VML_STATUS_OK";
        case VML_STATUS_BADSIZE: return "VML_STATUS_BADSIZE";
        case VML_STATUS_BADMEM: return "VML_STATUS_BADMEM";
        case VML_STATUS_ERRDOM: return "VML_STATUS_ERRDOM";
        case VML_STATUS_SING: return "VML_STATUS_SING";
        case VML_STATUS_OVERFLOW: return "VML_STATUS_OVERFLOW";
        case VML_STATUS_UNDERFLOW: return "VML_STATUS_UNDERFLOW";
        case VML_STATUS_ACCURACYWARNING: return "VML_STATUS_ACCURACYWARNING";
    }
    return "VML_STATUS_UNKNOWN";
}

/**
 * @brief Main function for VM API testing
 *
 * Main performs accuracy testing of all VM OMP offload math functions
 *
 * @param[in] argc         Number of arguments
 * @param[in] argv         Pointer to argument strings
 * @return                 0
 *
 */
int main (int argc, char **argv)
{
    /* Total errors */
    int err = 0;

    /* Error satatus */
    int st = VML_STATUS_OK;

    VmInputData  in;
    VmOutputData out;

    fprintf (stdout, "Classic c vm_all_funcs: started...\n"); fflush (stdout);

    own_allocate_data (length, &in, &out);

    vmlClearErrStatus();

    fprintf (stdout, "\t===========================================================\n");
    err += own_evaluate_func ("Asin",       -0.9,   0.9,      own_vm_asin,       asin,           casin,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Acos",       -0.9,   0.9,      own_vm_acos,       acos,           cacos,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Atan",       -10000, 10000,    own_vm_atan,       atan,           catan,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Atan2",      -10000, 10000,    own_vm_atan2,      atan2,          NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Asinh",      -10000, 10000,    own_vm_asinh,      asinh,          casinh,         kVtoV,      &in, &out);
    err += own_evaluate_func ("Acosh",       1.01,  1000,     own_vm_acosh,      acosh,          cacosh,         kVtoV,      &in, &out);
    err += own_evaluate_func ("Atanh",      -0.9,   0.9,      own_vm_atanh,      atanh,          catanh,         kVtoV,      &in, &out);
    err += own_evaluate_func ("Sin",        -10,    10,       own_vm_sin,        sin,            csin,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Cos",        -10,    10,       own_vm_cos,        cos,            ccos,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Tan",        -10,    10,       own_vm_tan,        tan,            ctan,           kVtoV,      &in, &out);
    err += own_evaluate_func ("SinCos",     -10000, 10000,    own_vm_sincos,     own_sincos,     NULL,           kVtoVV,     &in, &out);
    err += own_evaluate_func ("Sinh",       -50,    50,       own_vm_sinh,       sinh,           csinh,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Cosh",       -50,    50,       own_vm_cosh,       cosh,           ccosh,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Tanh",       -5,     5,        own_vm_tanh,       tanh,           ctanh,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Exp",        -75,    75,       own_vm_exp,        exp,            cexp,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Exp2",       -30,    30,       own_vm_exp2,       exp2,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Expm1",      -30,    30,       own_vm_expm1,      expm1,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Ln",          1.01,  100000,   own_vm_ln,         log,            clog,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Log2",        1.01,  100000,   own_vm_log2,       log2,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Log1p",       0.01,  100000,   own_vm_log1p,      log1p,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Pow",         0.1,   10,       own_vm_pow,        pow,            cpow,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Powr",        0.1,   10,       own_vm_powr,       pow,            NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Powx",        0.1,   10,       own_vm_powx,       pow,            cpow,           kVXtoV,     &in, &out);
    err += own_evaluate_func ("Pow2o3",      0.1,   10,       own_vm_pow2o3,     own_pow2o3,     NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Pow3o2",      0.1,   10,       own_vm_pow3o2,     own_pow3o2,     NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Sqrt",        0.1,   100,      own_vm_sqrt,       sqrt,           csqrt,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Cbrt",        0.1,   10000,    own_vm_cbrt,       cbrt,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("InvSqrt",     0.1,   10000,    own_vm_invsqrt,    own_invsqrt,    NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("InvCbrt",     0.1,   10000,    own_vm_invcbrt,    own_invcbrt,    NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Hypot",      -10000, 10000,    own_vm_hypot,      hypot,          NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Erf",        -5,     5,        own_vm_erf,        erf,            NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Erfc",       -2,     5,        own_vm_erfc,       erfc,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("LGamma",      0,     5,        own_vm_lgamma,     lgamma,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("TGamma",      0,     5,        own_vm_tgamma,     tgamma,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("ExpInt1",     0.1,   5,        own_vm_expint1,    own_expint1,    NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Ceil",       -10000, 10000,    own_vm_ceil,       ceil,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Floor",      -10000, 10000,    own_vm_floor,      floor,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Round",      -10000, 10000,    own_vm_round,      round,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Trunc",      -10000, 10000,    own_vm_trunc,      trunc,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Rint",       -10000, 10000,    own_vm_rint,       rint,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("NearbyInt",  -10000, 10000,    own_vm_nearbyint,  nearbyint,      NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Remainder",  -10000, 10000,    own_vm_remainder,  remainder,      NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("LinearFrac", -1000,  1000,     own_vm_linearfrac, own_linearfrac, NULL,           kVVXtoV,    &in, &out);
    err += own_evaluate_func ("Add",        -10000, 10000,    own_vm_add,        own_add,        own_cadd,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Sub",        -10000, 10000,    own_vm_sub,        own_sub,        own_csub,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Mul",        -10000, 10000,    own_vm_mul,        own_mul,        own_cmul,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Div",        -10000, 10000,    own_vm_div,        own_div,        own_cdiv,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Sqr",        -10000, 10000,    own_vm_sqr,        own_sqr,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Inv",        -10000, 10000,    own_vm_inv,        own_inv,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Modf",       -10000, 10000,    own_vm_modf,       own_modf,       NULL,           kVtoVV,     &in, &out);
    err += own_evaluate_func ("Fmod",       -10000, 10000,    own_vm_fmod,       fmod,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Fdim",       -10000, 10000,    own_vm_fdim,       fdim,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Fmax",       -10000, 10000,    own_vm_fmax,       fmax,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Fmin",       -10000, 10000,    own_vm_fmin,       fmin,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("MaxMag",     -10000, 10000,    own_vm_maxmag,     own_maxmag,     NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("MinMag",     -10000, 10000,    own_vm_minmag,     own_minmag,     NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("NextAfter",  -10000, 10000,    own_vm_nextafter,  nextafter,      NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("CopySign",   -10000, 10000,    own_vm_copysign,   copysign,       NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Frac",       -10000, 10000,    own_vm_frac,       own_frac,       NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Conj",       -10000, 10000,    own_vm_conj,       NULL,           conj,           kVtoV,      &in, &out);
    err += own_evaluate_func ("MulByConj",  -10000, 10000,    own_vm_mulbyconj,  NULL,           own_cmulbyconj, kVVtoV,     &in, &out);
    err += own_evaluate_func ("CIS",        -10000, 10000,    own_vm_cis,        NULL,           own_cis,        kVRtoVC,    &in, &out);
    err += own_evaluate_func ("Arg",        -10000, 10000,    own_vm_arg,        NULL,           own_carg,       kVCtoVR,    &in, &out);
    err += own_evaluate_func ("Abs",        -10000, 10000,    own_vm_abs,        fabs,           own_cabs,       kVCtoVR,    &in, &out);
    /* Functions with Intel-specific reference LIBM implementations */
#if (defined __INTEL_COMPILER) || (defined __INTEL_LLVM_COMPILER)
    err += own_evaluate_func ("Asinpi",     -0.9,   0.9,      own_vm_asinpi,     asinpi,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Acospi",     -0.9,   0.9,      own_vm_acospi,     acospi,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Atanpi",     -10000, 10000,    own_vm_atanpi,     atanpi,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Atan2pi",    -10000, 10000,    own_vm_atan2pi,    atan2pi,        NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Sind",       -10000, 10000,    own_vm_sind,       sind,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Cosd",       -10000, 10000,    own_vm_cosd,       cosd,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Tand",       -10000, 10000,    own_vm_tand,       tand,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Sinpi",      -10000, 10000,    own_vm_sinpi,      sinpi,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Cospi",      -10000, 10000,    own_vm_cospi,      cospi,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Tanpi",      -10000, 10000,    own_vm_tanpi,      tanpi,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Exp10",      -30,    30,       own_vm_exp10,      exp10,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Log10",       1.01,  100,      own_vm_log10,      log10,          clog10,         kVtoV,      &in, &out);
    err += own_evaluate_func ("ErfInv",     -0.9,   0.9,      own_vm_erfinv,     erfinv,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("ErfcInv",    -0.1,   1.9,      own_vm_erfcinv,    erfcinv,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("CdfNorm",    -4,     4,        own_vm_cdfnorm,    cdfnorm,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("CdfNormInv", -0.1,   0.9,      own_vm_cdfnorminv, cdfnorminv,     NULL,           kVtoV,      &in, &out);
#endif

    st   = vmlGetErrStatus();
    fprintf (stdout, "\t===========================================================\n");

    fprintf (stdout, "Classic vm_all_funcs: status: %s[%s], accuracy: %s\n\n",
        vml_status_string(st),
        (st >= VML_STATUS_OK) ? "expected" : "unexpected",
        err > 0 ? "over bounds" : "normal"
    );

    own_deallocate_data (&in, &out);

    return 0;
} /* main */

