/*
 *      Copyright (C) 1993,1994 Bas Laarhoven.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 $Source: /usr/src/distr/ftape-0.9.10/RCS/fdc-io.c,v $
 $Author: bas $
 *
 $Revision: 1.24 $
 $Date: 1994/02/20 15:26:55 $
 $State: Alpha $
 *
 *      This file contains the low-level floppy disk interface code
 *      for the QIC-40/80 tape streamer device driver.
 */

static char RCSid[] = "$Id: fdc-io.c,v 1.24 1994/02/20 15:26:55 bas Alpha bas $";

#include <linux/errno.h>
#include <linux/sched.h>
#include <asm/system.h>
#include <asm/io.h>
#include <asm/dma.h>

#include "ftape.h"
#include "fdc-io.h"
#include "fdc-isr.h"
#include "ftape-io.h"
#include "ftape-rw.h"
#include "calibr.h"

/*      Global vars.
 */
int ftape_unit = -1;
int ftape_motor = 0;
int current_cylinder = -1;
fdc_mode_enum fdc_mode = fdc_idle;

/*      Local vars.
 */
static int fdc_calibr_count;
static int fdc_calibr_time;
static int fdc_confused = 0;
static int fdc_status;
volatile byte fdc_head;         /* FDC head */
volatile byte fdc_cyl;          /* FDC track */
volatile byte fdc_sect;         /* FDC sector */

void
fdc_catch_stray_interrupts( unsigned count)
{
  unsigned long flags;

  save_flags( flags);
  cli();
  if (count == 0) {
    expected_stray_interrupts = 0;
  } else {
    expected_stray_interrupts += count;
  }
  restore_flags( flags);
}

/*  Wait during a timeout period for a given FDC status.
 *  If usecs == 0 then just test status, else wait at least for usecs.
 *  Returns -ETIME on timeout. Function must be calibrated first !
 */
int
fdc_wait( int usecs, byte mask, byte state)
{
  int count_1 = (fdc_calibr_count * usecs - 1) / fdc_calibr_time;

  do {
    fdc_status = inb_p( FDC_STATUS);
    if ((fdc_status & mask) == state) {
      return 0;
    }
  } while (count_1-- >= 0);
  return -ETIME;
}

int
fdc_ready_wait( int usecs)
{
  return fdc_wait( usecs, FDC_DATA_READY, FDC_DATA_READY);
}

static void
fdc_usec_wait( int usecs)
{
  fdc_wait( usecs, 0, 1);       /* will always timeout ! */
}

int
fdc_ready_out_wait( int usecs)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  return fdc_wait( usecs, FDC_DATA_OUT_READY, FDC_DATA_OUT_READY);
}

int
fdc_ready_in_wait( int usecs)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  return fdc_wait( usecs, FDC_DATA_OUT_READY, FDC_DATA_IN_READY);
}

int
fdc_wait_calibrate( void)
{
  return calibrate( "fdc_wait",
                   fdc_usec_wait, &fdc_calibr_count, &fdc_calibr_time);
}

/*  Wait for a (short) while for the FDC to become ready
 *  and transfer the next command byte.
 *  Return -ETIME on timeout on getting ready (depends on hardware!).
 */
int
fdc_write( byte data)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  if (fdc_wait( 150, FDC_DATA_READY_MASK, FDC_DATA_IN_READY) < 0) {
    return -ETIME;
  } else {
    outb( data, FDC_DATA);
    return 0;
  }
}

/*  Wait for a (short) while for the FDC to become ready
 *  and transfer the next result byte.
 *  Return -ETIME if timeout on getting ready (depends on hardware!).
 */
int
fdc_read( byte* data)
{
  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  if (fdc_wait( 150, FDC_DATA_READY_MASK, FDC_DATA_OUT_READY) < 0) {
    return -ETIME;
  } else {
    *data = inb( FDC_DATA);
    return 0;
  }
}

/*  Output a cmd_len long command string to the FDC.
 *  The FDC should be ready to receive a new command or
 *  an error (EBUSY) will occur.
 */
int
fdc_command( byte* cmd_data, int cmd_len)
{
  TRACE_FUN( 8, "fdc_command");
  int result = 0;
  unsigned long flags;
  int count = cmd_len;

  fdc_usec_wait( RQM_DELAY);    /* wait for valid RQM status */
  save_flags( flags);
  cli();
  fdc_status = inb( FDC_STATUS);
  if ((fdc_status & FDC_DATA_READY_MASK) == FDC_DATA_IN_READY) {
    int retry = 0;
    fdc_mode = *cmd_data;       /* used by isr */
    interrupt_seen = 0;
    while (count) {
      result = fdc_write( *cmd_data);
      if (result < 0) {
        TRACEx3( 6, "fdc_mode = %02x, status = %02x at index %d",
                (int) fdc_mode, (int) fdc_status, cmd_len - count);
        if (++retry <= 3) {
          TRACE( 2, "fdc_write timeout, retry");
        } else {
          TRACE( 1, "fdc_write timeout, fatal");
          fdc_confused = 1;
          /* recover ??? */
          break;
        }
      } else {
        --count;
        ++cmd_data;
      }
    }
  } else {
    TRACE( 1, "fdc not ready");
    result = -EBUSY;
  }
  restore_flags( flags);
  TRACE_EXIT;
  return result;
}

/*  Input a res_len long result string from the FDC.
 *  The FDC should be ready to send the result or an error
 *  (EBUSY) will occur.
 */
int
fdc_result( byte* res_data, int res_len)
{
  TRACE_FUN( 8, "fdc_result");
  int result = 0;
  unsigned long flags;
  int count = res_len;

  save_flags( flags);
  cli();
  fdc_status = inb( FDC_STATUS);
  if ((fdc_status & FDC_DATA_READY_MASK) == FDC_DATA_OUT_READY) {
    int retry = 0;
    while (count) {
      if (!(fdc_status & FDC_BUSY)) {
        TRACE( 1, "premature end of result phase");
      }
      result = fdc_read( res_data);
      if (result < 0) {
        TRACEx3( 6, "fdc_mode = %02x, status = %02x at index %d",
                (int) fdc_mode, (int) fdc_status, res_len - count);
        if (++retry <= 3) {
          TRACE( 2, "fdc_read timeout, retry");
        } else {
          TRACE( 1, "fdc_read timeout, fatal");
          fdc_confused = 1;
          /* recover ??? */
          break;
        }
      } else {
        --count;
        ++res_data;
      }
    }
  } else {
    TRACE( 1, "fdc not ready");
    result = -EBUSY;
  }
  restore_flags( flags);
  fdc_usec_wait( RQM_DELAY);    /* allow FDC to negate BSY */
  TRACE_EXIT;
  return result;
}

/*      Handle command and result phases for
 *      commands without data phase.
 */
int
fdc_issue_command( byte *out_data, int out_count,
                  byte *in_data, int in_count)
{
  TRACE_FUN( 8, "fdc_issue_command");
  int result;
  int t0, t1;

  if (out_count > 0) {
    result = fdc_command( out_data, out_count);
    if (result < 0) {
      TRACE( 1, "fdc_command failed");
      TRACE_EXIT;
      return result;
    }
  }
  /* will take 24 - 30 usec for fdc_sense_drive_status and
   * fdc_sense_interrupt_status commands.
   *    35 fails sometimes (5/9/93 SJL)
   * On a loaded system it incidentally takes longer than
   * this for the fdc to get ready ! ?????? WHY ??????
   * So until we know what's going on use a very long timeout.
   */
  t0 = timestamp();
  result = fdc_ready_out_wait( 500 /* usec */);
  t1 = timestamp();
  if (result < 0) {
    TRACEi( 1, "fdc_ready_out_wait failed after:", t0 - t1);
    TRACE_EXIT;
    return result;
  }
  if (in_count > 0) {
    result = fdc_result( in_data, in_count);
    if (result < 0) {
      TRACE( 1, "result phase aborted");
      TRACE_EXIT;
      return result;
    }
  }
  TRACE_EXIT;
  return 0;
}

/*      Wait for FDC interrupt with timeout.
 */
int
fdc_interrupt_wait( int time) {
  TRACE_FUN( 8, "fdc_interrupt_wait");
  unsigned long flags;
  struct wait_queue wait = { current, NULL };
  int result = -ETIME;
  
  if (wait_intr) {
    TRACE( 1, "error: nested call");
    return -EIO;                /* return error... */
  }
  save_flags( flags);
  cli();
  if (interrupt_seen == 0) {
    int signal_pending = 0;
   /*   timeout time will be between 0 and MSPT milliseconds too long ! */
    current->timeout = jiffies + 1 + (time + MSPT - 1) / MSPT;
    current->state = TASK_INTERRUPTIBLE;
  
    add_wait_queue( &wait_intr, &wait);
  
    while (1) {
      sti();                    /* allow interrupt_seen */
      while (!interrupt_seen && current->state != TASK_RUNNING) {
        schedule();             /* sets TASK_RUNNING on timeout */
      }
      cli();                    /* prevents late interrupt_seen */
      if (current->signal & ~current->blocked) {
        /* probably awakened by non-blocked signal, fdc probably still busy.
         * will create a mess if we don't reset it !
         */
#if 0
        disable_dma( FLOPPY_DMA);
        fdc_reset();
#endif
        /* hack (?) to allow gracefull exit
         * just mask every signal that interferes
         */
        current->blocked |= (current->signal & ~current->blocked) ;
        signal_pending = 1;
      } else {
        if (signal_pending) {
          TRACE( 1, "error: aborted on signal");
          current->timeout = 0;
          remove_wait_queue( &wait_intr, &wait);
          restore_flags( flags);
          TRACE_EXIT;
          return -EINTR;
        } else {
          break;
        }
      }
    }
    /* get here only on timeout or qic117 interrupt, TASK_RUNNING */
    if (interrupt_seen) {
      current->timeout = 0;     /* interrupt hasn't cleared this */
      result = 0;
    }
    remove_wait_queue( &wait_intr, &wait);
  } else {
    result = 0;
  }
  restore_flags( flags);
  interrupt_seen = 0;		/* should be somewhere else ! */
  TRACE_EXIT;
  return result;
}

/*      Start/stop drive motor. Enable DMA mode.
 */
void
fdc_motor( int motor)
{
  TRACE_FUN( 8, "fdc_motor");
  int unit = FTAPE_UNIT;
  int data = unit | FDC_RESET_NOT | FDC_DMA_MODE;

  ftape_motor = motor;
  if (ftape_motor) {
    data |= FDC_MOTOR_0 << unit;
    TRACEx1( 4, "turning motor %d on", unit);
  } else {
    TRACEx1( 4, "turning motor %d off", unit);
  }
  outb_p( data, FDC_CONTROL);
  ftape_sleep( 10 * MILLISECOND);
  TRACE_EXIT;
}

/*      Reset the floppy disk controller. Leave the ftape_unit selected.
 */
void
fdc_reset(void)
{
  TRACE_FUN( 8, "fdc_reset");
  int unit = FTAPE_UNIT;
  byte fdc_ctl = unit | FDC_DMA_MODE;

  expect_stray_interrupt = 1;
  if (ftape_motor) {
    fdc_ctl |= FDC_MOTOR_0 << unit;
  }
  /* Assert the reset line.  Leave the proper unit selected.
   */
  outb_p( fdc_ctl, FDC_CONTROL);
  ftape_sleep( MILLISECOND);    /* > 14 fdc clocks */
  /* Now release the reset line.
   */
  fdc_ctl |= FDC_RESET_NOT;
  outb_p( fdc_ctl, FDC_CONTROL);
  ftape_sleep( 10 * MILLISECOND); /* ??? */
  /* Select clock for fdc.  May need to be changed to select data rate.
   */
  outb_p( 0, FDC_VFO);          /* set high density (500 Kbps) mode */
  expect_stray_interrupt = 0;
  TRACE_EXIT;
}

/* When we're done, put the fdc into reset mode so that the regular
   floppy disk driver will figure out that something is wrong and
   initialize the controller the way it wants. */
void
fdc_disable( void)
{
  outb_p( FTAPE_UNIT, FDC_CONTROL);
}

/*      Specify FDC seek-rate
 */
int
fdc_set_seek_rate( int seek_rate)
{
  byte in[ 3];

  in[ 0] = FDC_SPECIFY;
  in[ 1] = ((16 - seek_rate) << 4) | 13 /* head_unload_time */;
  in[ 2] = (1 /* head_load_time */ << 1) | 0 /* non_dma */;

  return fdc_command( in, 3);
}

/*      Sense drive status: get unit's drive status (ST3)
 */
int
fdc_sense_drive_status( int *st3)
{
  TRACE_FUN( 8, "fdc_sense_drive_status");
  int result;
  byte out[ 2];
  byte in[ 1];

  out[ 0] = FDC_SENSED;
  out[ 1] = FTAPE_UNIT;
  result = fdc_issue_command( out, 2, in, 1);
  if (result < 0) {
    TRACE( 1, "issue_command failed");
  } else {
    *st3 = in[ 0];
    result = 0;
  }
  TRACE_EXIT;
  return result;
}

/*      Sense Interrupt Status command:
 *      should be issued at the end of each seek.
 *      get ST0 and current cylinder.
 */
int
fdc_sense_interrupt_status( int *st0, int *current_cylinder)
{
  TRACE_FUN( 8, "fdc_sense_interrupt_status");
  int result;
  byte out[ 1];
  byte in[ 2];

  out[ 0] = FDC_SENSEI;
  result = fdc_issue_command( out, 1, in, 2);
  if (result) {
    TRACE( 1, "issue_command failed");
  } else {
    *st0 = in[ 0];
    *current_cylinder = in[ 1];
    result = 0;
  }
  TRACE_EXIT;
  return result;
}

/*      step to track
 */
int
fdc_seek( int track)
{
  TRACE_FUN( 8, "fdc_seek");
  int result;
  byte out[3];
  int st0, pcn;

  ftape_sleep( MILLISECOND);    /* NEEDED to prevent problems */

  out[ 0] = FDC_SEEK;
  out[ 1] = FTAPE_UNIT;
  out[ 2] = track;
  seek_completed = 0;
  result = fdc_command( out, 3);
  if (result != 0) {
    TRACEi( 1, "failed, status =", result);
    TRACEx1( 4, "destination was: %d, resetting FDC...", track);
    /*  We really need this command to work !
     */
    fdc_reset();
    fdc_set_seek_rate( 2);
    TRACE_EXIT;
    return result;
  }

  /*    Handle interrupts until seek_completed or timeout.
   */
  for (;;) {
    result = fdc_interrupt_wait( 2 * SECOND);
    if (result < 0) {
      TRACEi( 2, "fdc_interrupt_wait timeout, status =", result);
      TRACE_EXIT;
      return result;
    } else if (seek_completed) {
      result = fdc_sense_interrupt_status( &st0, &pcn);
      if (result != 0) {
        TRACEi( 1, "fdc_sense_interrupt_status failed, status =", result);
        TRACE_EXIT;
        return result;
      }
      if ((st0 & ST0_SEEK_END) == 0) {
        TRACE( 1, "no seek-end after seek completion !??");
        TRACE_EXIT;
        return -EIO;
      }
      break;
    }
  }
  /*    Verify whether we issued the right tape command.
   */
  /* Verify that we seek to the proper track. */
  if (pcn != track) {
    TRACE( 1, "bad seek..");
    TRACE_EXIT;
    return -EIO;
  }
  current_cylinder = pcn;
  TRACE_EXIT;
  return 0;
}

/*      Recalibrate and wait until home.
 */
int
fdc_recalibrate( void)
{
  TRACE_FUN( 8, "fdc_recalibrate");
  int result;
  byte out[ 2];
  int st0;
  int pcn;
  int retry;

  result = fdc_set_seek_rate( 6);
  if (result) {
    TRACEi( 1, "fdc_set_seek_rate failed, status =", result);
    TRACE_EXIT;
    return result;
  }
  out[ 0] = FDC_RECAL;
  out[ 1] = FTAPE_UNIT;
  seek_completed = 0;
  result = fdc_command( out, 2);
  if (result) {
    TRACEi( 1, "fdc_command failed, status =", result);
    TRACE_EXIT;
    return result;
  }
  /*    Handle interrupts until seek_completed or timeout.
   */
  for (retry = 0;; ++retry) {
    result = fdc_interrupt_wait( 2 * SECOND);
    if (result < 0) {
      TRACE( 1, "fdc_interrupt_wait failed");
      TRACE_EXIT;
      return result;
    } else if (result == 0 && seek_completed) {
      result = fdc_sense_interrupt_status( &st0, &pcn);
      if (result != 0) {
        TRACEi( 1, "fdc_sense_interrupt_status failed, status =", result);
        TRACE_EXIT;
        return result;
      }
      if ((st0 & ST0_SEEK_END) == 0) {
        if (retry < 1) {
          continue;             /* some drives/fdc's give an extra interrupt */
        } else {
          TRACE( 1, "no seek-end after seek completion !??");
          TRACE_EXIT;
          return -EIO;
        }
      }
      break;
    }
  }
  current_cylinder = pcn;
  if (pcn != 0) {
    TRACEi( 1, "failed: resulting track =", pcn);
  }
  result = fdc_set_seek_rate( 2);
  if (result != 0) {
    TRACEi( 1, "fdc_set_seek_rate failed, status =", result);
    TRACE_EXIT;
    return result;
  }
  TRACE_EXIT;
  return 0;
}

/*      Setup Floppy Disk Controller and DMA to read or write the next cluster
 *      of good sectors from or to the current segment.
 */
int
setup_fdc_and_dma( buffer_struct* buff, unsigned char operation)
{
  TRACE_FUN( 8, "setup_fdc_and_dma");
  unsigned long flags;
  unsigned char out[ 9];
  int result;
  int dma_mode;

  if (operation == FDC_READ || operation == FDC_READ_DELETED) {
    dma_mode = DMA_MODE_READ;
    TRACEx2( 5, "xfer %d sectors to 0x%p", buff->sector_count, buff->ptr);
  } else if (operation == FDC_WRITE || operation == FDC_WRITE_DELETED) {
    dma_mode = DMA_MODE_WRITE;
    TRACEx2( 5, "xfer %d sectors from 0x%p", buff->sector_count, buff->ptr);
  } else {
    TRACE( -1, "bug: illegal operation parameter");
    TRACE_EXIT;
    return -EIO;
  }
  /* Program the DMA controller.
   */
  save_flags( flags);
  cli();                      /* could be called from ISR ! */
  disable_dma( FLOPPY_DMA);
  clear_dma_ff( FLOPPY_DMA);
  set_dma_mode( FLOPPY_DMA, dma_mode);
  set_dma_addr( FLOPPY_DMA, (unsigned) buff->ptr);
  set_dma_count( FLOPPY_DMA, SECTOR_SIZE * buff->sector_count);
#ifdef GCC_2_4_5_BUG
  /*  This seemingly stupid construction confuses the gcc-2.4.5
   *  code generater enough to create correct code.
   */
  if (1) {
    int i;

    for (i = 0 ; i < 1 ; ++i) {
      udelay( 1);
    }
  }
#endif
  enable_dma( FLOPPY_DMA);
  /* Issue FDC command to start reading/writing.
   */
  out[ 0] = operation;
  out[ 1] = FTAPE_UNIT;
  out[ 2] = buff->cyl;
  out[ 3] = buff->head;
  out[ 4] = buff->sect + buff->sector_offset;
  out[ 5] = 3;                  /* Sector size of 1K. */
  out[ 6] = out[ 4] + buff->sector_count - 1; /* last sector */
  out[ 7] = 109;                /* Gap length. */
  out[ 8] = 0xff;		/* No limit to transfer size. */
  restore_flags( flags);
  TRACEx4( 6, "c: 0x%02x, h: 0x%02x, s: 0x%02x, "
          "cnt: 0x%02x", out[ 2], out[ 3], out[ 4], out[ 6]- out[ 4]+ 1);
  result = fdc_command( out, 9);
  if (result != 0) {
    fdc_mode = fdc_idle;
    TRACE( 1, "fdc_command failed");
  }
  fdc_setup_error = result;
  TRACE_EXIT;
  return result;
}

