/* Panel managing.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include "global.h"
#include "dir.h"
#include "panel.h"
#include "util.h"
#include "color.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/screen.c,v 1.3 1994/05/14 21:34:59 miguel Exp $";

int show_mini_info = 1;		/* Show miniinfo */

int extra_info = show_perm;

static void set_colors (Panel *panel)
{
    wstandend (panel->win_file);
    if (hascolors){
	wattron (panel->win_file, NORMAL_COLOR);
    }
}

void panel_refresh (Panel *panel)
{
    if (panel->view_type & VIEW_DISABLED)
	return;
    wrefresh (panel->win_file);
}

int set_attr (Panel *panel, int hilight, int marked)
{
    int color;

    color = (marked * 2 + hilight);
    wstandend (panel->win_file);
    
    /* Does this work on B&W terminals? */
    if (hascolors)
	wattron (panel->win_file,sel_mark_color[color]|(marked ? A_BOLD : 0));
    else {
	if (hilight)
	    wattron (panel->win_file, A_REVERSE | (marked ? A_BOLD : 0));
	else
	    if (marked)
		wattron (panel->win_file, A_BOLD);
    }
}

/* Use current attributes to update the contents of the file */
void repaint_file (Panel *panel, int file_index)
{
    int c = ' ';
    int isdir = 0;
    int column;
    
#define entry (panel->dir [file_index].buf)
    
    if (S_ISDIR (entry.st_mode)){
	isdir = 1;
	c = '/';
    } else if (S_ISLNK (entry.st_mode))
	c = '@';
    else if (S_ISSOCK (entry.st_mode) || S_ISFIFO (entry.st_mode))
	c = '=';
    else if (is_exe (entry.st_mode))
	c = '*';
    
    switch (panel->view_type){
    case view_brief:
	column = (file_index - panel->top_file) / panel->lines;
	wmove (panel->win_file,
	       (file_index - panel->top_file) % panel->lines + 2,
	       (column * (COLS/4-1) + 1));
	wprintw (panel->win_file, "%c%-*s", c, COLS/4-3,
		 name_trunc (panel->dir [file_index].fname, COLS/4-3));
	if (column)
	    waddch (panel->win_file, ' ');
	else
	    waddch (panel->win_file, ACS_VLINE);
	break;

    case view_long:
	/* Only here temporarily */
	wmove (panel->win_file, file_index - panel->top_file + 2, 1);
	wprintw (panel->win_file,
		 "%10s %3d %8s %8s %11s %12s %-*s",
		 string_perm (entry.st_mode),
		 entry.st_nlink,
		 get_owner (entry.st_uid),
		 get_group (entry.st_gid),
		 size_trunc (entry.st_size),
		 file_date (entry.st_ctime),
		 COLS-60,
		 name_trunc (panel->dir [file_index].fname, COLS-60)
		 );

	break;
	
	
    case view_full:
	wmove (panel->win_file, file_index - panel->top_file + 2, 1);
	wprintw (panel->win_file, 
		   "%c%-*s", c, COLS/2-24, split_extension
		 (name_trunc (panel->dir [file_index].fname, COLS/2-24), COLS/2-24));
	waddch (panel->win_file, ACS_VLINE);
	wprintw (panel->win_file, "%9s",
		 isdir ? ">SUB-DIR<" : size_trunc (entry.st_size));
	waddch (panel->win_file, ACS_VLINE);

	switch (extra_info){
	case show_perm:
	    wprintw (panel->win_file, "%10s", string_perm (entry.st_mode));
	    break;

	case show_dates:
	    wprintw (panel->win_file, "No dates yet");
	    break;
	}
    }
#undef stat
}

void empty_entry (Panel *panel, int file_index)
{
    int column;
    
    switch (panel->view_type){
    case view_brief:
	column = (file_index - panel->top_file) / panel->lines;
	wmove (panel->win_file,
	       (file_index - panel->top_file) % panel->lines + 2,
	       (column * (COLS/4-1) + 1));
	wprintw (panel->win_file, "%*s", COLS/4-2, "");
	if (column)
	    waddch (panel->win_file, ' ');
	else
	    waddch (panel->win_file, ACS_VLINE);
	break;
	
    case view_full:
	wmove (panel->win_file, file_index - panel->top_file + 2, 1);
	wprintw (panel->win_file, "%*s", COLS/2-2, "");
	break;

    case view_long:
	wmove (panel->win_file, file_index - panel->top_file + 2, 1);
	wprintw (panel->win_file, "%*s", COLS-2, "");
    }
}

void display_mini_info (Panel *panel)
{
    if (show_mini_info){
	wmove (panel->win_file, panel->lines+3, 1);
	if (panel->marked){
	    wattrset (panel->win_file, MARKED_COLOR | A_BOLD);
	    wprintw (panel->win_file, "  %s bytes in %d files",
		     size_trunc (panel->total), panel->marked);
	} else {
	    set_colors (panel);
	    wprintw(panel->win_file,"%-*s", panel->cols,
		    name_trunc(panel->dir [panel->selected].fname, COLS/2-2));
	}
    }
}

paint_dir (Panel *panel)
{
    int i;
    int hilight;		/* Color used for hilighting */
    int marked;			/* Color used for marked file */
    int items;			/* Number of items */

    if (panel->view_type & VIEW_DISABLED)
	return;
    
    items = panel->view_type == view_brief ?
	panel->lines * 2 : panel->lines;
    
    for (i = 0; i < items; i++){
	if (i+panel->top_file >= panel->count){
	    set_attr (panel, 0, 0);
	    empty_entry (panel, i+panel->top_file);
	} else {
	    hilight = panel->selected==i+panel->top_file && panel->active;
	    marked  = panel->dir [i+panel->top_file].f.marked;
	    set_attr (panel, hilight, marked);
	    repaint_file (panel, i+panel->top_file);
	}
    }
    wstandend (panel->win_file);
}

void show_dir (Panel *panel)
{
    char tmp [81];

    set_colors (panel);
    box (panel->win_file, ACS_VLINE, ACS_HLINE);
    if (panel->active)
	wattron (panel->win_file, A_REVERSE);
    trim (strip_home (panel->cwd), tmp, panel->cols-3);
    mvwaddstr (panel->win_file, 0, 1, tmp);
    if (panel->active) wstandend (panel->win_file);
}

/* To be used only by long_frame and full_frame to adjust top_file */
static void adjust_top_file (Panel *panel)
{
    if (panel->selected - panel->top_file > panel->lines)
	panel->top_file = panel->selected;
}

brief_frame (Panel *panel)
{
    panel->cols = COLS/2-2;
    panel->win_file = panel->small_frame;
    werase (panel->win_file);
    show_dir (panel);

    wmove (panel->win_file, 1, 1);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, " %-17s", "     Name");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, " %-18s", "     Name");
}

full_frame (Panel *panel)
{
    panel->win_file = panel->small_frame;
    panel->cols = COLS/2-2;
    adjust_top_file (panel);
    werase (panel->win_file);
    show_dir (panel);

    wmove (panel->win_file, 1, 1);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, " %-*s", COLS/2-24, "     Name");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, "%9s", "Size  ");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, "%10s", "Permission");
}

long_frame (Panel *panel)
{
    panel->win_file = panel->big_frame;
    panel->cols = COLS-2;
    adjust_top_file (panel);
    werase (panel->win_file);
    show_dir (panel);

    wmove (panel->win_file, 1, 1);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, "Permission");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, "Lnk");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, " Owner  ");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, " Group  ");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, "    Size   ");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, "    Date    ");
    set_attr (panel, 0, 0);
    waddch (panel->win_file, ACS_VLINE);
    set_attr (panel, 0, 1);
    wprintw (panel->win_file, "%-*s", COLS-60, " Name");
}

paint_panel (Panel *panel)
{
    if (panel->view_type & VIEW_DISABLED)
	return;
    show_dir (panel);
    paint_dir (panel);
    display_mini_info (panel);
    if (show_mini_info){
	wstandend (panel->win_file);
	wmove (panel->win_file, panel->lines+2, 1);
	whline (panel->win_file, ACS_HLINE|NORMAL_COLOR, panel->cols);
    }
    panel_refresh (panel);
}

void Xtry_to_select (Panel *panel, char *name)
{
    int i;
    char *subdir;
    
    if (!name){
	panel->selected = 0;
	panel->top_file = 0;
	return;
    }

    /* We only want the last component of the directory */
    for (subdir = name + strlen (name) - 1; subdir >= name; subdir--){
	if (*subdir == '/'){
	    subdir++;
	    break;
	}
    }
    if (subdir < name)
	subdir = name;
    
    /* Search that subdirectory, if found select it */
    for (i = 0; i < panel->count; i++){
	if (strcmp (subdir, panel->dir [i].fname))
	    continue;

	if (i != panel->selected){
	    panel->selected = i;
	    panel->top_file = panel->selected - panel->lines/2;
	    if (panel->top_file < 0)
		panel->top_file = 0;
	}
	return;
    }
    panel->selected = 0;
    panel->top_file = 0;
}

init_panel (Panel *panel, int x1, int y1, int x2, int y2)
{
    getwd (panel->cwd);
    panel->sort_type = (sortfn *) sort_name;
    panel->win_file = newwin (y2 - y1, x2 - x1, y1, x1);
    panel->small_frame = panel->win_file;
    panel->big_frame   = newwin (y2 - y1, COLS, 0, 0); 
    
    set_colors (panel);
    wclr (panel->small_frame);
/*    wclr (panel->big_frame);  */

    panel->lines = y2-y1-3-(show_mini_info*2);
    panel->cols  = x2-x1-2;
    panel->view_type = view_full;
    panel->top_file = 0;
    panel->selected = 0;
    panel->is_status = 0;
    panel->marked = 0;
    panel->total = 0;
    full_frame (panel);

    /* Configure repaint mode for the panel */
    leaveok (panel->win_file, TRUE);
    leaveok (panel->big_frame, TRUE);
}

void panel_reload (Panel *panel)
{
    int i;
    struct stat current_stat;
    
    if (!PANEL_ISVIEW (panel))
	return;
    stat (panel->cwd, &current_stat);
    if (current_stat.st_ctime == panel->dir_stat.st_ctime &&
	current_stat.st_mtime == panel->dir_stat.st_mtime)
	return;
    chdir (panel->cwd);
    panel->count = do_reload_dir (&panel->dir, panel->sort_type, panel->count);
    panel->marked = 0;
    for (i = 0; i < panel->count; i++)
	if (panel->dir [i].f.marked)
	    panel->marked++;
    display_mini_info (panel);
}


