#include <ctype.h>
#include <strings.h>
#include <OI/oi.H>

/*
 *	seq entry check
 *
 *	This program was created to introduce the programmer to 
 *	providing their own entry, increment, and decrement routines
 *	for the oi_seq_entry_field object.
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *		- OI_d_tech
 *			set_associated_object()
 *			layout_associated_object()
 *		- OI_app_window
 *			oi_create_app_window()
 *		- OI_entry_field
 *			set_char_chk()
 *			set_entry_chk()
 *		- OI_seq_entry_field
 *			oi_create_seq_entry_field()
 *			allow_inc_dec_chk()
 *			set_inc()
 *			set_dec()
 */


/*
 *	define a list of objects to be passed to the
 *	entry check routines.
 */
typedef struct _ObjectList {
	char		**names ;
	OI_number	size ;
} ObjectList ;

char * DayNames[] =
{
	"Sunday",
	"Monday",
	"Tuesday",
	"Wednesday",
	"Thursday",
	"Friday",
	"Saturday",
};
ObjectList	DayList = { DayNames, OI_count( DayNames ) };

char * MonthNames[] =
{
	"January",
	"February",
	"March",
	"April",
	"May",
	"June",
	"July",
	"August",
	"September",
	"October",
	"November",
	"December",
};
ObjectList	MonthList = { MonthNames, OI_count( MonthNames ) };


/*
 *	lowercase()
 *
 *	This utility routine converts a string to lower case.
 */
static void
lowercase( char * p, int n )
{
	while ( n-- ) {
		if (isupper( *p ))
			*p = tolower( *p );
		++p;
	}
}


/*
 *	match text
 *
 *	This utility routine matches the entered character string
 *	against a list of character strings.  The list is assumed
 *	to have entries of the form: "Monday", that is, first character
 *	is upper case, all remaining characters are lower case.
 */
static int
match_text( char * day, char **list, int n )
{
	int	i ;
	int	rtn = -1 ;
	char	buf[ 256 ] ;

	if (day) {
		strcpy( buf, day );
		lowercase( buf, strlen( buf ) );
		if (islower( buf[0] ))
			buf[0] = toupper( buf[0] );
		for (i=0; i<n; i++, list++)
			if (!strcmp( buf, *list )) {
				rtn = i ;
				break ;
			}
	}
	return( rtn );
}


/*
 *	list entry increment
 *
 *	This routine takes a sequential entry field, and an ObjectList
 *	as input, and increments the selected object.  This routine will
 *	wrap from highest to lowest entry.
 */
static int
list_entry_inc( OI_seq_entry_field * sef, void * arg )
{
	OI_number	n;
	ObjectList	*info = (ObjectList *) arg;

	n = match_text( sef->part_text(), info->names, info->size );
	if (n < 0)
		n = 0;
	else {
		if (++n >= info->size) n = 0 ;
	}
	sef->set_text( info->names[ n ] );
	return( 1 );
}


/*
 *	list entry decrement
 *
 *	This routine takes a sequential entry field, and an ObjectList
 *	as input, and decrements the selected object.  This routine will
 *	wrap from lowest to highest entry.
 */
static int
list_entry_dec( OI_seq_entry_field * sef, void * arg )
{
	OI_number	n;
	ObjectList	*info = (ObjectList *) arg;

	n = match_text( sef->part_text(), info->names, info->size );
	if (n < 0)
		n = 0;
	else {
		if (--n < 0) n = info->size - 1;
	}
	sef->set_text( info->names[ n ] );
	return( 1 );
}


/*
 *	list entry check
 *
 *	This routine takes a sequential entry field, and an ObjectList
 *	as input, and validates the user entered text against permissible
 *	entries, as defined in the list.
 */
static OI_ef_entry_chk_status
list_entry_chk( OI_entry_field * ent, void * arg )
{
	ObjectList	*info = (ObjectList *) arg;
	if (match_text( ent->part_text(), info->names, info->size ) >= 0)
		return( OI_EF_ENTRY_CHK_OK );
	return( OI_EF_ENTRY_CHK_BAD );
}


void
main( int argc, char **argv )
{
	OI_app_window		*win ;
	OI_seq_entry_field	*days ;
	OI_seq_entry_field	*months ;
	OI_connection		*conp ;

	
	/*
	 *	Open a connection.
	 */
	if (conp = OI_init(&argc, argv, "SeqEntryTest")) {
		/*
		 *	create the application window.
		 *	set layout to row major.
		 */
		win = oi_create_app_window( "mainWin", 1, 1, "Seq Ent Field Demo" );
		win->set_layout( OI_layout_column );

		/*
		 *	create the day seq entry field
		 *	- allow increment, decrement check to be performed on the entries.
		 *	- provide an increment check function
		 *	- provide a decrement check function.
		 *	- disable the character check function, since non numberic 
		 *		characters will be typed.
		 *	- set entry check for user entered values.
		 */
		days = oi_create_seq_entry_field( "dayEntry", 12, "day: ", "Monday", 32 );
		days->allow_inc_dec_chk();
		days->set_inc( (OI_sequence_fnp) list_entry_inc, (void *) &DayList );
		days->set_dec( (OI_sequence_fnp) list_entry_dec, (void *) &DayList );
		days->set_char_chk( (OI_ef_char_chk_fnp) NULL, (void *) NULL );
		days->set_entry_chk( (OI_ef_entry_chk_fnp) list_entry_chk, (void *) &DayList );
		days->layout_associated_object( win, 0, 1, OI_ACTIVE );

		/*
		 *	create the month seq entry field
		 *	- allow increment, decrement check to be performed on the entries.
		 *	- provide an increment check function
		 *	- provide a decrement check function.
		 *	- disable the character check function, since non numberic 
		 *		characters will be typed.
		 *	- set entry check for user entered values.
		 */
		months = oi_create_seq_entry_field( "monthEntry", 12, "month: ", "January", 32 );
		months->allow_inc_dec_chk();
		months->set_inc( (OI_sequence_fnp) list_entry_inc, (void *) &MonthList );
		months->set_dec( (OI_sequence_fnp) list_entry_dec, (void *) &MonthList );
		months->set_char_chk( (OI_ef_char_chk_fnp) NULL, (void *) NULL );
		months->set_entry_chk( (OI_ef_entry_chk_fnp) list_entry_chk, (void *) &MonthList );
		months->layout_associated_object( win, 0, 2, OI_ACTIVE );

		/*
		 *	display main window.
		 *	begin interaction.
		 */
		win->set_associated_object( win->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE );
		OI_begin_interaction();
	}

	/*
	 *	Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
}
