#include <stdlib.h>
#include <stddef.h>	/* for offsetof() macro */
#include <values.h>
#include <OI/oi.H>


/*
 *	Hex Entry Field
 *	
 *	This program provides a complete subclassing example.
 *	This program demonstrates how to create a hex entry
 *	field subclass from the OI_entry_field base class.
 *	Additionally, this program demonstrates three additions
 *	to a simple subclass:
 *		- the class is registered so that class hierarchy
 *			queries will work.
 *		- resources have been added.
 *		- an interface function has been added in case of
 *			constructor failure.
 *
 *	Under normal situations, the class definitions would be 
 *	defined in a separate include file, but they have been
 *	incorporated into this file for completeness.
 *
 *	The reader should refer back to the README file for
 *	simple subclassing examples.
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_init()
 *			OI_begin_interaction()
 *		- OI_resources
 *		- OI_d_tech
 *			layout_associated_object()
 *			set_associated_object()
 *			set_layout()
 *		- OI_app_window
 *			oi_create_app_window()
 *		- OI_entry_field
 *			set_char_chk()
 *				OI_EF_CHAR_CHK_BAD
 *				OI_EF_CHAR_CHK_INSERT
 *			set_entry_chk()
 *				OI_EF_ENTRY_CHK_OK
 *				OI_EF_ENTRY_CHK_BAD
 *
 *	Please note that you can easily extend the HexEntryField
 *	class to provide ways to override any of the base class
 *	member functions, or read or change the maximum allowed
 *	value:
 *			int	maximum_value();
 *			void	set_maximum_value( int mx_val );
 *	Please refer to the get_resources member function for 
 *	indication on how to override a base class member function,
 *	but still call the base class member function.
 */

/*
 *	Define the HexEntryField class, derived from the
 *	OI_entry_field base class.
 *
 *	The class definition adds these data items:
 *		- mx_val
 *			the maximum value allowed.
 *	The class definition adds these member functions:
 *		- construct
 *		- char_chk
 *			this member function is used as the default
 *			character check function, and is called every
 *			time a character is typed to validate the
 *			character.
 *		- entry_chk
 *		- HexEntryField
 *			object constructor
 */
class	HexEntryField : public OI_entry_field {
	int			mx_val ;			/* max value allowed */
 private:
	void			construct(int) ;
	OI_ef_char_chk_status	char_chk(OI_entry_field*,void*,OI_number,char) ;
	OI_ef_entry_chk_status	entry_chk(OI_entry_field*,void*) ;

 protected:
				HexEntryField(OI_class *clsp, char *np, OI_number ln, char *lp, char *dp, int mx)
							: OI_entry_field(clsp,np,ln,lp,dp,ln) { construct(mx); }
 public:
	/*
	 *	public functions reserved for class usage
	*/
				HexEntryField(char*, OI_number, char*, char*, int) ;
				~HexEntryField();
	/*
	 *	normal public functions
	*/
	virtual	void		get_resources();
	int			max_value()		{ return(mx_val); }
} ;
HexEntryField::~HexEntryField() { }
HexEntryField		*create_HexEntryField(char*,OI_number,char* =NULL,char* ="0",int=0) ;



/*
 *	register the HexEntryField model class
 *
 *	register the class so that class hierarchy queries will work.
 */
static	OI_class	*my_classp ;		/* ptr to class object for this class */
void
reg_HexEntryField()
{
	my_classp = OI_register_class("HexEntryField", "OI_entry_field", NULL, NULL) ;
	return ;
}


/*
 *	create_HexEntryField
 *
 *	generic interface function that ensures that the constructor
 *	succeeded, and that no internal error occurred.
 *	Upon success, a pointer to the object is returned.  On failure,
 *	if the object was created it is destroyed, and a NULL is returned,
 *	thus permitting the programmer to use the following coding style:
 *
 *		if (hex = create_HexEntryField( ... )) {
 *			// success
 *		}
 *		else {
 *			// failure
 *		}
 *
 */
HexEntryField*
create_HexEntryField(char *unp, OI_number ln, char *lbp, char *dfp, int mx)
{
	HexEntryField	*p ;

	if ((p = new HexEntryField(unp,ln,lbp,dfp,mx)) && (OI_status<0)) {
		p->del() ;
		p = NULL ;
	}
	return (p) ;
}


/*
 *	HexEntryField::HexEntryField
 *
 *	The HexEntryField object constructor.  The HexEntryField
 *	constructor requires the following arguments:
 *		1. object name, 
 *		2. number of characters displayed in the entry field
 *		3. default entry field label,
 *		4. default value to display.
 *	The base class (OI_entry_field) uses these arguments along with
 *	a fifth argument (maximum number of characters).  In this example
 *	we will default the maximum number of characters to equal the
 *	number of characters displayed.
 *
 *	When invoked, the HexEntryField constructor invokes the
 *	base class (OI_entry_field) constructor with appropriate arguments.
 */	
HexEntryField::HexEntryField (
	char		*usr_namp,	/* name of object */
	OI_number	len,		/* # chars displayed in entry field */
	char		*labp,		/* ptr to label for field */
	char		*defp,		/* ptr to default string */
	int		mx)		/* max # chars in entry field */
	: OI_entry_field(my_classp,usr_namp,len,labp,defp,len)
{
	construct(mx) ;
}


/*
 *	HexEntryField::construct
 *
 *	Please note, that this member function exists for the following reasons:
 *	The basic object constructor (this member function) is entered two ways
 *		via the public constructor:	HexEntryField::HexEntryField( name, ...  )
 *		and,
 *		via the protected constructor:	HexEntryField::HexEntryField( classp, name, ... )
 *	The protected constructor exists to permit further subclassing off of this object.
 *
 *	If the object was created successfully (OI_status >=0), then
 *	set both the character and entry check callbacks for this
 *	object.
 */	
void
HexEntryField::construct (
	int	mx)		/* max value allowed */
{
	if (OI_status >= 0) {
		if (! (mx_val = mx))
			mx_val = MAXINT ;
		set_char_chk(this,(OI_ef_char_chk_memfnp)&HexEntryField::char_chk) ;
		set_entry_chk(this,(OI_ef_entry_chk_memfnp)&HexEntryField::entry_chk) ;
	}
}


/*
 *	HexEntryField::char_chk
 *
 *	This member function validates each character as typed.
 *	If the character is not a valid hexadecimal character
 *		[0-9, a-f, A-F]
 *	then an error (OI_EF_CHAR_CHK_BAD) is returned, otherwise
 *	success (OI_EF_CHAR_CHK_INSERT) is returned.
 *
 */
OI_ef_char_chk_status
HexEntryField::char_chk (OI_entry_field*, void*, OI_number, char c)
{
	OI_ef_char_chk_status	ok ;

	ok = OI_EF_CHAR_CHK_BAD ;
	if ((c != '\0') && isxdigit(c))
		ok = OI_EF_CHAR_CHK_INSERT ;
	return(ok) ;
}


/*
 *	HexEntryField::entry_chk
 *
 *	This member function validates the input text once the
 *	value has been entered (carriage return pressed).
 *	If the value is greater than or equal to the specified
 *	maximum value for this field, then an error is returned
 *	(OI_EF_ENTRY_CHK_BAD), otherwise success (OI_EF_ENTRY_CHK_OK)
 *
 */
OI_ef_entry_chk_status
HexEntryField::entry_chk (OI_entry_field*, void*)
{
	OI_ef_entry_chk_status	ok ;
	int			n ;

	ok = OI_EF_ENTRY_CHK_OK ;
	if (part_text()) {
		if ((n = atoi(part_text())) > mx_val)
			ok = OI_EF_ENTRY_CHK_BAD ;
	}
	return(ok) ;
}


/*
 *	Resources
 *
 *	define the resources available for this object.
 *	The OI_resource structure is further discussed in the documentation
 *	under OIresources, but for review:
 *		struct OI_resource {
 *			- resource name (OI_n_...)
 *				refer to	<OI/resname.H>
 *						<X11/StringDefs.h>
 *			- resource class (OI_c_...)
 *				refer to	<OI/resname.H>
 *						<X11/StringDefs.h>
 *			- representation type desired (OI_r_...)
 *				refer to	<OI/resname.H>
 *						<X11/StringDefs.h>
 *			- size in bytes of representation
 *			- offset from base to put resource value
 *				uses the macro:
 *					offsetof( class, parameter );
 *			- representation type of specified default (OI_r_...)
 *			- address of default resources
 *				two examples (representation type, address of resource)
 *					OI_r_Immediate, (XtPointer) 5
 *					OI_r_String, "black"
 *
 *			- callback member function for resources being set
 *				for example:
 *					OI_RESOURCE_MEMFN_CAST(&HexEntryField::res_maximum_value)
 *				please note that res_maximum_value would decode the (void * arg) as follows:
 *					set_maximum_value(*((int*)arg));
 *			- callback procedure for resource being set
 *			- callback member function for resource being gotten
 *				for example:
 *					OI_RESOURCE_GET_MEMFN_CAST(&HexEntryField::maximum_value)
 *			- callback procedure for resource being gotten
 *		}
 *	Note that one always gets resources for the superclass object first,
 *	before getting resources for the object in question.
 *
 */
void
HexEntryField::get_resources()
{
#define	OI_n_maximumValue	"maximumValue"
#define OI_c_MaximumValue	"MaximumValue"

	static OI_resource resources[] = {
		{ OI_n_maximumValue, OI_c_MaximumValue, OI_r_Int, sizeof(int), offsetof(HexEntryField,mx_val), 0, 0,
				NULL_PMF, NULL, NULL_PMF, NULL },
	};

	// get superclass resources first!
	OI_entry_field::get_resources() ;
	conp->rm()->get_resources(resources, OI_count(resources), this, set_resources(), this) ;
	return ;
}


/*
 *	Main
 *
 *	This procedure demonstrates the basic concepts of creating an
 *	OI application:
 *		1. initialize OI
 *		2. register the Hex Entry Field
 *		3. create the application window, and specify the layout mode.
 *		4. create a HexEntryField
 *		5. layout the HexEntryField within the application window.
 *		6. associate the application window with it's root.
 *		7. begin interaction.
 */
void
main (int argc, char **argv)
{
	OI_connection	*conp ;
	OI_app_window	*wp ;
	HexEntryField	*hex_efp ;

	
	/*
	 * Open a connection.
	 */
	if (conp = OI_init(&argc,argv,"OITest")) {
		/*
		 * register the HexEntryField class
		 */
		reg_HexEntryField() ;

		/*
		 * Create the hexef main window.  Make it row layout.
		 */
		wp = oi_create_app_window("main",10,10,"Hex Entry Field") ;
		wp->set_layout(OI_layout_row) ;

		/*
		 * Create the HexEntry Field object.  Layout this object
		 * into row 10, column 10 of the main window.
		 */
		hex_efp = create_HexEntryField("hexef",8,"Address: 0x") ;
		hex_efp->layout_associated_object(wp,10,10,OI_ACTIVE) ;

		/*
		 * OK, display main window.
		 */
		wp->set_associated_object(wp->root(),OI_DEF_LOC,OI_DEF_LOC,OI_ACTIVE) ;
		OI_begin_interaction() ;

		/*
		 * Cleanup.  Make sure that we cleanup the library.
		 */
		OI_fini();
	}
}
