/* This file implements the list object type */

#include <stdio.h>
#include "config.h"
#include "slang.h"
#include "_slang.h"

/* A list is simply a collection of objects.  Unlike an array, the size of the
 * list is not fixed and the elements of the list do not have to be of one
 * type.  That is, the list is heterogeneous.
 */

#define SL_LIST_TYPE 101
typedef struct List_Type
{
   struct List_Type *next;
   struct List_Type *prev;
   SLang_Object_Type obj;
}
List_Type;

typedef struct
{
   List_Type *list;
   unsigned int size;		       /* number of items in the list */
   List_Type *curr;		       /* pointer to a list element */
   unsigned int n;		       /* element that curr points to */
   List_Type *last;		       /* pointer to the LAST element of list */
}
SLList_Type;


static SLList_Type *new_list (void)
{
   SLList_Type *l;
   l = (SLList_Type *) SLMALLOC (sizeof (SLList_Type));
   if (l == NULL) SLang_Error = SL_MALLOC_ERROR;
   else
     MEMSET (l, 0, sizeof (SLList_Type));
   return l;
}

static List_Type *new_element (void)
{
   List_Type *l;
   l = (List_Type *) SLMALLOC (sizeof (List_Type));
   if (l == NULL) SLang_Error = SL_MALLOC_ERROR;
   return l;
}

/* This is a callbcak that will be used for deleting linked lists */
static void free_list (SLList_Type *lt)
{
   List_Type *l, *lnext;
   
   l = lt->list;
   while (l != NULL)
     {
	lnext = l->next;
	if (l->obj.main_type == SLANG_DATA)
	  {
	     if (l->obj.sub_type == STRING_TYPE) SLFREE (l->obj.v.s_val);
	     else if (l->obj.sub_type >= ARRAY_TYPE)
	       SLang_free_user_object (l->obj.v.uobj);
	  }
	SLFREE (l);
	l = l->next;
     }
   SLFREE (lt);
}



static void create_list (int *np)
{
   SLList_Type *lt;
   List_Type *l, *lnext;
   SLuser_Object_Type *u;
   int n = *np;
   
   if (n < 0) return;
   if ((lt = new_list ()) == NULL) return;
   lt->size = n;
   
   while (n-- > 0)
     {
	l = new_element ();
	if (l == NULL) break;
	l->next = lt->list;
	
	if (l->next != NULL) l->next->prev = l;
	else lt->last = l;
	
	l->prev = lt->list;
	lt->list = l;
	
	if (SLang_pop (&(l->obj))) break;
	/* Since we are popping it and using it, we are not going to free it. */
     }
   
   if (SLang_Error || (NULL == (u = SLang_create_user_object (SL_LIST_TYPE))))
     {
	free_list (lt);
	return;
     }
   
   u->obj = (long *) lt;
   SLang_push_user_object (u);
}

/* methods */
static int sizeof_list (void)
{
   SLList_Type *lt;
   SLuser_Object_Type *u;
   int n;
   if (NULL == (u = SLang_pop_user_object (SL_LIST_TYPE))) return 0;
   lt = (SLList_Type *) (u->obj);
   n = lt->n;
   SLang_free_user_object (u);
   return n;
}

/* returns nth element */
static List_Type *find_element (SLList_Type *lt, int sn)
{
   List_Type *l;
   unsigned int half1, half2, n;
   
   int diffa, diffb;

   if (lt->list == NULL)
     {
	lt->n = lt->size = 0;
	return NULL;
     }
   if ((sn < 0) || (sn >= lt->size))
     {
	lt->n = lt->size;
	lt->curr = l = lt->last;
	return NULL;
     }
   
   n = (unsigned int) sn;
   half1 = lt->n / 2;
   half2 = (lt->size + lt->n) / 2;

   if (n < lt->n)
     {
	if (n > half1) 
	  {
	     l = lt->curr;
	     while (n++ < lt->n) l = l->prev;
	  }
	else
	  {
	     l = lt->list;
	     while (n--) l = l->next;
	  }
     }
   else if (n < half2)
     {
	l = lt->curr;
	n = n - lt->n;
	while (n-- > 0) l = l->next;
     }
   else
     {
	l = lt->last;
	n = lt->size - n;
	while (n-- > 0) l = l->prev;
     }
   lt->curr = l;
   lt->n = sn;
   return l;
}

	
static void insert_element (int *np)
{
   SLList_Type *lt;
   SLuser_Object_Type *u;
   List_Type *l, *newl;
   int n = *np;
   
   if (NULL == (u = SLang_pop_user_object (SL_LIST_TYPE))) return;
   lt = (SLList_Type *) (u->obj);
   
   newl = new_element ();
   if (newl != NULL)
     {
	l = find_element (lt, n);
	if (l == NULL)
	  {
	     /* insert after last element */
	     l = lt->last;
	     newl->prev = l;
	     if (l == NULL) lt->list = newl;
	     else 
	       {
		  l->next = newl;
		  lt->n++;
	       }
	     lt->curr = lt->last = newl;
	  }
	else
	  {
	     newl->next = l;
	     newl->prev = l->prev;
	     if (l->prev == NULL) 
	       {
		  lt->list = newl;
	       }
	     else l->prev->next = newl;
	     l->prev = newl;
	     lt->curr = l;
	     /* No need to touch lt->n */
	  }
	lt->size++;
	SLang_pop (&(newl->obj));
     }
   
   SLang_free_user_object (u);
}


static void delete_element (int *np)
{
   SLList_Type *lt;
   SLuser_Object_Type *u;
   List_Type *l;
   int n = *np;
   
   if (NULL == (u = SLang_pop_user_object (SL_LIST_TYPE))) return;
   lt = (SLList_Type *) (u->obj);

   if ((lt->list != NULL) && (NULL != (l = find_element (lt, n))))
     {
	if (l->prev != NULL) l->prev->next = l->next;
	else
	  {
	     /* deleting first */
	     lt->list = l->next;
	  }
	if (l->next != NULL) 
	  {
	     l->next->prev = l->prev;
	     lt->curr = l->next;	       /* l->n stays the same */
	  }
	else 
	  {
	     /* deleting the last */
	     lt->last = l->prev;
	     lt->curr = l->prev;
	     lt->n--;
	  }
	lt->size--;
	free_list_element (l);
     }
   
   SLang_free_user_object (u);
}


   
