/* Chmod command -- for the Midnight Commander
   Copyright (C) 1994 Radek Doulik

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include <string.h>
#include <stdio.h>
#include <ncurses.h>
#include "mad.h"
#include "win.h"
#include "input.h"
#include "color.h"
#include "dlg.h"
#include "widget.h"
#include "dialog.h"	/* For do_refresh() */

/* Needed for the extern declarations of integer parameters */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>
#ifdef HAVE_UNISTD_H
    #include <unistd.h>
#endif
#include "dir.h"
#include "panel.h"		/* Needed for the externs */
#include "util.h"		/* Needed for the externs */
#include "file.h"
#include "main.h"
#include "chmod.h"

#define PX		5
#define PY		2

#define FX		40
#define FY		2

#define BX		6
#define BY		17

#define TX		40
#define TY		12

#define PERMISSIONS	12
#define BUTTONS		6

#define B_MARKED	B_USER
#define B_ALL		B_USER+1
#define B_SETMRK        B_USER+2
#define B_CLRMRK        B_USER+3

int mode_change, need_update;
int c_file, end_chmod;

umode_t and_mask, or_mask, c_stat;

static int normal_color;
static int title_color;
static int selection_color;

struct {
    mode_t mode;
    char *text;
    int selected;
    WCheck *check;
} check_perm[PERMISSIONS] = {

    {
	S_IXOTH, "execute/search by others", 0, 0,
    },
    {
	S_IWOTH, "write by others", 0, 0,
    },
    {
	S_IROTH, "read by others", 0, 0,
    },
    {
	S_IXGRP, "execute/search by group", 0, 0,
    },
    {
	S_IWGRP, "write by group", 0, 0,
    },
    {
	S_IRGRP, "read by group", 0, 0,
    },
    {
	S_IXUSR, "execute/search by owner", 0, 0,
    },
    {
	S_IWUSR, "write by owner", 0, 0,
    },
    {
	S_IRUSR, "read by owner", 0, 0,
    },
    {
	S_ISVTX, "sticky bit", 0, 0,
    },
    {
	S_ISGID, "set group ID on execution", 0, 0,
    },
    {
	S_ISUID, "set user ID on execution", 0, 0,
    },
};

struct {
    int ret_cmd, y, x;
    char *text;
    int hkey, hpos;
} chmod_but[BUTTONS] = {

    {
	B_CANCEL, 2, 33, "[ Cancel ]", 'c', 2,
    },
    {
	B_ENTER, 2, 17, "[ Set ]", 's', 2,
    },
    {
	B_CLRMRK, 0, 42, "[ Clear marked ]", 'l', 3,
    },
    {
	B_SETMRK, 0, 27, "[ Set marked ]", 'e', 3,
    },
    {
	B_MARKED, 0, 12, "[ Marked all ]", 'm', 2,
    },
    {
	B_ALL, 0, 0, "[ Set all ]", 'a', 6,
    },
};

static void chmod_toggle_select (void)
{
    int Id = ch_dlg->current->dlg_id - BUTTONS + single_set * 2;

    wattrset (ch_win, normal_color);
    check_perm[Id].selected ^= 1;
    mvwaddch (ch_win, PY + PERMISSIONS - Id, PX + 1, (check_perm[Id].selected) ? '*' : ' ');
    wmove (ch_win, PY + PERMISSIONS - Id, PX + 3);
}

static int chmod_callback (Dlg_head *h, int Par, int Msg)
{
    switch (Msg) {
    case DLG_ACTION:
	if (Par >= BUTTONS - single_set * 2) {
	    c_stat ^= check_perm[Par - BUTTONS + single_set * 2].mode;
	    wattrset (ch_win, SELECTED_COLOR);
	    mvwprintw (ch_win, FY + 4, FX + 2, "%o", c_stat);
	    chmod_toggle_select ();

	    mode_change = 1;
	}
	break;

    case DLG_KEY:
	if ((Par == 'T' || Par == 't' || Par == KEY_IC) &&
	    ch_dlg->current->dlg_id >= BUTTONS - single_set * 2) {
	    chmod_toggle_select ();
	    if (Par == KEY_IC)
		dlg_one_down (ch_dlg);
	    return 1;
	}
	break;
    }
    return 0;
}

static void init_chmod (void)
{
    int i;

    do_refresh ();
    end_chmod = c_file = need_update = 0;
    single_set = (cpanel->marked < 2) ? 2 : 0;

    if (use_colors){
	normal_color = COLOR_NORMAL;
	title_color  = COLOR_HOT_NORMAL;
	selection_color = COLOR_NORMAL;
    } else {
	normal_color = NORMAL_COLOR;
	title_color  = SELECTED_COLOR;
	selection_color = SELECTED_COLOR;
    }
    
#ifdef BUGGY_CURSES		/* See key.c: mi_getch /BUGGY_CURSES/ */
    touchwin (stdscr);
#endif

    ch_win = centerwin (22 - single_set, 70);

    ch_colors[0] = COLOR_NORMAL;
    ch_colors[1] = COLOR_FOCUS;
    ch_colors[2] = COLOR_HOT_NORMAL;
    ch_colors[3] = COLOR_HOT_FOCUS;
    
    ch_dlg = dlg_new (ch_win, ch_colors, chmod_callback,
			 winpos (22 - single_set, 70), "[Chmod]");

#define XTRACT(i) BY+chmod_but[i].y-single_set, BX+chmod_but[i].x, \
                  chmod_but[i].ret_cmd, chmod_but[i].text, chmod_but[i].hkey,                    chmod_but[i].hpos, 0, 0

    for (i = 0; i < BUTTONS; i++) {
	if (i == 2 && single_set)
	    break;
	else
	    add_widget (ch_dlg, button_new (XTRACT (i)));
    }


#define XTRACT2(i) 0, check_perm [i].text, 0,-1

    for (i = 0; i < PERMISSIONS; i++) {
	check_perm[i].check = check_new (PY + (PERMISSIONS - i), PX + 2,
					 XTRACT2 (i));
	add_widget (ch_dlg, check_perm[i].check);
    }
}

static void chmod_refresh (void)
{
    wattrset (ch_win, normal_color);
    wclr (ch_win);
    
    draw_box (ch_win, 1, 2, 20 - single_set, 66);
    draw_box (ch_win, PY, PX, PERMISSIONS + 2, 33);
    draw_box (ch_win, FY, FX, 10, 25);
    
    mvwprintw (ch_win, FY + 1, FX + 2, "Name");
    mvwprintw (ch_win, FY + 3, FX + 2, "Permissions (Octal)");
    mvwprintw (ch_win, FY + 5, FX + 2, "Owner name");
    mvwprintw (ch_win, FY + 7, FX + 2, "Group name");
    
    wattrset (ch_win, title_color);
    mvwprintw (ch_win, 1, 28, " Chmod command ");
    mvwprintw (ch_win, PY, PX + 1, " Permission ");
    mvwprintw (ch_win, FY, FX + 1, " File ");
    
    wattrset (ch_win, selection_color);
    
    mvwprintw (ch_win, TY, TX, "Use SPACE to change");
    mvwprintw (ch_win, TY + 1, TX, "an option, ARROW KEYS");
    mvwprintw (ch_win, TY + 2, TX, "to move between options");
    mvwprintw (ch_win, TY + 3, TX, "and T or INS to mark");
    
    wrefresh (ch_win);
}

int stat_file (char *filename, struct stat *st)
{
    if (stat (filename, st))
	return 0;
    if (!((st->st_mode & S_IFREG) || (st->st_mode & S_IFDIR) ||
	  (st->st_mode & S_IFLNK)))
	return 0;

    return 1;
}

static void chmod_done (void)
{
    if (need_update)
	update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

char *next_file (void)
{
    while (!cpanel->dir.list[c_file].f.marked)
	c_file++;

    return cpanel->dir.list[c_file].fname;
}

static void do_chmod (struct stat *sf)
{
    sf->st_mode &= and_mask;
    sf->st_mode |= or_mask;
    chmod (cpanel->dir.list [c_file].fname, sf->st_mode);
    cpanel->dir.list [c_file].f.marked = 0;
}

static void apply_mask (struct stat *sf)
{
    char *fname;

    need_update = end_chmod = 1;
    do_chmod (sf);
    cpanel->marked--;

    do {
	fname = next_file ();
	if (!stat_file (fname, sf))
	    return;
	c_stat = sf->st_mode;

	do_chmod (sf);
	cpanel->marked--;
    } while (cpanel->marked);
}

void chmod_cmd (void)
{
    char *fname;
    int i;
    struct stat sf_stat;

    if (!is_view_file_listing (cpanel->view_type))
	return;
    
    do {			/* do while any files remaining */
	init_chmod ();
	if (cpanel->marked)
	    fname = next_file ();	/* next marked file */
	else
	    fname = selection->fname;	/* single file */

	if (!stat_file (fname, &sf_stat))	/* get status of file */
	    break;
	
	c_stat = sf_stat.st_mode;

	mode_change = 0;	/* clear changes flag */

	/* set check buttons */
	for (i = 0; i < PERMISSIONS; i++){
	    check_perm[i].check->state = (c_stat & check_perm[i].mode) ? 1 : 0;
	    check_perm[i].selected = 0;
	}
	/* refresh screen */
	chmod_refresh ();

	/* display file info */
	mvwprintw (ch_win, FY + 2, FX + 2, "%s", name_trunc (fname, 21));
	mvwprintw (ch_win, FY + 4, FX + 2, "%o", c_stat);
	mvwprintw (ch_win, FY + 6, FX + 2, "%s",
		   name_trunc (get_owner (sf_stat.st_uid), 21));
	mvwprintw (ch_win, FY + 8, FX + 2, "%s",
		   name_trunc (get_group (sf_stat.st_gid), 21));

	run_dlg (ch_dlg);	/* retrieve an action */
	
	/* do action */
	switch (ch_dlg->ret_value){
	case B_ENTER:
	    if (mode_change)
		chmod (fname, c_stat);
	    need_update = 1;
	    break;
	    
	case B_CANCEL:
	    end_chmod = 1;
	    break;
	    
	case B_ALL:
	case B_MARKED:
	    and_mask = or_mask = 0;
	    and_mask = ~and_mask;

	    for (i = 0; i < PERMISSIONS; i++) {
		if (check_perm[i].selected || ch_dlg->ret_value == B_ALL)
		    if (check_perm[i].check->state & C_BOOL)
			or_mask |= check_perm[i].mode;
		    else
			and_mask &= ~check_perm[i].mode;
	    }

	    apply_mask (&sf_stat);
	    break;
	    
	case B_SETMRK:
	    and_mask = or_mask = 0;
	    and_mask = ~and_mask;

	    for (i = 0; i < PERMISSIONS; i++) {
		if (check_perm[i].selected)
		    or_mask |= check_perm[i].mode;
	    }

	    apply_mask (&sf_stat);
	    break;
	case B_CLRMRK:
	    and_mask = or_mask = 0;
	    and_mask = ~and_mask;

	    for (i = 0; i < PERMISSIONS; i++) {
		if (check_perm[i].selected)
		    and_mask &= ~check_perm[i].mode;
	    }

	    apply_mask (&sf_stat);
	    break;
	}

	if (cpanel->marked && ch_dlg->ret_value!=B_CANCEL) {
	    cpanel->dir.list[c_file].f.marked = 0;
	    cpanel->marked--;
	    need_update = 1;
	}
	destroy_dlg (ch_dlg);
	delwin (ch_win);
    } while (cpanel->marked && !end_chmod);
    chmod_done ();
}

void ch1_cmd (int id)
{
  if (advanced_chfns)
      chown_advanced_cmd ();
  else
      chmod_cmd ();
}

void ch2_cmd (int id)
{
  if (advanced_chfns)
      chown_advanced_cmd ();
  else
      chown_cmd ();
}
