/*
 * METALBASE 5.1
 *
 * Released January 1st, 1993 by Huan-Ti [ t-richj@microsoft.com ]
 *
 */

#include <mbase.h>
#include "internal.h"


/******************************************************************************
 *
 * PROTOTYPES
 *
 */

   static mb_err     _set_hack    XARGS( (relation *) );
   static void       _clr_hack    XARGS( (relation *) );
   static void       _lck_pause   XARGS( (void) );
   static bool       _fIsDead     XARGS( (relation *, int) );
   static void       _clrstrobe   XARGS( (relation *) );
   static bool       _set_flag    XARGS( (relation *) );
   static void       _clr_flag    XARGS( (relation *) );


/******************************************************************************
 *
 * LOCK TIMING
 *
 */

static void
_lck_pause ()
{
   ;   /* Place a small sleep command here if you have locking troubles */
}


/******************************************************************************
 *
 * EXCLUSIVE (RELATION-WIDE) LOCKS
 *
 */

mb_err
_chk_elck (rel, fFlag)    /* FAIR WARNING: This routine cannot be trusted    */
relation  *rel;           /*        unless you have a  temporary lock placed */
bool            fFlag;    /*        on the relation before calling it!       */
{
   ushort   pid;
   char     tchar;
   mb_time  start;

   for (;;)
      {
      lseek (rel->fhRel, POS_WORKFLAG, 0);
      readx (rel->fhRel, &tchar, 1);

      Assert_Rel ( ! (tchar & 5) );    /* Make sure no processes interrupted */
      Assert_Dat ( ! (tchar & 8) );    /* Make sure the datafile is okay     */


      if (rel->fExclusive & 1)  /* Does this process have an exclusive lock? */
         {
         if (fFlag && !_set_flag (rel))
            Error (MB_ABORTED);

         Error (MB_OKAY);
         }

      lseek (rel->fhLock, lckPOS_ELOCK, 0);
      readx (rel->fhLock, &pid, 2);
      if (! pid)                                 /* Does anyone have a lock? */
         {
         if (fFlag && !_set_flag (rel))
            Error (MB_ABORTED);

         Error (MB_OKAY);
         }

/*
 * The relation is locked by another user.  Call the service routine, and
 * abort if the user sez to.  Otherwise, remove our temporary lock, wait for
 * a second or two, reset the lock and try again.  Note that the former is the
 * default behaviour, if there's no callback service defined.
 *
 */

      if (! CallService (svcLOCKED))
         break;

      _clr_lck (rel);

      start = curtime();
      for ( ; elap_t(start) < 6; )     /* Wait at least five seconds */
         _lck_pause();

      if (_set_lck (rel) != MB_OKAY)
         Error (mb_errno);
      }

   SetError (MB_LOCKED);

lblERROR:
   return mb_errno;
}


mb_err
mb_unl   (rel)
relation *rel;
{
   ushort pid;

   if (_identify (rel) < 0)      Error (MB_BAD_REL);

   if (! (rel->fExclusive & 1))  Error (MB_OKAY);

   if (_set_lck (rel))           Error (mb_errno);


   lseek (rel->fhLock, lckPOS_ELOCK, 0);
   readx (rel->fhLock, &pid, 2);

   if (pid == rel->pid)
      {
      pid = 0;
      lseek (rel->fhLock, lckPOS_ELOCK, 0);
      writx (rel->fhLock, &pid, 2);
      }

   rel->fExclusive &= 6;  /* Clear the exclusive-lock bit */

   _clr_lck (rel);

   SetError (MB_OKAY);

lblERROR:
   return mb_errno;
}

mb_err
mb_lck   (rel)
relation *rel;
{
   ushort pid;

   if (_identify (rel) < 0)  Error (MB_BAD_REL);

   if (rel->fExclusive & 1)  Error (MB_OKAY);     /* We've already locked it */

   if (_set_lck (rel))       Error (mb_errno);


   lseek (rel->fhLock, lckPOS_ELOCK, 0);
   readx (rel->fhLock, &pid, 2);
   if (pid != 0)
      {
      Error_2 (MB_LOCKED);
      }

   lseek (rel->fhLock, lckPOS_ELOCK, 0);
   writx (rel->fhLock, &rel->pid, 2);          /* Write our PID as a lock    */

   rel->fExclusive |= 1;                       /* Set the exclusive-lock bit */

   SetError (MB_OKAY);

lblERROR_2:
   _clr_lck (rel);

lblERROR:
   return mb_errno;
}


/*****************************************************************************
 *
 * HACK LOCKS (CONCURRENCY CONTROL)
 *
 */

static void
_clr_hack (rel)
relation  *rel;
{
   ushort *pid;
   char    pids[6];

   lseek (rel->fhLock, lckPOS_HLOCK, 0);
   readx (rel->fhLock, pids, 6);

   if (*(pid = (ushort *)&pids[0]) == rel->pid)  *pid = 0L;
   if (*(pid = (ushort *)&pids[2]) == rel->pid)  *pid = 0L;
   if (*(pid = (ushort *)&pids[4]) == rel->pid)  *pid = 0L;

   lseek (rel->fhLock, lckPOS_HLOCK, 0);
   writx (rel->fhLock, pids, 6);
}

static mb_err
_set_hack (rel)
relation  *rel;
{
   ushort *pid;
   char    pids[6];
   bool    changed;
   mb_time timeStart;

   timeStart = curtime();

   for (;;)
      {
      if (elap_t (timeStart) > 5)
         {
         pids[0] = pids[1] = pids[2] = pids[3] = pids[4] = pids[5] = 0;
         lseek (rel->fhLock, lckPOS_HLOCK, 0);
         writx (rel->fhLock, pids, 6);
         timeStart = curtime();
         if (! CallService (svcTIMEOUT))
            {
            Error (MB_ABORTED);
            }
         continue;
         }

/*
 * FIRST ITERATION:
 *
 */

      changed = 0;
      lseek (rel->fhLock, lckPOS_HLOCK, 0);
      readx (rel->fhLock, pids, 6);

      if (*(pid = (ushort *)&pids[0]) == rel->pid) { *pid = 0; changed |= 1; }
      if (*pid != 0)  changed |= 2;
      if (*(pid = (ushort *)&pids[2]) == rel->pid) { *pid = 0; changed |= 1; }
      if (*pid != 0)  changed |= 2;
      if (*(pid = (ushort *)&pids[4]) == rel->pid) { *pid = 0; changed |= 1; }
      if (*pid != 0)  changed |= 2;

      if (! (changed & 2))
         {
         *pid = rel->pid;  changed |= 1;
         }

      if (changed & 1)
         {
         lseek (rel->fhLock, lckPOS_HLOCK, 0);
         writx (rel->fhLock, pids, 6);
         }

      if (changed & 2)
         {
         continue;
         }

/*
 * SECOND ITERATION:
 *
 */

      lseek (rel->fhLock, lckPOS_HLOCK, 0);
      readx (rel->fhLock, pids, 6);

      if (*(pid = (ushort *)&pids[0]) != 0)          continue; /* NOTE ORDER */
      if (*(pid = (ushort *)&pids[4]) != rel->pid)   continue; /* NOTE ORDER */
      if (*(pid = (ushort *)&pids[2]) != 0)          continue; /* NOTE ORDER */

      *pid = rel->pid;

      lseek (rel->fhLock, lckPOS_HLOCK, 0);
      writx (rel->fhLock, pids, 6);

/*
 * THIRD ITERATION:
 *
 */

      lseek (rel->fhLock, lckPOS_HLOCK, 0);
      readx (rel->fhLock, pids, 6);

      if (*(pid = (ushort *)&pids[4]) != rel->pid)   continue; /* NOTE ORDER */
      if (*(pid = (ushort *)&pids[2]) != rel->pid)   continue; /* NOTE ORDER */
      if (*(pid = (ushort *)&pids[0]) != 0)          continue; /* NOTE ORDER */

      *pid = rel->pid;

      lseek (rel->fhLock, lckPOS_HLOCK, 0);
      writx (rel->fhLock, pids, 6);

/*
 * FINAL CHECK:
 *
 */

      lseek (rel->fhLock, lckPOS_HLOCK, 0);
      readx (rel->fhLock, pids, 6);

      if (*(pid = (ushort *)&pids[4]) != rel->pid)   continue; /* NOTE ORDER */
      if (*(pid = (ushort *)&pids[2]) != rel->pid)   continue; /* NOTE ORDER */
      if (*(pid = (ushort *)&pids[0]) != rel->pid)   continue; /* NOTE ORDER */

      break;
      }

   SetError (MB_OKAY);

lblERROR:
   return mb_errno;
}

/*****************************************************************************
 *
 * TEMPORARY LOCKS/ QUEUEING
 *
 */

mb_err
_set_lck (rel)
relation *rel;
{
   char    pids[60];
   ushort *pid, tpid;
   int     i, j;
   bool    fDead;
   bool    fWarned;

/*
 * FLOW FOR GETTING   ( example queue:  12  13  14  00  19  22  00  00  00... )
 * INTO THE QUEUE:    (           pos:   0   1   2   3   4   5   6   7   8... )
 *
 *     set hacklock  -- This guarantees that only one process will try to get
 *                      into the queue at once--avoids race conditions.
 *
 * WAIT:
 *     pos = the first zero in the right-hand set of contiguous zeroes
 *           (position 6 in the example above)
 *     look for a queuehole (position 3 above): -- if we were to just set
 *        a lock when the queue has a hole in it, we'd possibly escalate
 *        the length of the queue, whereas if we wait a few seconds, it'll
 *        shrink itself (when process 19 wakes up and moves itself).
 *     if queuehole
 *        check strobe for our blocking process (pos 5, pid 22 in this case)
 *        if strobe hasn't changed and elapsed time > 3 seconds
 *           pos -= 1      -- move over the dead process and erase its hold on
 *           write PID, 0  -- the queue--try again and we'll start here.
 *        else
 *           pause         -- let other processes work without extra I/O
 *        goto WAIT        -- go check the queue for a hole again.
 *     if the queue's full (pos == 30 -- no free slots), return MB_BUSY
 *
 *     clear hacklock      -- we're assured this position in the queue now
 *
 */

   if (rel->fExclusive & 2)  Error (MB_OKAY);

   if (_set_hack (rel))      Error (mb_errno);

   _clrstrobe (rel);

   fWarned = FALSE;

lockwait:

   lseek (rel->fhLock, lckPOS_QUEUE, 0);
   readx (rel->fhLock, pids, 60);

   for (i = 29; i >= 0; i--)
      if (*(pid = (ushort *)&pids[i*2]) != 0)
         break;
   i++;           /* "i" now == first available zero. */

   if (i != 0)    /* Check for a queuehole before taking the slot. */
      {
      if (! fWarned)
         {
         fWarned = TRUE;
         if (! CallService (svcPAUSE))
            {
            Error_2 (MB_ABORTED);
            }
         }

      for (j = i-1; j >= 0; j--)
         if (*(pid = (ushort *)&pids[j*2]) == 0)
            break;

      if (j != -1)  /* If this != -1, there's a 0 right here in the queue */
         {
         if (! _fIsDead (rel, i-1))  /* If it's not dead, we expect it's     */
            {                   /* checking the guy before it, and so on--   */
            _lck_pause ();      /* and that eventually, someone will see the */
            _lck_pause ();      /* queuehole exists and will try to get it   */
            }                   /* filled.                                   */
         else
            {
            i--;       /* If it IS dead, though, move over it and try again. */
            tpid = 0;
            lseek (rel->fhLock, lckPOS_QUEUE +2*i, 0);
            writx (rel->fhLock, &tpid, 2);

            if (! CallService (svcTIMEOUT))
               {
               Error_2 (MB_ABORTED);
               }
            }
         goto lockwait; /* Look, GOTO was useful here, all right?  Sheesh... */
         }
      }
   if (i == 30)
      {
      Error_2 (MB_BUSY);
      }

   lseek (rel->fhLock, lckPOS_QUEUE +2*i, 0);
   writx (rel->fhLock, &rel->pid, 2);

   _clr_hack (rel);

/*
 * FLOW FOR WORKING OUR    ( example queue:   15  13  12  92  34  16  00... )
 * WAY UP THE QUEUE:       (           pos:    0   1   2   3   4   5   6... )
 *
 * (we're in slot #4, PID==34, in the example above):
 *
 * WAIT:
 *   If we're in slot 0, goto DONE
 *   Otherwise,
 *      Read pos OurPos-1 (#3)--check pid (92)
 *      If PID==0,                     -- The process that WAS there has moved,
 *      OR PID is dead                 -- or hasn't strobed in 3 seconds,
 *         Write our PID in that slot  -- move up over it.  This way, free
 *         Write zero in our last slot -- slots bubble upwards...
 *         Goto WAIT
 *      Strobe our position
 *      Goto WAIT
 *
 * DONE:
 *   We're finished, and a temporary lock is in place.  Make sure to strobe
 *   every second during operations, or you'll lose your lock.
 *
 */

   _clrstrobe (rel);

   if (i > 0 && !fWarned)
      {
      if (! CallService (svcPAUSE))
         {
         goto lblABORT;
         }
      }
   for (; i > 0; )
      {
      lseek (rel->fhLock, lckPOS_QUEUE +2*(i-1), 0);
      readx (rel->fhLock, &tpid, 2);

      fDead = FALSE;  /* In case we don't get to the "|| fDead()" bit */

      if (tpid == 0 || (fDead = _fIsDead (rel, i-1)))
         {
         tpid = 0;

         i--;
         lseek (rel->fhLock, lckPOS_QUEUE +2*i, 0);
         writx (rel->fhLock, &rel->pid, 2);
         writx (rel->fhLock, &tpid, 2);

         if (fDead && !CallService (svcTIMEOUT))
            {
            goto lblABORT;
            }
         continue;
         }

      _strobe (rel, i);  /* Don't let anyone think we're dead, but let */
      _lck_pause ();     /* other processes think for a second without */
      continue;          /* tons of I/O slowing everything down.       */
      }

   rel->fExclusive |= 2;

   Error (MB_OKAY);

lblERROR_2:
   _clr_hack (rel);

lblERROR:
   return mb_errno;


lblABORT:
   lseek (rel->fhLock, lckPOS_QUEUE +2*i, 0);
   writx (rel->fhLock, &tpid, 2);

   Error (MB_ABORTED);
}

void
_clr_lck (rel)
relation *rel;
{
   ushort tpid = 0;

   if (rel->fExclusive & 4)
      {
      _clr_flag (rel);
      }

   if (rel->fExclusive & 2)
      {
      rel->fExclusive &= 5;  /* Clear the temp lock bit */

      lseek (rel->fhLock, lckPOS_QUEUE, 0);
      writx (rel->fhLock, &tpid, 2);
      }
}

static bool
_fIsDead (rel, pos)
relation *rel;
int            pos;
{
   char  ch;

   lseek (rel->fhLock, lckPOS_STROBE +pos, 0);          /* Get just this one */
   readx (rel->fhLock, &ch, 1);                         /* position's strobe */

   if (rel->strobe[pos] != ch)             /* value's changed--if so, update */
      {
      rel->times[pos] = curtime();         /* times[] array to current time. */
      rel->strobe[pos] = ch;
      }

/*
 * Note: elap_t() will fail at midnight--it'll return a BIG negative number,
 *       which won't pass the IsItDead test below.  So it may take 6 seconds
 *       to detect if a process is dead, if successive checks occur right then.
 *
 * Now why 10 seconds?
 *    1-second granularity means two seconds are minimum.
 *    Previous value == current value adds two seconds for trigger (strobing
 *       process won't change it, even if it expects to--and won't try again
 *       for 1 sec, plus granularity safeguard).
 *    6-second safeguard (just to be SURE, 'cause it's a Bad Thing to be
 *       trigger happy, and a one-time timeout isn't worth fussing over).
 *
 */

   return (elap_t (rel->times[pos]) > 10) ? TRUE : FALSE;
}

void
_strobe  (rel, pos)
relation *rel;
int            pos;
{
   if (elap_t (rel->times[pos]) >= 1)
      {
      lseek (rel->fhLock, lckPOS_STROBE +pos, 0);
      rel->strobe[pos] = (char)( ((int)rel->strobe[pos] +1) % 255 );
      writx (rel->fhLock, &rel->strobe[pos], 1);
      rel->times[pos] = curtime();
      }
}

static void
_clrstrobe (rel)
relation   *rel;
{
   int     i;
   mb_time cur;

   cur = curtime();

   for (i = 0; i < 30; i++)
      rel->times[i] = cur;
}

/*****************************************************************************
 *
 * WORK FLAGS
 *
 * Bit 1(lsb)...set if a process is updating indices
 * Bit 2........set if relation has a datafile
 * Bit 3........set if an index is corrupt
 * Bit 4........set if the datafile is corrupt
 *
 */

static bool
_set_flag (rel)
relation  *rel;
{
   char tchar;

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   readx (rel->fhRel, &tchar, 1);

   tchar |= 1;
   rel->fExclusive |= 4;  /* Set the work-flag bit */

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   writx (rel->fhRel, &tchar, 1);

   return CallService (svcBUSY);
}

static void
_clr_flag (rel)
relation  *rel;
{
   char  tchar;

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   readx (rel->fhRel, &tchar, 1);

   tchar &= 14;
   rel->fExclusive &= 3;  /* Clear the work-flag bit */

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   writx (rel->fhRel, &tchar, 1);

   (void)CallService (svcOKAY);
}

