////////////////////////////////////////////////////////////////////////////////
//  Implementations of textures                                              //  
//  LAST EDIT: Tue Aug 16 16:11:14 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "useftext.h"

void rt_initTextureCommands( Tcl_Interp *) {
    RT_Texture3D::initNoise();
    RTM_command( RTN_NOISE_TEXTURE, RT_NoiseTexture::classCMD );
    RTM_command( RTN_POLY_BLOB_TEXTURE, RT_PolyBlobTexture::classCMD );
    RTM_command( RTN_UNI_COLOR_TEXTURE, RT_UniColorTexture::classCMD );
    RTM_command( RTN_WOOD_TEXTURE, RT_WoodTexture::classCMD );
    RTM_command( RTN_CHECKER_TEXTURE, RT_CheckerTexture::classCMD );
    RTM_command( RTN_GRID_TEXTURE, RT_GridTexture::classCMD );
    RTM_command( RTN_TURBULENCE_TEXTURE, RT_TurbulenceTexture::classCMD );
    RTM_command( RTN_INTERPOLATION_TEXTURE, RT_InterpolationTexture::classCMD );
    RTM_command( RTN_TRUNC_TEXTURE, RT_TruncTexture::classCMD );
    RTM_command( RTN_MARBLE_TEXTURE, RT_MarbleTexture::classCMD );
    RTM_command( RTN_X_LINEAR_MARBLE_TEXTURE, RT_XLinearMarbleTexture::classCMD );
    RTM_command( RTN_X_CUBIC_MARBLE_TEXTURE, RT_XCubicMarbleTexture::classCMD );
    RTM_command( RTN_LATTICE_TEXTURE, RT_LatticeTexture::classCMD );
}

#define RTM_TEXTURE_CMD(cls) \
    if ( res  == TCL_OK ) {  \
	if ( argc != 4 ) {\
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <texture> <texture>! ", 0 );\
	    return TCL_ERROR;\
	}\
	RT_Object *o1, *o2;\
	o1 = RT_Object::getObject( argv[2] );\
	o2 = RT_Object::getObject( argv[3] );\
	if ( o1 && o1->isA( RTN_TEXTURE_3D ) && o2 && o2->isA( RTN_TEXTURE_3D )) { \
	    new cls( argv[1], (RT_Texture3D*)o1, (RT_Texture3D*)o2 ); \
		RTM_classReturn;\
	}\
	Tcl_AppendResult( ip, "Bad parameters for the textures!", 0 );\
	return TCL_ERROR;\
    } \
    return res

#define RTM_X_TEXTURE_CMD(cls) \
    if ( res  == TCL_OK ) {  \
	if ( argc != 6 ) {\
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <texture> <texture> <texture> <texture>! ", 0 );\
	    return TCL_ERROR;\
	}\
        RT_Object *o1, *o2, *o3, *o4;\
	o1 = RT_Object::getObject( argv[2] );\
	o2 = RT_Object::getObject( argv[3] );\
	o3 = RT_Object::getObject( argv[4] );\
	o4 = RT_Object::getObject( argv[5] );\
	if ( o1 && o1->isA( RTN_TEXTURE_3D ) && o2 && o2->isA( RTN_TEXTURE_3D ) && o3 && o3->isA( RTN_TEXTURE_3D ) && o4 && o4->isA( RTN_TEXTURE_3D )) { \
            new cls( argv[1], (RT_Texture3D*)o1, (RT_Texture3D*)o2, (RT_Texture3D*)o3, (RT_Texture3D*)o4 ); \
            RTM_classReturn;\
	}\
	Tcl_AppendResult( ip, "Bad parameters for the textures!", 0 );\
	return TCL_ERROR;\
    } \
    return res

// utilitys:

int base( double r ) {
    r += rt_RayEps;
    int n = (int)r;
    return (n>r) ? n-1 : n; 
}

double dsqr( double d ) { return d * d; }

double dmod( double n, double p ) {
    assert( p!=0 );
    return ( n - base( n/p ) * p );
}

// PolyBlob:

const char *RTN_POLY_BLOB_TEXTURE = "PolyBlobTexture";

RT_Color RT_PolyBlobTexture::getSample( const RT_Vector &p) const {
    double r = ((double)(int)(getNoise( mat * p) *4))/4;
 
    if( r < 0.25 ) return RT_Color( 1, 0, 0 );  // red
    if( r < 0.5  ) return RT_Color( 0, 1, 0 );  // green
    if( r < 0.75 ) return RT_Color( 0, 0, 1 );  // blue
    return RT_Color( 1, 1, 0 );  // yellow
}

int RT_PolyBlobTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String} {Creates a texture {ARG 1 Object} simulating poly blobs.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	new RT_PolyBlobTexture( argv[1] );
	RTM_classReturn;
    } 
    return res;
}

// Noise Texture:

const char *RTN_NOISE_TEXTURE = "NoiseTexture";

RT_Color RT_NoiseTexture::colV;
int RT_NoiseTexture::colF, RT_NoiseTexture::colG;

RT_ParseEntry RT_NoiseTexture::table[] = {
    { "-color", RTP_COLOR, (char*)&colV, &colF, "Set a new {ARG 1 Color} for the texture.", RTPS_COLOR },
    { "-get_color", RTP_NONE, 0, &colG, "Return the current color of the texture.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_NoiseTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (colF) color( colV );
    if (colG) {
	char tmp[60];
	RT_color2string( get_color(), tmp );
	RT_Object::result( tmp );
    }
    return ret + colF + colG;
};

RT_Color RT_NoiseTexture::getSample( const RT_Vector &p) const {
    return col * getNoise( mat * p );
}

int RT_NoiseTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Color} {Creates a noise texture {ARG 1 Object} with an initial {ARG 2 Color}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 3 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <color[3]>! ", 0 );
	    return TCL_ERROR;
	}
	RT_Color cl; int dum;
	if (RT_getColor( &argv[2], cl, dum )) { 
	    new RT_NoiseTexture( argv[1], cl ); 
	    RTM_classReturn;
	}
	Tcl_AppendResult( ip, "Bad parameter for <color[3]>! ", 0 );
	return TCL_ERROR;
    } 
    return res;
};

// uni color:

const char *RTN_UNI_COLOR_TEXTURE = "UniColorTexture";

RT_Color RT_UniColorTexture::colV;
int RT_UniColorTexture::colF, RT_UniColorTexture::colG;

RT_ParseEntry RT_UniColorTexture::table[] = {
    { "-color", RTP_COLOR, (char*)&colV, &colF, "Set a new {ARG 1 Color} for the texture.", RTPS_COLOR },
    { "-get_color", RTP_NONE, 0, &colG, "Return the current color of the texture.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_UniColorTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (colF) color( colV );
    if (colG) {
	char tmp[60];
	RT_color2string( get_color(), tmp );
	RT_Object::result( tmp );
    }
    return ret + colF + colG;
};

int RT_UniColorTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Color} {Creates a unicolor texture {ARG 1 Object} with an initial {ARG 2 Color}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 3 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <color[3]>! ", 0 );
	    return TCL_ERROR;
	}
	RT_Color cl; int dum;
	if (RT_getColor( &argv[2], cl, dum )) { 
	    new RT_UniColorTexture( argv[1], cl ); 
	    RTM_classReturn;
	}
	Tcl_AppendResult( ip, "Bad parameter for <color[3]>! ", 0 );
	return TCL_ERROR;
    } 
    return res;
};

// wood:

const char *RTN_WOOD_TEXTURE = "WoodTexture";

int RT_WoodTexture::brV, RT_WoodTexture::brF, RT_WoodTexture::brG; 

RT_ParseEntry RT_WoodTexture::table[] = {
    { "-bright", RTP_INTEGER, (char*)&brV, &brF, "Specify via a {ARG 1 Boolean} if the bright or dark surface dominates.", RTPS_INTEGER },
    { "-get_bright", RTP_NONE, 0, &brG, "Return 1 if the bright component dominates, 0 else.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_WoodTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (brF) bright( brV );
    if (brG) RT_Object::result( get_bright() ? "1" : "0" );
    return ret + brF + brG;
};

int RT_WoodTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture} {Creates a wood {ARG 1 Texture} consisting of a {ARG 2 Bright} and a {ARG 3 Dark} texture.}}", 0 );
	return TCL_OK;
    }
    RTM_TEXTURE_CMD( RT_WoodTexture );
};

RT_Color RT_WoodTexture::getSample( const RT_Vector &p) const {
    RT_Vector q = mat * p;
    double radius  = sqrt( dsqr(q.x) + dsqr(q.z) );
    radius -= sin ( 2 * M_PI * radius / 2 ) / 6;
    radius += 0.1 * getNoise( RT_Vector( q.x/3, q.y, q.z ) );
    
    double grain = dmod(radius, 0.3) / 0.3; 
    grain = xlight ? dsqr( grain ) : sqrt( grain );  
    
    return tex1->getSample(p) * (1 - grain) + tex0->getSample(p) * (grain);
};

// checker:

const char *RTN_CHECKER_TEXTURE = "CheckerTexture";

int RT_CheckerTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture} {Creates a chessboard-like, but 3D {ARG 1 Texture} consisting of tow other textures: {ARG 2 Texture1} and {ARG 3 Texture2}.}}", 0 );
	return TCL_OK;
    }
    RTM_TEXTURE_CMD( RT_CheckerTexture );
};

RT_Color RT_CheckerTexture::getSample( const RT_Vector &p) const {
    RT_Vector q = mat * p;
    int checker = 
	fabs( base(q.x)%2) == fabs( base(q.y)%2) == fabs( base(q.z)%2);
    return checker ? tex0->getSample( p ) : tex1->getSample( p );
};

// grid:

const char *RTN_GRID_TEXTURE = "GridTexture";

double RT_GridTexture::wiV;
int RT_GridTexture::wiF, RT_GridTexture::wiG; 

RT_ParseEntry RT_GridTexture::table[] = {
    { "-width", RTP_DOUBLE, (char*)&wiV, &wiF, "Specify the {ARG 1 Width} for the grid in {RANGE 1 \"0 1\"}.", RTPS_DOUBLE },
    { "-get_width", RTP_NONE, 0, &wiG, "Return with of the grid.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_GridTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (wiF) width( wiV );
    if (wiG) {
	char tmp[20];
	RT_double2string( get_width(), tmp );
	RT_Object::result( tmp );
    }
    return ret + wiF + wiG;
};

int RT_GridTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture} {Creates a grid {ARG 1 Texture} consisting of two further textures: {ARG 2 Texture1} and {ARG 3 Texture2}.}}", 0 );
	return TCL_OK;
    }
    RTM_TEXTURE_CMD(RT_GridTexture);
};

RT_Color RT_GridTexture::getSample( const RT_Vector &pnt ) const {
    RT_Vector p = mat * pnt;
    RT_Vector q = p + RT_Vector( getNoise(p*0.1), getNoise(p*0.1), getNoise(p*0.1) ) * 2;
    
    int x = ( frac(q.x) < xwidth );
    int y = ( frac(q.y) < xwidth );
    int z = ( frac(q.z) < xwidth );
    
    if ( x || y || z) return getRGBNoise( q , 0.2 ) + tex0->getSample(pnt);
    return getRGBNoise( q , 0.2 ) + tex1->getSample(pnt);
};

// turbulence:

const char *RTN_TURBULENCE_TEXTURE = "TurbulenceTexture";

int RT_TurbulenceTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture} {Creates a turbulence {ARG 1 Texture} consisting of two further textures: {ARG 2 Texture1} and {ARG 3 Texture2}.}}", 0 );
	return TCL_OK;
    }
    RTM_TEXTURE_CMD( RT_TurbulenceTexture );
};

RT_Color RT_TurbulenceTexture::getSample( const RT_Vector &p ) const {
    double d = getTurbulence( mat * p, xiter );
    if( d < xlevel ) return tex0->getSample(p);
    d = (d - xlevel) / (1 - xlevel) * 0.6 + 0.4;
    return tex1->getSample(p) * d;
};

int RT_TurbulenceTexture::itF, RT_TurbulenceTexture::itV, RT_TurbulenceTexture::itG;
int RT_TurbulenceTexture::lvF, RT_TurbulenceTexture::lvG;
double RT_TurbulenceTexture::lvV;

RT_ParseEntry RT_TurbulenceTexture::table[] = {
    { "-iter", RTP_INTEGER, (char*)&itV, &itF, "Set the {ARG 1 Number} of iterations.", RTPS_INTEGER },
    { "-get_iter", RTP_NONE, 0, &itG, "Return number of iterations.", RTPS_NONE },
    { "-level", RTP_DOUBLE, (char*)&lvV, &lvF, "Set the {ARG 1 Level} of turbulence. The range is {RANGE 1 \"0 1\"}.", RTPS_DOUBLE },
    { "-get_level", RTP_NONE, 0, &lvG, "Return the level.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_TurbulenceTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (itF) iter( itV );
    if (itG) {
	char tmp[10];
	RT_int2string( get_iter(), tmp );
	RT_Object::result( tmp );
    }
    if (lvF) level( lvV );
    if (lvG) {
	char tmp[20];
	RT_double2string( get_level(), tmp );
	RT_Object::result( tmp );
    }
    return ret + lvF + lvG + itF + itG;
};

// interpolation:

const char *RTN_INTERPOLATION_TEXTURE = "InterpolationTexture";

int RT_InterpolationTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture} {Creates a {ARG 1 Texture} interpolating two further textures: {ARG 2 Texture1} and {ARG 3 Texture2}.}}", 0 );
	return TCL_OK;
    }
    RTM_TEXTURE_CMD( RT_InterpolationTexture );
};

int RT_InterpolationTexture::stF, RT_InterpolationTexture::stV, RT_InterpolationTexture::stG;

RT_ParseEntry RT_InterpolationTexture::table[] = {
    { "-steps", RTP_INTEGER, (char*)&stV, &stF, "Set the {ARG 1 Number} of steps for interpolations.", RTPS_INTEGER },
    { "-get_steps", RTP_NONE, 0, &stG, "Return number of interpolations.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_InterpolationTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (stF) steps( stV );
    if (stG) {
	char tmp[10];
	RT_int2string( get_steps(), tmp );
	RT_Object::result( tmp );
    }
    return ret + stF + stG;
};

RT_Color RT_InterpolationTexture::getSample( const RT_Vector &p ) const {
    double r = getNoise( mat * p );
    r = (double)( ((int)(r*2*(xsteps-1)) + 1) / 2 ) / (xsteps-1);
    return tex0->getSample(p) * (1-r) + tex1->getSample(p) * r;
};

// trunc Texture:

const char *RTN_TRUNC_TEXTURE = "TruncTexture";

RT_Color RT_TruncTexture::getSample( const RT_Vector &p) const {
    RT_Color c = tex->getSample( p );
    c.r = base( c.r * xsteps ) / (double)xsteps;
    c.g = base( c.g * xsteps ) / (double)xsteps;
    c.b  = base( c.b * xsteps ) / (double)xsteps;
    return c;
};

int RT_TruncTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture} {Creates a {ARG 1 Texture} that truncs an another {ARG 2 Texture1}.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 3 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <texture>! ", 0 );
	    return TCL_ERROR;
	}
	RT_Object *o;
	o = RT_Object::getObject( argv[2] );
	if ( o && o->isA( RTN_TEXTURE_3D )) { 
	    new RT_TruncTexture( argv[1], (RT_Texture3D*)o ); 
	    RTM_classReturn;
	}
	Tcl_AppendResult( ip, "Bad parameters for the texture!", 0 );
	return TCL_ERROR;
    } 
    return res;
};

int RT_TruncTexture::stF, RT_TruncTexture::stV, RT_TruncTexture::stG;

RT_ParseEntry RT_TruncTexture::table[] = {
    { "-steps", RTP_INTEGER, (char*)&stV, &stF, "Set the number of {ARG 1 Steps}.", RTPS_INTEGER },
    { "-get_steps", RTP_NONE, 0, &stG, "Return number of steps.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_TruncTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (stF) steps( stV );
    if (stG) {
	char tmp[10];
	RT_int2string( get_steps(), tmp );
	RT_Object::result( tmp );
    }
    return ret + stF + stG;
};

// marble Texture:

const char *RTN_MARBLE_TEXTURE = "MarbleTexture";

int RT_MarbleTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture} {Creates a {ARG 1 Texture} simulating marble by linear interpolating two further textures: {ARG 2 Texture1} and {ARG 3 Texture2}.}}", 0 );
	return TCL_OK;
    }
    RTM_TEXTURE_CMD( RT_MarbleTexture );
};

RT_Color RT_MarbleTexture::getSample( const RT_Vector &p) const {
    RT_Vector q = mat * p;
    double d = frac( q.x + getNoise( q * 0.33) ) * w[3];
    RT_Color n = getRGBNoise( q, 0.2 );

    if (d<w[0]) return tex1->getSample(p) + n;
    if (d<w[1]) return (tex1->getSample(p) * (w[1]-d) + tex0->getSample(p) *  (d-w[0])) * (1.0/(w[1]-w[0])) + n;
    if (d<w[2]) return tex0->getSample(p) + n;

    return (tex1->getSample(p) * (d-w[2]) + tex0->getSample(p) * (w[3]-d)) * (1.0/(w[3]-w[2])) + n;
};

int RT_MarbleTexture::wiG, RT_MarbleTexture::wiF;
char *RT_MarbleTexture::wiV;

RT_ParseEntry RT_MarbleTexture::table[] = {
    { "-widths", RTP_STRING, (char*)&wiV, &wiF, "Set the widths of the various sections. {ARG 1 List} is a vector consisting of four doubles. The first specifys the width of Tex1, the second the width of the linear Tex1 to Tex2 interpolation, third: Tex2 and the forth the Tex2 to Tex1 interpolation width.", "Double_List" },
    { "-get_widths", RTP_NONE, 0, &wiG, "Return widths of the various stripes.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_MarbleTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (wiF) {
	wiF = 0;
	int argc; char **argv;
	double wx[4]; 
	if (Tcl_SplitList( rt_Ip, wiV, &argc, &argv) == TCL_OK) {
	    if (argc == 4) 
		if ( RT_string2double( argv[0], wx[0] ))
		    if ( RT_string2double( argv[1], wx[1] ))
			if ( RT_string2double( argv[2], wx[2] ))
			    if ( RT_string2double( argv[3], wx[3] )) { widths( wx ); wiF = 1; }
	    if (!wiF) rt_Output->errorVar( get_name(), ": Bad arguments for widths. Must be four doubles.", 0 );
	    free((char*)argv);
	}
    }
    if (wiG) {
	char tmp[80];
	double *d = get_widths();
	sprintf( tmp, "{%lf %lf %lf %lf}", d[0], d[1], d[2], d[3] );
	RT_Object::result( tmp );
    }
    return ret + wiF + wiG;
};

// extended linear marble:

const char *RTN_X_LINEAR_MARBLE_TEXTURE = "XLinearMarbleTexture";

int RT_XLinearMarbleTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture Texture Texture} {Creates a {ARG 1 Texture} by linear interpolating four further textures: {ARG 2 Texture1}, {ARG 3 Texture2}, {ARG 4 Texture3} and {ARG 5 Texture4}.}}", 0 );
	return TCL_OK;
    }
    RTM_X_TEXTURE_CMD( RT_XLinearMarbleTexture );
};

RT_Color RT_XLinearMarbleTexture::getSample( const RT_Vector &p) const {
    RT_Vector q = mat * p;
    double d = frac( q.x + getNoise(q*0.63) + 0.3*getNoise(q*3) ) * w[6];
    RT_Color n = getRGBNoise(q,0.1);

    if( d < w[0] ) return tex0->getSample(p) + n;
    if( d < w[1] ) return ( tex0->getSample(p)*(w[1]-d) + tex1->getSample(p)*(d-w[0])) * (1.0/(w[1]-w[0])) + n;
    if( d < w[2] ) return tex1->getSample(p) + n;
    if( d < w[3] ) return ( tex1->getSample(p)*(w[3]-d) + tex2->getSample(p) * (d-w[2]) ) * (1.0/(w[3]-w[2])) + n;
    if( d < w[4] ) return (tex2->getSample(p) *(w[4]-d) + tex3->getSample(p)*(d-w[3]) ) * (1.0/(w[4]-w[3])) + n;
    if( d < w[5] ) return tex3->getSample(p) + n;
    return ( tex3->getSample(p)*(w[6]-d) + tex0->getSample(p) * (d-w[5])) * (1.0/(w[6]-w[5])) + n;
};

int RT_XLinearMarbleTexture::wiG, RT_XLinearMarbleTexture::wiF;
char *RT_XLinearMarbleTexture::wiV;

RT_ParseEntry RT_XLinearMarbleTexture::table[] = {
    { "-widths", RTP_STRING, (char*)&wiV, &wiF, "Set the widths of the various sections. {ARG 1 List} is a vector consisting of four doubles. 1st: Tex1, 2nd: Tex1 to Tex2, 3rd: Tex2, 4th: Tex2 to Tex3, 5th: Tex3 to Tex4, 6th: Tex4 and 7th: Tex4 to Tex1.", "Double_List" },
    { "-get_widths", RTP_NONE, 0, &wiG, "Return widths of the various stripes.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_XLinearMarbleTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (wiF) {
	wiF = 0;
	int argc; char **argv;
	double wx[7]; 
	if (Tcl_SplitList( rt_Ip, wiV, &argc, &argv) == TCL_OK) {
	    if (argc == 7) 
		if ( RT_string2double( argv[0], wx[0] ))
		    if ( RT_string2double( argv[1], wx[1] ))
			if ( RT_string2double( argv[2], wx[2] ))
			    if ( RT_string2double( argv[3], wx[3] ))
				if ( RT_string2double( argv[4], wx[4] ))
				    if ( RT_string2double( argv[5], wx[5] ))
					if ( RT_string2double( argv[6], wx[6] )) { widths( wx ); wiF = 1; }
	    if (!wiF) rt_Output->errorVar( get_name(), ": Bad arguments for widths. Must be seven doubles.", 0 );
	    free((char*)argv);
	}
    }
    if (wiG) {
	char tmp[120];
	double *d = get_widths();
	sprintf( tmp, "{%lf %lf %lf %lf %lf %lf %lf}", d[0], d[1], d[2], d[3], d[4], d[5], d[6] );
	RT_Object::result( tmp );
    }
    return ret + wiF + wiG;
};

// extended cubic marble:

const char *RTN_X_CUBIC_MARBLE_TEXTURE = "XCubicMarbleTexture";

int RT_XCubicMarbleTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Texture Texture Texture Texture} {Creates a {ARG 1 Texture} by linear interpolating four further textures: {ARG 2 Texture1}, {ARG 3 Texture2}, {ARG 4 Texture3} and {ARG 5 Texture4}.}}", 0 );
	return TCL_OK;
    }
    RTM_X_TEXTURE_CMD( RT_XCubicMarbleTexture );
};

RT_Color RT_XCubicMarbleTexture::getSample( const RT_Vector &p) const {
    // Hermite interpolation of the colors c[0]-c[4] to the interval [0,w4]
    RT_Vector q = mat * p;
    double   d = frac( q.x + getNoise(q*0.63) + 0.3*getNoise(q*3) ) * w[4];
    RT_Color n( getRGBNoise(q,0.2) );
    double dd;

    if (d<w[0]) dd = 0;
    else if (d<w[1]) dd = (d-w[0])/(w[1]-w[0]);
    else if (d<w[2]) dd = (d-w[1])/(w[2]-w[1]);
    else if (d<w[3]) dd = (d-w[2])/(w[3]-w[2]);
    else dd = (d-w[3])/(w[4]-w[3]);

    dd=dd*dd*(3-2*dd);

    if (d<w[0]) return tex0->getSample(p) + n;
    if (d<w[1]) return tex1->getSample(p) * dd + tex0->getSample(p) * (1-dd) + n;
    if (d<w[2]) return tex2->getSample(p) * dd + tex1->getSample(p) * (1-dd) + n;
    if (d<w[3]) return tex3->getSample(p) * dd + tex2->getSample(p) * (1-dd) + n;
    return tex0->getSample(p) * dd + tex3->getSample(p) * (1-dd) + n;
};

int RT_XCubicMarbleTexture::wiG, RT_XCubicMarbleTexture::wiF;
char *RT_XCubicMarbleTexture::wiV;

RT_ParseEntry RT_XCubicMarbleTexture::table[] = {
    { "-widths", RTP_STRING, (char*)&wiV, &wiF, "Set the widths of the various sections. {ARG 1 List} is a vector consisting of four doubles. 1st: Tex1, 2nd: Tex1 to Tex2, 3rd: Tex2 to Tex3, 4th: Tex3 to Tex4, and 5th: Tex4.", "Double_List" },
    { "-get_widths", RTP_NONE, 0, &wiG, "Return widths of the various stripes.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_XCubicMarbleTexture::objectCMD(char *argv[]) {
    int ret = RT_Texture3D::objectCMD( argv );
    RT_parseTable( argv, table );
    if (wiF) {
	wiF = 0;
	int argc; char **argv;
	double wx[5]; 
	if (Tcl_SplitList( rt_Ip, wiV, &argc, &argv) == TCL_OK) {
	    if (argc == 5) 
		if ( RT_string2double( argv[0], wx[0] ))
		    if ( RT_string2double( argv[1], wx[1] ))
			if ( RT_string2double( argv[2], wx[2] ))
			    if ( RT_string2double( argv[3], wx[3] ))
					if ( RT_string2double( argv[4], wx[4] )) { widths( wx ); wiF = 1; }
	    if (!wiF) rt_Output->errorVar( get_name(), ": Bad arguments for widths. Must be five doubles.", 0 );
	    free((char*)argv);
	}
    }
    if (wiG) {
	char tmp[100];
	double *d = get_widths();
	sprintf( tmp, "{%lf %lf %lf %lf %lf}", d[0], d[1], d[2], d[3], d[4] );
	RT_Object::result( tmp );
    }
    return ret + wiF + wiG;
};

// Lattice bump

const char *RTN_LATTICE_TEXTURE = "LatticeTexture";

RT_Vector RT_LatticeTexture::getNormal( const RT_Vector &p1) const {
    RT_Vector p = mat * p1;
    double dx = frac( p.x ); double dy = frac( p.y );
    RT_Vector bm = RT_Vector( 0,0,1 );
    if ( dx < 0.1 ) { if ( dx < dy && (1.0 - dy) > dx  ) bm.x = 1; }
    else if ( dx > 0.9 ) { if ( dx > dy && (1.0 - dy) < dx  )  bm.x = -1; }
    if ( dy < 0.1 ) { if ( dy < dx && (1.0 - dx) > dy ) bm.y = 1; }
    else if ( dy > 0.9 ) { if ( dy > dx && (1.0 - dx) < dy ) bm.y = -1; }
    return bm;
}

int RT_LatticeTexture::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) {
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String} {Creates a texture {ARG 1 Object} - no additional arguments.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	new RT_LatticeTexture( argv[1] );
	RTM_classReturn;
    } 
    return res;
}

