////////////////////////////////////////////////////////////////////////////////
//  implementations of utility functions.                                     //  
//  LAST EDIT: Fri Aug  5 08:55:23 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "utils.h"
#include "matrix.h"
#include "primitiv.h"
#include "high/lookat.h"
#include "surface.h"

#ifndef RTD_CPP_INCLUDES
extern "C" {
#endif

#include <stdio.h>
#include <string.h>
#include <tcl.h>

#ifndef RTD_CPP_INCLUDES
}
#endif

#define max(a,b) ((a)<(b))?(b):(a) 

void RT_clearArgv( char **argv, int start, int diff) {
    if (start >= 0 ) 
	for (int i = start; argv[i] ; i++ ) 
	    argv[i] = argv[i+diff+1];
}

int RT_findString(char **argv, const char *str) {
    int i = 0;
    while (*argv) {
	if (!strcmp( *argv, str)) return i;
	argv++; i++;
    }
    return -1;
}

int RT_getInt(char **argv, int &in, int &nr) {
    nr = 0;
    if (*argv) {
	int i;
	if ( sscanf( *argv, "%i", &i )) {
	    in = i;
	    nr = 1;
	    return 1;
	}
	return 0;
    }
    return 0;
}

int RT_getVector(char **s, RT_Vector &v, int &nr) {
    nr = 0;
    if (!*s) return 0;
    int argc; char **argv;
    if (Tcl_SplitList( rt_Ip, *s, &argc, &argv) == TCL_OK) {
	int ok = 0;
	if (argc == 3) {
	    if ( RT_string2double( argv[0], v.x))
		if (RT_string2double( argv[1], v.y))
		    if (RT_string2double( argv[2], v.z)) ok = 1;
	}
	free((char*)argv);
	if (ok) {
	    nr = 1;
	    return 1; 
	}
    }
    return 0;
}

int RT_getFloat(char **argv, float &fl, int &nr) {
    nr = 0;
    if (*argv) {
	float f;
	if ( sscanf( *argv, "%f", &f )) {
	    fl = f;
	    nr = 1;
	    return 1;
	}
	return 0;
    }
    return 0;
}

int RT_getSurface(char **argv, RT_Surface &su, int &nr) {
    nr = 0;
    if (*argv) {
	RT_Surface s;
	if (s.set( *argv )) {
	    su = s;
	    nr = 1;
	    return 1;
	}
	return 0;
    }
    return 0;
}

int RT_getColor(char **argv, RT_Color &co, int &nr) {
    nr = 0;
    if (*argv) {
	RT_Color c;
	if (c.set( *argv )) {
	    co = c;
	    nr = 1;
	    return 1;
	}
	return 0;
    }
    return 0;
}

int RT_getMatrix(char **s, RT_Matrix &v, int &nr) {
    nr = 0;
    if (!*s) return 0;
    int argc; char **argv;
    if (Tcl_SplitList( rt_Ip, *s, &argc, &argv) == TCL_OK) {
	int ok = 0;
	if (argc == 16) {
	    for (int j = 0; j < 4; j++) 
		for (int i = 0; i < 4; i++) {
		    double d;
		    ok = RT_string2double( argv[j * 4 + i], d );
		    v.set(j, i, d);
		    if (!ok) break;
		}
	}
	free((char*)argv);
	if (ok) {
	    nr = 1;
	    return 1; 
	}
    }
    return 0;
}

int RT_getDouble(char **argv, double &fl, int &nr) {
    nr = 0;
    if (*argv) {
	double d;
	if ( sscanf( *argv, "%lf", &d )) {
	    fl = d;
	    nr = 1;
	    return 1;
	}
	return 0;
    }
    return 0;
}

int RT_getString(char **argv, char *&str, int &nr) {
    nr = 0;
    if (*argv) {
	str = argv[0];
	nr = 1;
	return 1;
    }
    return 0;
}

void RT_double2string(double d, char *s) {   sprintf( s, "%lf", d ); }

void RT_float2string(float f, char *s) {   sprintf( s, "%f", f ); }

void RT_int2string(int i, char *s) {   sprintf( s, "%i", i ); }

void RT_long2string(long l, char *s) {   sprintf( s, "%l", l ); }

void RT_vector2string(const RT_Vector &v, char *s) {
    sprintf( s, "{ %lf %lf %lf } ", v.x, v.y, v.z ); 
}

void RT_bounds2string(const RT_Bounds &b, char *s) {
    sprintf( s, "{ %lf %lf %lf %lf %lf %lf } ", b.min.x, b.max.x, b.min.y, b.max.y, b.min.z, b.max.z  );
}

void RT_color2string(const RT_Color &v, char *s) {
    strcpy( s, v.get() );
}

void RT_surface2string(const RT_Surface &x, char *s) {
    strcpy( s, x.get() );
}

void RT_matrix2string(const RT_Matrix &m, char *s) {
    sprintf( s, "{ %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf } ",
	    m.get(0,0), m.get(0,1), m.get(0,2), m.get(0,3), 
	    m.get(1,0), m.get(1,1), m.get(1,2), m.get(1,3), 
	    m.get(2,0), m.get(2,1), m.get(2,2), m.get(2,3), 
	    m.get(3,0), m.get(3,1), m.get(3,2), m.get(3,3)
	    ); 
}

int RT_getFacet(char **s, RT_Facet &fac, int &nr) {
    nr = 0;
    if (!*s) return 0;
    int argc; char **argv;
    if (Tcl_SplitList( rt_Ip, *s, &argc, &argv) == TCL_OK) {
	int ok = 0;
	int *tmp = 0;
	if (argc) tmp = new int[argc];
	for (int i=0; i<argc; i++) {
	    if ( RT_string2int(argv[i], tmp[i])) ok=1;
		else { ok=0; break; }
	}
	free((char*)argv);
	if (ok) {
	    nr = 1;
	    fac.set(argc, tmp);
	    if (tmp) delete tmp;
	    return 1; 
	}
    }
    return 0;
}

void RT_parseTable(char **argv, RT_ParseEntry *entry) {
    static char tmp[1000];
    if (argv[0] && !strcmp ( argv[0], RT_HELP )) {
	while (entry->type != RTP_END) {
	    if (entry->flag) *entry->flag = 0; // reset the entry flags
	    sprintf( tmp, "{%s {%s} {%s}}\n", entry->name, entry->tcltypes, entry->info ); 
	    RT_Object::result( tmp );
	    entry++;
	}
	return;
    }
    while (entry->type != RTP_END ) {
	if (entry->flag) *entry->flag = 0;
	int start, diff;
	if ((entry->type != RTP_SPECIAL) && (start = RT_findString( argv, entry->name)) >= 0) {
	    diff = 0;
	    switch ( entry->type ) {
	      case RTP_NONE:
		*entry->flag = 1;
		break;
	      case RTP_STRING: 
		char *fnd;
		if ( RT_getString( &argv[ start +1 ], fnd, diff) ) {
		    *(char**)entry->ptr = fnd; 
		    *entry->flag = 1;
		}
		break;
	      case RTP_MATRIX: {
		  RT_Matrix ma;
		  if ( RT_getMatrix( &argv[ start +1 ], ma, diff) ) {
		      *(RT_Matrix*)entry->ptr = ma; 
		      *entry->flag = 1;
		  }
	      }
		break;
	      case RTP_VECTOR: {
		  RT_Vector vc;
		  if ( RT_getVector( &argv[ start +1 ], vc, diff) ) {
		      *(RT_Vector*)entry->ptr = vc; 
		      *entry->flag = 1;
		  }
	      }
		break;
	      case RTP_COLOR: {
		  RT_Color vc;
		  if ( RT_getColor( &argv[ start +1 ], vc, diff) ) {
		      *(RT_Color*)entry->ptr = vc; 
		      *entry->flag = 1;
		  }
	      }
		break;
	      case RTP_DOUBLE:
		double d;
		if (RT_getDouble( &argv[start+1], d, diff )) {
		    *(double*)entry->ptr = d; 
		    *entry->flag = 1;
		}
		break;
	      case RTP_INTEGER:
		int i;
		if (RT_getInt( &argv[start+1], i, diff )) {
		    *(int*)entry->ptr = i; 
		    *entry->flag = 1;
		}
		break;
	    }
	    RT_clearArgv( argv, start, diff );
	}
	entry++;
    }
}

int RT_string2double(const char *s, double &dou) {
    double d;
    if ( sscanf( (char*)s, "%lf", &d )) {
	dou = d;
	return 1;
    }
    return 0;
}

int RT_string2float(const char *s, float &fl) {
    float f;
    if ( sscanf( (char*)s, "%f", &f )) {
	fl = f;
	return 1;
    }
    return 0;
}

int RT_string2int(const char *s, int &in) {
    int i;
    if ( sscanf( (char*)s, "%i", &i )) {
	in = i;
	return 1;
    }
    return 0;
}

int RT_strings2vector(const char **ss, RT_Vector &vec) {
    double x, y, z;
    if ( !sscanf( (char*)ss[0], "%lf", &x )) return 0;
    if ( !sscanf( (char*)ss[1], "%lf", &y )) return 0;
    if ( !sscanf( (char*)ss[2], "%lf", &z )) return 0;
    vec.x = x;
    vec.y = y;
    vec.z = z;
    return 1;
}

void RT_calcNorm( int n, RT_Vector *points, RT_Vector &norm ) {
    double xu, yu, zu, ru, xv, yv, zv, rv, xn, yn, zn, rn;
    double deltax, deltay, deltaz;
    int arete, aretex, aremix, aremiy, aremiz;
    RT_Vector pta, ptb, ptc;
    float xmin, xmax, ymin, ymax, zmin, zmax;

    xmin = ymin = zmin = 99999. ;
    xmax = ymax = zmax = -99999. ;
    for ( int i = 0; i < n; i++ ) {
	if ( points[i].x < xmin ) {
	    aremix = i;
	    xmin = points[i].x;
	}
	if ( points[i].y < ymin ) {
	    aremiy = i;
	    ymin = points[i].y;
	}
	if ( points[i].z < zmin ) {
	    aremiz = i;
	    zmin = points[i].z;
	}
	xmax = max( points[i].x,xmax );
	ymax = max( points[i].y,ymax );
	zmax = max( points[i].z,zmax );
    }

    deltax = xmax - xmin;
    deltay = ymax - ymin;
    deltaz = zmax - zmin;
    if ( deltax >= deltay && deltay >= deltaz ) aretex = aremix;
    else if ( deltax >= deltaz && deltaz >= deltay ) aretex = aremix;
    else if ( deltay >= deltax && deltax >= deltaz ) aretex = aremiy;
    else if ( deltay >= deltaz && deltaz >= deltax ) aretex = aremiy;
    else if ( deltaz >= deltax && deltax >= deltay ) aretex = aremiz;
    else if ( deltaz >= deltay && deltay >= deltax ) aretex = aremiz;
    pta = points[aretex];
    arete = aretex + 1;
    if (arete >= n) arete = 0;
    ptb = points[arete];

    xu = ptb.x - pta.x; yu = ptb.y - pta.y; zu = ptb.z - pta.z;
    ru = sqrt( xu*xu + yu*yu + zu*zu );
    if (fabs( ru) > rt_RayEps) {xu /= ru; yu /= ru; zu/= ru; }
    do {
	arete++; if ( arete >= n ) arete = 0;
	ptc = points[arete];
	xv = ptc.x - ptb.x; yv = ptc.y - ptb.y; zv = ptc.z - ptb.z;
	rv = sqrt(xv*xv + yv*yv + zv*zv);
	if (fabs( rv) > rt_RayEps) { xv /= rv; yv /= rv; zv/= rv; }
	xn = yu*zv - zu*yv;
	yn = zu*xv - xu*zv;
	zn = xu*yv - yu*xv;
	rn = sqrt(xn*xn + yn*yn + zn*zn);
	if (fabs( rn) > rt_RayEps) { xn /= rn; yn /= rn; zn/= rn;}
    }
    while ( rn < 0.01 && (xu*xv + yu*yv + zu*zv) > 0.);

    norm.x = -xn; // clockwise
    norm.y = -yn;
    norm.z = -zn;
}

// fixed Coordinates transformations:

int RT_fixedCoordRot(RT_Primitive *p, double angle, char axis) {
    // individualize the matrices:
    p->translate( RT_Vector(0,0,0) );

    RT_Matrix tmat = p->getModel()->get_globalMatrix() * p->getModel()->get_matrix();
    RT_Matrix transMat = tmat;

    // set the projection to 0
    transMat.set(3,0,0);
    transMat.set(3,1,0);
    transMat.set(3,2,0);
    transMat.set(3,3,1);
    
    // set the translation to 0
    transMat.set(0,3,0);
    transMat.set(1,3,0);
    transMat.set(2,3,0);
    
    RT_Matrix rotMat;
    switch (axis) {
      case 'x': {
	  rotMat = rotMat.ROTATION_X(angle);
	  break;    
      }
      case 'y': {
	  rotMat = rotMat.ROTATION_Y(angle);
	  break;    
      }
      case 'z': {
	  rotMat = rotMat.ROTATION_Z(angle);
	  break;    
      }
      default: {
	  char str[4];
	  sprintf(str, "%c", axis);
	  rt_Output->errorVar( "Wrong axis: ", str," in fixedCoordRot!", 0);
	  return 0;
      }
    }
    RT_Matrix result;
    result = rotMat * transMat;
    
    // get the projection terms:
    result.set(3,0,tmat.get(3,0));
    result.set(3,1,tmat.get(3,1));
    result.set(3,2,tmat.get(3,2));
    result.set(3,3,tmat.get(3,3));

    // get the translation terms:	
    result.set(0,3,tmat.get(0,3));
    result.set(1,3,tmat.get(1,3));
    result.set(2,3,tmat.get(2,3));

    result = p->getModel()->get_invGlobalMatrix() * result;
    p->matrix(result);
    return 1;
}

int RT_fixedCoordRotCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 4) {
	Tcl_AppendResult( ip, argv[0], " need three args: ",
			  "<primitive> <angle> <axis> ." , 0 );
	return TCL_ERROR;
    }
    double angle; 
    if (!RT_string2double(argv[2], angle)) {
	Tcl_AppendResult( ip, argv[0], " wrong value of angle: ",argv[2],
			 " it must be an integer!" , 0 );
	return TCL_ERROR;
    }
    RT_Object *o;
    if (!(o=(RT_Object *)RT_Object::getObject(argv[1]))) {
	Tcl_AppendResult( ip, argv[0], ": there is no object ", argv[1], "." , 0 );
	return TCL_ERROR;
    }
    if( !o->isA( RTN_PRIMITIVE )) {
	Tcl_AppendResult( ip, argv[0], ": object ",argv[1],
			 " is not a primitive !" , 0 );
	return TCL_ERROR;
    }
    return RT_fixedCoordRot((RT_Primitive *) o,angle,(*argv[3])) ? TCL_OK : TCL_ERROR;
}

// translate a primitive in the initial coord system
int RT_fixedCoordTrans(RT_Primitive *p, const RT_Vector &tv) {
    // individualize the matrices:
    p->translate( RT_Vector(0,0,0) );
    RT_Matrix tmat = p->getModel()->get_globalMatrix() * p->getModel()->get_matrix();
    RT_Matrix transMat = tmat;
    
    transMat.set(3,0,0);
    transMat.set(3,1,0);
    transMat.set(3,2,0);
    transMat.set(3,3,1);

    RT_Matrix trMat;
    trMat = trMat.TRANSLATION(tv);

    RT_Matrix result;
    result=trMat*transMat;
    result.set(3,0,tmat.get(3,0));
    result.set(3,1,tmat.get(3,1));
    result.set(3,2,tmat.get(3,2));
    result.set(3,3,tmat.get(3,3));

    result=p->getModel()->get_invGlobalMatrix() * result;
    p->matrix(result);
    return 1;
}

int RT_fixedCoordTransCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 3) {
	Tcl_AppendResult( ip, argv[0], " need two args: ",
			 "<primitive> <vector> ." , 0 );
	return TCL_ERROR;
    }
    RT_Vector tv;
    int n =0;
    if (!RT_getVector(&argv[2], tv, n)) {
	Tcl_AppendResult( ip, argv[0], " wrong value of vector: ",argv[2],
			 " it must be {<tx> <ty> <tz>}!" , 0 );
	return TCL_ERROR;
    }
    RT_Object *o;
    if(!(o=(RT_Object *)RT_Object::getObject(argv[1]))) {
	Tcl_AppendResult( ip, argv[0], ": There is no object ",argv[1], "!" , 0 );
	return TCL_ERROR;
    }
    if (!(o->isA( RTN_PRIMITIVE ))) {
	Tcl_AppendResult( ip, argv[0], ": Object ",argv[1],
			 " is not a primitive!" , 0 );
	return TCL_ERROR;
    }
    return RT_fixedCoordTrans((RT_Primitive *) o,tv) ? TCL_OK : TCL_ERROR;
}

// scale a primitive in the initial coord system:

int RT_fixedCoordScale(RT_Primitive *p, const RT_Vector &tv) {
    // individualize the matrices:
    p->translate( RT_Vector(0,0,0) );
    
    RT_Matrix tmat = p->getModel()->get_globalMatrix() * p->getModel()->get_matrix();
    RT_Matrix transMat = tmat;
    
    transMat.set(3,0,0);
    transMat.set(3,1,0);
    transMat.set(3,2,0);
    transMat.set(3,3,1);
    
    RT_Matrix trMat;
    trMat = trMat.SCALE(tv);
    
    RT_Matrix result;
    result = trMat*transMat;
    result.set(3,0,tmat.get(3,0));
    result.set(3,1,tmat.get(3,1));
    result.set(3,2,tmat.get(3,2));
    result.set(3,3,tmat.get(3,3));

    result = p->getModel()->get_invGlobalMatrix() * result;
    p->matrix(result);
    return 1;
}

int RT_fixedCoordScaleCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 3) {
	Tcl_AppendResult( ip, argv[0], " need two args: ",
			  "<primitive> <vector> ." , 0 );
	return TCL_ERROR;
    }
    RT_Vector tv;
    int n =0;
    if (!RT_getVector(&argv[2], tv, n)) {
	Tcl_AppendResult( ip, argv[0], ": wrong value of vector: ",argv[2],
			 " it must be {<tx> <ty> <tz>}!" , 0 );
	return TCL_ERROR;
    }
    RT_Object *o;
    if (!(o=(RT_Object *)RT_Object::getObject(argv[1]))) {
	Tcl_AppendResult( ip, argv[0], ": There is no object ",argv[1], "!" , 0 );
	return TCL_ERROR;
    }
    if (!(o->isA( RTN_PRIMITIVE ))) {
	Tcl_AppendResult( ip, argv[0], ": Object ",argv[1],
			 " is not a primitive !" , 0 );
	return TCL_ERROR;
    }
    return RT_fixedCoordScale((RT_Primitive *) o,tv) ? TCL_OK : TCL_ERROR;
}

// rotate a primitiv around a point in the initial coord system
// p ist the manipulated primitive
// point is the rotation poin
// vec is the rotationvector
// is the camera with define the initial coordinatesystem

int RT_fixedPointRot(RT_Primitive *p, const RT_Vector &point, const RT_Vector &vec, RT_LookatCamera *cam) {
    // individualize the matrices:
    p->translate( RT_Vector(0,0,0) );

    RT_Matrix tmat = cam->getMatrix() * p->getModel()->get_globalMatrix() * p->getModel()->get_matrix();
    RT_Matrix transMat = tmat;

    transMat.set(3,0,0);
    transMat.set(3,1,0);
    transMat.set(3,2,0);
    transMat.set(3,3,1);

    // accumulate the rotation point in the world coord system
    RT_Matrix tmpMat = transMat;
    tmpMat.set(0,3,0);
    tmpMat.set(1,3,0);
    tmpMat.set(2,3,0);
    tmpMat.set(3,3,1);

    RT_Vector wpoint;
    wpoint = tmpMat*point;

    // build the translation matrixes to the point coord system
    RT_Matrix t1Mat,t2Mat;
    t1Mat = t1Mat.TRANSLATION(wpoint*-1);
    t2Mat = t2Mat.TRANSLATION(wpoint);
    
    // build the rotation matrix;
    RT_Matrix rMat;
    rMat = rMat.ROTATION(vec);

    // define a translation matrix with the inital translation of the element
    RT_Matrix tr;
    tr.set(0,3,tmat.get(0,3));
    tr.set(1,3,tmat.get(1,3));
    tr.set(2,3,tmat.get(2,3));
    
    // define a translation matrix with the invers initial translation of the element
    RT_Matrix trm;
    trm.set(0,3,-tmat.get(0,3));
    trm.set(1,3,-tmat.get(1,3));
    trm.set(2,3,-tmat.get(2,3));

    // now build the transformation
    // 1. translate with the initial translation
    // 2. translate to the point coord system
    // 3. rotate
    // 4. retranslate from point coord system
    // 6. retranslate to the initial translation
    // 7. rotate around the initial rotation matrix
    
    RT_Matrix result;
    result = tr*t2Mat*rMat*t1Mat*trm*transMat;
    
    // get the projection terms	
    result.set(3,0,tmat.get(3,0));
    result.set(3,1,tmat.get(3,1));
    result.set(3,2,tmat.get(3,2));
    result.set(3,3,tmat.get(3,3));
    
    // calculate the local matrix of the object
    RT_Matrix imat;
    imat = cam->getMatrix() * p->getModel()->get_globalMatrix();
    imat = imat.INVERS();
    result = imat*result;
    p->matrix(result);
    return 1;
}

int RT_fixedPointRotCMD(ClientData, Tcl_Interp *ip, int argc, char *argv[]) {
    if (argc != 5) {
	Tcl_AppendResult( ip, argv[0], " need four args: ",
			  "<primitive> <point> <vector> <lookatcamera>." , 0 );
	return TCL_ERROR;
    }
    RT_Vector point,vec;
    int n =0;
    if(!RT_getVector(&argv[2], point, n)) {
	Tcl_AppendResult( ip, argv[0], ": wrong value of vector: ",argv[2],
			 " it must be {<px> <py> <pz>}!" , 0 );
	return TCL_ERROR;
    }
    if(!RT_getVector(&argv[3], vec, n)) {
	Tcl_AppendResult( ip, argv[0], ": wrong value of vector: ",argv[2],
			  " it must be {<tx> <ty> <tz>}!" , 0 );
	return TCL_ERROR;
	}
    RT_Object *p;
    if(!(p=(RT_Object *)RT_Object::getObject(argv[1]))) {
	Tcl_AppendResult( ip, argv[0], ": There is no object ",argv[1], "!" , 0 );
	return TCL_ERROR;
    }
    if (!(p->isA( RTN_PRIMITIVE ))) {
	Tcl_AppendResult( ip, argv[0], ": Object ",argv[1],
			  " is not a primitive!" , 0 );
	return TCL_ERROR;
    }
    RT_Object *c;
    if(!(c=(RT_Object *)RT_Object::getObject(argv[4]))) {
	Tcl_AppendResult( ip, argv[0], ": There is no object ",argv[4], "!" , 0 );
	return TCL_ERROR;
    }
    if ( !(c->isA( RTN_LOOKAT_CAMERA ))) {
	Tcl_AppendResult( ip, argv[0], ": Object ",argv[1],
			  " is not a lookat camera!" , 0 );
	return TCL_ERROR;
    }
    return RT_fixedPointRot((RT_Primitive *) p,point,vec,(RT_LookatCamera *)&c) ? TCL_OK : TCL_ERROR;
}

// try to find the possibility to intersect the segment from s to s+r  with p + (0,i) 

int RT_intersect_xy(const RT_Vector &p, const RT_Vector &s, const RT_Vector &r) {
    if (r.x == 0.0) return 0;
    double b = (p.x - s.x)/r.x;
    if (b <= 0.0 || b > 1.0) return 0;
    if ((s.y + (b * r.y) - p.y) < 0.0) return 0;
    return 1;
}

int RT_intersect_xz(const RT_Vector &p, const RT_Vector &s, const RT_Vector &r) {
    if (r.x == 0.0) return 0;
    double b = (p.x - s.x)/r.x;
    if (b <= 0.0 || b > 1.0) return 0;
    if ((s.z + (b * r.z) - p.z) < 0.0) return 0;
    return 1;
}

int RT_intersect_yz(const RT_Vector &p, const RT_Vector &s, const RT_Vector &r) {
    if (r.y == 0.0) return 0;
    double b = (p.y - s.y)/r.y;
    if (b <= 0.0 || b > 1.0) return 0;
    if ((s.z + (b * r.z) - p.z) < 0.0) return 0;
    return 1;
}

