/****** POVtoOFF.c **********************************************
 *
 * PURPOSE:
 * 	Reads a POV file created by POVCAD and converts it to an OFF format
 *   of polyhedra from YART.  It works for triangles only!
 * RETURNS: 0 for OK or error number
 * GLOBALS:
 * CALLED BY:  System
 * FUNCTIONS CALLED:
 * INPUT FILES: first argument
 * OUTPUT FILES: second argument
 * PROCESS/ALGORITHM:
 *	Read the input file looking for a "triangle" tag
 *   parse the follong three 3D points into a data structure
 *   for each point
 *      if it already exists (from another polygon) use that as its index
 *      else add it to the list of points and use that last index point
 *   After all triangles have been processed
 *    print to output
 *    #points #polydons #of polygon point
 *   write one point (3 numbers) per line
 *   write polygon indexes
 *  
 * CAVEATS:
 *	Can only handle triangles
 *   Max of 1000 polygons may be processed
 *   Max of 500 unique points (verticies)
 *
 *              AUTHOR                  DATE            PROJECT LEADER
 *    ORIGINAL: P. Bryan Heidorn       9-29-94                                        
 *    ALTERED : 
 *    REASON:
 *    ALTERED :
 *    REASON:
 *
 **************************************************************************/

#include "stdio.h"
#include "string.h"

struct polygon {
    int nvert;    	/* Number of vertices in this polygon */
    int verts[4];	/* Vertices in this polygon, I really only */
				/* needed three since I'm only doing triangles */
};
struct point {
	double x, y, z;
	};

struct OFFObject {
    int npts;           /* The number of points */
    int npolys;         /* The number of polygons */
    int nsegs;          /* The number of segments */
    struct point *points[1000]; /* list of pointers to the vertex values */
    struct polygon *polys[2000]; /* list of pointers to polygon index values */
};
/* Macro */
#define ptEqual(X,Y) (((X.x != Y.x) || (X.y != Y.y) || (X.z != Y.z))?0:1)
#define NOTFOUND -1
#define ENDIF }
#define ENDELSE }
#define ENDWHILE }
#define ENDFOR }
/****************************************************************************/
main(int argc, char *argv[])
{
	int Index;
	struct OFFObject obj;
	char POVFileName[80];
	char Usage[] = "Usage: POVtoOFF File\nWhere File is the name of the file to be converted.  The result is sent to standard output.";
	char LineBuffer[256];
	struct point PtTemp[3];
	struct polygon *CurrentPolygon;
	FILE *fpOut;
	
    FILE *fpIn;
    int i;

	if (argc != 2 ) {
		fprintf (stderr, "%s", Usage );
		return -1;
	ENDIF  
	strcpy(POVFileName,argv[1]);
    /* printf("Before open\n"); getchar(); */
    if (!(fpIn = fopen(POVFileName,"r"))) {           /* Open the .geom file */
		fprintf (stderr, "Could not find the file %s\n", POVFileName);
		fprintf (stderr, "%s", Usage );
        return -1;
	ENDIF  

	obj.npts = obj.npolys = 0;
	while (1) 
	{	
	/* Break when we hit the end of the file */
	if (!fgets( LineBuffer, 256, fpIn)) break;
	/* see if the key word triangle is in the file */
	if (strstr (LineBuffer,"triangle")) {
		/* Allocate space for a polygon */
		obj.polys[obj.npolys] = (struct polygon *)malloc(sizeof(struct polygon));
		obj.polys[obj.npolys]->nvert = 3;
		sscanf(LineBuffer," triangle {< %lf , %lf , %lf > , < %lf , %lf , %lf > , < %lf , %lf , %lf > }",
             &PtTemp[0].x, &PtTemp[0].y, &PtTemp[0].z,
             &PtTemp[1].x, &PtTemp[1].y, &PtTemp[1].z,
             &PtTemp[2].x, &PtTemp[2].y, &PtTemp[2].z ); 
		/* check for degenerate triangles.  That is where two or more points are
		 * the same forming a line or point.  
		 * Skip the polygon in this case 
		 * Boy, it would be nice to be using C++ and an overloaded == here
		 */
		if ( ptEqual( PtTemp[0], PtTemp[1]) || 
			 ptEqual( PtTemp[0], PtTemp[2]) || 
			 ptEqual( PtTemp[1], PtTemp[2])) {
			free (obj.polys[obj.npolys]);
		} else { 
		/* Incrament polygon count */
     	obj.npolys++; 
		/*  For each of the points in the triangle, get an index number */
		/* if one does not exist, make it */
		for (i = 0; i < 3; i++) {
			if ((Index = GetPointIndex(obj.points,&PtTemp[i], obj.npts)) == NOTFOUND) {
				/* Allocate a point structure and tack it on obj.points and fill it */
				obj.points[obj.npts] = (struct point *)malloc(sizeof(struct point));
				obj.points[obj.npts]->x = PtTemp[i].x;
				obj.points[obj.npts]->y = PtTemp[i].y;
				obj.points[obj.npts]->z = PtTemp[i].z;
				Index = obj.npts++;
			ENDIF  
			obj.polys[obj.npolys-1]->verts[i] = Index;
		ENDFOR
		ENDELSE
	ENDIF
	ENDWHILE
	 
close (fpIn);
/* Print the number of points, Number of Polygons, 
 * Number of indexes in the polygon section (the 3 is because they are all triangles! 
 */
printf ("%d %d %d\n", obj.npts, obj.npolys, obj.npolys*3 );
/* free data structures */
for (i = 0; i < obj.npts; i++) {
	printf ("%lf %lf %lf\n", obj.points[i]->x, obj.points[i]->y, obj.points[i]->z);
	free (obj.points[i]);
	}
for (i = 0; i < obj.npolys; i++) {
	printf ("3 %d %d %d\n", 
		obj.polys[i]->verts[0]+1, obj.polys[i]->verts[1]+1, obj.polys[i]->verts[2]+1);
	free (obj.polys[i]);
	}
}
/****************************************************************************/
/* Search through the points (vertexes) in the list for a point that matches
 * if one is not found return -1 else the index to it
 */
int GetPointIndex(struct point *ptList[], struct point *pt, int NumPoints) {
int Index;

Index = -1;
for (Index = 0; Index < NumPoints; Index++) {
	if ( ptList[Index]->x == pt->x && ptList[Index]->y == pt->y 
		&& ptList[Index]->z == pt->z ) {
		return Index;
	}
}
return -1;
}

/****************************************************************************/
int fptEqual ( struct point *pt1, struct point *pt2 ) {

if ( pt1->x != pt2->x || pt1->y != pt2->y || pt1->x != pt2->z )
	return 0;
return 1;
}

