/*======================================================================

    A driver for PCMCIA serial devices

    Written by David Hinds, dhinds@allegro.stanford.edu
    
======================================================================*/

#include <linux/config.h>

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/tty.h>
#include <linux/serial.h>
#include <linux/major.h>
#include <asm/io.h>
#include <asm/system.h>

#include "version.h"
#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "ds.h"
#include "cisreg.h"

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
static char *version =
    "serial_cs.c 1.41 1995/05/15 04:46:01 (David Hinds)\n";
#endif

/*====================================================================*/

/* Parameters that can be set with 'insmod' */

/* Bit map of interrupts to choose from */
static u_long irq_mask = 0xdeb8;

/*====================================================================*/

static void serial_config(dev_link_t *link);
static void serial_release(u_long arg);
static int serial_event(event_t event, int priority,
			event_callback_args_t *args);

static dev_info_t dev_info = "serial_cs";

static dev_link_t *serial_attach(void);
static void serial_detach(dev_link_t *);

static dev_link_t *dev_list = NULL;

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*======================================================================

    If we're linked into the kernel, this gets called from init.c.
    
======================================================================*/

#ifndef MODULE
u_long serial_init(u_long kmem_start, u_long kmem_end)
{
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    register_pcmcia_driver(&dev_info, &serial_attach, &serial_detach);
    return kmem_start;
}
#endif

/*======================================================================

    serial_attach() creates an "instance" of the driver, allocating
    local data structures for one device.  The device is registered
    with Card Services.

======================================================================*/

static dev_link_t *serial_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("serial_attach()\n");
#endif

    /* Create new serial device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &serial_release;
    link->release.data = (u_long)link;
    link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
    link->io.NumPorts1 = 8;
    link->io.IOAddrLines = 3;
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    link->irq.IRQInfo2 = irq_mask;
    link->conf.Attributes = CONF_ENABLE_IRQ | CONF_ENABLE_SPKR;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;
    link->conf.Status = CCSR_AUDIO_ENA;
    link->priv = kmalloc(sizeof(int), GFP_KERNEL);
    *(int *)link->priv = -1;
    
    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask =
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &serial_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != CS_SUCCESS) {
	cs_error(RegisterClient, ret);
	serial_detach(link);
	return NULL;
    }
    
    return link;
} /* serial_attach */

/*======================================================================

    This deletes a driver "instance".  The device is de-registered
    with Card Services.  If it has been released, all local data
    structures are freed.  Otherwise, the structures will be freed
    when the device is released.

======================================================================*/

static void serial_detach(dev_link_t *link)
{
    dev_link_t **linkp;
    long flags;
    int ret;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("serial_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    save_flags(flags);
    cli();
    if (link->state & DEV_RELEASE_PENDING) {
	del_timer(&link->release);
	link->state &= ~DEV_RELEASE_PENDING;
    }
    restore_flags(flags);
    
    if (link->state & DEV_CONFIG)
	serial_release((u_long)link);
    
    if (link->handle) {
	ret = CardServices(DeregisterClient, link->handle);
	if (ret != CS_SUCCESS)
	    cs_error(DeregisterClient, ret);
    }
    
    /* Unlink device structure, free bits */
    *linkp = link->next;
    kfree_s(link->priv, sizeof(int));
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* serial_detach */

/*======================================================================

    serial_config() is scheduled to run after a CARD_INSERTION event
    is received, to configure the PCMCIA socket, and to make the
    serial device available to the system.

======================================================================*/

void serial_config(dev_link_t *link)
{
    client_handle_t handle;
    tuple_t tuple;
    u_char tuple_data[255];
    cisparse_t parse;
    struct serial_struct serial;
    int i, line, ibm = 0;

    sti();
    handle = link->handle;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("serial_config(0x%p)\n", link);
#endif
    
    tuple.TupleData = tuple_data;
    tuple.TupleOffset = 0; tuple.TupleDataMax = 255;
    do {
	/* Get configuration register information */
	tuple.Attributes = 0;
	tuple.DesiredTuple = CISTPL_CONFIG;
	i = CardServices(GetFirstTuple, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(GetTupleData, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(ParseTuple, handle, &tuple, &parse);
	if (i != CS_SUCCESS) break;
	link->conf.ConfigBase = (caddr_t)parse.config.base;
	link->conf.Present = parse.config.rmask[0];
	
    } while (0);
    
    if (i != 0) {
	printk("serial_cs: unable to parse CIS\n");
	link->state &= ~DEV_CONFIG_PENDING;
	return;
    }
    
    /* Is this an IBM modem? */
    tuple.DesiredTuple = CISTPL_MANFID;
    if (CardServices(GetFirstTuple, handle, &tuple) == CS_SUCCESS) {
	if (CardServices(GetTupleData, handle, &tuple) == CS_SUCCESS)
	    if ((tuple_data[0] == 0xa4) && (tuple_data[1] == 0x00))
		ibm = 1;
	}
    
    /* Configure card */
    link->state |= DEV_CONFIG;

    do {

	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	i = CardServices(GetFirstTuple, handle, &tuple);
	while (i == CS_SUCCESS) {
	    i = CardServices(GetTupleData, handle, &tuple);
	    if (i != CS_SUCCESS) break;
	    i = CardServices(ParseTuple, handle, &tuple, &parse);
	    if (i != CS_SUCCESS) break;
	    link->conf.ConfigIndex = parse.cftable_entry.index;
	    link->io.BasePort1 = parse.cftable_entry.io.win[0].base;
	    i = CardServices(RequestIO, link->handle, &link->io);
	    if (i == CS_SUCCESS) break;
	    i = CardServices(GetNextTuple, handle, &tuple);
	}
	if (i != CS_SUCCESS) {
	    cs_error(RequestIO, i);
	    break;
	}
	
	i = CardServices(RequestIRQ, link->handle, &link->irq);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIRQ, i);
	    break;
	}
	i = CardServices(RequestConfiguration, link->handle, &link->conf);
	if (i != CS_SUCCESS) {
	    cs_error(RequestConfiguration, i);
	    break;
	}
	serial.port = link->io.BasePort1;
	serial.irq = link->irq.AssignedIRQ;

	if (ibm) {
	    conf_reg_t reg = { CS_READ, 0x800, 0 };
	    i = CardServices(AccessConfigurationRegister,
			     link->handle, &reg);
	    if (i != CS_SUCCESS) {
		cs_error(AccessConfigurationRegister, i);
		break;
	    }
	    reg.Action = CS_WRITE;
	    reg.Value = reg.Value | 1;
	    i = CardServices(AccessConfigurationRegister,
			     link->handle, &reg);
	    if (i != CS_SUCCESS) {
		cs_error(AccessConfigurationRegister, i);
		break;
	    }
	}
	
	line = register_serial(&serial);
	if (line < 0) {
	    printk("serial_cs: register_serial() at 0x%04x, irq %d "
		   "failed\n", serial.port, serial.irq);
	    i = -1;
	    break;
	}
	*(int *)link->priv = line;
	sprintf(link->dev_name, "cua%d", line);
	link->major = TTYAUX_MAJOR;
	link->minor = 0x40+line;
	    
    } while (0);

    link->state &= ~DEV_CONFIG_PENDING;
    if (i != 0)
	serial_release((u_long)link);

} /* serial_config */

/*======================================================================

    After a card is removed, serial_release() will unregister the net
    device, and release the PCMCIA configuration.  If the device is
    still open, this will be postponed until it is closed.
    
======================================================================*/

void serial_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;
    sti();
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("serial_release(0x%p)\n", link);
#endif

    if (*(int *)link->priv >= 0) {
	link->dev_name[0] = '\0';
	unregister_serial(*(int *)link->priv);
    }
    
    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);
    
    link->state &= ~DEV_CONFIG;

} /* serial_release */

/*======================================================================

    The card status event handler.  Mostly, this schedules other
    stuff to run after an event is received.  A CARD_REMOVAL event
    also sets some flags to discourage the serial drivers from
    talking to the ports.
    
======================================================================*/

int serial_event(event_t event, int priority,
		event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("serial_event()\n");
#endif
    
    switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
	if (pc_debug)
	    printk("serial_cs: registration complete\n");
	break;
#endif
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    link->release.expires = 5;
	    link->state |= DEV_RELEASE_PENDING;
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	serial_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	if (link->state & DEV_CONFIG)
	    CardServices(ReleaseConfiguration, link->handle);
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (DEV_OK(link))
	    CardServices(RequestConfiguration, link->handle, &link->conf);
	break;
    }
    return 0;
} /* serial_event */

/*====================================================================*/

#ifdef MODULE
char kernel_version[] = UTS_RELEASE;

int init_module(void)
{
    servinfo_t serv;
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk("serial_cs: Card Services release does not match!\n");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &serial_attach, &serial_detach);
    return 0;
}

void cleanup_module(void)
{
    printk("serial_cs: unloading\n");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL)
	serial_detach(dev_list);
}
#endif /* MODULE */
