/*======================================================================

    A PCMCIA ethernet driver for the 3com 3c589 card.
    
    Written by David Hinds, dhinds@allegro.stanford.edu

    The network driver code is based on Donald Becker's 3c589 code:
    
    Written 1994 by Donald Becker.
    Copyright 1993 United States Government as represented by the
    Director, National Security Agency.  This software may be used and
    distributed according to the terms of the GNU Public License,
    incorporated herein by reference.
    Donald Becker may be reached at becker@cesdis1.gsfc.nasa.gov
    
======================================================================*/

#include <linux/config.h>

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/bitops.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>

#include "version.h"
#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "ds.h"

/* To minimize the size of the driver source I only define operating
   constants if they are used several times.  You'll need the manual
   if you want to understand driver details. */
/* Offsets from base I/O address. */
#define EL3_DATA	0x00
#define EL3_CMD		0x0e
#define EL3_STATUS	0x0e
#define ID_PORT		0x100
#define	EEPROM_READ	0x80

#define EL3WINDOW(win_num) outw(SelectWindow + (win_num), ioaddr + EL3_CMD)

/* The top five bits written to EL3_CMD are a command, the lower
   11 bits are the parameter, if applicable. */
enum c509cmd {
    TotalReset = 0<<11, SelectWindow = 1<<11, StartCoax = 2<<11,
    RxDisable = 3<<11, RxEnable = 4<<11, RxReset = 5<<11, RxDiscard = 8<<11,
    TxEnable = 9<<11, TxDisable = 10<<11, TxReset = 11<<11,
    FakeIntr = 12<<11, AckIntr = 13<<11, SetIntrMask = 14<<11,
    SetReadZero = 15<<11, SetRxFilter = 16<<11, SetRxThreshold = 17<<11,
    SetTxThreshold = 18<<11, SetTxStart = 19<<11, StatsEnable = 21<<11,
    StatsDisable = 22<<11, StopCoax = 23<<11,
};

/* The SetRxFilter command accepts the following classes: */
enum RxFilter {
    RxStation = 1, RxMulticast = 2, RxBroadcast = 4, RxProm = 8
};

/* Register window 1 offsets, the window used in normal operation. */
#define TX_FIFO		0x00
#define RX_FIFO		0x00
#define RX_STATUS 	0x08
#define TX_STATUS 	0x0B
#define TX_FREE		0x0C	/* Remaining free bytes in Tx buffer. */

#define WN0_IRQ		0x08	/* Window 0: Set IRQ line in bits 12-15. */
#define WN4_MEDIA	0x0A	/* Window 4: Various transcvr/media bits. */
#define MEDIA_TP	0x00C0	/* Enable link beat and jabber for 10baseT. */

struct el3_private {
    struct enet_statistics stats;
};

static char *if_names[] = {"10baseT", "AUI", "undefined", "BNC"};

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
static char *version =
    "3c589_cs.c 1.41 1995/04/11 18:25:31 (David Hinds)\n";
#endif

#ifdef TC589_DEBUG
static int tc589_debug = TC589_DEBUG;
#else
static int tc589_debug = 3;
#endif

/*====================================================================*/

/* Parameters that can be set with 'insmod' */

/* Special hook for setting if_port when module is loaded */
static int if_port = -1;

/* Bit map of interrupts to choose from */
static u_long irq_mask = 0xdeb8;

/*====================================================================*/

int tc589_probe(struct device *dev);

static void tc589_config(dev_link_t *link);
static void tc589_release(u_long arg);
static int tc589_event(event_t event, int priority,
		       event_callback_args_t *args);

static ushort read_eeprom(short ioaddr, int index);
static void tc589_reset(struct device *dev);
static int el3_config(struct device *dev, struct ifmap *map);
static int el3_open(struct device *dev);
static int el3_start_xmit(struct sk_buff *skb, struct device *dev);
static void el3_interrupt(int reg_ptr, struct pt_regs *regs);
static void update_stats(int addr, struct device *dev);
static struct enet_statistics *el3_get_stats(struct device *dev);
static int el3_rx(struct device *dev);
static int el3_close(struct device *dev);
static void set_multicast_list(struct device *dev, int num_addrs, void *addrs);

static dev_info_t dev_info = "3c589_cs";

static dev_link_t *tc589_attach(void);
static void tc589_detach(dev_link_t *);

static dev_link_t *dev_list = NULL;

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*======================================================================

    If we're linked into the kernel, this gets called from the net
    driver initialization code in Space.c.  It returns failure,
    since all 3c589 devices are dynamically allocated.
    
======================================================================*/

#ifndef MODULE
int tc589_probe(struct device *dev)
{
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    register_pcmcia_driver(&dev_info, &tc589_attach, &tc589_detach);
    return -1;
}
#endif

/*======================================================================

    We never need to do anything when a tc589 device is "initialized"
    by the net software, because we only register already-found cards.
    
======================================================================*/

static int tc589_init(struct device *dev)
{
    return 0;
}

/*======================================================================

    tc589_attach() creates an "instance" of the driver, allocating
    local data structures for one device.  The device is registered
    with Card Services.

======================================================================*/

static dev_link_t *tc589_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    struct device *dev;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("3c589_attach()\n");
#endif

    /* Create new ethernet device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &tc589_release;
    link->release.data = (u_long)link;
    link->io.NumPorts1 = 16;
    link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
    link->io.IOAddrLines = 4;
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    link->irq.IRQInfo2 = irq_mask;
    link->conf.Attributes = CONF_ENABLE_IRQ;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;
    link->conf.ConfigIndex = 1;
    link->conf.Present = PRESENT_OPTION;
    
    dev = kmalloc(sizeof(struct device), GFP_KERNEL);
    memset(dev, 0, sizeof(struct device));
    
    /* Make up a EL3-specific-data structure. */
    dev->priv = kmalloc(sizeof(struct el3_private), GFP_KERNEL);
    memset(dev->priv, 0, sizeof(struct el3_private));
    
    /* The EL3-specific entries in the device structure. */
    dev->hard_start_xmit = &el3_start_xmit;
    dev->set_config = &el3_config;
    dev->get_stats = &el3_get_stats;
    dev->set_multicast_list = &set_multicast_list;
    ether_setup(dev);
    dev->name = link->dev_name;
    dev->init = &tc589_init;
    dev->open = &el3_open;
    dev->stop = &el3_close;
    dev->tbusy = 1;
    link->priv = dev;
    
    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask =
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &tc589_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != 0) {
	cs_error(RegisterClient, ret);
	tc589_detach(link);
	return NULL;
    }
    
    return link;
} /* tc589_attach */

/*======================================================================

    This deletes a driver "instance".  The device is de-registered
    with Card Services.  If it has been released, all local data
    structures are freed.  Otherwise, the structures will be freed
    when the device is released.

======================================================================*/

static void tc589_detach(dev_link_t *link)
{
    dev_link_t **linkp;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("3c589_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    if (link->state & DEV_CONFIG) {
	tc589_release((u_long)link);
	if (link->state & DEV_STALE_CONFIG) {
	    link->state |= DEV_STALE_LINK;
	    return;
	}
    }
    
    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free bits */
    *linkp = link->next;
    if (link->priv) {
	struct device *dev = link->priv;
	if (dev->priv)
	    kfree_s(dev->priv, sizeof(struct el3_private));
	kfree_s(link->priv, sizeof(struct device));
    }
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* tc589_detach */

/*======================================================================

    tc589_config() is scheduled to run after a CARD_INSERTION event
    is received, to configure the PCMCIA socket, and to make the
    ethernet device available to the system.
    
======================================================================*/

static void tc589_config(dev_link_t *link)
{
    client_handle_t handle;
    struct device *dev;
    tuple_t tuple;
    cisparse_t parse;
    u_char buf[64];
    int i;
    short ioaddr, *phys_addr;
    
    handle = link->handle;
    dev = link->priv;
    phys_addr = (short *)dev->dev_addr;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("3c589_config(0x%p)\n", link);
#endif

    do {
	tuple.DesiredTuple = CISTPL_CONFIG;
	i = CardServices(GetFirstTuple, handle, &tuple);
	if (i != CS_SUCCESS) break;
	tuple.TupleData = buf;
	tuple.TupleDataMax = 64;
	tuple.TupleOffset = 0;
	i = CardServices(GetTupleData, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(ParseTuple, handle, &tuple, &parse);
	if (i != CS_SUCCESS) break;
	link->conf.ConfigBase = (caddr_t)parse.config.base;
    } while (0);
    if (i != CS_SUCCESS) {
	printk("3c589_cs: unable to parse CIS\n");
	link->state &= ~DEV_CONFIG_PENDING;
	return;
    }
    
    /* Configure card */
    link->state |= DEV_CONFIG;

    do {
	i = CardServices(RequestIO, link->handle, &link->io);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIO, i);
	    break;
	}
	i = CardServices(RequestIRQ, link->handle, &link->irq);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIRQ, i);
	    break;
	}
	i = CardServices(RequestConfiguration, link->handle, &link->conf);
	if (i != CS_SUCCESS) {
	    cs_error(RequestConfiguration, i);
	    break;
	}
	i = request_irq(link->irq.AssignedIRQ, &el3_interrupt,
			0, "3c589_cs");
	if (i != CS_SUCCESS) {
	    printk("3c589_cs: unable to allocate irq %ld\n",
		   link->irq.AssignedIRQ);
	    break;
	}
	dev->irq = link->irq.AssignedIRQ;
	dev->base_addr = link->io.BasePort1;
	dev->tbusy = 0;
	i = register_netdev(dev);
	if (i != 0) {
	    printk("3c589_cs: register_netdev() failed\n");
	    break;
	}
    } while (0);
    
    link->state &= ~DEV_CONFIG_PENDING;
    if (i != 0) {
	tc589_release((u_long)link);
	return;
    }

    ioaddr = dev->base_addr;
    EL3WINDOW(0);

    /* Read the ethernet address and fill in the 3c589 registers.  Unlike
       other EtherLink III adaptors the on-board EEPROM isn't automatically
       read at power-on time.
       We must swap the ethernet address on little-endian machines. */
    for (i = 0; i < 3; i++)
	phys_addr[i] = htons(read_eeprom(ioaddr, i));
    
    /* The address and resource configuration register aren't loaded from
       the EEPROM and *must* be set to 0 and IRQ3 for the PCMCIA version. */
    outw(0x3f00, ioaddr + 8);

    /* The if_port symbol can be set when the module is loaded */
    if (if_port != -1)
	dev->if_port = if_port;
    else
	dev->if_port = read_eeprom(ioaddr, 8) >> 14;
    
    /* This is vital: the transceiver used must be set in the resource
       configuration register.  It took me many hours to discover this. */
    outw(dev->if_port << 14, ioaddr + 6);
    
    printk("%s: 3Com 3c589, port %#3lx, irq %d, %s port, ", dev->name,
	   dev->base_addr, dev->irq, if_names[dev->if_port]);
    for (i = 0; i < 6; i++)
	printk(" %02X", dev->dev_addr[i]);
    printk("\n");
    
} /* tc589_config */

/*======================================================================

    After a card is removed, tc589_release() will unregister the net
    device, and release the PCMCIA configuration.  If the device is
    still open, this will be postponed until it is closed.
    
======================================================================*/

static void tc589_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;
    struct device *dev = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("3c589_release(0x%p)\n", link);
#endif
    
    if (link->open) {
	printk("3c589_cs: release postponed, '%s' still open\n",
	       link->dev_name);
	link->state |= DEV_STALE_CONFIG;
	return;
    }
    
    if (link->dev_name[0] != '\0')
	unregister_netdev(dev);
    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);
    if (dev->irq != 0) {
	free_irq(dev->irq);
	irq2dev_map[dev->irq] = NULL;
    }
    
    link->state &= ~DEV_CONFIG;
    if (link->state & DEV_STALE_LINK)
	tc589_detach(link);
    
} /* tc589_release */

/*======================================================================

    The card status event handler.  Mostly, this schedules other
    stuff to run after an event is received.  A CARD_REMOVAL event
    also sets some flags to discourage the net drivers from trying
    to talk to the card any more.
    
======================================================================*/

static int tc589_event(event_t event, int priority,
		       event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;
    struct device *dev = link->priv;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("3c589_event()\n");
#endif
    
    switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
	if (pc_debug)
	    printk("3c589_cs: registration complete\n");
	break;
#endif
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    dev->tbusy = 1; dev->start = 0;
	    link->release.expires = 5;
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	tc589_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	if (link->state & DEV_CONFIG) {
	    if (link->open) {
		dev->tbusy = 1; dev->start = 0;
	    }
	    CardServices(ReleaseConfiguration, link->handle);
	}
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (link->state & DEV_CONFIG) {
	    CardServices(RequestConfiguration, link->handle, &link->conf);
	    if (link->open) {
		tc589_reset(dev);
		dev->tbusy = 0; dev->start = 1;
	    }
	}
	break;
    }
    return 0;
} /* tc589_event */

/*====================================================================*/

/* Read a word from the EEPROM using the regular EEPROM access register.
   Assume that we are in register window zero.
 */
static ushort read_eeprom(short ioaddr, int index)
{
    outw(EEPROM_READ + index, ioaddr + 10);
    /* Pause for at least 162 us. for the read to take place. */
    udelay(300L);
    return inw(ioaddr + 12);
}

/* Reset and restore all of the 3c589 registers. */
static void tc589_reset(struct device *dev)
{
    ushort ioaddr = dev->base_addr;
    int i;
    
    EL3WINDOW(0);
    outw(0x0001, ioaddr + 4);			/* Activate board. */ 
    outw(dev->if_port << 14, ioaddr + 6);	/* Set the xcvr. */
    outw(0x3f00, ioaddr + 8);			/* Set the IRQ line. */
    
    /* Set the station address in window 2. */
    EL3WINDOW(2);
    for (i = 0; i < 6; i++)
	outb(dev->dev_addr[i], ioaddr + i);
    
    if (dev->if_port == 3)
	/* Start the thinnet transceiver. We should really wait 50ms...*/
	outw(StartCoax, ioaddr + EL3_CMD);
    else if (dev->if_port == 0) {
	/* 10baseT interface, enabled link beat and jabber check. */
	EL3WINDOW(4);
	outw(inw(ioaddr + WN4_MEDIA) | MEDIA_TP, ioaddr + WN4_MEDIA);
    }

    /* Switch to the stats window, and clear all stats by reading. */
    outw(StatsDisable, ioaddr + EL3_CMD);
    EL3WINDOW(6);
    for (i = 0; i < 9; i++)
	inb(ioaddr+i);
    inb(ioaddr + 10);
    inb(ioaddr + 12);
    
    /* Switch to register set 1 for normal use. */
    EL3WINDOW(1);

    /* Accept b-cast and phys addr only. */
    outw(SetRxFilter | RxStation | RxBroadcast, ioaddr + EL3_CMD);
    outw(StatsEnable, ioaddr + EL3_CMD); /* Turn on statistics. */
    outw(RxEnable, ioaddr + EL3_CMD); /* Enable the receiver. */
    outw(TxEnable, ioaddr + EL3_CMD); /* Enable transmitter. */
    /* Allow status bits to be seen. */
    outw(SetReadZero | 0xff, ioaddr + EL3_CMD);
    outw(AckIntr | 0x69, ioaddr + EL3_CMD); /* Ack IRQ */
    outw(SetIntrMask | 0x98, ioaddr + EL3_CMD); /* Set interrupt mask. */
}

static int el3_config(struct device *dev, struct ifmap *map)
{
    if (map->port != (u_char)(-1)) {
	dev->if_port = map->port;
	printk("%s: switched to %s port\n",
	       dev->name, if_names[dev->if_port]);
    }
    return 0;
}

static int el3_open(struct device *dev)
{
    int ioaddr = dev->base_addr;
    dev_link_t *link;
    
    for (link = dev_list; link; link = link->next)
	if (link->priv == dev) break;
    if (!DEV_OK(link))
	return -ENODEV;

    link->open++;
#ifdef MODULE
    MOD_INC_USE_COUNT;
#endif

    EL3WINDOW(0);
    
    irq2dev_map[dev->irq] = dev;
    
    dev->interrupt = 0;
    dev->tbusy = 0; dev->start = 1;
    
    tc589_reset(dev);
    
    if (tc589_debug > 3)
	printk("%s: opened, status %4.4x.\n",
	       dev->name, inw(ioaddr + EL3_STATUS));
    
    return 0;					/* Always succeed */
}

static int el3_start_xmit(struct sk_buff *skb, struct device *dev)
{
    struct el3_private *lp = (struct el3_private *)dev->priv;
    int ioaddr = dev->base_addr;
    
    /* Transmitter timeout, serious problems. */
    if (dev->tbusy) {
	int tickssofar = jiffies - dev->trans_start;
	if (tickssofar < 10)
	    return 1;
	printk("%s: transmit timed out, tx_status %2.2x status %4.4x.\n",
	       dev->name, inb(ioaddr + TX_STATUS), inw(ioaddr + EL3_STATUS));
	dev->trans_start = jiffies;
	/* Issue TX_RESET and TX_START commands. */
	outw(TxReset, ioaddr + EL3_CMD);
	outw(TxEnable, ioaddr + EL3_CMD);
	dev->tbusy = 0;
    }
    
    if (skb == NULL) {
	dev_tint(dev);
	return 0;
    }
    
    if (skb->len <= 0)
	return 0;
    
    if (tc589_debug > 4) {
	printk("%s: el3_start_xmit(length = %ld) called, status %4.4x.\n",
	       dev->name, skb->len, inw(ioaddr + EL3_STATUS));
    }
#ifndef final_version
    {	/* Error-checking code, delete for 1.30. */
	ushort status = inw(ioaddr + EL3_STATUS);
	if (status & 0x0003 		/* IRQ line active, missed one. */
	    && inw(ioaddr + EL3_STATUS) & 3) { 			/* Make sure. */
	    printk("%s: missed interrupt, status then %04x now %04x"
		   "  Tx %2.2x Rx %4.4x.\n", dev->name, status,
		   inw(ioaddr + EL3_STATUS), inb(ioaddr + TX_STATUS),
		   inw(ioaddr + RX_STATUS));
	    if (status & 0x0002) /* Adaptor failure */
		tc589_reset(dev);
	    /* Fake interrupt trigger by masking, acknowledge interrupts. */
	    outw(SetReadZero | 0x00, ioaddr + EL3_CMD);
	    outw(AckIntr | 0x69, ioaddr + EL3_CMD); /* Ack IRQ */
	    outw(SetReadZero | 0xff, ioaddr + EL3_CMD);
	}
    }
#endif
    
    /* Avoid timer-based retransmission conflicts. */
    if (set_bit(0, (void*)&dev->tbusy) != 0)
	printk("%s: transmitter access conflict.\n", dev->name);
    else {
	/* Put out the doubleword header... */
	outw(skb->len, ioaddr + TX_FIFO);
	outw(0x00, ioaddr + TX_FIFO);
	/* ... and the packet rounded to a doubleword. */
	outsl(ioaddr + TX_FIFO, skb->data, (skb->len + 3) >> 2);
	
	dev->trans_start = jiffies;
	if (inw(ioaddr + TX_FREE) > 1536) {
	    dev->tbusy = 0;
	} else
	    /* Interrupt us when the FIFO has room for max-sized packet. */
	    outw(SetTxThreshold + 1536, ioaddr + EL3_CMD);
    }

    dev_kfree_skb (skb, FREE_WRITE);
    
    /* Clear the Tx status stack. */
    {
	short tx_status;
	int i = 4;
	
	while (--i > 0	&&	(tx_status = inb(ioaddr + TX_STATUS)) > 0) {
	    if (tx_status & 0x38) lp->stats.tx_aborted_errors++;
	    outb(0x00, ioaddr + TX_STATUS); /* Pop the status stack. */
	    if (tx_status & 0x30) outw(TxReset, ioaddr + EL3_CMD);
	    if (tx_status & 0x3C) outw(TxEnable, ioaddr + EL3_CMD);
	}
    }
    return 0;
}

/* The EL3 interrupt handler. */
static void el3_interrupt(int reg_ptr, struct pt_regs *regs)
{
    int irq = -(regs->orig_eax+2);
    struct device *dev = (struct device *)(irq2dev_map[irq]);
    int ioaddr, status;
    int i = 0;
    
    if (dev == NULL) {
	printk ("el3_interrupt(): irq %d for unknown device.\n", irq);
	return;
    }
    
    if (dev->interrupt) {
	printk("%s: re-entering the interrupt handler.\n", dev->name);
	return;
    }
    dev->interrupt = 1;
    
    ioaddr = dev->base_addr;
    status = inw(ioaddr + EL3_STATUS);
    
    if (tc589_debug > 4)
	printk("%s: interrupt, status %4.4x.\n", dev->name, status);
    
    while ((status = inw(ioaddr + EL3_STATUS)) & 0x91) {
	if ((dev->start == 0) || ((status & 0xe000) != 0x2000)) {
	    printk("%s: interrupt from dead card\n", dev->name);
	    break;
	}
	
	if (status & 0x10)
	    el3_rx(dev);
	
	if (status & 0x08) {
	    if (tc589_debug > 5)
		printk("    TX room bit was handled.\n");
	    /* There's room in the FIFO for a full-sized packet. */
	    outw(AckIntr | 0x08, ioaddr + EL3_CMD);
	    dev->tbusy = 0;
	    mark_bh(NET_BH);
	}
	if (status & 0x80)		/* Statistics full. */
	    update_stats(ioaddr, dev);
	
	if (++i > 10) {
	    printk("%s: infinite loop in interrupt, status %4.4x.\n",
		   dev->name, status);
	    /* Clear all interrupts */
	    outw(AckIntr | 0xFF, ioaddr + EL3_CMD);
	    break;
	}
	/* Acknowledge the IRQ. */
	outw(AckIntr | 0x41, ioaddr + EL3_CMD); /* Ack IRQ */
    }
    
    if (tc589_debug > 4) {
	printk("%s: exiting interrupt, status %4.4x.\n", dev->name,
	       inw(ioaddr + EL3_STATUS));
    }
    
    dev->interrupt = 0;
    return;
}


static struct enet_statistics *el3_get_stats(struct device *dev)
{
    struct el3_private *lp = (struct el3_private *)dev->priv;
    unsigned long flags;
    
    save_flags(flags);
    cli();
    update_stats(dev->base_addr, dev);
    restore_flags(flags);
    return &lp->stats;
}

/*  Update statistics.  We change to register window 6, so this should be run
    single-threaded if the device is active. This is expected to be a rare
    operation, and it's simpler for the rest of the driver to assume that
    window 1 is always valid rather than use a special window-state variable.
 */
static void update_stats(int ioaddr, struct device *dev)
{
    struct el3_private *lp = (struct el3_private *)dev->priv;
    
    if (tc589_debug > 5)
	printk("%s: updating the statistics.\n", dev->name);
    /* Turn off statistics updates while reading. */
    outw(StatsDisable, ioaddr + EL3_CMD);
    /* Switch to the stats window, and read everything. */
    EL3WINDOW(6);
    lp->stats.tx_carrier_errors 	+= inb(ioaddr + 0);
    lp->stats.tx_heartbeat_errors	+= inb(ioaddr + 1);
    /* Multiple collisions. */	   	inb(ioaddr + 2);
    lp->stats.collisions		+= inb(ioaddr + 3);
    lp->stats.tx_window_errors		+= inb(ioaddr + 4);
    lp->stats.rx_fifo_errors		+= inb(ioaddr + 5);
    lp->stats.tx_packets		+= inb(ioaddr + 6);
    /* Rx packets   */			inb(ioaddr + 7);
    /* Tx deferrals */			inb(ioaddr + 8);
    inw(ioaddr + 10);	/* Total Rx and Tx octets. */
    inw(ioaddr + 12);
    
    /* Back to window 1, and turn statistics back on. */
    EL3WINDOW(1);
    outw(StatsEnable, ioaddr + EL3_CMD);
    return;
}

static int el3_rx(struct device *dev)
{
    struct el3_private *lp = (struct el3_private *)dev->priv;
    int ioaddr = dev->base_addr;
    int boguscnt = 10;
    short rx_status;
    
    if (tc589_debug > 5)
	printk("%s: in rx_packet(), status %4.4x, rx_status %4.4x.\n",
	       dev->name, inw(ioaddr+EL3_STATUS), inw(ioaddr+RX_STATUS));
    while ((rx_status = inw(ioaddr + RX_STATUS)) > 0) {
	if (rx_status & 0x4000) { /* Error, update stats. */
	    short error = rx_status & 0x3800;
	    lp->stats.rx_errors++;
	    switch (error) {
	    case 0x0000:		lp->stats.rx_over_errors++; break;
	    case 0x0800:		lp->stats.rx_length_errors++; break;
	    case 0x1000:		lp->stats.rx_frame_errors++; break;
	    case 0x1800:		lp->stats.rx_length_errors++; break;
	    case 0x2000:		lp->stats.rx_frame_errors++; break;
	    case 0x2800:		lp->stats.rx_crc_errors++; break;
	    }
	} else {
	    short pkt_len = rx_status & 0x7ff;
	    struct sk_buff *skb;
	    
	    skb = alloc_skb(pkt_len+3, GFP_ATOMIC);
	    if (tc589_debug > 4)
		printk("    Receiving packet size %d status %4.4x.\n",
		       pkt_len, rx_status);
	    if (skb != NULL) {
		skb->len = pkt_len;
		skb->dev = dev;
		
		/* 'skb->data' points to the start of sk_buff data area. */
		insl(ioaddr+RX_FIFO, skb->data,
		     (pkt_len + 3) >> 2);
		netif_rx(skb);
		outw(RxDiscard, ioaddr + EL3_CMD); /* Pop top Rx packet. */
		lp->stats.rx_packets++;
		continue;
	    } else if (tc589_debug)
		printk("%s: couldn't allocate a sk_buff of size %d.\n",
		       dev->name, pkt_len);
	}
	lp->stats.rx_dropped++;
	outw(RxDiscard, ioaddr + EL3_CMD); /* Rx discard */
	while (--boguscnt > 0 && inw(ioaddr + EL3_STATUS) & 0x1000)
	    printk("    Waiting for 3c589 to discard packet, status %x.\n",
		   inw(ioaddr + EL3_STATUS) );
	if (--boguscnt < 0)
	    break;
    }
    
    return 0;
}

/* Set or clear the multicast filter for this adaptor.
   num_addrs == -1	Promiscuous mode, receive all packets
   num_addrs == 0	Normal mode, clear multicast list
   num_addrs > 0	Multicast mode, receive normal and MC packets, and do
			best-effort filtering.
 */
static void
set_multicast_list(struct device *dev, int num_addrs, void *addrs)
{
    short ioaddr = dev->base_addr;
    if (tc589_debug > 1)
	printk("%s: setting Rx mode to %d addresses.\n", dev->name, num_addrs);
    if (num_addrs > 0) {
	outw(SetRxFilter|RxStation|RxMulticast|RxBroadcast, ioaddr + EL3_CMD);
    } else if (num_addrs < 0) {
	outw(SetRxFilter | RxStation | RxMulticast | RxBroadcast | RxProm,
	     ioaddr + EL3_CMD);
    } else
	outw(SetRxFilter | RxStation | RxBroadcast, ioaddr + EL3_CMD);
}

static int el3_close(struct device *dev)
{
    int ioaddr = dev->base_addr;
    dev_link_t *link;

    for (link = dev_list; link; link = link->next)
	if (link->priv == dev) break;
    if (link == NULL)
	return -ENODEV;
    
    if (tc589_debug > 2)
	printk("%s: shutting down ethercard.\n", dev->name);
    
    /* Turn off statistics ASAP.  We update lp->stats below. */
    outw(StatsDisable, ioaddr + EL3_CMD);
    
    /* Disable the receiver and transmitter. */
    outw(RxDisable, ioaddr + EL3_CMD);
    outw(TxDisable, ioaddr + EL3_CMD);
    
    if (dev->if_port == 3)
	/* Turn off thinnet power.  Green! */
	outw(StopCoax, ioaddr + EL3_CMD);
    else if (dev->if_port == 0) {
	/* Disable link beat and jabber, if_port may change ere next open(). */
	EL3WINDOW(4);
	outw(inw(ioaddr + WN4_MEDIA) & ~MEDIA_TP, ioaddr + WN4_MEDIA);
    }
    
    /* Switching back to window 0 disables the IRQ. */
    EL3WINDOW(0);
    /* But we explicitly zero the IRQ line select anyway. */
    outw(0x0f00, ioaddr + WN0_IRQ);
        
    /* Check if the card actually exists before updating the statistics. */
    if (inw(ioaddr+EL3_STATUS) != 0xffff)
	update_stats(ioaddr, dev);

    link->open--;
    dev->start = 0;
    if (link->state & DEV_STALE_CONFIG)
	tc589_release((u_long)link);
    
#ifdef MODULE
    MOD_DEC_USE_COUNT;
#endif
    
    return 0;
}

/*====================================================================*/

#ifdef MODULE
char kernel_version[] = UTS_RELEASE;

int init_module(void)
{
    servinfo_t serv;
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk("serial_cs: Card Services release does not match!");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &tc589_attach, &tc589_detach);
    return 0;
}

void cleanup_module(void)
{
    printk("3c589_cs: unloading\n");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL)
	tc589_detach(dev_list);
}
#endif /* MODULE */
