/*
 *      Copyright (C) 1993,1994 Bas Laarhoven.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 $Source: /home/bas/distr/ftape-1.14d/RCS/ftape-read.c,v $
 $Author: bas $
 *
 $Revision: 1.13 $
 $Date: 1994/12/07 20:49:11 $
 $State: Beta $
 *
 *      This file contains the reading code
 *      for the QIC-117 floppy-tape driver for Linux.
 */

static char RCSid[] =
"$Id: ftape-read.c,v 1.13 1994/12/07 20:49:11 bas Beta bas $";


#include <linux/string.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <asm/dma.h>
#include <asm/segment.h>
#include <asm/system.h>

#include "ftape.h"
#include "ftape-read.h"
#include "qic117.h"
#include "ftape-io.h"
#include "ftape-rw.h"
#include "ftape-write.h"
#include "ftape-eof.h"
#include "ecc.h"

/*      Global vars.
 */
int buf_pos_rd = 0;
int buf_len_rd = 0;
int just_before_eof = 0;

/*      Local vars.
 */

void
ftape_zap_read_buffers( void)
{
  int i;

  for (i = 0; i < NR_ITEMS( buffer); ++i) {
    buffer[ i].address = ((byte*) tape_buffer) + i * BUFF_SIZE;
    buffer[ i].status = empty;
    buffer[ i].bytes = 0;
    buffer[ i].skip = 0;
    buffer[ i].retry = 0;
  }
  buf_len_rd = 0;
  buf_pos_rd = 0;
  eof_mark = 0;
  ftape_state = idle;
}

int
correct_and_copy( unsigned int tail, byte* destination)
{
  TRACE_FUN( 8, "correct_and_copy");
  struct memory_segment mseg;
  int result;
  BAD_SECTOR read_bad;

  mseg.read_bad = buffer[ tail].crc_error_map;
  mseg.marked_bad = 0;          /* not used... */
  mseg.blocks = buffer[ tail].bytes / SECTOR_SIZE;
  mseg.data = buffer[ tail].address;
  /*    If there are no data sectors we can skip this segment.
   */
  if (mseg.blocks <= 3) {
    TRACE( 4, "empty segment");
    TRACE_EXIT;
    return 0;
  }
  read_bad = mseg.read_bad;
  history.crc_errors += count_ones( read_bad);
  result = ecc_correct_data( &mseg);
  if (read_bad != 0 || mseg.corrected != 0) {
    TRACElx( 4, "crc error map:", read_bad);
    TRACElx( 4, "corrected map:", mseg.corrected);
    history.corrected += count_ones( mseg.corrected);
  }
  if (result == ECC_CORRECTED || result == ECC_OK) {
    if (result == ECC_CORRECTED) {
      TRACEi( 3, "ecc corrected segment:", buffer[ tail].segment_id);
    }
    memcpy( destination, mseg.data, (mseg.blocks - 3) * SECTOR_SIZE);
    if ((read_bad ^ mseg.corrected) & mseg.corrected) {
      /* sectors corrected without crc errors set */
      history.crc_failures++;
    }
    TRACE_EXIT;
    return (mseg.blocks - 3) * SECTOR_SIZE;
  } else {
    TRACEi( 1, "ecc failure on segment", buffer[ tail].segment_id);
    history.ecc_failures++;
    TRACE_EXIT;
    return -EAGAIN;             /* should retry */
  }
  TRACE_EXIT;
  return 0;
}

/*      Read given segment into buffer at address.
 */
int
read_segment( unsigned segment_id, byte* address, int* eof_mark,
             int read_ahead)
{
  TRACE_FUN( 5, "read_segment");
  int read_done= 0;
  int result = 0;
  int bytes_read = 0;
  int retry = 0;
  
  TRACEi( 5, "segment_id =", segment_id);
  if (ftape_state != reading) {
    if (ftape_state == writing) {
      ftape_flush_buffers();    /* flush write buffer */
      TRACE( 5, "calling ftape_abort_operation");
      result = ftape_abort_operation( &current_segment);
      if (result < 0) {
        TRACE( 1, "ftape_abort_operation failed");
        TRACE_EXIT;
        return -EIO;
      }
    } else {
      /* clear remaining read buffers */
      ftape_zap_read_buffers();
    }
    ftape_state = reading;
  }
  if (segment_id >= segments_per_track * tracks_per_tape) {
    TRACE( 5, "reading past end of tape");
    TRACE_EXIT;
    return -ENOSPC;
  }
  for (;;) {
    /*    Search all full buffers for the first matching the wanted segment.
     *    Clear other buffers on the fly.
     */
    while (!read_done && buffer[ tail].status == full) {
      if (buffer[ tail].segment_id == segment_id) {
        unsigned eof_sector;
        unsigned sector_count = 0;
        unsigned long bsm = bad_sector_map[ segment_id];
        int i;

        /*        If out buffer is already full, return its contents.
         */
        if (buffer[ tail].deleted) {
          TRACEi( 5, "found segment in cache :", segment_id);
          TRACE_EXIT;
          /*  Return a value that read_header_segment understands.
           *  As this should only occur when searching for the header
           *  segments it shouldn't be misinterpreted elsewhere.
           */
          return 0;
        }
        TRACEi( 5, "found segment in cache :", segment_id);
        eof_sector = check_for_eof( segment_id);
        if (eof_sector > 0) {
          TRACEi( 5, "end of file mark in sector:", eof_sector);
          for (i = 1; i < eof_sector; ++i) {
            if ((bsm & 1) == 0) {
              ++sector_count;
            }
            bsm >>= 1;
          }
          *eof_mark = 1;
        }
        if (eof_sector != 1) {  /* not found or gt 1 */
          result = correct_and_copy( tail, address);
          TRACEi( 5, "segment contains (bytes) :", result);
          if (result < 0) {
            if (result != -EAGAIN) {
              TRACE_EXIT;
              return result;
            }
            /* keep read_done == 0, will trigger ftape_abort_operation
             * because reading wrong segment.
             */
            TRACE( 1, "ecc failed, retry");
            ++retry;
          } else {
            read_done = 1;
          }
        } else {
          read_done = 1;
        }
        if (eof_sector > 0) {
          bytes_read = sector_count * SECTOR_SIZE;
          TRACEi( 5, "partial read count:", bytes_read);
        } else {
          bytes_read = result;
        }
      } else {
        TRACEi( 5, "zapping segment in cache :", buffer[ tail].segment_id);
      }
      buffer[ tail].status = empty;
      next_buffer( &tail);
    }
    if (!read_done && buffer[ tail].status == reading) {
      if (buffer[ tail].segment_id == segment_id) {
        int result = wait_segment( reading);
        if (result < 0) {
          if (result == -EINTR) {
            TRACE_EXIT;
            return result;
          }
          TRACE( 1, "wait_segment failed while reading");
          ftape_abort_operation( &current_segment);
        }
      } else {
        /*        We're reading the wrong segment, stop runner.
         */
        ftape_abort_operation( &current_segment);
      }
    }
    /*    if just passed the last segment on a track, wait for BOT or EOT mark.
     */
    if (runner_status == logical_eot) {
      int status;
      result = ftape_ready_wait( 10 * SECOND, &status);
      if (result < 0) {
        TRACE( 1, "ftape_ready_wait waiting for eot/bot failed");
      }
      if (status & (QIC_STATUS_AT_BOT | QIC_STATUS_AT_EOT) == 0) {
        TRACE( 1, "eot/bot not reached");
      }
      runner_status = end_of_tape;
    }
    /*    should runner stop ?
     */
    if (runner_status == aborting || runner_status == buffer_overrun ||
        runner_status == end_of_tape) {
      if (runner_status != end_of_tape) {
        ftape_smart_stop( &location_known, &current_segment);
      }
      if (runner_status == aborting) {
        if (buffer[ head].status == reading || buffer[ head].status == error) {
          buffer[ head].status = empty;
        }
      }
      runner_status = idle;       /* aborted ? */
    }
    /*    If segment to read is empty, do not start runner for it,
     *    but wait for next read call.
     */
    if (bad_sector_map[ segment_id] == EMPTY_SEGMENT) {
      bytes_read = 0;           /* flag empty segment */
      read_done = 1;
    }
    /*  Allow escape from this loop on signal !
     */
    if (current->signal & ~_SIG_BLOCK) {
      TRACE( 2, "interrupted by signal");
      TRACE_EXIT;
      return -EINTR;
    }
    /*    If we got a segment: quit, or else retry up to limit.
     *    Allow at least three attempts, First causing the error,
     *    Second for possibly failing (fast) ftape_start_tape, Third
     *    for recovery of ftape_start_tape.
     */
    if (read_done || retry > RETRIES_ON_ECC_ERROR) {
      break;
    }
    /*    Now at least one buffer is empty !
     *    Restart runner & tape if needed.
     */
    TRACEx3( 8, "head: %d, tail: %d, runner_status: %d",
            head, tail, runner_status);
    TRACEx2( 8, "buffer[].status, [head]: %d, [tail]: %d",
            buffer[ head].status, buffer[ tail].status);
    if (buffer[ tail].status == empty) {
      setup_new_segment( &buffer[ head], segment_id, -1, SECTORS_PER_SEGMENT);
      if (!read_ahead) {
        buffer[ head].next_segment = 0; /* disable read-ahead */
      }
      calc_next_cluster( &buffer[ head]);
      if (runner_status == idle) {
        result= ftape_start_tape( segment_id);
        if (result < 0) {
          TRACEx1( 1, "Error: segment %d unreachable", segment_id);
          TRACE_EXIT;
          return result;
        }
        runner_status = running;
      }
      buffer[ head].status = reading;
      setup_fdc_and_dma( &buffer[ head], FDC_READ);
    }
  }
  if (read_done) {
    TRACE_EXIT;
    return bytes_read;
  } else {
    TRACE( 1, "too many retries");
    TRACE_EXIT;
    return -EIO;
  }
}

int
read_header_segment( byte* address)
{
  TRACE_FUN( 5, "read_header_segment");
  int i;
  int result;
  int header_segment = -1;
  unsigned int tape_segments_per_track;
  unsigned int tape_tracks_per_cartridge;
  unsigned int max_floppy_side;
  unsigned int max_floppy_track;
  unsigned int max_floppy_sector;
  int status;

  result = ftape_report_drive_status( &status);
  if (result < 0) {
    TRACE( 1, "error: error_status or report failure");
    TRACE_EXIT;
    return -EIO;
  }
  TRACE( 5, "reading...");
  ftape_last_segment.id = 68;   /* will allow us to read the header ! */
  /*  We're looking for the first header segment.
   *  A header segment cannot contain bad sectors, therefor at the
   *  tape start, segments with bad sectors are (according to QIC-40/80)
   *  written with deleted data marks and must be skipped.
   */
  for (i = 0; i < ftape_last_segment.id; ++i) {
    bad_sector_map[ i] = 0x00000000;
  }
  result = 0;
  for (header_segment = 0;
       header_segment < ftape_last_segment.id && result == 0;
       ++header_segment) {
    /*  Set no read-ahead, the isr will force read-ahead whenever
     *  it encounters deleted data !
     */
    result = read_segment( header_segment, address, &status, 0);
  }
  if (result < 0 || header_segment >= ftape_last_segment.id) {
    TRACE( 1, "header segment not found");
    TRACE_EXIT;
    return -EIO;
  }
  result = ftape_abort_operation( &current_segment);
  if (result < 0) {
    TRACE( 1, "ftape_abort_operation failed");
    TRACE_EXIT;
    return -EIO;
  }
  if (GET4( address, 0) != 0xaa55aa55) {
    TRACE( 1, "tape not formatted (wrong signature)");
    TRACE_EXIT;
    return -EIO;
  }
  header_segment =
  header_segment_1 = GET2( address, 6);
  header_segment_2 = GET2( address, 8);
  TRACEx2( 2, "header segments are %d and %d",
          header_segment_1, header_segment_2);
  /*    Verify tape parameters...
   *    QIC-40/80 spec:                 tape_parameters:
   *
   *    segments-per-track              segments_per_track
   *    tracks-per-cartridge            tracks_per_tape
   *    max-floppy-side                 (segments_per_track *
   *                                    tracks_per_tape - 1) /
   *                                    segments_per_head
   *    max-floppy-track                segments_per_head /
   *                                    segments_per_cylinder - 1
   *    max-floppy-sector               segments_per_cylinder *
   *                                    SECTORS_PER_SEGMENT
   */

  tape_segments_per_track = GET2( address, 24);
  tape_tracks_per_cartridge = *(address + 26);
  max_floppy_side = *(address + 27);
  max_floppy_track = *(address + 28);
  max_floppy_sector = *(address + 29);

  TRACEx5( 4, "(spt/tpc/fhm/ftm/fsm) = %d/%d/%d/%d/%d",
          tape_segments_per_track, tape_tracks_per_cartridge,
          max_floppy_side, max_floppy_track, max_floppy_sector);
  if (tape_segments_per_track == 0 && tape_tracks_per_cartridge == 0 &&
      max_floppy_side == 0 && max_floppy_track == 0 &&
      max_floppy_sector == 0) {
    /*  QIC-40 Rev E and earlier has no values in the header.
     */
    tape_segments_per_track = 68; 
    tape_tracks_per_cartridge = 20;
    max_floppy_side = 1;
    max_floppy_track = 169;
    max_floppy_sector = 128;
  }
  /*  This test will compensate for the wrong parameter on tapes
   *  formatted by Conner software.
   */
  if (tape_segments_per_track == 150 &&
      tape_tracks_per_cartridge == 28 &&
      max_floppy_side == 7 &&
      max_floppy_track == 149 &&
      max_floppy_sector == 128) {
    TRACE( -1, "the famous CONNER bug: max_floppy_side off by one !");
    max_floppy_side = 6;
  }
  /*
   *    in case report_drive_configuration command failed,
   *    we set the parameters according to the tape data.
   */
  if (unknown_drive_config) {
    segments_per_track = tape_segments_per_track;
    tracks_per_tape = tape_tracks_per_cartridge;
    segments_per_head = ( (tracks_per_tape + max_floppy_side) /
                         (max_floppy_side + 1) ) * segments_per_track;
  }
  /*
   *    Verify drive_configuration with tape parameters
   */
  if ( (segments_per_track != tape_segments_per_track) ||
      (tracks_per_tape != tape_tracks_per_cartridge) ||
      ((segments_per_track * tracks_per_tape - 1) / segments_per_head
       != max_floppy_side) ||
      (segments_per_head / segments_per_cylinder - 1 != max_floppy_track) ||
      (segments_per_cylinder * SECTORS_PER_SEGMENT != max_floppy_sector) ) {
    TRACE( 1, "Tape parameters error");
    TRACE_EXIT;
    return -EIO;
  }
  first_data_segment = GET2( address, 10); /* first data segment */
  TRACEi( 4, "first data segment:", first_data_segment);
  /* Copy the bad sector map into our local buffer.
   */
  memcpy( bad_sector_map, address + 2 * SECTOR_SIZE, sizeof( bad_sector_map));
#if 0
  /* for testing of bad sector handling at end of tape
   */
  bad_sector_map[ segments_per_track * tracks_per_tape - 3] = 0x000003e0;
  bad_sector_map[ segments_per_track * tracks_per_tape - 2] = 0xff3fffff;
  bad_sector_map[ segments_per_track * tracks_per_tape - 1] = 0xffffe000;
#endif
  /*  Find the highest segment id that allows still one full
   *  deblock_buffer to be written to tape.
   */
  ftape_last_segment.size = 0;
  for (i = segments_per_track * tracks_per_tape - 1; i >= 0; --i) {
    int space = SECTORS_PER_SEGMENT - 3 - count_ones( bad_sector_map[ i]);
    if (space > 0) {
      ftape_last_segment.size += space; /* sectors free */
      ftape_last_segment.free = (ftape_last_segment.size -
                                 sizeof( deblock_buffer) / SECTOR_SIZE);
      if (ftape_last_segment.free >= 0) {
        ftape_last_segment.id = i;
        TRACEx2( 4, "`last' segment is %d, %d Kb",
                ftape_last_segment.id, ftape_last_segment.size);
        break;
      }
    }
  }
#if 0
  /*  Enable to test error recovery on Verbatim tape
   *  with bad area.
   */
  bad_sector_map[ 16] = 0xffffff80;
  bad_sector_map[ 17] = 0xffffffff;
  bad_sector_map[ 18] = 0x0000ffff;
#endif
#if 0
  /*  Enable to test bad sector handling
   */
  bad_sector_map[ 30] = 0xfffffffe;
  bad_sector_map[ 32] = 0x7fffffff;
  bad_sector_map[ 34] = 0xfffeffff;
  bad_sector_map[ 36] = 0x55555555;
  bad_sector_map[ 38] = 0xffffffff;
  bad_sector_map[ 50] = 0xffff0000;
  bad_sector_map[ 51] = 0xffffffff;
  bad_sector_map[ 52] = 0xffffffff;
  bad_sector_map[ 53] = 0x0000ffff;
#endif
#if 0
  /*  Enable when testing multiple volume tar dumps.
   */
  for (i = first_data_segment; i <= ftape_last_segment.id - 7; ++i) {
    bad_sector_map[ i] = EMPTY_SEGMENT;
  }
#endif
#if 0
  /*  Enable when testing bit positions in crc_error_map
   */
  for (i = first_data_segment; i <= ftape_last_segment.id; ++i) {
    bad_sector_map[ i] |= 0x00ff00ff;
  }
#endif
    
  if (tracing > 2) {    
    unsigned int map;
    int good_sectors = 0;
    int bad_sectors;
    unsigned int total_bad = 0;

    for (i = first_data_segment;
         i < segments_per_track * tracks_per_tape; ++i) {
      map = bad_sector_map[ i];
      bad_sectors = count_ones( map);
      if (bad_sectors > 0) {
        TRACEx2( 6, "bsm for segment %4d: 0x%08x", i, map);
        if (bad_sectors > SECTORS_PER_SEGMENT - 3) {
          bad_sectors = SECTORS_PER_SEGMENT - 3;
        }
        total_bad += bad_sectors;
      }
      good_sectors += SECTORS_PER_SEGMENT - 3 - bad_sectors;
    }
    TRACEx1( 3, "%d Kb usable on this tape",
            good_sectors - ftape_last_segment.free);
    if (total_bad == 0) {
      TRACE( 1, "WARNING: this tape has no bad blocks registered !");
    } else {
      TRACEx1( 2, "%d Sectors are marked as bad", total_bad);
    }
  }
  /* Copy the failed sector log into our local buffer.
   */
  if (!ftape_validate_label( &deblock_buffer[ 30])) {
    TRACE( -1, "This tape has no `Linux raw format' label,\n"
          "***** Use `mt' to erase this tape if you want to use file marks !");
  } else {
    extract_file_marks( address);
  }
  ftape_reset_position();
  TRACE_EXIT;
  return 0;
}

int
_ftape_read( char* buff, int req_len)
{
  TRACE_FUN( 5, "_ftape_read");
  int result = 0;
  int cnt;
  int to_do = req_len;
  static int remaining;
  int bytes_read = 0;

  if (ftape_offline || !formatted || no_tape) {
    TRACEx3( -1, "offline = %d, formatted = %d, no_tape = %d",
            ftape_offline, formatted, no_tape);
    result = -EIO;
  } else {
    history.used |= 1;
    if (first_data_segment == -1) {
      result = read_header_segment( deblock_buffer);
    }
  }
  if (result < 0) {
    TRACE_EXIT;
    return result;
  }
  /*  As GNU tar doesn't accept partial read counts when the multiple
   *  volume flag is set, we make sure to return the requested amount
   *  of data. Except, of course, at the end of the tape or file mark.
   */
  while (to_do > 0) {           /* don't return with a partial count ! */ 
    /*  If we're reading the `last' segment(s) on tape, make sure we don't
     *  get more than 29 Kb from it (As it only contains this much).
     *  This works only for sequential access, so random access should
     *  stay away from this `last' segment.
     *  Note: ftape_seg_pos points to the next segment what will be
     *        read, so it's one too hight here!
     */
    if (!eof_mark && ftape_seg_pos - 1 >= ftape_last_segment.id) {
      TRACEi( 5, "remaining of last segment:", remaining);
      if (to_do > remaining) {
        to_do = remaining;      /* fake a smaller request */
        TRACE( 5, "clipped request to remaining");
      }
    }
    while (!eof_mark && buf_len_rd == 0) {
      /*  When starting to read the `last' segment, set remaining
       */
      if (ftape_seg_pos == ftape_last_segment.id) {
        remaining = sizeof( deblock_buffer);
        TRACEi( 5, "remaining set to:", remaining);
      }
      result = read_segment( ftape_seg_pos, deblock_buffer, &eof_mark, 1);
      if (result < 0) {
        TRACEx1( 4, "read_segment result: %d", result);
        TRACE_EXIT;
        return result;
      }
      /*  Allow escape from this loop on signal !
       */
      if (current->signal & ~_SIG_BLOCK) {
        TRACE( 2, "interrupted by signal");
        TRACE_EXIT;
        return -EINTR;
      }
      buf_pos_rd = 0;
      buf_len_rd = result;
      ++ftape_seg_pos;
    }
    /*  Take as much as we can use
     */
    cnt = (buf_len_rd < to_do) ? buf_len_rd : to_do;
    TRACEi( 7, "nr bytes just read:", cnt);
    result = verify_area( VERIFY_WRITE, buff, cnt);
    if (result) {
      TRACE( 1, "verify_area failed");
      TRACE_EXIT;
      return -EIO;
    }
    memcpy_tofs( buff, deblock_buffer + buf_pos_rd, cnt);
    buff += cnt;
    to_do -= cnt;               /* what's left from req_len */
    remaining -= cnt;           /* what remains on this tape */
    bytes_read += cnt;          /* what we got so far */
    buf_pos_rd += cnt;          /* index in buffer */
    buf_len_rd -= cnt;          /* remaining bytes in buffer */
    if (eof_mark && buf_len_rd == 0) { /* nothing left */
      just_before_eof = 1;
      TRACE( 5, "partial count because of eof mark");
      if (bytes_read == 0) {
        eof_mark = 0;           /* no need for mark next read */
      }
      break;
    }
  }
  TRACE_EXIT;
  return bytes_read;
}
