/*
 * Copyright 1992 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of the example below.
 */

/*
 *
 *  source file:   ./xtpanel/graph.c
 *
 * Steve Cole, Dave Nichols (SEP), November 20 1992
 *      Inserted this sample edit history entry.
 *      Please log any further modifications made to this file:
 * Steve Cole (SEP), May 5 1993
 *	Rewrote to use form instead of box.
 *	Took out label and other annotation; better if users specify.
 */

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Scrollbar.h>

#include "object.h" 
#include "tree.h"
#include "builders.h"
#include "string_buf.h"

#include <stdio.h>
#include <string.h>

typedef struct graphinfo {
    float minval;
    float maxval;
    char* format;
    int nsamp;
    Widget *widgets;
} _ginfo;

/* translation table used for graph widget */
char graph_trans[] =
  "<Leave>:         update_graph()";

extern void graph_callback();
extern void graph_scroll_callback();
extern void graph_update();
extern void parse_valuestring();

void build_graph(root,parent)
     entry *root;
     Widget parent;
{
    Objdef *object;
    entry *curr;
    char* label;
    struct graphinfo *graph_info;
    Arg args[20], orientarg[1];
    int narg, nargsv;
    Widget form, scrollbar, prev;
    float valmin,valmax,val,top;
    float *values;
    char *valstring;
    int nsamp,isamp;
    char text[100];
    int width,height;
    char *orient;
    XtOrientation xtorient;
    char defname[12];
    static int numgraph=1;
    
    /* create new object */
    object = new_object();

    /* construct default graph name */
    sprintf(defname,"graph%d",numgraph++);
    
    /* find name, action, nsamp in tree */
    object->name = get_value(root,"name",defname);
    label = get_value(root,"label",object->name);
    object->action = parse_actions(object->name,root);
    graph_info = (struct graphinfo*) malloc( sizeof( struct graphinfo ) );
    object->info = graph_info;
    graph_info->minval = ((float) atof(get_value(root,"min","0")));
    graph_info->maxval = ((float) atof(get_value(root,"max","1")));
    graph_info->nsamp = ((int) atoi(get_value(root,"nsamp","10")));

    /* form widget holds the scrollbars*/
    narg = 0;
    common_tags(parent,root,args,&narg,SET_POSITION | SET_COLOR | SET_PIXMAP );
    /* no space in between scrollbars */
    XtSetArg(args[narg], XtNdefaultDistance, 0); narg++;

    form = XtCreateManagedWidget("graphbox",formWidgetClass,parent,args,narg);
    
    /* use translations to update value whenever mouse leaves the form */
    XtAugmentTranslations(form,XtParseTranslationTable(graph_trans));

    /* allocate space to hold slider values and widget names */
    values = (float*) malloc ( graph_info->nsamp * sizeof(float));
    graph_info->widgets = (Widget*) 
              malloc ( graph_info->nsamp * sizeof(Widget));

    /* get the string containing default values */
    valstring = get_value(root,"value","");

    /* decode */
    parse_valuestring(valstring,values,graph_info->nsamp," \n\t");

    graph_info->format = get_value(root,"format","%f");
    height = (int) atoi(get_value(root,"height","5"));
    width = (int) atoi(get_value(root,"width","100"));
    orient = get_value(root,"orientation","horizontal");

    narg = 0;
    /* common parameters */
    common_tags(form,root,args,&narg,SET_COLOR | SET_ORIENT);

    /* height, width */
    /* done here instead of in common_tags so we could have defaults */
    XtSetArg(args[narg], XtNlength, width); narg++;
    XtSetArg(args[narg], XtNthickness, height); narg++;
    /* turn off border */
    XtSetArg(args[narg], XtNborderWidth, 0); narg++;
    /* solid color thumb if foreground color requested */
    if (is_specified(root,"foreground")) {
       XtSetArg(args[narg], XtNthumb, None); narg++;
       }
    /* let pixmap of the form show through the scrollbars */
    XtSetArg(args[narg], XtNbackgroundPixmap, ParentRelative); narg++;

    nargsv = narg;

    /* loop over sliders */
    for (isamp=0; isamp<graph_info->nsamp; isamp++) {
  
    narg = nargsv;

    /* determine the correct starting point for the sliders */
    top = values[isamp]/graph_info->maxval;
    if (sizeof(float) > sizeof(XtArgVal))
      {
	  XtSetArg(args[narg], XtNtopOfThumb, top); narg++;
      }
    else
      {
	  XtArgVal * l_top = (XtArgVal *) &top;
	  XtSetArg(args[narg], XtNtopOfThumb, *l_top); narg++;
      }

    /* arrangement in form is relative to previous scrollbar
     * and depends upon orientation */
    if (isamp != 0) {
      XtSetArg(orientarg[0], XtNorientation, &xtorient); 
      XtGetValues( prev, orientarg, ONE );
      if (xtorient == XtorientHorizontal) {
    	XtSetArg(args[narg], XtNfromVert, prev); narg++;
      } else {
    	XtSetArg(args[narg], XtNfromHoriz, prev); narg++;
      }
    }

    /* slider is actually an athena scrollbar widget */
    scrollbar = XtCreateManagedWidget(object->name,scrollbarWidgetClass,
                   form,args,narg);
    
    /* scroll callback is for incremental scrolling with left and 
       right buttons */
    XtAddCallback( scrollbar, XtNscrollProc, graph_scroll_callback, 
		  (XtPointer) object );
    
    /* set to the correct starting point */
    (void) XawScrollbarSetThumb(scrollbar,
    (values[isamp]-graph_info->minval)/
            (graph_info->maxval-graph_info->minval),-1.);
    
    graph_info->widgets[isamp] = scrollbar;
    prev = scrollbar;
    }
    
    object->value = strdupl(valstring);
    object->widgetname = form;
    object->updater = graph_update;
    
}

void
  graph_scroll_callback(widget, client_data, pos_ptr)
Widget widget;
XtPointer client_data, pos_ptr;
{
    Arg arg[1];
    float top;
    int pos;
    pos = (int) pos_ptr;
    /* get the current position of the graph */
    XtSetArg( arg[0], XtNtopOfThumb, &top );
    XtGetValues( widget, arg, ONE );
    /* now compute new position - 5% change */
    top -= pos/abs(pos) * 0.05;
    if (top > 1.) top = 1.;
    if (top < 0.) top = 0.;
    /* update the graph */
    if (sizeof(float) > sizeof(XtArgVal))
      {
	  XtSetArg(arg[0], XtNtopOfThumb, top);
      }
    else
      {
	  XtArgVal * l_top = (XtArgVal *) &top;
	  XtSetArg(arg[0], XtNtopOfThumb, *l_top);
      }
    XtSetValues( widget, arg, ONE );
}

void
  graph_update(object, value)
Objdef *object;
char *value;
{
    struct graphinfo *graph_info;
    Arg arg[1];
    char text[10];
    float top,val;
    Dimension len;
    int pos;
    float *values;
    int isamp;

    graph_info = (struct graphinfo *) object->info;
    values = (float*) malloc ( graph_info->nsamp * sizeof(float));

    /* update value string */
    object->value = strdupl(value);

    /* decode */
    parse_valuestring(value,values,graph_info->nsamp," \n\t");

    /* loop over sliders */
    for (isamp=0; isamp<graph_info->nsamp; isamp++) {

    top = (values[isamp] - graph_info->minval)/
             (graph_info->maxval-graph_info->minval);
    if (top > 1.) top = 1.;
    if (top < 0.) top = 0.;
    /* compute new value */
    sprintf( text, graph_info->format, 
	    (float) graph_info->minval +
	    top*(graph_info->maxval - graph_info->minval));
    /* update the graph */
    if (sizeof(float) > sizeof(XtArgVal))
      {
	  XtSetArg(arg[0], XtNtopOfThumb, top);
      }
    else
      {
	  XtArgVal * l_top = (XtArgVal *) &top;
	  XtSetArg(arg[0], XtNtopOfThumb, *l_top);
      }
    XtSetValues( graph_info->widgets[isamp], arg, ONE );
    }
    perform_actions(object->name,object->action,0);
}

void parse_valuestring(valstring,values,nsamp,separator) 
    char *valstring;
    float *values;
    int nsamp;
    char *separator;
{
    char *item;
    int isamp,jsamp;

    for( item=strtok(valstring,separator),isamp=0;
         item != (char*)0 && isamp <nsamp ;
         item =strtok( (char*)0, separator ),isamp++){
		sscanf(item,"%f",&values[isamp]);
    }

    /* if valstring does not contain enough samples, reuse the last one */
    if (isamp < nsamp) {
	for (jsamp=isamp; jsamp<nsamp; jsamp++) {
		values[jsamp] = values[isamp-1];
	}
    }
}

/* callback used when graph button is pressed or cursor leaves graph */
void
graph_callback(widget, client_data, callData)
Widget widget;
XtPointer client_data, callData;
{
    Objdef *object;
    struct graphinfo *graph_info;
    Arg arg[1];
    char text[64];
    float top;
    Dimension len;
    int isamp;
    char *valstring;
    string_buf *buffer;

    object = (Objdef *) client_data;
    graph_info = (struct graphinfo *) object->info;

    /* loop over widgets */
    strcpy(object->value,"");
    buffer = buf_start();

    for (isamp=0; isamp<graph_info->nsamp; isamp++) {

    /* get the current position of the graph */
    XtSetArg( arg[0], XtNtopOfThumb, &top );
    XtGetValues( graph_info->widgets[isamp], arg, ONE );

    /* turn it into text, add to value string */
    sprintf( text, graph_info->format, 
	    (float) graph_info->minval +
	    top*(graph_info->maxval - graph_info->minval));
    buf_cat( buffer,text,strlen(text));
    buf_cat( buffer," ",1);
    }
    object->value = buf_fetch(buffer);
/*    buf_free(buffer);*/

    perform_actions(object->name,object->action,1);
}

/*
 * this routine gets called whenever we leave
 * a graph box, to be sure the graph's value is up to date
 */
void graph_update_callback( w, event, params, num_params)
     Widget w;
     XEvent *event;
     String *params;
     Cardinal *num_params;
{
    Objdef* object;
    struct graphinfo *graph_info;
    char *value;
    Arg arg[1];
    int narg;
    int isamp;
    float top;
    char text[64];
    Dimension len;
    char *valstring;
    string_buf *buffer;

    object = find_by_widget(w);
    graph_info = (struct graphinfo *) object->info;

    /* loop over widgets */
    strcpy(object->value,"");
    buffer = buf_start();

    for (isamp=0; isamp<graph_info->nsamp; isamp++) {

    /* get the current position of the graph */
    XtSetArg( arg[0], XtNtopOfThumb, &top );
    XtGetValues( graph_info->widgets[isamp], arg, ONE );

    /* turn it into text, add to value string */
    sprintf( text, graph_info->format, 
	    (float) graph_info->minval +
	    top*(graph_info->maxval - graph_info->minval));
    buf_cat( buffer,text,strlen(text));
    buf_cat( buffer," ",1);
    }
    object->value = buf_fetch(buffer);

}
