/*
 * Copyright 1992 the Board of Trustees of the Leland Stanford Junior
 * University. Official permission to use this software is included in
 * the documentation. It authorizes you to use this file for any
 * non-commercial purpose, provided that this copyright notice is not
 * removed and that any modifications made to this file are commented
 * and dated in the style of the example below.
 */

/*
 *
 *  source file:   ./xtpanel/parse.c
 *
 * Steve Cole, Dave Nichols (SEP), August 28 1992
 *      Inserted this sample edit history entry.
 *      Please log any further modifications made to this file:
 * Steve Cole (SEP), November 20 1992
 *      added toggle, scrollbar, graph objects
 * Steve Cole (SEP), February 23 1993
 *      added field object
 * Steve Cole (SEP), February 24 1993
 *      added -U to list of cpp instructions; put -cpp in self-doc
 * Steve Cole (SEP), May 6 1993
 *      added timer, startup, grid objects.
 * Steve Cole (SEP), May 10 1993
 *      parse_itemlist passes back separator.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include "tree.h"
#include <stdio.h>
#include <string.h>
#include "string_buf.h"

#ifndef CPP_COM
#define CPP_COM "/lib/cpp -P"
#endif

#ifdef _POSIX_SOURCE

#include <limits.h>
#ifdef MAX_PATH
#define MAXPATHLEN PATH_MAX
#else
#define MAXPATHLEN 255
#endif

#include <unistd.h>

#else /* not posix */

#include <sys/param.h>

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif
extern   char* getenv();

#endif /* not posix */

extern char* progname;

static void cpp_arg_scan();
static void parse_file_named();
void parse_file();
static void usage();
static void parse_item();
static void parse_slider();
static void parse_message();
static void parse_field();
static void parse_dialog();
static void parse_text();
static void parse_variable();
static void parse_button();
static void parse_menubutton();
static void parse_choice();
static void parse_list();
static void parse_scrollbar();
static void parse_graph();
static void parse_grid();
static void parse_timer();
static void parse_startup();


/* parse the command line */
void parse_args(  argc, argv, root  )
int argc;
char** argv;
entry *root;
{
  entry *ent, *subent;
  int use_cpp;
  char *cpp_args;

/* no arguments and stdin is a terminal - do self-doc */
if (argc == 0 && isatty(fileno(stdin))) {
      usage();
      exit(0);
}

/* first argument -tty - take input from stdin if it is a tty */
if (argc > 0) {
   if (!strcmp( *argv, "-tty" ) ){
      if( isatty( fileno(stdin))) {
        (void) parse_file(stdin,root);
      }
      argv++; argc--;
   }
}

/* first argument -help - run the help script */
if (argc > 0) {
   if (!strcmp( *argv, "-help" ) ){
        system("xtpanel -file help/help");
        argv++; argc--;
        if (argc == 0) exit(0);
   }
}

/* scan over the argument list looking for the -cpp flag and any
 * C-preprocessor arguments, -I... and -D....
 */
cpp_arg_scan( &argc, &argv, &use_cpp, &cpp_args );


/* process the rest of the command line */
while(argc>0){

   ent = (entry*)0;

   if( !strcmp( *argv, "-quit" ) ){
      ent = new_entry("button",6); 
      subent = new_entry( "name", 4 );
      subent->value = strdupl("QUIT");
      add_child( ent, subent );
      subent = new_entry( "action", 6 );
      subent->value = strdupl("QUIT");
      add_child( ent, subent );
      argv++; argc--;

   }else if( !strcmp(  *argv, "-file" ) ){
      argv++; argc--;
      if( argc >0 ){
         char* filename; 
	 filename = *argv; argv++; argc--;
         parse_file_named( filename, root, use_cpp, cpp_args );
      }else{
	  fprintf(stderr,"expecting filename after \"-file\"\n");
          usage(); exit(-1);
      }

   }else if( !strcmp(  *argv, "-button" ) ){
      ent = new_entry("button", 6 ); argv++; argc--;
      parse_button( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-slider" ) ){
      ent = new_entry("slider", 6 ); argv++; argc--;
      parse_slider( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-menubutton" ) ){
      ent = new_entry("menubutton", 10 ); argv++; argc--;
      parse_menubutton( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-text" ) ){
      ent = new_entry("text", 4 ); argv++; argc--;
      parse_text( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-field" ) ){
      ent = new_entry("field", 5 ); argv++; argc--;
      parse_field( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-dialog" ) ){
      ent = new_entry("dialog", 6 ); argv++; argc--;
      parse_dialog( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-choice" ) ){
      ent = new_entry("choice", 6 ); argv++; argc--;
      parse_choice( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-list" ) ){
      ent = new_entry("list", 4 ); argv++; argc--;
      parse_list( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-message" ) ){
      ent = new_entry("message", 7 ); argv++; argc--;
      parse_message( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-var" ) ){
      ent = new_entry("var", 3 ); argv++; argc--;
      parse_variable( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-toggle" ) ){
      ent = new_entry("toggle", 6 ); argv++; argc--;
      parse_button( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-scrollbar" ) ){
      ent = new_entry("scrollbar", 9 ); argv++; argc--;
      parse_scrollbar( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-graph" ) ){
      ent = new_entry("graph", 5 ); argv++; argc--;
      parse_graph( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-grid" ) ){
      ent = new_entry("grid", 4 ); argv++; argc--;
      parse_grid( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-togglegrid" ) ){
      ent = new_entry("grid", 10 ); argv++; argc--;
      parse_grid( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-timer" ) ){
      ent = new_entry("timer", 5 ); argv++; argc--;
      parse_timer( &argc, &argv, ent );

   }else if( !strcmp(  *argv, "-startup" ) ){
      ent = new_entry("startup", 7 ); argv++; argc--;
      parse_startup( &argc, &argv, ent );

   }else{
      fprintf( stderr," unrecognised argument %s \n",*argv);
      usage();
      exit(-1);
   }

   if( ent != (entry*)0 ) {	
      add_child( root, ent );
   }

}
}


/* scan over the argument list looking for the -cpp flag and any
 * C-preprocessor arguments, -I... and -D....
 * remove all relevant arguments from the argument list
 */

static void cpp_arg_scan( pargc, pargv, use_cpp, cpp_args )
int *pargc;
char*** pargv;
int *use_cpp;
char** cpp_args;
{

int *keep_args,i;
int num_keep=0;
char** locargv= *pargv;
int argc= *pargc;

*cpp_args = malloc(1024);
strcpy( *cpp_args, "");

   keep_args = (int*)malloc( argc * sizeof(int) );

   *use_cpp = 0;

   for( i=0; i<argc; i++ ){
      if( !strcmp( *locargv, "-cpp" ) ){
	*use_cpp = 1;
      }else if( !strncmp( *locargv, "-I", 2 )){
	strcat( *cpp_args, *locargv );
	strcat( *cpp_args, " " );
      }else if( !strncmp( *locargv, "-D", 2 )){
	strcat( *cpp_args, *locargv );
	strcat( *cpp_args, " " );
      }else if( !strncmp( *locargv, "-U", 2 )){
	strcat( *cpp_args, *locargv );
	strcat( *cpp_args, " " );
      }else{
	keep_args[num_keep] = i;
        num_keep++;
      }
      locargv++; 
   }

   /* now compress the argument list */
   for( i=0 ; i<num_keep; i++ ){
      (*pargv)[i] = (*pargv)[keep_args[i]];
   }
   *pargc = num_keep;
	
}

static char *find_file();

static void parse_file_named( filename , root, use_cpp, cpp_args )
char *filename;
entry *root;
int use_cpp;
char *cpp_args;
{
char* name;
FILE* file;
char cpp_command[1024];

   if( (name = find_file( filename )) == NULL ){
      fprintf(stderr,"unable to find file %s\n",filename); exit(-1);
   }else{
       if( use_cpp ){
	  sprintf(cpp_command," %s <%s %s ",CPP_COM,name,cpp_args);
	  file = popen( cpp_command, "r" );
       }else{
          file = fopen( name, "r" );
       }

       if( file == (FILE*)0 ){
          fprintf(stderr,"unable to open file %s\n",name); exit(-1);
       }else{
          parse_file( file, root );
       }

       if( use_cpp ){
          pclose( file );
       }else{
          fclose( file );
       }
   }
 
}

/* parse a file */
#define PARSE_LEN 5000
char parse_buf[PARSE_LEN]; /* PARSE_LEN is the size of chunks to read */

void parse_file( file , root )
FILE* file;
entry* root;
{
   int num;
   string_buf * buffer;
   char *charry;

   buffer = buf_start();

   while( (num = fread( parse_buf, 1, PARSE_LEN, file )) > 0 ){
     buf_cat( buffer, parse_buf, num );
   };
   if( num <0 ){
	fprintf(stderr," error reading input \n");
	exit(-1);
   }

   /* trim off trailing whitespace */
   buf_trim( buffer, " \t\n" );

   charry = buf_fetch( buffer );

   if( strlen(charry) >0 ) xtpanel_scan( charry, root );

   free(charry);

}


/* make an entry for a given tag, the value will be the next argument */
static entry* parse_next(  pargc, pargv , tag )
int *pargc;
char*** pargv;
char* tag;
{
  entry* newent;

  if( *pargc == 0 || **pargv[0] == '-' ){
     /* no more arguments for this object */
     newent = (entry*)0;
  }else{
     /* else get the next argument an make an entry */
     newent = new_entry( tag, strlen(tag) );
     newent->value = strdupl( **pargv );
     (*pargc)--; (*pargv)++;
  }
  return newent;
}

/* parse a button, it has a label, action, name, value */
static void parse_button( pargc, pargv, ent )
int *pargc;
char*** pargv;
entry *ent;
{

entry *newent;
 
  if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

}

/* parse a variable, it has a name, action and a value */
static void parse_variable( pargc, pargv, ent )
int *pargc;
char*** pargv;
entry *ent;
{

entry *newent;
 
  if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

}

/* parse a text item: value, action, name, width, height, editType */
static void parse_text( pargc, pargv, ent )
int *pargc;
char*** pargv;
entry *ent;
{
entry *newent;

  if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "width" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "height" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "editType" )) == (entry*) 0 ) return;
  else add_child( ent, newent );
}

/* parse a field item: action, name, value */
static void parse_field( pargc, pargv, ent )
int *pargc;
char*** pargv;
entry *ent;
{
entry *newent;

  if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
  else add_child( ent, newent);

}

/* parse a dialog item: label, action, name, value */
static void parse_dialog( pargc, pargv, ent )
int *pargc;
char*** pargv;
entry *ent;
{
entry *newent;

  if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
  else add_child( ent, newent);

}

/* parse a mesage item: value, name */
static void parse_message( pargc, pargv, ent )
int *pargc;
char*** pargv;
entry *ent;
{
entry *newent;

  if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

  if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
  else add_child( ent, newent );

}

/* parse a slider item, it just has lots of arguments*/
static void parse_slider( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "min" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "max" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "format" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "width" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "height" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
}

/* parse a scrollbar item, like slider but no label */
static void parse_scrollbar( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "min" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "max" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "format" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "width" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "height" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
}

/* parse a graph item, it is even worse than a slider */
static void parse_graph( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "nsamp" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "min" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "max" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "format" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "width" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "height" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
}

/* parse a grid item, it is even worse than a graph */
static void parse_grid( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "nx" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "ny" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "xmin" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "ymin" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "dx" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "dy" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "format" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv,"formtype")) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "width" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "height" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
}

/* parse a timer item */
static void parse_timer( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "interval" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "running" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
}

/* parse a startup item */
static void parse_startup( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );

    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
}

/* generic item, used in choice, list, and menubutton */
static void parse_item( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) 
      { fprintf( stderr," expecting label for item \n"); exit(-1); }
    add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) 
      { fprintf( stderr," expecting value for item \n");exit(-1); }
    add_child( ent, newent );
    
}

static void parse_menubutton( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( *pargc >0 ){
	int numitem;
	numitem = atoi( **pargv ); (*pargc)--; (*pargv)++;
	while( numitem >0 ){
	    entry* newitem;
	    newitem = new_entry("item", 4);
	    parse_item(  pargc, pargv, newitem );
	    add_child( ent, newitem );
	    numitem--;
	}
    }else{
	(*pargv)--;
	fprintf( stderr, " unexpected end of arguments after %s\n", 
		(**pargv) );
	fprintf( stderr, " expecting number of items in menu \n");
	usage();
	exit(-1);
    }
}


static void parse_choice( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( *pargc >0 ){
	int numitem;
	numitem = atoi( **pargv ); (*pargc)--; (*pargv)++;
	while( numitem >0 ){
	    entry* newitem;
	    newitem = new_entry("item", 4);
	    parse_item(  pargc, pargv, newitem );
	    add_child( ent, newitem );
	    numitem--;
	}
    }else{
	(*pargv)--;
	fprintf( stderr, " unexpected end of arguments after %s\n", 
		(**pargv) );
	fprintf( stderr, " expecting number of items in choice \n");
	usage();
	exit(-1);
    }
}


static void parse_list( pargc, pargv, ent )
     int *pargc;
     char*** pargv;
     entry *ent;
{
    entry *newent;
    
    if( (newent=parse_next( pargc, pargv, "label" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "action" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "name" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( (newent=parse_next( pargc, pargv, "value" )) == (entry*) 0 ) return;
    else add_child( ent, newent );
    
    if( *pargc >0 ){
	int numitem;
	numitem = atoi( **pargv ); (*pargc)--; (*pargv)++;
	while( numitem >0 ){
	    entry* newitem;
	    newitem = new_entry("item", 4);
	    parse_item(  pargc, pargv, newitem );
	    add_child( ent, newitem );
	    numitem--;
	}
    }else{
	(*pargv)--;
	fprintf( stderr, " unexpected end of arguments after %s\n", 
		(**pargv) );
	fprintf( stderr, " expecting number of items in list \n");
	usage();
	exit(-1);
    }
}


static void usage()
{
    fprintf( stderr,"USAGE\n-----\n%s [ < config_file ] \n",progname);
    fprintf( stderr,"\t[ -file \t config_file2 ] \n");
    fprintf( stderr,"\t[ -help ] \n");
    fprintf( stderr,"\t[ -tty ] \n");
    fprintf( stderr,"\t[ -cpp \t cpp_args ] \n");
    fprintf( stderr,"\t[ -quit ] \n");
    fprintf( stderr,"\t[ -message \t value name ] \n");
    fprintf( stderr,"\t[ -text \t value action name width height");
    fprintf( stderr," editType ]\n");
    fprintf( stderr,"\t[ -button \t label action name value ] \n");
    fprintf( stderr,"\t[ -toggle \t label action name value ] \n");
    fprintf( stderr,"\t[ -field \t action name value ]\n");
    fprintf( stderr,"\t[ -dialog \t label action name value ]\n");
    fprintf( stderr,"\t[ -slider \t label action name value min max");
    fprintf( stderr," format width height ]\n");
    fprintf( stderr,"\t[ -scrollbar \t action name value min max");
    fprintf( stderr," format width height ]\n");
    fprintf( stderr,"\t[ -choice \t label action name value numchoice");
    fprintf( stderr," label value ... ]\n");
    fprintf( stderr,"\t[ -list \t label action name value numchoice");
    fprintf( stderr," label value ... ]\n");
    fprintf( stderr,"\t[ -menubutton \t label action name value numchoice");
    fprintf( stderr," label value ... ]\n");
    fprintf( stderr,"\t[ -graph \t nsamp action name value min max");
    fprintf( stderr," format width height ]\n");
    fprintf( stderr,"\t[ -grid \t nx ny action name value xmin ymin");
    fprintf( stderr," dx dy format width height ]\n");
    fprintf( stderr,"\t[ -togglegrid \t nx ny action name value xmin ymin");
    fprintf( stderr," dx dy format formtype width height ]\n");
    fprintf( stderr,"\t[ -var \t\t name value action ]\n");
    fprintf( stderr,"\t[ -startup \t\t action name value ]\n");
    fprintf( stderr,"\t[ -timer \t\t action interval name value running ]\n");
    fprintf( stderr,"\n\n\n");
}


static char	*get_file_path();
static char *find_file( name )
     char    *name;
{
    static char     file_name[ MAXPATHLEN ];
    struct stat     s;
    int             path_index;
    char            *path;
    
    /* name begins with / */
    if( name[0] == '/' )
      return name;
    
    /* search path */
    path_index = 0;
    while( ( path = get_file_path( path_index ) ) != NULL )
      {
	  strcpy( file_name, path );
	  strcat( file_name, "/" );
	  strcat( file_name, name );
	  if( stat( file_name, &s ) != -1 ) return file_name;
	  path_index++;
      }
    
    return name;
}

#ifndef SYS_XTPANELDIR
#define SYS_XTPANELDIR "/usr/local/lib/xtpanel"
#endif

static char* get_file_path( index )
     int     index;
{
    static char path[ MAXPATHLEN ];
    char    *file_path = NULL;
    char    *p;
    char    *q;
    
    /*  build the path */
    if( ( file_path = getenv( "XTPANEL_PATH" ) ) != NULL ){
	strcpy( path, file_path );
	strcat( path, ":"  );
    }else{
	strcpy( path, "./:" );
    }
    strcat( path, getenv("HOME") );
    strcat( path, "/.xtpanel:"  );
    strcat( path, SYS_XTPANELDIR );
    
    p = path;
    
    while( index-- >= 0 )
      {
	  q = p;
	  p = strchr( p, ':' );
	  if( p == NULL )
	    break;
	  else
	    p++;
      }
    
    if( index >= 0 )
      return NULL;
    
    if( p != NULL ) p--;
    if( p != NULL && *p == ':' ) *p = '\0';
    
    return q;
    
}


/* parse an itemlist into separate items */
void parse_itemlist( root, separator )
     entry *root;
     char **separator;
{
    entry *curr;
    entry *itement, *leaf;
    char* itemlist;
    char* item;
    
    
    /* for all itemlist entries in "root" , convert them into items */
    for (curr = get_next(root,"itemlist",(entry*) 0); curr != ((entry*) 0); 
	 curr = get_next(root,"itemlist",curr) ){
	
	itemlist = get_value(  curr, "list", ""  );
	*separator = get_value( curr, "separator", " " );
	
	for( item=strtok(itemlist,*separator); 
	    item != (char*)0 ;
	    item =strtok( (char*)0, *separator ) ){
	    
	    /* make a new item entry */
	    itement = new_entry( "item", 4 );
	    add_child( root, itement );
	    
	    /* make a new label entry */
	    leaf = new_entry( "label", 5 );
	    leaf->value = strdupl( item );
	    add_child( itement, leaf );
	    
	}
    }
    
}
