#!/usr/local/bin/perl
#  categories.pl - functions to implement categories
#
#  Written by Curtis Olson.  Started September 29, 1994.
#
#  Copyright (C) 1994  Curtis L. Olson  - curt@sledge.mn.org
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

# $Id: categories.pl,v 1.4 1994/11/07 23:13:52 clolson Exp $
# (Log is kept at end of this file)


# Global variables

# %CATS - an associative array of categories
# @CATKEYS - a sorted list of category keys (for traversing the cat list)
# $sorted_catkeys - specifies whether the list in @CATKEYS is valid


# initialize the categories list
sub init_cats {
    # out: result

    $sorted_catkeys = 0;

    return "ok";
}


# set @CATKEYS = sorted list of transaction keys
sub sort_catkeys {
    $sorted_catkeys = 1;

    print DEBUG "sort_catkeys()\n" if $debug;
    @CATKEYS = sort(keys %CATS);
}


# edit a category in the category list
sub edit_cat {
    # in: category
    # out: category

    local($cat) = @_;

    $sorted_catkeys = 0;
    ($key, $desc, $tax) = split(/:/, $cat);

    $CATS{$key} = "$desc:$tax";

    print DEBUG "cat-edit:  $cat\n" if $debug;

    return "$cat";
}


# insert a category into the category list
sub insert_cat {
    # in: category
    # out: category

    local($cat) = @_;

    $sorted_catkeys = 0;
    ($key, $desc, $tax) = split(/:/, $cat);

    $CATS{$key} = "$desc:$tax";

    print DEBUG "cat-insert:  $cat\n" if $debug;

    return "$cat";
}


# delete a category from the category list
sub delete_cat {
    # in: category

    local($cat) = @_;

    $sorted_catkeys = 0;
    ($key, $desc, $tax) = split(/:/, $cat);

    delete $CATS{$key};

    print DEBUG "cat-deleted:  $cat\n" if $debug;

    return "$cat";
}


# attempt to find a category matching the key
# incomplete keys are allowed
sub find_cat {
    # in: key
    # out: category

    local($key) = @_;

    if ($sorted_catkeys == 0) {
	&sort_catkeys();
    }

    if ( $key ne "" ) {
	# escape any '[' and ']' in $key
	$key =~ s/\[/\\\[/g;
	$key =~ s/\]/\\\]/g;
	print DEBUG "$key\n" if $debug;
        foreach $catkey (@CATKEYS) {
	    if ( $catkey =~ m/^$key/i ) {
		print DEBUG "found $catkey\n" if $debug;
	        return $catkey;
	    }
        }
    }

    print DEBUG "found none\n" if $debug;
    return "none";
}


# returns the entire category list in one big chunk.
sub all_cats {
    # out: category list

    local($key);

    $| = 0;				# turn off buffer flushing

    if ($sorted_catkeys == 0) {
	&sort_catkeys();
    }

    foreach $key (@CATKEYS) {
	print ("$key:$CATS{$key}\n");
    }

    $| = 1;				# turn buffer flushing back on

    return "none";
}


# load a categories list
sub load_cats {
    # in: file base name
    # out: result

    local($file) = @_;

    $sorted_catkeys = 0;

    dbmclose(CATS);
    dbmopen(%CATS, $file, 0666);

    return "ok";
}


# load a categories list
sub import_cats {
    # in: file base name
    # out: result

    local($file) = @_;

    $sorted_catkeys = 0;

    open(LOADCATS, "<$file") || return "error";

    while ( <LOADCATS> ) {
	chop;
	&insert_cat($_);
    }

    close(LOADCATS);

    return "ok";
}


# save the category list
sub save_cats {
    # in: file base name
    # out: result

    local($file) = @_;

    if ($sorted_catkeys == 0) {
	&sort_catkeys();
    }

    open(SAVECATS, ">$file.cat") || return "error";

    foreach $key (@CATKEYS) {
        print( SAVECATS "$key:$CATS{$key}\n" );
    }

    close(SAVECATS);

    return "ok";
}


&init_cats();


# ----------------------------------------------------------------------------
# $Log: categories.pl,v $
# Revision 1.4  1994/11/07  23:13:52  clolson
# Working on interactive category viewing/editing
#
# Revision 1.3  1994/11/01  22:38:58  clolson
# Added log.pl --> implemented logging.
# Worked on install script.
#
# Revision 1.2  1994/10/31  23:14:38  clolson
# Started tracking towards cbbsh (the text only version).  Chris Browne has
# some good ideas, so I am trying to incorperate them, and make my stuff
# compatible with his.
#
# Revision 1.1  1994/10/31  15:18:51  curt
# These files resulted from the splitting of engine.pl
#
