#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Toggle.h>

#include <stdlib.h>

#include <except.h>
#include <mymalloc.h>
#include <myxlib.h>

#include "analyze.h"
#include "games.h"
#include "gospel.h"
#include "utils.h"
#include "xgospel.h"

#define XTPOINTER_TO_INT(x) ((int)(long) (x))
#define INT_TO_XTPOINTER(x) ((XtPointer) (long) (x))

struct _Analyze {
    struct _Analyze *Next, *Previous;
    char           *Name;
    char           *SgfFile, *PsFile;
    Gamelog        *Log;
    int             Level, Size, BaseMove;
    Widget          Model, TopWidget, BoardWidget, ScrollWidget, SgfOverwrite;
    Widget          SgfFromStart, PsOverwrite, PsFromStart;
    Widget          ErrorBeep, ErrorRaise, Copy;
};

static Analyze AnalyzeBase = { &AnalyzeBase, &AnalyzeBase };

static void CallDestroyAnalyze(Widget w,
                               XtPointer clientdata, XtPointer calldata)
{
    Analyze *Ana;

    Ana = (Analyze *) clientdata;

    if (Ana->Model) {
        XtRemoveCallback(Ana->Model, XtNdestroyCallback, CallCleanWidget,
                         (XtPointer) &Ana->Model);
        XtRemoveCallback(Ana->Model, XtNdestroyCallback, CallDestroy,
                         (XtPointer) Ana->Copy);
    }
    Ana->Previous->Next = Ana->Next;
    Ana->Next->Previous = Ana->Previous;
    FreeGamelog(Ana->Log);
    myfree(Ana->SgfFile);
    myfree(Ana->PsFile);
    myfree(Ana->Name);
    myfree(Ana);
}

static void ResetAnalyze(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Analyze *Ana;
    Gamelog *log;
    Widget   bw;
    int      i, x, y, node, nodes;

    Ana   = (Analyze *) clientdata;
    log   = Ana->Log;
    bw    = Ana->BoardWidget;
    node  = NodeNumber(log)-1;
    nodes = NumberNodes(log)-2;

    for (i=node; i<nodes; i++) DownGamelog(log);
    for (i=nodes; i>0; i--)    DeleteNode(log);

    OpenBoard(bw);
    SetBoard(bw, GamelogToBoard(Ana->Log));
    CloseBoard(bw);
    
    i = XTPOINTER_TO_INT(FindComment(log, &LastMoveFun));
    if (i<0) x = y = -1;
    else {
        y = i / log->SizeX;
        x = i % log->SizeX;
    }
    LastMove(bw, x, y);
    
    XawScrollbarSetThumb(Ana->ScrollWidget, 0.0, 1.0);
}

static void UndoAnalyze(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Analyze   *Ana;
    Gamelog   *log;
    Widget     board;
    int        i, x, y, node, nodes;
    float      Step;
    
    Ana   = (Analyze *) clientdata;
    log   = Ana->Log;
    node  = NodeNumber(log)-1;
    nodes = NumberNodes(log)-2;

    if   (nodes == 0) {
        IfBell(Ana->ErrorBeep);
        IfRaise(Ana->ErrorRaise, Ana->ErrorRaise);
    } else {
        for (i=node; i<nodes; i++) DownGamelog(log);
        DeleteNode(log);
        board = Ana->BoardWidget;
        OpenBoard(board);
        SetBoard(board, GamelogToBoard(log));
        CloseBoard(board);

        i = XTPOINTER_TO_INT(FindComment(log, &LastMoveFun));
        if (i<0) x = y = -1;
        else {
            y = i / log->SizeX;
            x = i % log->SizeX;
        }
        LastMove(Ana->BoardWidget, x, y);

        Step = 1.0 / nodes;
        XawScrollbarSetThumb(Ana->ScrollWidget, (nodes-1)*Step, Step);
    }
}

static void AnalyzeAnalyze(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Analyze *Ana;
    Gamelog *log;
    int      node;

    Ana = (Analyze *) clientdata;
    log = Ana->Log;
    node  = NodeNumber(log)-1;
    OpenAnalyze(Ana->BoardWidget, Ana->Name,
                Ana->Level+1, Ana->Size, Ana->BaseMove+node);
}

static Exception NotEmpty = { "Position is not empty" };

static void AnalyzeButton(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Analyze   *Ana;
    Gamelog   *log;
    int        i, node, nodes, move;
    float      Step;
    GoButton  *Button;
    char     **Board;
    StoneList  Stone;

    Button = (GoButton *) calldata;
    Ana = (Analyze *)     clientdata;
    log = Ana->Log;
    node  = NodeNumber(log)-1;
    nodes = NumberNodes(log)-2;
    for (i=node; i<nodes; i++) DownGamelog(log);

    w   = Ana->BoardWidget;
    if (Button->params == 1) {
        OpenBoard(w);
        WITH_HANDLING {
            move = Ana->BaseMove+node;
            switch(atoi(Button->str[0])) {
              case 0:
                DoMove(log, Button->x, Button->y, move & 1 ? White : Black);
                LastMove(w, Button->x, Button->y);
                i = Button->y * log->SizeX + Button->x;
                AddComment(log, &LastMoveFun, INT_TO_XTPOINTER(i));
                break;
              case 1:
		Board = GamelogToBoard(log);
		if (Board[Button->y][Button->x] == Empty) Raise(NotEmpty);
		else {
		    Stone.x     = Button->x;
		    Stone.y     = Button->y;
		    Stone.Color = Empty;
		    Stone.Next  = NULL;
		    SetStones(log, &Stone);
		}
                LastMove(w, -1, -1);
                AddComment(log, &LastMoveFun, INT_TO_XTPOINTER(-1));
                break;
              case 2:
                DoMove(log, Button->x, Button->y, move & 1 ? Black : White);
                LastMove(w, Button->x, Button->y);
                i = Button->y * log->SizeX + Button->x;
                AddComment(log, &LastMoveFun, INT_TO_XTPOINTER(i));
                break;
              default:
                Warning("Invalid parameter '%s' to Button action\n",
                        Button->str[0]);
                break;
            }
        } ON_EXCEPTION {
            IfBell(Ana->ErrorBeep);
            IfRaise(Ana->ErrorRaise, Ana->ErrorRaise);
/*          Warning(ExceptionName()); */
        } END_HANDLING;
        SetBoard(w, GamelogToBoard(log));
        CloseBoard(w);

        Step = 1.0 / (NumberNodes(log)-1);
        node = NodeNumber(log)-1;
        XawScrollbarSetThumb(Ana->ScrollWidget, node*Step, Step);
    } else Warning("Invalid parameters to Button action\n");
}

static void ScrollAnalyze(Widget scrollbar, XtPointer client_data,
                          XtPointer pos)
{
    Analyze   *Ana;
    Gamelog   *log;
    int        i, x, y, Pos, Pixels, node, Nodes, Target;
    float      Step;
    Dimension  Length;
    Widget     w;

    Ana = (Analyze *) client_data;
    w   = Ana->BoardWidget;
    log = Ana->Log;
    Pos = XTPOINTER_TO_INT(pos);

    XtVaGetValues(scrollbar, XtNlength, (XtArgVal) &Length, NULL);

    node   = NodeNumber(log);
    Nodes  = NumberNodes(log)-1;
    if (Nodes > 1) Pixels = Length / (Nodes-1);
    else           Pixels = Length;
    if (Pixels < appdata.ScrollUnit) Pixels = appdata.ScrollUnit;
    if        (Pos > 0) {
        Target = node + 1 +   Pos /Pixels;
        if (Target > Nodes) Target = Nodes;
        for (i= node; i<Target; i++) DownGamelog(log);
    } else if (Pos < 0) {
        Target = node - 1 - (-Pos)/Pixels;
        if (Target < 1) Target = 1;
        for (i= node; i>Target; i--) UpGamelog(log);
    }
    
    OpenBoard(w);
    SetBoard(w, GamelogToBoard(log));
    CloseBoard(w);

    i = XTPOINTER_TO_INT(FindComment(log, &LastMoveFun));
    if (i<0) x = y = -1;
    else {
        y = i / log->SizeX;
        x = i % log->SizeY;
    }
    LastMove(w, x, y);

    Step = 1.0 / (NumberNodes(log)-1);
    XawScrollbarSetThumb(scrollbar, (NodeNumber(log)-1)*Step, Step);
}

static void JumpAnalyze(Widget scrollbar, XtPointer client_data, XtPointer per)
{
    Analyze   *Ana;
    Gamelog   *log;
    int        x, y, i, Pos, Now, nodes;
    float      Percent, Step;
    Widget     w;

    Ana     = (Analyze *) client_data;
    w       = Ana->BoardWidget;
    log     = Ana->Log;
    Percent = * (float *) per;
    nodes   = NumberNodes(log)-1;
    Now     = NodeNumber(log)-1;
    Step    = 1.0 / nodes;
    Pos     = Percent * nodes;
    if      (Pos < 0) Pos = 0;
    else if (Pos >= nodes) Pos = nodes-1;

    if (Pos > Now)
        for (i=Now; i<Pos; i++) DownGamelog(log);
    else if (Pos < Now)
        for (i=Now; i>Pos; i--) UpGamelog(log);

    OpenBoard(w);
    SetBoard(w, GamelogToBoard(log));
    CloseBoard(w);

    i = XTPOINTER_TO_INT(FindComment(log, &LastMoveFun));
    if (i<0) x = y = -1;
    else {
        y = i / log->SizeX;
        x = i % log->SizeX;
    }
    LastMove(w, x, y);

    XawScrollbarSetThumb(scrollbar, Pos*Step, Step);
}

/* Toggle equal time blink */
void ToggleBlink(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Widget board;
    int    timeout;

    board = (Widget) clientdata;
    if (False == (Boolean) XTPOINTER_TO_INT(calldata))
        XtVaSetValues(board, XtNoffTimeout, (XtArgVal) 0, NULL);
    else {
        XtVaGetValues(board, XtNonTimeout,  (XtArgVal) &timeout, NULL);
        XtVaSetValues(board, XtNoffTimeout, (XtArgVal)  timeout, NULL);
    }
}

extern Observe *FindBoardWidget(Widget w);
static void CallCopy(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Widget   board;
    Analyze *Ana, *Here;
    Gamelog *Log;
    char   **Board;
    int      Size, move;
    Observe *observe;

    Ana = (Analyze *) clientdata;
    board = Ana->BoardWidget;
    if (board) {
        FreeGamelog(Ana->Log);
        Size = Ana->Size;
        Ana->Log  = Log = AllocGamelog(Size, Size);
        AddComment(Log, &LastMoveFun, INT_TO_XTPOINTER(Size-1));
        w = Ana->Model;
        if (w) {
            observe = FindBoardWidget(w);
            if (observe) move = ObserveMove(observe);
            else {
                for (Here = AnalyzeBase.Next;
                     Here != &AnalyzeBase;
                     Here = Here->Next)
                    if (Here->BoardWidget == w) {
                        move = Here->BaseMove + NumberNodes(Here->Log)-2;
                        goto found;
                    }
                Raise1(AssertException, "Could not find widget to copy from");
            }
          found:
            Board = AllocBoard(Size, Size);
            WITH_UNWIND {
                GetBoard(w, Board);
                PositionToNode(Log, Board, Empty);
            } ON_UNWIND {
                FreeBoard(Board, Size);
            } END_UNWIND;
        } else SetStones(Log, NULL);
        AddComment(Log, &LastMoveFun, INT_TO_XTPOINTER(-Size-1));

        OpenBoard(board);
        SetBoard(board, GamelogToBoard(Ana->Log));
        CloseBoard(board);
        LastMove(board, -1, -1);
        XawScrollbarSetThumb(Ana->ScrollWidget, 0.0, 1.0);
        Ana->BaseMove = move;
    }
}

/* Analyze should have pos in structure (share with observe ?) -Ton */
static void SaveGame(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Analyze *Ana;
    Gamelog *log;
    Boolean  Overwrite, FromStart;
    int      i, Now, node;

    Ana   = (Analyze *) clientdata;
    log   = Ana->Log;
    Now   = NodeNumber(log)-1;

    w = Ana->SgfFromStart;
    if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &FromStart, NULL);
    else   FromStart = True;

    if (FromStart != False) {
        node = 0;
        if (Now > node) for (i = Now; i > node; i--) UpGamelog(log);
    } else node = Now;

    WITH_UNWIND {
        w = Ana->SgfOverwrite;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &Overwrite, NULL);
        else   Overwrite = False;

        SaveWrite(Ana->SgfFile, Overwrite, Ana->ErrorBeep, Ana->ErrorRaise, 
                  "Sgf save error", WriteSgfFun, (XtPointer) log);
    } ON_UNWIND {
        if (Now > node) for (i = node; i < Now; i++) DownGamelog(log);
    } END_UNWIND;
}

static void SavePs(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Analyze *Ana;
    Gamelog *log;
    Boolean  Overwrite, FromStart;
    int      i, Now, node;

    Ana   = (Analyze *) clientdata;
    log   = Ana->Log;
    Now   = NodeNumber(log)-1;

    w = Ana->PsFromStart;
    if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &FromStart, NULL);
    else   FromStart = True;

    if (FromStart != False) {
        node = 0;
        if (Now > node) for (i = Now; i > node; i--) UpGamelog(log);
    } else node = Now;

    WITH_UNWIND {
        w = Ana->PsOverwrite;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &Overwrite, NULL);
        else   Overwrite = False;

        SaveWrite(Ana->PsFile, Overwrite, Ana->ErrorBeep, Ana->ErrorRaise,
		  "Postscript save error", WritePsFun, (XtPointer) log);
    } ON_UNWIND {
        if (Now > node) for (i = node; i < Now; i++) DownGamelog(log);
    } END_UNWIND;
}

static Analyze *RealOpenAnalyze(Analyze *Ana)
{
    const char *Name;
    char     title[80], icon[80];
    int      Level, Size;
    Boolean  state;
    Widget   Root, quit, Reset, Analyse, board, undo, Scroll, SgfSave, PsSave;
    Widget   SgfFile, SgfOverwrite, SgfFromStart;
    Widget   PsFile, PsOverwrite, PsFromStart;
    Widget   ErrorBeep, ErrorRaise, Blink, Copy;

    Level = Ana->Level;
    Name  = Ana->Name;
    Size  = Ana->Size;

    if (Level > 0) {
        sprintf(title, "%d", Ana->Level+1);
        Ana->SgfFile   = StringToFilename(appdata.AnalyzeFilename,
                                          (int) 'T', SGFEXTENSION,
                                          (int) 't', ".",
                                          (int) 'N', "Analyzing",
                                          (int) 'n', "_",
                                          (int) 'B', "", (int) 'b', "",
                                          (int) 'W', "", (int) 'w', "",
                                          (int) 'U', "", (int) 'V', "",
                                          (int) 'L', title, (int) 'l', "_", 0);
        Ana->PsFile    = StringToFilename(appdata.AnalyzeFilename,
                                          (int) 'T', PSEXTENSION,
                                          (int) 't', ".",
                                          (int) 'N', "Analyzing",
                                          (int) 'n', "_",
                                          (int) 'B', "", (int) 'b', "",
                                          (int) 'W', "", (int) 'w', "",
                                          (int) 'U', "", (int) 'V', "",
                                          (int) 'L', title, (int) 'l', "_", 0);
        sprintf(title, "Analyzing**%d %s", Ana->Level+1, Name);
        sprintf(icon,  "Analyzing**%d %s", Ana->Level+1, Name);
    } else {
        Ana->SgfFile   = StringToFilename(appdata.AnalyzeFilename,
                                          (int) 'T', SGFEXTENSION,
                                          (int) 't', ".",
                                          (int) 'N', "Analyzing",
                                          (int) 'n', "_",
                                          (int) 'B', "", (int) 'b', "",
                                          (int) 'W', "", (int) 'w', "",
                                          (int) 'U', "",
                                          (int) 'V', "",
                                          (int) 'L', "", (int) 'l', "", 0);
        Ana->PsFile    = StringToFilename(appdata.AnalyzeFilename,
                                          (int) 'T', PSEXTENSION,
                                          (int) 't', ".",
                                          (int) 'N', "Analyzing",
                                          (int) 'n', "_",
                                          (int) 'B', "", (int) 'b', "",
                                          (int) 'W', "", (int) 'w', "",
                                          (int) 'U', "",
                                          (int) 'V', "",
                                          (int) 'L', "", (int) 'l', "", 0);
        sprintf(title, "Analyzing %s", Name);
        sprintf(icon,  "Analyzing %s", Name);
    }
                                      

    Root = MyVaCreateManagedWidget("analyzer", toplevel,
                                   XtNtitle,       (XtArgVal) title,
                                   XtNiconName,    (XtArgVal) icon,
                                   XtNboardSize,   (XtArgVal) Size,
                                   NULL);
    quit            = XtNameToWidget(Root, "*quit");
    Reset           = XtNameToWidget(Root, "*reset");
    undo            = XtNameToWidget(Root, "*undo");
    Copy            = XtNameToWidget(Root, "*copy");
    Analyse         = XtNameToWidget(Root, "*analyze");
    Scroll          = XtNameToWidget(Root, "*scroll");
    board           = XtNameToWidget(Root, "*board");

    SgfSave         = XtNameToWidget(Root, "*sgfSave");
    SgfFile         = XtNameToWidget(Root, "*sgfFile");
    SgfOverwrite    = XtNameToWidget(Root, "*sgfOverwrite");
    SgfFromStart    = XtNameToWidget(Root, "*sgfFromStart");

    PsSave          = XtNameToWidget(Root, "*psSave");
    PsFile          = XtNameToWidget(Root, "*psFile");
    PsOverwrite     = XtNameToWidget(Root, "*psOverwrite");
    PsFromStart     = XtNameToWidget(Root, "*psFromStart");

    ErrorBeep       = XtNameToWidget(Root, "*errorBeep");
    ErrorRaise      = XtNameToWidget(Root, "*errorRaise");
    Blink           = XtNameToWidget(Root, "*blink");
    XawScrollbarSetThumb(Scroll, 0.0, 1.0);

    XtAddCallback(Root,    XtNdestroyCallback,
                  CallDestroyAnalyze, (XtPointer) Ana);
    XtAddCallback(quit,    XtNcallback,   CallDestroy,   (XtPointer) Root);
    XtAddCallback(Reset,   XtNcallback,   ResetAnalyze,  (XtPointer) Ana);
    XtAddCallback(undo,    XtNcallback,   UndoAnalyze,   (XtPointer) Ana);
    XtAddCallback(Analyse, XtNcallback,   AnalyzeAnalyze,(XtPointer) Ana);
    XtAddCallback(Scroll,  XtNjumpProc,   JumpAnalyze,   (XtPointer) Ana);
    XtAddCallback(Scroll,  XtNscrollProc, ScrollAnalyze, (XtPointer) Ana); 
    XtAddCallback(board,   XtNbuttonUp,   AnalyzeButton, (XtPointer) Ana);
    if (SgfSave) XtAddCallback(SgfSave, XtNcallback, SaveGame,
                               (XtPointer) Ana);
    if (SgfFile) XtAddCallback(SgfFile, XtNcallback, ChangeSgfFilename,
                               (XtPointer) &Ana->SgfFile);
    if (PsSave)  XtAddCallback(PsSave,  XtNcallback, SavePs,
                               (XtPointer) Ana);
    if (PsFile)  XtAddCallback(PsFile, XtNcallback, ChangePsFilename,
                               (XtPointer) &Ana->PsFile);
    if (Blink && board) {
        XtAddCallback(Blink, XtNcallback, ToggleBlink, (XtPointer) board);
            XtVaGetValues(Blink, XtNstate, (XtArgVal) &state, NULL);
        if (state == False)
            XtVaSetValues(board, XtNoffTimeout, (XtArgVal) 0, NULL);
    }
    if (Copy) XtAddCallback(Copy, XtNcallback, CallCopy, (XtPointer) Ana);

    MyRealizeWidget(Root);

    OpenBoard(board);
    SetBoard(board, GamelogToBoard(Ana->Log));
    CloseBoard(board);
    LastMove(board, -1, -1);

    Ana->TopWidget       = Root;
    Ana->ScrollWidget    = Scroll;
    Ana->BoardWidget     = board;
    Ana->SgfOverwrite    = SgfOverwrite;
    Ana->SgfFromStart    = SgfFromStart;
    Ana->PsOverwrite     = PsOverwrite;
    Ana->PsFromStart     = PsFromStart;
    Ana->ErrorBeep       = ErrorBeep;
    Ana->ErrorRaise      = ErrorRaise;
    Ana->Copy            = Copy;
    
    return Ana;
}

Analyze *OpenAnalyze(Widget w, const char *Name, int Level, int Size, int Move)
{
    Analyze *Ana;
    Gamelog *Log;
    char   **Board;

    Ana = mynew(Analyze);
    WITH_HANDLING {
        Ana->Model    = 0; 
        Ana->Name     = Ana->SgfFile = Ana->PsFile = NULL;
        Ana->Log      = NULL;
        Ana->Next     =  AnalyzeBase.Next;
        Ana->Previous = &AnalyzeBase;
        Ana->Next->Previous = Ana->Previous->Next = Ana;
        Ana->Level    = Level;
        Ana->Size     = Size;
        Ana->BaseMove = Move;
        Ana->Name     = mystrdup(Name);

        Ana->Log  = Log = AllocGamelog(Size, Size);
        AddComment(Log, &LastMoveFun, INT_TO_XTPOINTER(Size-1));
        if (w) {
            Board = AllocBoard(Size, Size);
            WITH_UNWIND {
                GetBoard(w, Board);
                PositionToNode(Log, Board, Empty);
            } ON_UNWIND {
                FreeBoard(Board, Size);
            } END_UNWIND;
        } else SetStones(Log, NULL);
        AddComment(Log, &LastMoveFun, INT_TO_XTPOINTER(-Size-1));
        
        RealOpenAnalyze(Ana);
        if (w) {
            XtAddCallback(w, XtNdestroyCallback, CallCleanWidget,
                          (XtPointer) &Ana->Model);
            XtAddCallback(w, XtNdestroyCallback, CallDestroy,
                          (XtPointer) Ana->Copy);
            Ana->Model = w;
        } else XtDestroyWidget(Ana->Copy);
    } ON_EXCEPTION {
        CallDestroyAnalyze(0, (XtPointer) Ana, 0);
        ReRaise();
    } END_HANDLING;

    return Ana;
}

Analyze *AnalyzeBoard(char **Board, const char *Name,
                      int Level, int Size, int Move)
{
    Analyze  *Ana;
    Gamelog  *Log;

    Ana = mynew(Analyze);
    WITH_HANDLING {
        Ana->Name     = Ana->SgfFile = Ana->PsFile = NULL;
        Ana->Log      = NULL;
        Ana->Model    = 0;
        Ana->Name     = mystrdup(Name);
        Ana->Next     =  AnalyzeBase.Next;
        Ana->Previous = &AnalyzeBase;
        Ana->Next->Previous = Ana->Previous->Next = Ana;
        Ana->Level    = Level+1;
        Ana->Size     = Size;
        Ana->BaseMove = Move;
        Ana->Log = Log = AllocGamelog(Size, Size);
        AddComment(Log, &LastMoveFun, INT_TO_XTPOINTER(Size-1));

        PositionToNode(Log, Board, Empty);
        AddComment(Log, &LastMoveFun, INT_TO_XTPOINTER(-Size-1));

        RealOpenAnalyze(Ana);
        XtDestroyWidget(Ana->Copy);
/*      Ana->Copy = 0; */
    } ON_EXCEPTION {
        CallDestroyAnalyze(0, (XtPointer) Ana, 0);
        ReRaise();
    } END_HANDLING;
    return Ana;
}
