#include <string.h>
#if !STDC_HEADERS && HAVE_MEMORY_H
# include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#include <stddef.h>

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Scrollbar.h>

#include <except.h>
#include <mymalloc.h>
#include <myxlib.h>

#include "analyze.h"
#include "connect.h"
#include "observe.h"
#include "games.h"
#include "gospel.h"
#include "tell.h"
#include "utils.h"
#include "xgospel.h"

#define XTPOINTER_TO_INT(x) ((int)(long) (x))
#define INT_TO_XTPOINTER(x) ((XtPointer) (long) (x))

#ifdef    HAVE_NO_MEMCHR_PROTO
extern void *memchr(const void *s, int c, size_t n);
#endif /* HAVE_NO_MEMCHR_PROTO */

struct _Observe {
    struct _Observe *Next;
    Game	   *Game;
    int             Node;
    int             ReplayRate, ReplayCount;
    char           *SgfFile, *PsFile, *KibitzFile;
    Widget          TopWidget,  BoardWidget, InfoWidget, InputWidget;
    Widget          TimeWidget, Pass, Done, CaptureWidget, ScrollWidget;
    Widget          KomiWidget, HandicapWidget, MoveWidget, Replay, Title;
    Widget          MoveBeepWidget, MoveRaiseWidget, ErrorBeep, ErrorRaise;
    Widget          KibitzBeep, KibitzRaise;
    Widget          SgfOverwrite, KibitzOverwrite, PsOverwrite;
    Widget          SgfFromStart, PsFromStart;
};

static void SendKibitz(), ChangeTitle();

static XtActionsRec actionTable[] = {
    { "kibitz", SendKibitz },
    { "changetitle", ChangeTitle },
};

static void CallDestroyObserve(Widget w,
                               XtPointer clientdata, XtPointer calldata);
static void ButtonUp(Widget w, XtPointer clientdata, XtPointer calldata);

int ObserveMove(const Observe *observe)
{
    Game    *game;
    Gamelog *log;
    int      i, nodes, Now, Pos;

    game    = observe->Game;
    log     = GameGamelog(game);
    if (!log) Raise1(AssertException, "Saving observation without a game !?");
    nodes   = NumberNodes(log);
    Now     = NodeNumber(log);
    Pos     = observe->Node;
    
    if      (Pos < 1)      Pos = 1;
    else if (Pos >= nodes) Pos = nodes-1;
    
    if      (Pos > Now) for (i=Now; i<Pos; i++) DownGamelog(log);
    else if (Pos < Now) for (i=Now; i>Pos; i--) UpGamelog(log);
    return XTPOINTER_TO_INT(FindComment(log, &MoveFun));
}

static void ToggleReplay(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Observe *observe;
    Game    *game;
    Gamelog *log;
    int      nodes;
    
    observe = (Observe *) clientdata;
    if (False == (Boolean) XTPOINTER_TO_INT(calldata)) observe->ReplayRate = 0;
    else {
        observe->ReplayRate  = appdata.ReplayTimeout;
        game  = observe->Game;
        log   = GameGamelog(game);
        nodes = NumberNodes(log)-1;
        if (observe->Node == nodes) {
            GotoObserve(observe, 1);
            observe->ReplayCount = 0;
        } else observe->ReplayCount = observe->ReplayRate-1;
    }
}

static void AnalyzeGame(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Observe *observe;
    Game    *game;
    
    observe = (Observe *) clientdata;
    game    = observe->Game;
    OpenAnalyze(observe->BoardWidget, GameDescription(game),
                0, GameXSize(game), ObserveMove(observe));
}

static void DupGame(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Observe *observe, *newobserve;
    
    observe = (Observe *) clientdata;
    newobserve = OpenObserve(observe->Game);
    GotoObserve(newobserve, observe->Node);
}

static void SaveGame(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Observe *observe;
    Game    *game;
    Gamelog *log;
    Boolean  Overwrite, FromStart;
    int      i, Now, node;

    observe = (Observe *) clientdata;
    game    = observe->Game;
    log     = GameGamelog(game);
    if (log) {
        Now     = NodeNumber(log);

        w = observe->SgfFromStart;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &FromStart, NULL);
        else   FromStart = True;

        if (FromStart != False) node = 1;
        else                    node = observe->Node;

        if      (Now < node) for (i = Now; i < node; i++) DownGamelog(log);
        else if (Now > node) for (i = Now; i > node; i--) UpGamelog(log);

        w = observe->SgfOverwrite;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &Overwrite, NULL);
        else   Overwrite = False;
        
        SaveWrite(observe->SgfFile, Overwrite,
                  observe->ErrorBeep, observe->ErrorRaise,
                  "Sgf save error", WriteSgfFun, (XtPointer) log);
    } else Raise1(AssertException, "Saving observation without a game !?");
}

static void SavePs(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Observe *observe;
    Game    *game;
    Gamelog *log;
    Boolean  Overwrite, FromStart;
    int      i, Now, node;

    observe = (Observe *) clientdata;
    game    = observe->Game;
    log     = GameGamelog(game);
    if (log) {
        Now     = NodeNumber(log);

        w = observe->PsFromStart;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &FromStart, NULL);
        else   FromStart = True;

        if (FromStart != False) node = 1;
        else                    node = observe->Node;

        if      (Now < node) for (i = Now; i < node; i++) DownGamelog(log);
        else if (Now > node) for (i = Now; i > node; i--) UpGamelog(log);

        w = observe->PsOverwrite;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &Overwrite, NULL);
        else   Overwrite = False;
        
        SaveWrite(observe->PsFile, Overwrite,
                  observe->ErrorBeep, observe->ErrorRaise,
                  "Postscript save error", WritePsFun, (XtPointer) log);
    } else Raise1(AssertException, "Saving observation without a game !?");
}

static void SaveKibitz(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Widget     kibitz;
    Observe   *observe;
    Boolean    Overwrite;

    observe = (Observe *) clientdata;
    kibitz = observe->InfoWidget;
    if (kibitz) {
        w = observe->KibitzOverwrite;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &Overwrite, NULL);
        else   Overwrite = False;

        SaveWrite(observe->KibitzFile, Overwrite,
                  observe->ErrorBeep, observe->ErrorRaise,
                  "Kibitz save error", SaveTextFun, (XtPointer) kibitz);
    } else {
        IfBell(observe->ErrorBeep);
        IfRaise(observe->ErrorRaise, observe->ErrorRaise);
        PopMessage("Kibitz save error", "Kibitzes were ignored so there is "
                   "nothing to be saved");
    }
}

void GotoObserve(Observe *observe, int Pos)
{
    Game    *game;
    Gamelog *log;
    int      i, x, y, Now, nodes, move;
    float    Step;
    Widget   w;
    char     Text[10];

    game    = observe->Game;
    log     = GameGamelog(game);
    if (log) {
        nodes   = NumberNodes(log);
        Now     = NodeNumber(log);
        Step    = 1.0 / (nodes-1);
        w = observe->BoardWidget;

        if      (Pos < 1)      Pos = 1;
        else if (Pos >= nodes) Pos = nodes-1;

        if      (Pos > Now) for (i=Now; i<Pos; i++) DownGamelog(log);
        else if (Pos < Now) for (i=Now; i>Pos; i--) UpGamelog(log);

        if (Pos != observe->Node) {
            OpenBoard(w);
            SetBoard(w, GamelogToBoard(log));
            CloseBoard(w);
        }
        /* These are outside the previous if so that a non accepted move's
           blinking cross gets reset */
        XYFromLast(game, &x, &y, FindComment(log, &LastMoveFun));
        LastMove(w, x, y);

        observe->Node = Pos;

        if (observe->ScrollWidget)
            XawScrollbarSetThumb(observe->ScrollWidget, (Pos-1)*Step, Step);

        if (observe->MoveWidget) {
            move = XTPOINTER_TO_INT(FindComment(log, &MoveFun));
            sprintf(Text, "%3d/%-3d", move, GameMove(game));
            XtVaSetValues(observe->MoveWidget,
                          XtNlabel, (XtArgVal) Text, NULL);
        }
    } else Raise1(AssertException, "Scrolling in a window without game !?");
}

static void ScrollObserve(Widget scrollbar, XtPointer client_data,
                          XtPointer pos)
{
    Observe   *observe;
    int        Pos, Pixels;
    Dimension  Length;

    observe = (Observe *) client_data;
    Pos = XTPOINTER_TO_INT(pos);

    XtVaGetValues(scrollbar, XtNlength, (XtArgVal) &Length, NULL);

    Pixels = Length / (NumberNodes(GameGamelog(observe->Game))-1);
    if (Pixels < appdata.ScrollUnit) Pixels = appdata.ScrollUnit;
    if   (Pos < 0) Pos = Pos/Pixels-1;
    else           Pos = Pos/Pixels+1;
    GotoObserve(observe, observe->Node+Pos);
}

static void JumpObserve(Widget scrollbar, XtPointer client_data, XtPointer per)
{
    Observe *observe;
    int      Pos;

    observe = (Observe *) client_data;
    Pos     = * (float *) per * (NumberNodes(GameGamelog(observe->Game))-1);

    GotoObserve(observe, Pos+1);
}

static void RefreshGame(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Observe *observe;
    int      Id;

    observe = (Observe *) clientdata;
    Id = GameServerId(observe->Game);
    if (Id >= 0)
        if (ScoringP(observe->Game)) SendCommand(NULL, NULL, "status %d", Id);
        else SendCommand(NULL, NULL, "moves %d", Id);
    else if (observe->InfoWidget)
        BatchAddText(observe->InfoWidget, ".....            "
                     "Refresh not sent, game is gone\n");
    else Output("Refresh not sent, game is gone\n");
}

static void SendMove(Widget w, XtPointer clientdata, const char *Text)
{
    Observe *observe;
    Game    *game;
    int      Id;

    observe = (Observe *) clientdata;
    game    = observe->Game;

    Id = GameServerId(game);
    if (Id >= 0) MoveCommand(NULL, Text);
    else if (observe->InfoWidget)
        BatchAddText(observe->InfoWidget, ".....            "
                     "%s not sent, game is gone\n", Text);
    else Outputf("%s not sent, game is gone\n", Text);
}

static void SendUndo(Widget w, XtPointer clientdata, XtPointer calldata)
{
    SendMove(w, clientdata, "undo");
}

static void SendPass(Widget w, XtPointer clientdata, XtPointer calldata)
{
    SendMove(w, clientdata, "pass");
}

static void SendDone(Widget w, XtPointer clientdata, XtPointer calldata)
{
    SendMove(w, clientdata, "done");
}

static void CallResume(Widget w, XtPointer clientdata, XtPointer calldata)
{
    const Game *game;

    game = (Game *) clientdata;
    SendCommand(NULL, NULL, "load %s-%s",
                GameWhiteName(game), GameBlackName(game));
}

static void CallObservers(Widget w, XtPointer clientdata, XtPointer calldata)
{
    const Game    *game;
    const Observe *observe;
    int            Id;

    observe = (const Observe *) clientdata;
    game = observe->Game;
    Id = GameServerId(game);
    if (Id < 0) {
        if (observe->InfoWidget)
            BatchAddText(observe->InfoWidget, ".....            "
                         "Observers not sent, game is gone\n");
    } else SendCommand(NULL, (XtPointer) 2, "all %d", Id);
}

static void SetObserveDescription(Observe *observe, const Game *game)
{
    char          *Name;

    SetGameHeaders(observe->TopWidget, game);

    Name = GameName(appdata.SgfFilename, SGFEXTENSION, game);
    myfree(observe->SgfFile);
    observe->SgfFile = Name;

    Name = GameName(appdata.PsFilename, PSEXTENSION, game);
    myfree(observe->PsFile);
    observe->PsFile = Name;

    Name = GameName(appdata.KibitzFilename, "kibitz", game);
    myfree(observe->KibitzFile);
    observe->KibitzFile = Name;
}

void SetObserveDescriptions(Observe *observers, const Game *game)
{
    Observe *observe;

    for (observe = observers; observe; observe = observe->Next)
        SetObserveDescription(observe, game);
}

Observe *OpenObserve(Game *game)
{
    Observe    *observe;
    int         sizex, sizey, Playing, Reviewing, move;
    char       *title, HandicapString[10], MoveText[10];
    const char *BlackName, *WhiteName, *gameTitle;
    Boolean     state;
    Widget      Root, Collect, Info, Com, Input, quit, MoveBeep, MoveRaise;
    Widget      ErrorBeep, ErrorRaise, KibitzBeep, KibitzRaise;
    Widget      Refresh, Captures, analyze, Tim, undo, Pass, Scroll, Dup;
    Widget      Komi, Handicap, Move, SgfSave, PsSave, KibitzSave;
    Widget      SgfOverwrite, PsOverwrite, KibitzOverwrite;
    Widget      SgfFromStart, SgfFile, PsFromStart, PsFile, KibitzFile;
    Widget      StatsWhite, StatsBlack;
    Widget      Blink, done, TalkBlack, TalkWhite, resume;
    Widget      adjourn, declineAdjourn, Observers, replay, Title;
  
    if (DebugFun) {
        printf("OpenObserve(%s(%d))\n",
               GameDescription(game), GameServerId(game));
        fflush(stdout);
    }

    sizex   = GameXSize(game);
    sizey   = GameYSize(game);
    Playing = PlayingP(game);
    if (Playing) Reviewing = 0;
    else Reviewing = ReviewP(game); 

    /* initialization */
    sprintf(HandicapString, "%d", GameHandicap(game));
    move = GameMove(game);
    sprintf(MoveText, "%3d/%-3d", move, move);
    Root = MyVaCreateManagedWidget(Playing ? "play" :
                                   Reviewing ? "review" : "observe",
                                   toplevel,
                                   XtNboardSize, (XtArgVal) sizex,
                                   "captures",   (XtArgVal) GameCaptures(game),
                                   "move",       (XtArgVal) MoveText,
                                   "komi",       (XtArgVal) GameKomi(game),
                                   "handicap",   (XtArgVal) HandicapString,
                                   "title",      (XtArgVal) "",
                                   NULL);
    Collect         = XtNameToWidget(Root, "*collect");
    quit            = XtNameToWidget(Root, "*quit");
    MoveBeep        = XtNameToWidget(Root, "*moveBeep");
    MoveRaise       = XtNameToWidget(Root, "*moveRaise");
    KibitzBeep      = XtNameToWidget(Root, "*kibitzBeep");
    KibitzRaise     = XtNameToWidget(Root, "*kibitzRaise");
    ErrorBeep       = XtNameToWidget(Root, "*errorBeep");
    ErrorRaise      = XtNameToWidget(Root, "*errorRaise");
    Refresh         = XtNameToWidget(Root, "*refresh");
    Observers       = XtNameToWidget(Root, "*observers");
    analyze         = XtNameToWidget(Root, "*analyze");
    Move            = XtNameToWidget(Root, "*move");
    Komi            = XtNameToWidget(Root, "*komi");
    Handicap        = XtNameToWidget(Root, "*handicap");
    Captures        = XtNameToWidget(Root, "*captures");
    Com             = XtNameToWidget(Root, "*board");
    KibitzFile      = XtNameToWidget(Root, "*kibitzFile");
    KibitzSave      = XtNameToWidget(Root, "*kibitzSave");
    KibitzOverwrite = XtNameToWidget(Root, "*kibitzOverwrite");
    SgfFromStart    = XtNameToWidget(Root, "*sgfFromStart");
    SgfFile         = XtNameToWidget(Root, "*sgfFile");
    SgfSave         = XtNameToWidget(Root, "*sgfSave");
    SgfOverwrite    = XtNameToWidget(Root, "*sgfOverwrite");
    PsSave          = XtNameToWidget(Root, "*psSave");
    PsFromStart     = XtNameToWidget(Root, "*psFromStart");
    PsFile          = XtNameToWidget(Root, "*psFile");
    PsOverwrite     = XtNameToWidget(Root, "*psOverwrite");
    StatsBlack      = XtNameToWidget(Root, "*statsBlack");
    StatsWhite      = XtNameToWidget(Root, "*statsWhite");
    Blink           = XtNameToWidget(Root, "*blink");
    replay          = XtNameToWidget(Root, "*replay");
    TalkBlack       = XtNameToWidget(Root, "*talkBlack");
    TalkWhite       = XtNameToWidget(Root, "*talkWhite");

    BlackName = GameBlackName(game);
    WhiteName = GameWhiteName(game);
    if (BlackName == WhiteName) {
        if (StatsBlack && StatsWhite) {
            XtDestroyWidget(StatsWhite);
            StatsWhite = 0;
        }
        if (TalkBlack && TalkWhite) {
            XtDestroyWidget(TalkWhite);
            TalkWhite = 0;
        }
    }

    if (Playing) {
        Title   = XtNameToWidget(Root, "*titlePlay");
        undo    = XtNameToWidget(Root, "*undo");
        Pass    = XtNameToWidget(Root, "*pass");
        done    = XtNameToWidget(Root, "*done");
        resume  = XtNameToWidget(Root, "*resume");
        adjourn = XtNameToWidget(Root, "*adjourn");
        declineAdjourn = XtNameToWidget(Root, "*declineAdjourn");
    } else Title = XtNameToWidget(Root, "*titleObserve");

    Dup    = XtNameToWidget(Root, "*dup");
    Scroll = XtNameToWidget(Root, "*scroll");
    Info   = XtNameToWidget(Root, "*info");
    Input  = XtNameToWidget(Root, "*input");
    Tim    = XtNameToWidget(Root, "*time");
    if (Tim) AddText(Tim, GetTime(game));

    WITH_HANDLING {
        observe = mynew(Observe);
        observe->SgfFile = observe->PsFile = observe->KibitzFile = NULL;
        observe->TopWidget       = Root;
        SetObserveDescription(observe, game);
        observe->Game            = game;
        observe->Node            = 0;
        observe->InfoWidget      = Info;
        observe->Title           = Title;
        observe->ScrollWidget    = Scroll;
        observe->BoardWidget     = Com;
        observe->InputWidget     = Input;
        observe->TimeWidget      = Tim;
        observe->MoveBeepWidget  = MoveBeep;
        observe->MoveRaiseWidget = MoveRaise;
        observe->KibitzBeep      = KibitzBeep;
        observe->KibitzRaise     = KibitzRaise;
        observe->ErrorBeep       = ErrorBeep;
        observe->ErrorRaise      = ErrorRaise;
        observe->KomiWidget      = Komi;
        observe->HandicapWidget  = Handicap;
        observe->CaptureWidget   = Captures;
        observe->MoveWidget      = Move;
        observe->SgfOverwrite    = SgfOverwrite;
        observe->PsOverwrite     = PsOverwrite;
        observe->KibitzOverwrite = KibitzOverwrite;
        observe->SgfFromStart    = SgfFromStart;
        observe->PsFromStart     = PsFromStart;
        observe->Replay          = replay;
        observe->Next            = GameToObservers(game);
        GameToObservers(game)    = observe;

        XtAddCallback(Root,    XtNdestroyCallback, 
                      CallDestroyObserve, (XtPointer) observe);
        XtAddCallback(quit,    XtNcallback, CallDestroy, (XtPointer) Root);
        XtAddCallback(Refresh, XtNcallback, RefreshGame, (XtPointer) observe);
        XtAddCallback(analyze, XtNcallback, AnalyzeGame, (XtPointer) observe);
        XtAddCallback(Dup,     XtNcallback, DupGame,     (XtPointer) observe);
        XtAddCallback(Com,     XtNbuttonUp, ButtonUp,    (XtPointer) observe);
        XtAddCallback(Scroll,  XtNjumpProc, JumpObserve, (XtPointer) observe);
        XtAddCallback(Scroll,  XtNscrollProc, ScrollObserve,
                      (XtPointer) observe);

        if (Playing) {
            if (undo) XtAddCallback(undo, XtNcallback, SendUndo,
                                    (XtPointer) observe);
            if (Pass) XtAddCallback(Pass, XtNcallback, SendPass,
                                    (XtPointer) observe);
            if (done) {
                XtAddCallback(done, XtNcallback, SendDone,
                              (XtPointer) observe);
                XtUnmanageChild(done);
            }
            if (resume) XtAddCallback(resume, XtNcallback, CallResume,
                                      (XtPointer) game);
            if (adjourn) XtAddCallback(adjourn, XtNcallback, CallSendCommand,
                                       (XtPointer) "adjourn");
            if (declineAdjourn)
                XtAddCallback(declineAdjourn, XtNcallback, CallSendCommand,
                              (XtPointer) "decline adjourn");
            observe->Pass = Pass;
            observe->Done = done;
        }
        if (Observers) XtAddCallback(Observers, XtNcallback, CallObservers,
                                     (XtPointer) observe);
        if (SgfSave) XtAddCallback(SgfSave, XtNcallback, SaveGame,
                                   (XtPointer) observe);
        if (SgfFile) XtAddCallback(SgfFile, XtNcallback, ChangeSgfFilename,
                                   (XtPointer) &observe->SgfFile);
        if (PsSave) XtAddCallback(PsSave, XtNcallback, SavePs,
                                  (XtPointer) observe);
        if (PsFile) XtAddCallback(PsFile, XtNcallback, ChangePsFilename,
                                  (XtPointer) &observe->PsFile);
        if (KibitzSave) XtAddCallback(KibitzSave, XtNcallback, SaveKibitz,
                                      (XtPointer) observe);
        if (KibitzFile) XtAddCallback(KibitzFile, XtNcallback,
                                      ChangeKibitzFilename,
                                      (XtPointer) &observe->KibitzFile);
        if (TalkBlack) {
            XtAddCallback(TalkBlack, XtNcallback, GetTell,
                          (XtPointer) GameBlackName(game));
            XtVaGetValues(TalkBlack, XtNlabel, (XtArgVal) &title, NULL);
            title = GameTemplateDescription(game, title);
            XtVaSetValues(TalkBlack, XtNlabel, (XtArgVal) title, NULL); 
            myfree(title);
        }

        if (TalkWhite) {
            XtAddCallback(TalkWhite, XtNcallback, GetTell,
                          (XtPointer) GameWhiteName(game));
            XtVaGetValues(TalkWhite, XtNlabel, (XtArgVal) &title, NULL);
            title = GameTemplateDescription(game, title);
            XtVaSetValues(TalkWhite, XtNlabel, (XtArgVal) title, NULL); 
            myfree(title);
        }

        if (StatsBlack) {
            XtAddCallback(StatsBlack, XtNcallback, GetStats,
                          (XtPointer) GameBlackName(game));
            XtVaGetValues(StatsBlack, XtNlabel, (XtArgVal) &title, NULL);
            title = GameTemplateDescription(game, title);
            XtVaSetValues(StatsBlack, XtNlabel, (XtArgVal) title, NULL); 
            myfree(title);
        }

        if (StatsWhite) {
            XtAddCallback(StatsWhite, XtNcallback, GetStats,
                          (XtPointer) GameWhiteName(game));
            XtVaGetValues(StatsWhite, XtNlabel, (XtArgVal) &title, NULL);
            title = GameTemplateDescription(game, title);
            XtVaSetValues(StatsWhite, XtNlabel, (XtArgVal) title, NULL); 
            myfree(title);
        }

        if (Blink && Com) {
            XtAddCallback(Blink, XtNcallback, ToggleBlink, (XtPointer) Com);
            XtVaGetValues(Blink, XtNstate, (XtArgVal) &state, NULL);
            if (state == False)
                XtVaSetValues(Com, XtNoffTimeout, (XtArgVal) 0, NULL);
        }
        if (replay) {
            XtAddCallback(replay, XtNcallback, ToggleReplay,
                          (XtPointer) observe);
            XtVaGetValues(replay, XtNstate, (XtArgVal) &state, NULL);
            ToggleReplay(0, (XtPointer) observe,
                         INT_TO_XTPOINTER((int) state));
        }
        if (Title)
            if (TeachingP(game) || Reviewing) {
                gameTitle = GameTitle(game);
                if (gameTitle) AddText(Title, gameTitle);
            } else {
                XtDestroyWidget(Title);
                observe->Title = 0;
            }
       
        XtSetKeyboardFocus(Collect, Input);
        MyRealizeWidget(Root);
    } ON_EXCEPTION {
        XtDestroyWidget(Root);
        ReRaise();    
    } END_HANDLING;
    return observe;
}

void PassToDone(const Game *game, int Pass, int Done)
{
    Observe *observe;

    if (DebugFun) {
        if (game) printf("PassToDone(%s, %d, %d)\n",
                         GameDescription(game), Pass, Done);
        else      printf("PassToDone(NULL, %d, %d)\n", Pass, Done);
        fflush(stdout);
    }

    if (game) {                 /* Should be impossible, but who knows */
        for (observe = GameToObservers(game); observe; observe = observe->Next)
            if (!observe->Pass || !observe->Done)
                Warning("Called PassToDone on an incomplete observe\n");
            else VaSetManagementChildren(observe->Pass, Pass, observe->Done,
                                         Done, (Widget) 0);
    } else Warning("Called PassToDone(NULL)\n");
}

static void ButtonUp(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Observe  *observe;
    Game     *game;
    Gamelog  *log;
    BWPiece   Color;
    GoButton *Button;
    char      Move[4];
    int       i, x, y, node, Nodes, Pos;

    Button = (GoButton *) calldata;
    x      = Button->x;
    y      = Button->y;
    observe= (Observe  *) clientdata;
    game   = observe->Game;
    log    = GameGamelog(game);
    if (!log) Raise1(AssertException, "Move in game without gamelog");
    Nodes  = NumberNodes(log)-1;

    if (MyGameP(game)) {
        GotoObserve(observe, Nodes);
        if (0 <= x && x < log->SizeX && 0 <= y && y < log->SizeY) {
            GoMoveFromXY(Move, x, y);
            LastMove(w, x, y);
            SendMove(w, clientdata, Move);
        } else {
            IfBell(observe->ErrorBeep);
            IfRaise(observe->ErrorRaise, observe->ErrorRaise);
        }
    } else if (0 <= x && x < log->SizeX && 0 <= y && y < log->SizeY) {
        node = NodeNumber(log);
        Pos  = observe->Node;

        if      (Pos < 1)      Pos = 1;
        else if (Pos >= Nodes) Pos = Nodes-1;
        
        if      (Pos > node) for (i=node; i<Pos; i++) DownGamelog(log);
        else if (Pos < node) for (i=node; i>Pos; i--) UpGamelog(log);
        
        Color = GetStone(log, x, y);
        while (Pos > 0) {
            UpGamelog(log);
            Pos--;
            if (Color != GetStone(log, x, y)) {
                DownGamelog(log);
                Pos++;
                goto found;
            }
        }
        while (Pos < Nodes) {
            DownGamelog(log);
            Pos++;
            if (Color != GetStone(log, x, y)) goto found;
        }
      found:
        GotoObserve(observe, Pos);
    } else GotoObserve(observe, Nodes);
}

Observe *ObserveFindWidget(Observe *observers, Widget w, int offset)
{
    Observe *observe;

    for (observe = observers; observe; observe = observe->Next)
        if (w == *(Widget *)(offset + (char *) observe)) return observe;
    return NULL;
}

extern Observe *FindObserveWidget(Widget w, int offset);

Observe *ObserveBoardWidget(Observe *observers, Widget w)
{
    Observe *observe;

    for (observe = observers; observe; observe = observe->Next)
        if (w == observe->BoardWidget) return observe;
    return NULL;
}

Observe *FindBoardWidget(Widget w)
{
    return FindObserveWidget(w, offsetof(Observe, BoardWidget));
}

static void RealSendKibitz(Game *game, int Id, const char *Text, int length)
{
    UserSendCommand(NULL, NULL, "kibitz %d %.*s", Id, length, Text);
    Kibitz(game, Me, Text, length);
}

static void SendKibitz(Widget w, XEvent *event, String *string, Cardinal *n)
{
    Observe    *observe;
    Game       *game;
    int         Id;
    String      Buffer;
    const char *From, *To, *End;

    observe = FindObserveWidget(w, offsetof(Observe, InputWidget));
    if (observe) {
        game = observe->Game;
        XtVaGetValues(w, XtNstring, &Buffer, NULL);
        Id = GameServerId(game);
        if (Id >= 0) {
            End = strchr(Buffer, 0);
            for (From = Buffer;
                 (To = memchr((char *)From, '\n', End-From)) != NULL;
                 From = To+1)
                RealSendKibitz(game, Id, From, To-From);
            RealSendKibitz(game, Id, From, End-From);
        } else if (observe->InfoWidget)
            BatchAddText(observe->InfoWidget, ".....            "
                         "Kibitz not sent, game is gone\n"); 
        else Output("Kibitz not sent, game is gone\n");
        XtVaSetValues(w, XtNstring, "", NULL);
    } else WidgetWarning(w, "kibitz() called on invalid widget");
}

void Kibitz(const Game *game, const Player *player,
            const char *kibitz, int length)
{
    const char *Desc;
    Observe    *observe;
    int         Failed;

    Desc = PlayerString(player);
    AddGameComment(game, "%16s: %.*s", Desc, length, kibitz);
    Failed = 0;
    for (observe = GameToObservers(game); observe; observe = observe->Next) {
        if (observe->InfoWidget)
            BatchAddText(observe->InfoWidget, "%16s: %.*s\n",
                         Desc, length, kibitz);
        else Failed = 1;
        IfBell(observe->KibitzBeep);
        IfRaise(observe->KibitzRaise, observe->KibitzRaise);
    }
    if (Failed) Outputf("%16s: %.*s\n", Desc, length, kibitz);
}

void SetCaptures(const Game *game)
{
    XtArgVal Text;
    Observe *observe;

    Text = (XtArgVal) GameCaptures(game);
    for (observe = GameToObservers(game); observe; observe = observe->Next)
        if (observe->CaptureWidget)
            XtVaSetValues(observe->CaptureWidget, XtNlabel, Text, NULL);
}

void SetKomi(const Observe *observers, const char *Text)
{
    const Observe *observe;

    for (observe = observers; observe; observe = observe->Next)
        if (observe->KomiWidget)
            XtVaSetValues(observe->KomiWidget, XtNlabel, Text, NULL);
}

void SetTitle(const Observe *observers, const char *Text)
{
    const Observe *observe;

    for (observe = observers; observe; observe = observe->Next)
        if (observe->Title) {
            XtVaSetValues(observe->Title, XtNstring, (XtArgVal) "", NULL);
            if (Text) AddText(observe->Title, Text);
        }
}

static void ChangeTitle(Widget w, XEvent *event, String *string, Cardinal *n)
{
    Observe *observe;
    Game       *game;
    int         Id;
    String      Buffer;
    const char *gameTitle;

    observe = FindObserveWidget(w, offsetof(Observe, Title));
    if (observe) {
        game = observe->Game;
        gameTitle = GameTitle(game);
        if (gameTitle == NULL) gameTitle = "";
        XtVaGetValues(w, XtNstring, &Buffer, NULL);
        Id = GameServerId(game);
        if (strcmp(gameTitle, Buffer)) {
            if (Id >= 0)
                if (Buffer[0]) UserSendCommand(NULL, NULL, "title %s", Buffer);
                else if (observe->InfoWidget)
                    BatchAddText(observe->InfoWidget, "Cannot unset title\n");
                else Outputf("Cannot unset title\n");
            else if (observe->InfoWidget)
                BatchAddText(observe->InfoWidget, ".....            "
                             "Change title not sent, game is gone\n"); 
            else Output("Change title not sent, game is gone\n");
            XtVaSetValues(w, XtNstring, "", NULL);
            if (gameTitle[0]) AddText(w, gameTitle);
        }
    } else WidgetWarning(w, "changetitle() called on invalid widget");
}

void SetHandicap(const Observe *observers, int handicap)
{
    const Observe *observe;
    char           Handicap[10];

    sprintf(Handicap, "%d", handicap);
    for (observe = observers; observe; observe = observe->Next)
        if (observe->HandicapWidget)
            XtVaSetValues(observe->HandicapWidget, XtNlabel, Handicap, NULL);
}

void SetObserveTime(const Observe *observers, const char *Text)
{
    const Observe  *observe;
    Widget          Source;
    XawTextBlock    block;
    XawTextEditType type;

    block.firstPos = 0;
    block.length   = strlen(Text);
    block.ptr      = (char *) Text;
    block.format   = FMT8BIT;
    for (observe = observers; observe; observe = observe->Next)
        if (observe->TimeWidget) {
            XtVaGetValues(observe->TimeWidget,
                          XtNeditType,       (XtArgVal) &type,
                          XtNtextSource,     (XtArgVal) &Source,
                          NULL);
            XtVaSetValues(observe->TimeWidget,
                          XtNeditType, (XtArgVal) XawtextEdit, NULL);
            XawTextReplace(observe->TimeWidget, 0,
                           XawTextSourceScan(Source, 0, XawstAll,
                                             XawsdRight, 1, True),
                           &block);
            XtVaSetValues(observe->TimeWidget,
                          XtNeditType, (XtArgVal) type, NULL);
        }
}

#ifndef   HAVE_NO_STDARG_H
void GameMessage(const Game *game, const char *Prefix, 
                 const char *Format, ...)
#else  /* HAVE_NO_STDARG_H */
void GameMessage(game, Prefix, va_alist)
const Game *game;
const char *Prefix;
va_dcl
#endif /* HAVE_NO_STDARG_H */
{
    char     Text[2048];
    Observe *observe;
    int      Fail, Mine;
    va_list  args;

#ifndef   HAVE_NO_STDARG_H
    va_start(args, Format);
#else  /* HAVE_NO_STDARG_H */
    const char *Format;

    va_start(args);
    Format    = va_arg(args, const char *);
#endif /* HAVE_NO_STDARG_H */
    sprintf(Text, "%-17s", Prefix);
    vsprintf(strchr(Text, 0), Format, args);
    va_end(args);

    AddGameComment(game, "%s", Text);
    Fail = 0;
    Mine = MyGameP(game);
    for (observe = GameToObservers(game); observe; observe = observe->Next) {
        if (observe->InfoWidget)
            BatchAddText(observe->InfoWidget, "%s\n", Text);
        else Fail = 1;
        if (Mine) GotoObserve(observe, observe->Node);
    }
    if (Fail) Outputf("%s\n", Text);
}

static void CallDestroyObserve(Widget w,
                               XtPointer clientdata, XtPointer calldata)
{
    Observe *observe, *before;
    Game    *game;

    observe = (Observe *) clientdata;
    game = observe->Game;
    
    /* The next line is non portable but should work "almost" everywhere :-) */
    before = (Observe *)((char *) &GameToObservers(game)-
                         offsetof(Observe, Next));
    if (before->Next == observe && observe->Next == NULL) StopObserve(game);
    else while (before->Next != observe) before = before->Next;
    before->Next = observe->Next;
    myfree(observe->KibitzFile);
    myfree(observe->SgfFile);
    myfree(observe->PsFile);
    myfree(observe);
    
    TestDeleteGame(game);
}

void ShowPosition(const Game *game)
{
    Observe *observe;
    Gamelog *log;
    Boolean  Beep;
    int      nodes, Me;
    float    Step;

    log = GameGamelog(game);
    nodes = NumberNodes(log)-1;
    Step  = 1.0 / nodes;
    Me    = MyGameP(game);
    for (observe = GameToObservers(game); observe; observe = observe->Next)
        if (observe->Node == nodes-1 || Me) {
            GotoObserve(observe, nodes);
            if (observe->MoveBeepWidget) {
                XtVaGetValues(observe->MoveBeepWidget,
                              XtNstate, (XtArgVal) &Beep, NULL);
                if (Beep != False) StoneSound(observe->BoardWidget);
            }
            IfRaise(observe->MoveRaiseWidget, observe->MoveRaiseWidget);
        } else GotoObserve(observe, observe->Node);
}

void InitObserve(Widget Toplevel)
{
    XtAppAddActions(XtWidgetToApplicationContext(Toplevel),
                    actionTable, XtNumber(actionTable));
}

void DestroyObservers(Observe **observers)
{
    Observe *next, *observe;

    for (observe = *observers; observe; observe = next) {
        next = observe->Next;
        XtDestroyWidget(observe->TopWidget);
        myfree(observe);
    }
    *observers = NULL;
}

void ReplayTime(unsigned long diff, Observe *observers, int nodes)
{
    Observe *observe;

    for (observe = observers; observe; observe = observe->Next)
        if (observe->ReplayRate) {
            observe->ReplayCount += diff;
            if (observe->ReplayCount >= observe->ReplayRate) {
                observe->ReplayCount = 0;
                if (observe->Node == nodes) {
                    observe->ReplayRate = 0;
                    if (observe->Replay)
                        XtVaSetValues(observe->Replay, XtNstate,
                                      (XtArgVal) False, NULL);
                } else GotoObserve(observe, observe->Node+1);
            }
        }
}
