#include <myxlib.h>
#include <mymalloc.h>
#include <except.h>

#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Toggle.h>

#include "SmeBell.h"

#include "utils.h"
#include "xgospel.h"

static void NextText();
static void EndText();

static XtActionsRec actionTable[] = {
    { "nexttext", NextText },
    { "endtext",  EndText },
};

static Exception OpenException = { "Could not open file",
                                    0, ErrnoExceptionAction };

FILE *OpenWrite(const char *FileName, Boolean Overwrite)
{
    FILE *fp;

    /* Very naive but portable test */
    if (Overwrite == False && (fp = fopen(FileName, "r")) != NULL) {
        fclose(fp);
        return NULL;
    }

    fp = fopen(FileName, "w");
    if (!fp) Raise2(OpenException, ExceptionCopy(FileName), "for write");
    return fp;
}

XtPointer SaveWrite(const char *FileName, Boolean Overwrite,
                    Widget Bell, Widget Raize, const char *ErrorType,
                    void (*WriteFun(FILE *fp, XtPointer Closure)),
                    XtPointer Closure)
{
    FILE       *fp;
    const char *Error;
    XtPointer   Result;

    Result = NULL;
    Error  = NULL;
    WITH_HANDLING {
        fp = OpenWrite(FileName, Overwrite);
        if (fp)
            WITH_UNWIND {
                Result = WriteFun(fp, Closure);
            } ON_UNWIND {
                if (ExceptionPending()) {
                    if (ferror(fp)) {
                        Error = strerrno();
                        ClearException();
                    }
                    fclose(fp);
                } else if (fclose(fp)) Error = strerrno();

                if (Error) {
                    IfBell(Bell);
                    IfRaise(Raize, Raize);
                    PopMessage(ErrorType, "Could not write file %s: %s",
                               FileName, Error);
                }
            } END_UNWIND
        else {
            IfBell(Bell);
            IfRaise(Raize, Raize);
            PopMessage(ErrorType,
                       "Will not overwrite existing file %s",
                       FileName);
        }
    } ON_EXCEPTION {
        if (ExceptionP(OpenException)) {
            ClearException();
            IfBell(Bell);
            IfRaise(Raize, Raize);
            PopMessage(ErrorType, "Could not open file %s: %s",
                       FileName, StrExceptionErrno());
        } else ReRaise();
    } END_HANDLING;
    return Result;
}

XtPointer SaveTextFun(FILE *fp, XtPointer Closure)
{
    return (XtPointer) MySaveText(fp, (Widget) Closure);
}

static Boolean ReturnTrue(String Name)
{
    return True;
}

#ifndef HAVE_NO_STDARG_H
char *StringToFilename(const char *Filename, ...)
#else  /* HAVE_NO_STDARG_H */
char *StringToFilename(va_alist)
va_dcl
#endif /* HAVE_NO_STDARG_H */
{
    va_list          Args;
    int              NrSubs;
    SubstitutionRec *Subs, *Sub;
    char            *Result, *Temp;

#ifndef HAVE_NO_STDARG_H
    va_start(Args, Filename);
#else  /* HAVE_NO_STDARG_H */
    char *Filename;
    
    va_start(Args);
    Filename = va_arg(Args, char **);
#endif /* HAVE_NO_STDARG_H */
    for (NrSubs = 2; va_arg(Args, int); NrSubs++) (void) va_arg(Args, char *);
    va_end(Args);

    Subs = mynews(SubstitutionRec, NrSubs);

#ifndef HAVE_NO_STDARG_H
    va_start(Args, Filename);
#else  /* HAVE_NO_STDARG_H */
    va_start(Args);
    Filename = va_arg(Args, char **);
#endif /* HAVE_NO_STDARG_H */
    for (Sub = Subs; (Sub->match = va_arg(Args, int)) != 0; Sub++)
        Sub->substitution = va_arg(Args, char *);
    va_end(Args);
    Sub->match        = 'D';
    Sub->substitution = appdata.Directory;
    Sub++;
    Sub->match        = 'd';
    Sub->substitution = appdata.Directory;
    Temp = XtFindFile((char *) Filename, Subs, NrSubs, ReturnTrue);
    myfree(Subs);
    Result = mystrdup(Temp);
    XtFree(Temp);

    return Result;
}

#ifndef HAVE_NO_STDARG_H
Widget ChangeFilename(Widget Toplevel, const char *Type, const char *Title,
                      char **Filename, ...)
#else  /* HAVE_NO_STDARG_H */
Widget ChangeFilename(Toplevel, Type, Title, va_alist)
Widget Toplevel;
const char *Type;
const char *Title;
va_dcl
#endif /* HAVE_NO_STDARG_H */
{
    va_list       args;
    int           count;
    XtVarArgsList Args;
    Widget        Root;

#ifndef HAVE_NO_STDARG_H
    va_start(args, Filename);
#else  /* HAVE_NO_STDARG_H */
    char **Filename;
    
    va_start(args);
    FileName = va_arg(args, char **);
#endif /* HAVE_NO_STDARG_H */
    count = CountVarArgs(args);
    va_end(args);
#ifndef HAVE_NO_STDARG_H
    va_start(args, Filename);
#else  /* HAVE_NO_STDARG_H */
    char **Filename;
    
    va_start(args);
    FileName = va_arg(args, char **);
#endif /* HAVE_NO_STDARG_H */
    Args = VarArgsToList(args, count);
    va_end(args);

    Root = AskString(Toplevel, 0, NULL, Title, Type, Filename, Args, NULL);
    myfree(Args);
    return Root;
}

void ChangeSgfFilename(Widget w,
                       XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "sgfFilename", "Enter name of sgf file",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

void ChangePsFilename(Widget w,
                      XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "psFilename",
                          "Enter name of postscript file",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

void ChangeKibitzFilename(Widget w,
                          XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "kibitzFilename",
                          "Enter name of kibitzfile",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

/***************************************************************************/

void IfBell(Widget w)
{
    Boolean Beep;

    if (w)
        if (XtIsSubclass(w, smeBellObjectClass) != False) SmeBell(w);
        else {
            Beep = True;
            XtVaGetValues(w, XtNstate, (XtArgVal) &Beep, NULL);
            if (Beep != False) XBell(XtDisplayOfObject(w), 20);
        }
    else XBell(XtDisplay(toplevel), 20);
}

void IfRaise(Widget w, Widget root)
{
    Boolean Raise;
    Widget  Here;

    if (w) {
        Raise = False;
        XtVaGetValues(w, XtNstate, (XtArgVal) &Raise, NULL);
        if (Raise == False) return;
        for (Here = root; Here; Here = XtParent(Here))
            if (XtIsShell(Here) != False &&
                XtIsSubclass(Here, simpleMenuWidgetClass) == False) {
                XtPopup(Here, XtGrabNone);
                return;
            }
    }
    XBell(XtDisplay(toplevel), 20);
}
/***************************************************************************/

#ifndef   HAVE_NO_STDARG_H
Widget PopMessage(const char *Title, const char *Format, ...)
#else  /* HAVE_NO_STDARG_H */
Widget PopMessage(Title, va_alist)
const char *Title;
va_dcl
#endif /* HAVE_NO_STDARG_H */
{
    va_list rest;
    char    Text[2048];
    Widget  Root;

#ifndef   HAVE_NO_STDARG_H
    va_start(rest, Format);
#else  /* HAVE_NO_STDARG_H */
    const char *Format;

    va_start(rest);
    Format = va_arg(rest, const char *);
#endif /* HAVE_NO_STDARG_H */
    vsprintf(Text, Format, rest);
    va_end(rest);
    Root = MyVaCreateManagedWidget("popMessage", toplevel,
                                   "text", Text, "title", Title, NULL);
    MyRealizeWidget(Root);
    return Root;
}

/***************************************************************************/

typedef struct {
    String Title, IconName;
} IconTitle, *IconTitlePtr;

#define offset(field) XtOffset(IconTitlePtr, field)
static XtResource resources[] = {
    { XtNtitle, XtCTitle, XtRString, sizeof(String),
      offset(Title), XtRString, NULL }, 
    { XtNiconName, XtCIconName, XtRString, sizeof(String),
      offset(IconName), XtRString, NULL }, 
};
#undef offset

void SetWidgetTitles(Widget w,
                     char *(*fun)(const char *Pattern, XtPointer Closure),
                     XtPointer Closure)
{
    IconTitle  iconTitle;
    char      *Title, *IconName;

    XtGetApplicationResources(w, &iconTitle, resources, XtNumber(resources),
                              NULL, 0);
    Title = (*fun)(iconTitle.Title,    Closure);
    WITH_UNWIND {
        IconName = (*fun)(iconTitle.IconName, Closure);
        XtVaSetValues(w,
                      XtNtitle,    (XtArgVal) Title,
                      XtNiconName, (XtArgVal) IconName,
                      NULL);
        myfree(IconName);
    } ON_UNWIND {
        myfree(Title);
    } END_UNWIND;
}

typedef struct {
    String Name;
} PropName, *PropNamePtr;

#define offset(field) XtOffset(PropNamePtr, field)
static XtResource PResources[] = {
    { NULL, NULL, XtRString, sizeof(String),
      offset(Name), XtRString, NULL }, 
};
#undef offset

void SetWidgetProperty(Widget w, const char *Name, const char *Class,
		       char *(*fun)(const char *Pattern, XtPointer Closure),
		       XtPointer Closure)
{
    PropName Prop;
    char    *PName;

    PResources[0].resource_name  = (String) Name;
    PResources[0].resource_class = (String) Class;
    XtGetApplicationResources(w, &Prop, PResources, XtNumber(PResources),
                              NULL, 0);
    PName = (*fun)(Prop.Name, Closure);
    XtVaSetValues(w, (String) Name, (XtArgVal) PName, NULL);
    myfree(PName);
}

/***************************************************************************/

typedef struct {
    void    (*Callback)(XtPointer);
    XtPointer Closure;
    int       Count;
    Widget    Collect, Root;
} AskData;

typedef struct {
    AskData *askdata;
    char   **Address;
    Widget   NextText;
} AskAddress;

void InitUtils(Widget TopLevel)
{
    XtAppAddActions(XtWidgetToApplicationContext(TopLevel),
                    actionTable, XtNumber(actionTable));
}

void CallSetString(Widget w, XtPointer clientdata, XtPointer calldata)
{
    String text;
    char *Address;
    AskAddress *askaddress;
    AskData    *askdata;

    XtVaGetValues(w, XtNstring, &text, NULL);
    askaddress = (AskAddress *) clientdata;
    Address = *askaddress->Address;
    WITH_UNWIND {
        *askaddress->Address = mystrdup(text);
    } ON_UNWIND {
        myfree(Address);
        askdata = askaddress->askdata;
        if (0 == --askdata->Count) {
            if (askdata->Callback) (*askdata->Callback)(askdata->Closure);
            myfree(askdata);
        }
        myfree(askaddress);
    } END_UNWIND;
}

static void EndText(Widget w, XEvent *event, String *string, Cardinal *n)
{
    XtPointer client_data;

    if (FindCallback(w, XtNdestroyCallback, CallSetString, &client_data))
        XtDestroyWidget(((AskAddress *) client_data)->askdata->Root);
    else WidgetWarning(w, "endtext() called on invalid widget");
}

static void NextText(Widget w, XEvent *event, String *string, Cardinal *n)
{
    XtPointer client_data;
    AskAddress  *askaddress;
    Widget       Nexttext, Collect;

    if (FindCallback(w, XtNdestroyCallback, CallSetString, &client_data)) {
        askaddress = (AskAddress *) client_data;
        Collect    = askaddress->askdata->Collect;
        if (*n > 0) {
            Nexttext = XtNameToWidget(Collect, string[0]);
            if (!Nexttext) Nexttext = askaddress->NextText;
        } else Nexttext = askaddress->NextText;
        if (Nexttext) {
            XawTextDisplayCaret(w, False);
            XtSetKeyboardFocus(Collect, Nexttext);
            XawTextDisplayCaret(Nexttext, True);
        } else XtDestroyWidget(askaddress->askdata->Root);
    } else WidgetWarning(w, "nexttext() called on invalid widget");
}

#ifndef HAVE_NO_STDARG_H
Widget AskString(Widget TopLevel, void (*Callback)(XtPointer),
                 XtPointer Closure, const char *title, ...)
#else  /* HAVE_NO_STDARG_H */
Widget AskString(Widget TopLevel, Callback, Closure, va_alist)
void (*Callback)(XtPointer);
XtPointer Closure;
va_dcl
#endif /* HAVE_NO_STDARG_H */
{
    va_list args;
    const char   *Name;
    char        **Address;
    XtVarArgsList Args;
    Widget        Root, collect, entry, text, oldest;
    AskData      *askdata;
    AskAddress   *askaddress;

#ifndef HAVE_NO_STDARG_H
    askdata = mynew(AskData);
    WITH_HANDLING {
        va_start(args, title);
        WITH_UNWIND {
#else  /* HAVE_NO_STDARG_H */
    askdata = mynew(Askdata);
    WITH_HANDLING {
        const char *title;

        va_start(args);
        WITH_UNWIND {
            title = va_arg(args, const char *);
#endif /* HAVE_NO_STDARG_H */
            Root =
                MyVaCreateManagedWidget("askString", TopLevel,
                                        "title", (XtArgVal) title,
                                        NULL);
            askdata->Callback = Callback;
            askdata->Count    = 0;
            askdata->Root     = Root;
            askdata->Closure  = Closure;
            WITH_HANDLING {
                askaddress = NULL;
                while ((Name = va_arg(args, const char *)) != NULL) {
                    Address = va_arg(args, char **);
                    if (Address) *Address = NULL;
                    Args = va_arg(args, XtVarArgsList);
                    if (Args)
                        entry = MyVaCreateManagedWidget(Name, Root,
                                                        XtVaNestedList, Args,
                                                        NULL);
                    else entry = MyVaCreateManagedWidget(Name, Root, NULL);
                    text  = XtNameToWidget(entry, "*text");
                    if (text) {
                        AddText(text, ""); /* Forces resize */
                        XawTextSetInsertionPoint(text,
                            XawTextSourceScan(XawTextGetSource(text), 0,
                                              XawstAll, XawsdRight, 1, True));
                        if (Address) {
                            if (askaddress) askaddress->NextText = text;
                            else            oldest               = text;
                            askaddress = mynew(AskAddress);
                            askaddress->askdata = askdata;
                            askaddress->Address = Address;
                            askdata->Count++;
                            XtAddCallback(text, XtNdestroyCallback,
                                          CallSetString,
                                          (XtPointer) askaddress);
                        }
                    }
                }
                if (askaddress) askaddress->NextText = /* oldest */ 0;
                if ((collect = XtNameToWidget(Root, "*collect")) != 0 &&
                    askaddress) {
                    XtSetKeyboardFocus(collect, oldest);
                    XawTextDisplayCaret(oldest, True);
                }
                askdata->Collect = collect;
                MyRealizeWidget(Root);
            } ON_EXCEPTION {
                XtDestroyWidget(Root);
                ReRaise();
            } END_HANDLING;
        } ON_UNWIND {
            va_end(args);
        } END_UNWIND;
    } ON_EXCEPTION {
        myfree(askdata);
        ReRaise();
    } END_HANDLING;
    return Root;
}
