/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*++ seekdbb.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/lib/RCS/seekdbb.c,v $
 *
 * DESCRIPTION:
 *
 *	Seek on an NQS database file without throwing away the current
 *	block cache, if the block in the cache is the one that we are
 *	interested in.  This function should only be called by the NQS
 *	daemon.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.3 $ $Date: 1992/12/22 15:46:28 $ $State: Exp $)
 * $Log: seekdbb.c,v $
 * Revision 1.3  1992/12/22  15:46:28  jrroma
 * Version 3.30
 *
 * Revision 1.2  92/06/18  13:24:40  jrroma
 * Added gnu header
 * 
 * Revision 1.1  92/06/18  12:47:43  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <errno.h>
#include <unistd.h>

/*** seekdbb
 *
 *
 *	void seekdbb():
 *
 *	Seek on an NQS database file without throwing away the current
 *	block cache, if the block in the cache is the one that we are
 *	interested in.  This function should only be called by the NQS
 *	daemon.
 */
void seekdbb (struct confd *file, long position)
{
	long blockoffset;
	register int i;
	register char *cp;

	if (position > file->size) position = file->size;
	else if (position < 0) position = 0;
	blockoffset = blockdb (position) * ATOMICBLKSIZ;
	if (position < file->size) {
		/*
		 *  There is a block in the file that we can read.
		 */
		if (file->rposition - ATOMICBLKSIZ != blockoffset) {
			/*
			 *  We do not have the proper block in the cache.
			 *  Correct this situation!
			 */
			lseek (file->fd, blockoffset, 0);
			errno = 0;	/* Set to zero, so that partial reads */
					/* do not cause a spurious error */
					/* message in nqs_abort(). */
			if (read (file->fd, file->cache->v.chars,
				  ATOMICBLKSIZ) != ATOMICBLKSIZ) {
				printf ("E$Read error in seekdbb().\n");
				fflush (stdout);
			}
			file->rposition = blockoffset + ATOMICBLKSIZ;
		}
	}
	else {
		/*
		 *  We are at the very end of the file.  Fill the cache
		 *  buffer with all zero.  (allodb uses this fact).
		 */
		lseek (file->fd, blockoffset, 0);
		file->rposition = blockoffset;
		cp = file->cache->v.chars;
		for (i=0; i < ATOMICBLKSIZ; i++) *cp++ = '\0';
	}
	file->vposition = position;
}
