/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*	V01.3	JRR	17-Jun-1992	Added header.
*	V01.4	JRR	03-Sep-1992	Added support for remote qlimits.
*	V01.5	JRR	23-Feb-1993	Modified for C prototypes.
*	V01.6	JRR	18-Aug-1993	Miniscule change for includes.
*	V01.7	JRR	02-Mar-1994	Fixed up NETDB.
*/
/*++ sholbymach.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/lib/RCS/sholbymach.c,v $
 *
 * DESCRIPTION:
 *
 *	Print on stdout the limits that are meaningful for a machine.
 *
 *
 *	Author:
 *	-------
 *	Robert W. Sandstrom, Sterling Software Incorporated.
 *	January 13, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.7 $ $Date: 1994/03/30 20:32:44 $ $State: Exp $)
 * $Log: sholbymach.c,v $
 * Revision 1.7  1994/03/30  20:32:44  jrroma
 * Version 3.35.6
 *
 * Revision 1.6  93/09/10  13:55:23  jrroma
 * Version 3.35
 * 
 * Revision 1.5  93/07/13  21:31:36  jrroma
 * Version 3.34
 * 
 * Revision 1.4  92/12/22  15:46:31  jrroma
 * Version 3.30
 * 
 * Revision 1.3  92/06/18  13:24:48  jrroma
 * Added gnu header
 * 
 * Revision 1.2  92/01/16  17:17:22  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/16  17:16:29  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"			/* NQS types and definitions */
#include <netdb.h>			/* Network database header file; */
#include "netpacket.h"			/* NQS remote message packets */

/*** sholbymach
 *
 *
 *	int sholbymach():
 *
 *	Print on stdout the limits that are meaningful
 *	for the named machine.
 *
 *	Returns: 0 if output was produced.
 *		-1 if no output was produced.
 */
int sholbymach (
	struct confd *paramfile,	/* This pointer useful for */
					/* local case only */
	char *itsname,			/* The machine in question */
	long flags)			/* SHO_? */
{
	Mid_t itsmid;
	Mid_t mymid;
	struct hostent *ent;		/* Host table entry structure */
	struct passwd *passwd;          /* Whose request it is */
        int sd;                         /* Socket descriptor */
        short timeout;                  /* Seconds between tries */
        long transactcc;                /* Holds establish() return value */
	uid_t cuid;		    	/* Mapped owner user-id */
	int extrach;			/* Number of chs already read */
	short output;			/* Boolean */

	if ((ent = gethostbyname (itsname)) == (struct hostent *) 0) {
		fprintf (stderr, "Invalid hostname specification: ");
		fprintf (stderr, "%s.\n", itsname);
		return (0);
	}
	switch (nmap_get_mid (ent, &itsmid)) {
	    case NMAP_SUCCESS:		/* Successfully got machine-id */
		break;
	    case NMAP_ENOMAP:		/* What?  No mid defined! */
		fprintf (stderr, "Invalid hostname specification: ");
		fprintf (stderr, "%s.\n", itsname);
		return (0);
	    case NMAP_ENOPRIV:		/* No privilege */
		fprintf (stderr, "Unexpected error in nmap_get_mid(), ");
		fprintf (stderr, "machine = %s.\n", itsname);
		return (0);
	    default:
		fprintf (stderr, "Unexpected error in nmap_get_mid(), ");
		fprintf (stderr, "machine = %s.\n", itsname);
		return (0);
	}
	if (localmid (&mymid) != 0) {
		fprintf (stderr, "Unexpected error in localmid().\n");
		return (0);
	}
	if (itsmid == mymid) {
		if (telldb (paramfile) != -1) {	/* if we have read before, */
			seekdb (paramfile, 0L);	/* seek to beginning */
		}
		return (shoalllim (paramfile, flags));
	} else {
            cuid = getuid();
            if ((passwd = fetchpwuid (cuid)) == (struct passwd *) 0) {
                fprintf (stderr, "Unable to determine caller's username\n");
                exit (1);
            }
	    closepwdb();
	    interclear();
	    interw32i ((long) cuid);
            interwstr (passwd->pw_name);
	    interw32i ((long) flags);
            sd = establish (NPK_QLIMIT, itsmid, cuid, passwd->pw_name, 
				&transactcc);
            if (sd < 0) {
                if (sd == -2) {
                    /*
                     * Retry is in order.
                     */
                    timeout = 1;
                    do {
                        nqssleep (timeout);
                        interclear ();
                        interw32i ((long) cuid);
                        interwstr (passwd->pw_name);
                        interw32i ((long) flags);
                        sd = establish (NPK_QLIMIT, itsmid,
                             cuid, passwd->pw_name, &transactcc);
                        timeout *= 2;
                    } while (sd == -2 && timeout <= 16);
	            /*
	             *  Beyond this point, give up on retry.
	             */
	            if (sd < 0) {
	                analyzetcm (transactcc, stderr, "");
			exit (0);
	            }
		} else {
		    /*
		     *  Retry would surely fail.
		     */
		     /*
		      * We seem to get TCMP_CONNBROKEN here for some reason.
		      * But it gets reported as "Undefined transaction code..."
		      * Hmmmm.
		      * printf("Transact = %o\n",  transactcc);
		      */ 
		     analyzetcm (transactcc, stderr, "");
		     exit (0);
		}
	    }
	    /*
	     *  The server liked us.
	     */
	    output = -1;
	    /*
	     *  First, display any characters that
	     *  were already read inside establish().
	     */
	    if (extrach = extrasockch ()) {
		output = 0;
		write (STDOUT, bufsockch (), extrach);
	    }
	    /*
	     *  Then display any characters that the server
	     *  may still want to send us.
	     */
	    if (filecopyentire (sd, STDOUT)) return (0);
	    else return (output);
        }
}
