/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *     Copyright (C) 1995 Pascal Haible (haible@IZFM.Uni-Stuttgart.DE) and 
 *             Michael Weller (eowmob@exp-math.uni-essen.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Dipl.-Ing. Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 * Michael Weller (eowmob@exp-math.uni-essen.de or
 * mat42b@aixrs1.hrz.uni-essen.de) Heiderhoefen 116b,
 * D 46049 Oberhausen, Germany.
 */

/*
 * This file is heavily based on fb.c copyright 10/94 Pascal Haible.
 * Still after countless patches by Michael Weller, there is not much left
 * of the original code.
 */

/*
 * These two MUST be first!
 */

#include <linux/autoconf.h>
#include <linux/module.h>

#include <linux/types.h>
#include <linux/version.h>	/* printk() */
#include <linux/kernel.h>	/* printk() */
#include <linux/mm.h>		/* invalidate() */
#include <linux/fs.h>		/* struct file_operations */
#include <linux/fcntl.h>	/* O_NOCTTY and friends */
#include <linux/tty.h>
#include <linux/vt.h>
#include <linux/kd.h>
#include <linux/mman.h>		/* PROT_* */
#include <linux/errno.h>
#include <linux/major.h>
#include <linux/malloc.h>	/* kmalloc() */

#include <asm/io.h>		/* out*() */
#include <asm/segment.h>	/* put_user_long() etc. */
#include <asm/pgtable.h>	/* PAGE_* */
#include <asm/page.h>		/* PAGE_* */

/*#define DEBUG*/	/* Emit some debug messages */
/*#define VERBOSE_DEB*/ /* Emit even more debug messages.. in fact so many that syslog will easily
			   be overrun !! */

#ifndef DEBUG
/* #define NDEBUG */ /* define for production kernels (No sanity checks -> less code) */
#endif

/*
 * currently unused as of include/linux/major.h (< 1.2.x)
 *
 * Is actually used for this on 680x0, but was still illegally acquired by a f* cdrom now
 * leave as true int variable for external insmod config.
 */
int fb_module_major = 29; /* currently unused as of include/linux/major.h */
			  /* Michael: Is actually used for this on 680x0 */
			  /* Michael: Still illegally acquired by a f* cdrom now */

#define DISPLAY_SETTLE_TIME 10

#define FB_NOSIGBUS /* Do not touch.. only for time measurements.. */

/*
 * End of config section:
 * Do not touch any #define's below.
 */

#include "fb_debug.h"

#include "fb.h"
#include "config.h"
#include "chipset.h"

#define CHECK_VT_AND_RETURN_ON_ERR(sess) {\
	int errorcode; \
	 \
	errorcode = fb_check_ready(sess); \
	if (errorcode) \
		return errorcode; \
}

#ifdef DEBUG
#define static
#define inline
#endif

/*
 * We must include the kernel idenfication string to install the module.
 */

#ifndef CONFIG_MODVERSIONS
char kernel_version[] = UTS_RELEASE;
#endif

static void *fb_savestate_buf = NULL, *fb_fontbuffer = NULL;
static void *panic_save_away = NULL;

static volatile struct {
	int got_save_state:1;		/* fb_savestate_buf contains valid data */
	int got_font:1;			/* fb_fontbuffer  contains valid data */
	int panic_is_vmall:1;		/* panic_save_away vmalloc'ed, not kmalloc'ed */
	int have_panic_buff:1;		/* allocated a panic_save_away buffer */
	int mmaping:1;			/* Currently modifying page tables */
	int unmap_pending:1;		/* We must unmap page tables due to a console switch
					   as soon as mmaping is reset */
	} fb_flags = {0, 0, 0, 0, 0, 0};

static int fb_active_sessions = 0; /* We use this as local MOD_USE_COUNT coz that one may be changed
				      externally as well */

/*
 * this is the pte of a faked non existing swap page. We use SHM_SWP_TYPE
 * to ensure the swap module does not try to find it in the swap space.
 * This should not happen however, as we ensure that swap_free is never
 * called on this pte.
 */

static const pte_t empty_pte = __pte(SHM_SWP_TYPE << 1);

static inline unsigned long min(unsigned long a, unsigned long b)
{
	if(a < b)
		return a;
	else	return b;
}

/*
 * Descriptor for mappings of one process on a specific virtual screen.
 */

struct fb_sess_t {
	struct fb_sess_t *sess_next, *sess_prev;
	struct fb_vt_t *sess_vt;
	struct vm_area_struct *sess_vm_first;
	struct fb_vm_operations *sess_ops;
	struct {
		int orphaned: 1;
		} sess_flags;
	};
/*
 * Descriptor per virtual screen:
 */

struct fb_vt_t {
	struct fb_vt_t *vt_next, *vt_prev;
	struct fb_sess_t *vt_master;
	int vt_num;
	struct {
		int active: 1;
		int graphmode: 1;
		int dedicated: 1;
		int dont_care4screen: 1;
		int alloc_on_the_fly: 1; 
		} vt_flags;
	};

/*
 * We append some additional info to the vm_operations we use
 */

struct fb_vm_operations {
	struct vm_operations_struct fb_vm_ops;
	struct fb_sess_t *fb_vm_sess;
	};

static struct fb_vt_t *fb_vt_list = NULL;

static int fb_entervt(int curr_vc, int dest_vc, int tid);
static int fb_leavevt(int curr_vc, int dest_vc, int tid);
static int fb_prep_tty_ioctl(struct tty_struct *tty, struct file *file,
	int perm, unsigned int cmd, unsigned long arg);
static void fb_did_tty_ioctl(struct tty_struct *tty, struct file *file,
	int perm, unsigned int cmd, unsigned long arg);
static struct vt_handler_operations fb_vt_hand = {
	NULL, NULL,
	fb_entervt, fb_leavevt, fb_prep_tty_ioctl, fb_did_tty_ioctl,
	NULL};

/*
 * taken from sbcd.c:
 */

static inline void fb_sleep(unsigned int jifs) /* jif = 10ms */
{
	current->state = TASK_INTERRUPTIBLE;
	current->timeout = jiffies + jifs;
	schedule();
}

/* call b4 changing page tables: */
static inline void begin_map(void) {
	fb_flags.mmaping = 1;
}

/*
 * Poor man's map_page_range.. This is IMHO missing from kernel...
 * This is a quite simple implementation but should be enough for now.
 * (Someone could try to eliminate some of the unneeded pagetable lookups,
 * Look how the real zero_map or unmap works.. but not me..) - Michael.
 * Ok.. I did it.. would be easier if I could get rid of the accesses
 * to the task struct as well.. but there is no way to do it in a portable way.
 */

static inline void map_page_range(struct task_struct * tsk,
		 unsigned long addr, unsigned long size, pte_t entry)
{
	pte_t *ptep;
	unsigned long pte_end;

#ifdef VERBOSE_DEB
	DEB(printk(KERN_DEBUG "map_page_range: address=%lx, size=%lx, entry=%lx\n",
		addr, size, pte_val(entry));)
#endif

	size += addr;

	while(addr < size) {
		ptep = pte_offset( pmd_offset( pgd_offset(tsk, addr), addr), addr);
		pte_end = min(size, (addr + PMD_SIZE) & PMD_MASK);

		while(addr < pte_end) {
			*ptep++ = entry;
			addr += PAGE_SIZE;
		}
	}
}

/*
 * Poor man's remap page range. This keeps me from making a kernel patch at this early
 * stage.. in addition the kernel's remap page range want's to be smart and f*s up most
 * stuff by making too many (pseudo) smart assumptions.
 *
 * PORTABILITY-NOTE!: We rely on pte_val(pte) += PAGE_SIZE being equivalent to:
 * pte = mk_pte(pte_page(pte) + PAGE_SIZE, prot);
 * with prot = pte_prot(pte) (which doesn't even exist)
 */
	
static inline void myremap_page_range(struct task_struct * tsk,
		 unsigned long addr, unsigned long dest, size_t size, pgprot_t prot)
{
	pte_t pte = mk_pte(dest, prot);
	pte_t *ptep;
	unsigned long pte_end;

	size += addr;

	while(addr < size) {
		ptep = pte_offset( pmd_offset( pgd_offset(tsk, addr), addr), addr);
		pte_end = min(size, (addr + PMD_SIZE) & PMD_MASK);

		while(addr < pte_end) {
			*ptep++ = pte;
			pte_val(pte) += PAGE_SIZE;
			addr += PAGE_SIZE;
		}
	}
}

static inline void unmap_bank(struct chip_bank_desc *bank_desc) {
	struct mapping_desc *map_ptr = bank_desc -> mappings;
	struct mapping_desc *map_end = bank_desc -> end_maps;

	/*
	 * In the case of a paging dead lock there will be exactly one mapping
	 * (at least after some dead lock loops) and the faulting pte will still be
	 * old.
	 */

	while(map_ptr < map_end) {
		if (map_ptr -> tsk) {
#ifndef FB_NOSIGBUS
			/* The check is only reliable when it is in the active task: */
			if (map_ptr -> tsk == current) {
				if (!pte_young(*(map_ptr -> faulting_pte))) {
					/* This signal may not be blocked! */
        				current->blocked &= ~(1<<(SIGBUS-1));
        				send_sig(SIGBUS, current, 1);
				}
			}
#endif
			map_page_range(map_ptr -> tsk, map_ptr -> start,
					map_ptr -> size, empty_pte);
			if (map_ptr -> equiv)
				map_ptr -> equiv -> tsk = NULL;
			map_ptr -> tsk = NULL;
		}
		map_ptr++;
	}
}

static inline void unmap_area(unsigned long start, unsigned long end) {
	int num_banks = chip_num_banks;
	struct chip_bank_desc *bdesc = chip_banks;
	struct mapping_desc *map_ptr, *map_end;

	while(num_banks--) {
		map_ptr = bdesc -> mappings;
		map_end = bdesc -> end_maps;
		while(map_ptr < map_end) {
			if((map_ptr -> tsk == current) &&
					(start < map_ptr -> start + map_ptr -> size) &&
					(end > map_ptr -> start)) {
				map_page_range(map_ptr -> tsk, map_ptr -> start,
					map_ptr -> size, empty_pte);
				map_ptr -> tsk = NULL;
				if (map_ptr -> equiv)
					map_ptr -> equiv -> tsk = NULL;
			}
			map_ptr++;
		}
		bdesc++;
	}
}

static inline void make_new_map(struct chip_bank_desc *bdesc, struct vm_area_struct *vma,
		int next_also) {
	register unsigned long mapsize, mapstart;
	struct mapping_desc *map_ptr = bdesc -> next_map;

	if(map_ptr -> tsk) {
		/* release old map */
		map_page_range(map_ptr -> tsk, map_ptr -> start,
			map_ptr -> size, empty_pte);
		if (map_ptr -> equiv)
			map_ptr -> equiv -> equiv = NULL;
		}

	map_ptr -> tsk = current;
	map_ptr -> start = (bdesc -> value) * (bdesc -> pages << PAGE_SHIFT) -
				vma -> vm_offset +
				vma -> vm_start;
	mapsize = (bdesc -> pages << PAGE_SHIFT);
	mapstart = bdesc -> base;
	if (map_ptr -> start < vma -> vm_start) {
		mapsize -= vma -> vm_start - map_ptr -> start;
		mapstart += vma -> vm_start - map_ptr -> start;
		map_ptr -> start = vma -> vm_start;
	}
	if (map_ptr -> start + mapsize > vma -> vm_end)
		mapsize = vma -> vm_end - map_ptr -> start;
	map_ptr -> size = mapsize;

#ifdef VERBOSE_DEB
	DEB(printk(KERN_DEBUG "make_new_map: start=%lx, size=%lx\n",
		map_ptr -> start, mapsize);)
#endif

	if(next_also) {
		/* record this mapping for the next bank also */
		map_ptr = bdesc[1].next_map;
		if(map_ptr -> tsk) {
			/* release old map */
			map_page_range(map_ptr -> tsk, map_ptr -> start,
				mapsize, empty_pte);

			if (map_ptr -> equiv)
				map_ptr -> equiv -> equiv = NULL;
		}
		*map_ptr = *(bdesc -> next_map);
		/* record mapping equivalence: */
		map_ptr -> equiv = bdesc -> next_map;
		bdesc -> next_map -> equiv = map_ptr;

		if((++bdesc[1].next_map) >= bdesc[1].end_maps)
			bdesc[1].next_map = bdesc[1].mappings;
	} else {
		map_ptr -> equiv = NULL;
	}

	myremap_page_range(current, map_ptr -> start,
                                mapstart,
                                mapsize,
				vma->vm_page_prot);

	if((++bdesc[0].next_map) >= bdesc[0].end_maps)
		bdesc[0].next_map = bdesc[0].mappings;
}

static inline void
fb_graphics_on(struct fb_vt_t *vt)
{
	ENTERING;
	if (!vt->vt_flags.graphmode) {
		if(vt_cons[vt->vt_num]) {
#ifndef NDEBUG
			if(vt_cons[vt->vt_num]->vc_mode != KD_TEXT) {
				printk( KERN_DEBUG "fb: vt %d changed mode behind my back\n",
					vt->vt_num);
			}
#endif
			vt_cons[vt->vt_num]->vc_mode = KD_GRAPHICS;
			do_blank_screen(1);
		} else {
			printk( KERN_CRIT "fb: vt %d disallocated behind my back\n", vt->vt_num);
		}
		vt->vt_flags.graphmode = 1;
		if(!fb_flags.got_save_state) {
			chip_savestate(fb_savestate_buf, fb_flags.got_font ? NULL : fb_fontbuffer);
			fb_flags.got_save_state = 1;
			fb_flags.got_font = 1;
		}
		chip_blank();
		chip_graphics();
		/* chip_blank(); */
		fb_sleep(DISPLAY_SETTLE_TIME);
		/* return with screen blanked */
		/* chip_unblank(); */
	}
	LEAVING;
}

static inline void
fb_graphics_off(struct fb_vt_t *vt)
{
	ENTERING;
	if (vt->vt_flags.graphmode) {
		chip_blank();
		chip_restorestate(fb_savestate_buf, fb_fontbuffer);
		chip_unblank();
		vt->vt_flags.graphmode = 0;
		if(vt_cons[vt->vt_num]) {
#ifndef NDEBUG
			if(vt_cons[vt->vt_num]->vc_mode != KD_GRAPHICS) {
				printk( KERN_DEBUG "fb: vt %d changed mode behind my back\n",
					vt->vt_num);
			}
#endif
			vt_cons[vt->vt_num]->vc_mode = KD_TEXT;
			do_unblank_screen();
		} else {
			printk( KERN_CRIT "fb: vt %d disallocated behind my back\n", vt->vt_num);
		}
	}
	LEAVING;
}

static void release_vt(struct fb_vt_t *vt) {
	ENTERING;
	if (!vt)
		return;
	vt->vt_next->vt_prev = vt->vt_prev;
	vt->vt_prev->vt_next = vt->vt_next;

	if(vt == fb_vt_list) {
		fb_vt_list = vt->vt_next;
		if(vt == fb_vt_list)
			fb_vt_list = NULL;
	}
	fb_graphics_off(vt);
	vt_handler_handle_vc(vt->vt_num, VT_HANDLER_VIDEO, VT_HANDLER_REL);
	kfree_s(vt, sizeof(*vt));
#ifdef DEBUG
	{
		int banknum;
		struct chip_bank_desc *bdesc = chip_banks;
		struct mapping_desc *map_ptr, *map_end;

		for(banknum = 0; banknum < chip_num_banks; banknum++) {
			map_ptr = bdesc -> mappings;
			map_end = bdesc -> end_maps;
			while(map_ptr < map_end) {
				if(map_ptr -> tsk == current) {
					printk(KERN_DEBUG "(/dev/fb[%d]: Orphaned? mapping of bank "
							"%d on close: st:0x%08lx, "
							"len:0x%08lx\n",
						map_ptr -> tsk -> pid,
						banknum,
						map_ptr -> start,
						map_ptr -> size);
					map_ptr -> tsk = NULL;
				}
				map_ptr++;
			}
			bdesc++;
		}
	}
#endif
	LEAVING;
}

static void release_session(struct fb_sess_t *session) {
	struct fb_vt_t *myvt = session->sess_vt;
	/* unlink from vt: */

	if(myvt) {
		session->sess_next->sess_prev = session->sess_prev;
		session->sess_prev->sess_next = session->sess_next;

		if(myvt->vt_master == session) {
			myvt->vt_master = session->sess_next;
			if(session->sess_next == session) {
				myvt->vt_master = NULL;
				release_vt(myvt);
			} else {
				/* New master.. perform actions */
			}
			kfree_s(session, sizeof(*session));
		}
	}

	/* one user less: */
	MOD_DEC_USE_COUNT;
	fb_active_sessions--;
	if (fb_active_sessions == 1) {
		int retval;

		retval = unregister_vt_handler(&fb_vt_hand);
		if (!retval) {
			MOD_DEC_USE_COUNT;
			fb_active_sessions--;
		} else if (retval != EAGAIN) {
			printk(KERN_CRIT "%s: Aiee.. unable to unregister vt_handler",
				FB_MODULE_NAME);
		}
		/* release an orphaned panic buffer: */
		if (fb_flags.have_panic_buff) {
			if (fb_flags.panic_is_vmall)
				vfree(panic_save_away);
			else	kfree_s(panic_save_away, video_mem_term - video_mem_base);
			fb_flags.have_panic_buff = 0;
			panic_save_away = NULL;
		}
	}
}

/*****************************************************************************
 * This gets called on a page fault with !PAGE_PRESENT but entry != 0.
 * Unfortunately the address where the page fault happened is not passed
 * to *swapin.
 *
 * This is no longer true. Thus we use this info now happily - Michael
 *****************************************************************************/
static pte_t
fb_vm_ops_swapin(struct vm_area_struct * vma, unsigned long address, unsigned long entry)
/*****************************************************************************/
{
	int new_bank, new_page;
	struct chip_bank_desc *bank_desc;
	pte_t new_entry, *pte_ptr;

#ifdef VERBOSE_DEB
DEB(printk(KERN_DEBUG "%s: fb_vm_ops_swapin: address=%lx, entry=%lx\n",FB_MODULE_NAME,
	address, entry);)
DEB(printk(KERN_DEBUG "%s: fb_vm_ops_swapin: vma->vm_offset = %lx\n", FB_MODULE_NAME,
	vma->vm_offset);)
#endif

	/* Adjust to page boundary: */
	address &= PAGE_MASK;

	/*
	 * Use the banking window that was banked least recently (depends on what
	 * the driver does)
	 */
	new_bank = chip_choose_bank((int) vma->vm_flags, address);
	bank_desc = chip_banks + new_bank;

	/*
	 * Note that the happening of a page fault means that the accessed
	 * memory is not visible in any of the two banks,
	 * so no need to test for such a case.
	 *
	 * As we support multiple mmap areas in a single process this is nolonger true - Michael.
	 */

	/*
	 * We check if we need to change the page, and if so remove any yet existing mapping.
	 */

	new_page = address >> bank_desc[0].shift;
	if(new_page != bank_desc[0].value) {
		unmap_bank(bank_desc);
	}

	/*
	 * If we need to use the next bank also and it had a different setting yet
	 * unmap it as well:
	 */

	if((new_bank & NEEDS_NEXT_BANK) && (new_page != bank_desc[1].value))
		unmap_bank(bank_desc + 1);

	/*
	 * This does the actual mapping:
	 */

	if(new_bank & NEEDS_NEXT_BANK)
		(bank_desc -> set_both_banks)(new_page);
	else	(bank_desc -> set_bank)(new_page);

	/*
	 * The page where the page fault occurred must not be changed, this
	 * must be done by the calling function (silly eh?)
	 */

	/* First reconstruct the faulting linear address: */
	address = (address + vma->vm_start - vma->vm_offset) & PAGE_MASK;
	pte_ptr = pte_offset( pmd_offset( pgd_offset(current, address), address), address);

	/* Record the address of the pte: */
	bank_desc -> next_map -> faulting_pte = pte_ptr;

	/* set paging tables */
	make_new_map(bank_desc, vma, new_bank & NEEDS_NEXT_BANK);

	/* restore original entry: */
	new_entry = *pte_ptr;
	pte_val(*pte_ptr) = entry;

	/*
	 * Set the entry at the faulting position to old, if everything works ok the
         * processor will make it young when restarting the offending operation.
	 */
	new_entry = pte_mkold(new_entry);
	
	/*
	 * Well.. we could leave that away.. it is done later (but only for *pte_ptr,
	 * but then in intel i386 it does invalidate all pagetables anyway.. however
	 * we leave it in for later compatibility to other machines.
	 */

	invalidate();

	/* Undo things done later in memory.c: */
	--vma->vm_task->mm->maj_flt;
	--vma->vm_task->mm->rss;
	/* and do the right thing: */
	++vma->vm_task->mm->min_flt;

#ifdef VERBOSE_DEB
	DEB(printk(KERN_DEBUG "%s: leaving fb_vm_ops_swapin: new_entry=%lx old_entry=%lx\n",
		FB_MODULE_NAME, pte_val(new_entry), pte_val(*pte_ptr)););
#endif VERBOSE_DEB

	return new_entry;
}

/*****************************************************************************/
static void
fb_vm_ops_unmap(struct vm_area_struct *area, unsigned long address, size_t length)
/*****************************************************************************/
{
DEB(printk(KERN_DEBUG "%s: fb_vm_ops_unmap: addr=%x, len=%x, a:%x, e:%x, o:%x\n",
	FB_MODULE_NAME, (unsigned)address, (unsigned)length,
			(unsigned)area->vm_start, (unsigned)area->vm_end,
			(unsigned)area->vm_offset);)

	/*
	 * Release any mapped VGA pages in the affected area:
         * (We could split the existing mappings, but i want keep it simple first..)
	 */

	unmap_area(address, address + length);

	/*
	 * Invalidate affected pagetable entries.
	 */

	map_page_range(current, address, length, __pte(0));
}

/*****************************************************************************/
static void
fb_vm_ops_open(struct vm_area_struct *area)
/*****************************************************************************/
{
	struct fb_sess_t *mysess = ((struct fb_vm_operations *)area->vm_ops)->fb_vm_sess;
	struct vm_area_struct *area_ptr;

	ENTERING;
	DEB(printk(KERN_DEBUG "area: %08x\n", (unsigned)area);)
	DEB(printk(KERN_DEBUG "sess: %08x\n", (unsigned)mysess);)

	/* link into session's vm_area which is sorted by offset values */
	area_ptr = mysess->sess_vm_first;
	if (!area_ptr) {
		mysess->sess_vm_first = area;
		area->vm_next_share = area->vm_prev_share = area;
		goto end;
		return;
	}
	
	do {
		if (area_ptr->vm_offset > area->vm_offset) {
			if (area_ptr == mysess->sess_vm_first)
				mysess->sess_vm_first = area;
			break;
		}
		area_ptr = area_ptr->vm_next_share;
	} while(area_ptr != mysess->sess_vm_first);
	/* Insert b4 area_ptr */
	area->vm_next_share = area_ptr;
	area->vm_prev_share = area_ptr->vm_prev_share;
	area->vm_prev_share->vm_next_share = area;
	area_ptr->vm_prev_share = area;

	end:
	DEB(
		area_ptr = mysess->sess_vm_first;
		do {
			printk(KERN_DEBUG
			 "a: %08x p: %08x, n: %08x, o:%08x\n",
			 (unsigned)area_ptr, (unsigned)area_ptr->vm_prev_share,
			 (unsigned)area_ptr->vm_next_share, (unsigned)area_ptr->vm_offset);
			area_ptr = area_ptr->vm_next_share;
		} while(area_ptr != mysess->sess_vm_first); )
			
	LEAVING;
}

/*****************************************************************************/
static void
fb_vm_ops_close(struct vm_area_struct * area)
/*****************************************************************************/
{
	struct fb_sess_t *mysess = ((struct fb_vm_operations *)area->vm_ops)->fb_vm_sess;

	ENTERING;
	DEB(printk(KERN_DEBUG "area: %08x\n", (unsigned)area);)
	DEB(printk(KERN_DEBUG "prev: %08x, next: %08x\n", (unsigned)area->vm_prev_share,
						(unsigned)area->vm_next_share);)
	DEB(printk(KERN_DEBUG "sessptr b4: %08x\n", (unsigned)mysess->sess_vm_first);)

	/*
	 * This is usually either not required or already done by fb_vm_ops_unmap.
	 * However when unmapping due to process exit, fb_vm_ops_unmap is not called. Thus we have
	 * to live with the performance hit even in the other cases.
	 */
	unmap_area(area->vm_start, area->vm_end);

	/* Unlink from session: */
	area->vm_next_share->vm_prev_share = area->vm_prev_share;
	area->vm_prev_share->vm_next_share = area->vm_next_share;
	if (mysess->sess_vm_first == area) {
		if(area->vm_next_share == area) {
			/* We were the last .. */
			mysess->sess_vm_first = NULL;
			if (mysess->sess_flags.orphaned)
				release_session(mysess);
		} else
			mysess->sess_vm_first = area->vm_next_share;
	}

	DEB(printk(KERN_DEBUG "sessptr after: %08x\n", (unsigned)mysess->sess_vm_first);)

	LEAVING;
}

static struct vm_operations_struct default_fb_vm_ops = {
	fb_vm_ops_open,		/* open */
	fb_vm_ops_close,	/* close */
	fb_vm_ops_unmap,	/* unmap */
	NULL,			/* protect */
	NULL,			/* sync */
	NULL,			/* advise */
	NULL,			/* nopage */
	NULL,			/* wppage */
	NULL,			/* swapout */
	fb_vm_ops_swapin,	/* swapin */
};

static int
get_vcnum(dev_t ttydev) {
	int vcnum;

	if (MAJOR(ttydev) != TTY_MAJOR)
			return -1;
	vcnum = MINOR(ttydev);
	if ((vcnum < 1) || (vcnum > MAX_NR_CONSOLES ))
		return -1;
	return vcnum - 1;
}

static int
get_vt_struct(struct fb_sess_t *session, int vcnum, int exclusive) {
	struct fb_vt_t *vtptr;

#ifndef NDEBUG
	if ((!session) || (session->sess_vt))
		printk(KERN_CRIT "fb: get_vt_struct on illegal session %08x, vt: %08x\n",
			(unsigned)session, session ? 0 : (unsigned)(session->sess_vt));
#endif
	if (fb_vt_list) {
		vtptr = fb_vt_list;
		do {
			if (vtptr->vt_num == vcnum) {
				if (exclusive || vtptr->vt_flags.dedicated)
					return -EBUSY;
				session->sess_next = vtptr->vt_master;
				session->sess_prev = vtptr->vt_master->sess_prev;
				session->sess_prev->sess_next = session;
				vtptr->vt_master->sess_prev = session;
				return 1;
			}
			vtptr = vtptr->vt_next;
		} while(vtptr != fb_vt_list);
	}

	if (vt_cons[vcnum]) {
		if ((vt_cons[vcnum] -> vc_mode != KD_TEXT) ||
			(vt_cons[vcnum] -> vt_mode.mode != VT_AUTO))
				return -EBUSY; /* Oldstyle console graphix running  */
	}
#ifndef NDEBUG
	else {
		printk(KERN_DEBUG "fb: Yuck... fb_open on unallocated console.\n");
	}
#endif

	vtptr = kmalloc(sizeof(struct fb_vt_t), GFP_KERNEL);
	if (!vtptr)
		return -ENOMEM;
	vtptr->vt_num = vcnum;
	vtptr->vt_flags.graphmode = 0;
	vtptr->vt_flags.active = 0;
	vtptr->vt_flags.dedicated = 0;
	/* need a way to fine tune the next two from the outside.. later with ONOCTTY and
           new ioctl. */
	vtptr->vt_flags.dont_care4screen = 0;
	vtptr->vt_flags.alloc_on_the_fly = 0;

	if ( ! ( vtptr->vt_flags.dont_care4screen || vtptr->vt_flags.alloc_on_the_fly ||
			fb_flags.have_panic_buff) )
		{
		/* get a panic buffer */

		void *tmpbuf;
		int flags, releasebuf = 0;

		tmpbuf = vmalloc(video_mem_term - video_mem_base);
		
		/* Avoid races.. */
		save_flags(flags);
		cli();
		if (tmpbuf && !fb_flags.have_panic_buff) {
			fb_flags.have_panic_buff = 1;
			fb_flags.panic_is_vmall = 1;
			panic_save_away = tmpbuf;
			tmpbuf = NULL;
		} else {
			/* Oops.. someone allocate a buffer behind our back */
			releasebuf = 1;
		}
		restore_flags(flags);

		if (releasebuf && tmpbuf)
			vfree(tmpbuf);
		/* deallocation is not done in irq! Thus no race */
		if (!fb_flags.have_panic_buff) {
			kfree_s(vtptr, sizeof(struct fb_vt_t));
			return -ENOMEM;
		}
	}

	/* Atomically register and check if someone uses this console already */
	if (vt_handler_handle_vc(vcnum, VT_HANDLER_VIDEO, VT_HANDLER_ACQ)) {
		/* Tsk.. tsk.. busy */
		kfree_s(vtptr, sizeof(struct fb_vt_t));
		/* A panic buffer may remain.. it is taken care of elsewhere.. */
		return -EBUSY;
	}
	if (fb_vt_list) {
		vtptr->vt_next = fb_vt_list;
		vtptr->vt_prev = fb_vt_list->vt_prev;
		vtptr->vt_next->vt_prev = vtptr;
		vtptr->vt_prev->vt_next = vtptr;
	} else {
		fb_vt_list = vtptr->vt_next = vtptr->vt_prev = vtptr;
	}
	session->sess_vt = vtptr;
	vtptr->vt_master = session->sess_next = session->sess_prev = session;

	return 0;
}

static int
fb_check_ready(struct fb_sess_t *sess) {
	if (!(sess -> sess_vt))
		return -ENOTTY;
	return 0;
}

/*****************************************************************************/
static int
fb_open( struct inode* ino, struct file* filep)
/*****************************************************************************/
{
	struct fb_sess_t *new_sess;
	int error_code = -EIO;

	DOING;
	DEB(printk(KERN_DEBUG "fb_open: flags=0x%x\n", (unsigned)filep->f_flags));

	new_sess = kmalloc(sizeof(struct fb_sess_t), GFP_KERNEL);
	if (!new_sess)
		return -ENOMEM;

	new_sess->sess_vt = NULL;

	new_sess->sess_ops = kmalloc(sizeof(struct fb_vm_operations), GFP_KERNEL);
	if (!new_sess->sess_ops) {
		error_code = -ENOMEM;
	    release_mem_on_err:
		if (new_sess->sess_vt && !new_sess->sess_vt->vt_master)
			release_vt(new_sess->sess_vt);
		if(new_sess->sess_ops)
			kfree_s(new_sess->sess_ops, sizeof(struct fb_vm_operations));
		kfree_s(new_sess, sizeof(struct fb_sess_t));
		return error_code;
	}

	if (!(filep->f_flags & O_NOCTTY)) {
		if (current->tty) {
			int vcnum;

			vcnum = get_vcnum(current->tty->device);
			if(vcnum >= 0) {
				error_code = get_vt_struct(new_sess, vcnum, 1);
				if (error_code < 0)
					goto release_mem_on_err;
			}
		}

		if (!new_sess->sess_vt) {
			error_code = -ENOTTY;
			goto release_mem_on_err;
		}
	}

	/* MOD_INC_USE_COUNT makes sure that the driver memory is not freed */
	/* while the device is in use. */
	if (!fb_active_sessions) {
		int retval;

		retval = register_vt_handler(&fb_vt_hand);
		if (!retval) {
			MOD_INC_USE_COUNT;
			fb_active_sessions++;
		} else if (retval == EAGAIN) {
			release_vt(new_sess->sess_vt);
			error_code = -EAGAIN;
			goto release_mem_on_err;
		} else {
			printk(KERN_CRIT "%s: Aiee.. unable to register vt_handler",
				FB_MODULE_NAME);
		}
	}
	MOD_INC_USE_COUNT;
	fb_active_sessions++;
	
	new_sess->sess_vm_first = NULL;
	new_sess->sess_ops->fb_vm_ops = default_fb_vm_ops;
	new_sess->sess_ops->fb_vm_sess = new_sess;
	new_sess->sess_flags.orphaned = 0;

	filep->private_data = new_sess;

	return 0;
}

/*****************************************************************************/
static void
fb_close( struct inode* ino, struct file* filep)
/*****************************************************************************/
{
	struct fb_sess_t *mysess = filep->private_data;

	DOING;

	mysess->sess_flags.orphaned = 1;

	if (!mysess->sess_vm_first)
		release_session(mysess);
}

/*****************************************************************************/
static int
fb_mmap( struct inode * inode, struct file * filep, struct vm_area_struct * vma)
/*****************************************************************************/
{
	struct fb_sess_t *mysess = filep->private_data;
	unsigned long size;
	
DEB(printk(KERN_DEBUG "%s: fb_mmap: vma->vm_start=%lx, vma->vm_end=%lx.\n",FB_MODULE_NAME,vma->vm_start,vma->vm_end);)

	/*
	 * Correct page alignment was checked by caller..
	 * .. almost:
	 */
	if (vma -> vm_offset & ~(PAGE_MASK))
		return -EINVAL;

	CHECK_VT_AND_RETURN_ON_ERR(mysess);

	size = vma->vm_end - vma->vm_start;

	/*
	 * Fire up vga, if not already done.
	 */
	fb_graphics_on(mysess->sess_vt);

	/*
	 * Let the kernel zeromap the area. This way we don't have to worry about not
         * present page tables etc.
	 */

	if (zeromap_page_range(vma->vm_start, size, __pgprot(0) ))
		return -EAGAIN;

	/*
	 * This was a nice trick.. however the swapper will get annoyed from the
	 * faked swapped out null spaces.. so we fake a swapped out reserved
         * page instead:
	 * This will be done in the open call below.
	 */

	DEB(printk(KERN_DEBUG "%s: fb_mmap: vma->vm_page_prot = %lx\n", FB_MODULE_NAME,
			pgprot_val(vma->vm_page_prot)));
	DEB(printk(KERN_DEBUG "%s: fb_mmap: vma->vm_flags = %x\n", FB_MODULE_NAME,
			vma->vm_flags));

	/* TODO: support read/write only mappings. */
	vma->vm_page_prot = PAGE_SHARED;

	/*
	 * We do NOT! provide an inode. This allows us to misuse the vm_next_share,
	 * vm_prev_share function pointers for our own housekeeping. Since they
	 * are in the vma they are always present and allow us to always find info
	 * about the current virtual screen and framebuffer. The inode would not
	 * allow us to find this info as there may be more then one device node
	 * pointing to the same device and actually different virtual screens
	 * may be realized via the same inode.
	 *
	 * We do set the VM_SHM bit. It will ensure that the kernel merges only vma
	 * mappings with adjacent offsets on the screen even w/o the inode set.
	 */
	
	vma->vm_flags |= VM_SHM;

	vma->vm_ops = &(mysess->sess_ops->fb_vm_ops);

	/* link into area list */
	vma->vm_ops->open(vma);
	/* Insert faked paged out page, s.t. we got notified if they are accessed. */
	map_page_range(current, vma->vm_start,
	  			vma->vm_end - vma->vm_start,
				empty_pte);
	invalidate();

	return 0;
}

/*****************************************************************************/
static int
fb_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		unsigned long arg)
/*****************************************************************************/
{
	int ret;
	unsigned long size;

#define VERIFY_AREA(_cmd, _arg, _size, _ret)				\
	if (_cmd & IOC_IN) {						\
		_ret = verify_area(VERIFY_READ, (void *) _arg, _size);	\
		if (_ret)						\
			return _ret;					\
	}								\
	if (_cmd & IOC_OUT) {						\
		_ret = verify_area(VERIFY_WRITE, (void *) _arg, _size);	\
		if (_ret)						\
			return _ret;					\
	}

	size = (cmd & IOCSIZE_MASK) >> IOCSIZE_SHIFT;

	switch (cmd) {
		case FB_GETINFO:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_GETINFO).\n",FB_MODULE_NAME);)
			VERIFY_AREA(cmd, arg, size, ret);
			memcpy_tofs((void *)arg, &chip_info, sizeof(chip_info));
			return 0;
		case FB_MODES_GET_INFO:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_MODES_GET_INFO).\n",FB_MODULE_NAME);)
			VERIFY_AREA(cmd, arg, size, ret);
			memcpy_tofs((void *)arg, chip_modesinfo,
				chip_info.number_modes * sizeof(fb_mode_t));
			return 0;
		case FB_MODE_SET:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_MODE_SET).\n",FB_MODULE_NAME);)
			VERIFY_AREA(cmd, arg, size, ret);
			ret = get_user_long((void *)arg);
			if(ret >= chip_info.number_modes)
				return -EINVAL;
			chip_mode = ret;
			return 0;
		case FB_SET_ORIGIN:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_SET_ORIGIN).\n",FB_MODULE_NAME);)
			VERIFY_AREA(cmd, arg, size, ret);
			return chip_setorigin((fb_origin_t *)arg);
		case FB_SET_COLORMAP:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_SET_COLORMAP).\n",FB_MODULE_NAME);)
			VERIFY_AREA(cmd, arg, size, ret);
			return chip_setcolormap((fb_cmap_t *)arg);
		case FB_GET_COLORMAP:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_GET_COLORMAP).\n",FB_MODULE_NAME);)
			VERIFY_AREA(cmd, arg, size, ret);
			return chip_getcolormap((fb_cmap_t *)arg);
		/* no args */
		case FB_BLANK:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_BLANK).\n",FB_MODULE_NAME);)
			return chip_blank();
		case FB_UNBLANK:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_UNBLANK).\n",FB_MODULE_NAME);)
			return chip_unblank();
		/* invalid ioctl */
		case FB_TEST_TEST:
DEB(printk(KERN_DEBUG "%s: fb_ioctl(FB_TEST_TEST).\n",FB_MODULE_NAME);)
			chip_test_test();
			return 0;
		default: ;
DEB(printk(KERN_DEBUG "%s: fb_ioctl(--invalid--).\n",FB_MODULE_NAME);)
			return -ENOIOCTLCMD;
	}

#undef VERIFY_AREA

}

/*****************************************************************************/

static struct file_operations fb_fops = {
	NULL,		/* lseek */
	NULL,		/* read */
	NULL,		/* write */
	NULL,		/* readdir */
	NULL,		/* select */
	fb_ioctl,	/* ioctl */
	fb_mmap,	/* mmap */
	fb_open,	/* open */
	fb_close,	/* release */
	NULL,		/* fsync */
	NULL,		/* fasync */
	NULL,		/* check_media_change */
	NULL		/* revalidate */
};

/*
 * And now the modules code and kernel interface.
 */
int
init_module( void)
{
	unsigned char *chipset_name;
	int chip_state_size, retval;

	/* test if svga with compiled in chipset is present */
	if (NULL==(chipset_name=chip_test_chipset()))
		return -ENODEV;

	printk(KERN_INFO "%s: %s detected.\n",FB_MODULE_NAME,chipset_name);
	chip_state_size = chip_savestate_getsize();

	fb_savestate_buf = kmalloc(chip_state_size, GFP_KERNEL);

	/* We use vmalloc instead of kmalloc or a static array coz these would require
	   continuous pages which we don't really need. */

	fb_fontbuffer = vmalloc(FB_FONT_BUFFER_SIZE);

	if ((!fb_savestate_buf) || (!fb_fontbuffer)) {
		printk(KERN_ERR "%s: no memory for buffers.\n",FB_MODULE_NAME);
		retval = -ENOMEM;
	    free_buffers:
		if (fb_savestate_buf)
			kfree_s(fb_savestate_buf, chip_state_size);
		fb_savestate_buf = NULL;
		if (fb_fontbuffer)
			vfree(fb_fontbuffer);
		fb_fontbuffer = NULL;
		return retval;
	}

	if (register_chrdev(fb_module_major, FB_MODULE_NAME, &fb_fops)) {
		printk(KERN_ERR "%s: register_chrdev failed.\n",FB_MODULE_NAME);

		retval = -EBUSY;
	    	goto free_buffers;
	}

	/* let the chipset code set some chip_* variables */
	if ((retval = chip_init())) {
		unregister_chrdev(fb_module_major, FB_MODULE_NAME);
		goto free_buffers;
	}

	printk(KERN_INFO "%s: driver installed.\n",FB_MODULE_NAME);

	return 0;
}

void
cleanup_module(void)
{
	/* as rmmod takes care of the module being in use,
	   this won't be called while in use */

	if (unregister_chrdev(fb_module_major, FB_MODULE_NAME))
		printk(KERN_CRIT "%s: Beware! Unable to deregister chrdev.\n", FB_MODULE_NAME);

	if (fb_savestate_buf)
		kfree(fb_savestate_buf); /* Alas... we forgot the size of fb_savestate_buf */
	if (fb_fontbuffer)
		vfree(fb_fontbuffer);
	fb_savestate_buf = NULL;
	fb_fontbuffer = NULL;

	/* Give the chipset code a chance to remove any buffers */
	chip_cleanup();
	printk(KERN_INFO "%s: removed.\n",FB_MODULE_NAME);
}

static inline struct fb_vt_t *find_vt(int vc_num) {
	struct fb_vt_t *ptr;

	ptr = fb_vt_list;
	if (!ptr)
		return NULL;
	do	{
		if (ptr->vt_num == vc_num)
			return ptr;
		ptr = ptr->vt_next;
	} while (ptr != fb_vt_list);
	return NULL;
}

static int fb_entervt(int curr_vc, int dest_vc, int tid) {
	struct fb_vt_t *dest_vt;

	dest_vt = find_vt(dest_vc);
	if (dest_vt)
		printk(KERN_DEBUG "fb_entervt[%d]: %d -> %d\n", tid, curr_vc, dest_vc);
	
	return 0;
}

static int fb_leavevt(int curr_vc, int dest_vc, int tid) {
	struct fb_vt_t *curr_vt;

	curr_vt = find_vt(curr_vc);
	if (!curr_vt || !(curr_vt->vt_flags.active))
		return 0;
	printk(KERN_DEBUG "fb_leavevt[%d]: %d -> %d\n", tid, curr_vc, dest_vc);

	curr_vt->vt_flags.active = 0;
	if (fb_flags.mmaping)
		fb_flags.unmap_pending = 1;
	return 0;
}

static int fb_prep_tty_ioctl(struct tty_struct *tty, struct file *file,
				int perm, unsigned int cmd, unsigned long arg) {
	DEB(printk(KERN_DEBUG "fb: Catched tty ioctl %04x!\n", cmd));
	return 0;
}

static void fb_did_tty_ioctl(struct tty_struct *tty, struct file *file,
				int perm, unsigned int cmd, unsigned long arg) {
	switch(cmd){
		case PIO_FONT:
			DEB(printk(KERN_DEBUG "fb: Intercepted succesful set font\n"));
			memcpy_fromfs(fb_fontbuffer, (void *)arg, FB_FONT_BUFFER_SIZE);
			break;
	}
}
