/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *      Copyright (C) 1995 Michael Weller (eowmob@exp-math.uni-essen.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Michael Weller (eowmob@exp-math.uni-essen.de or
 * mat42b@aixrs1.hrz.uni-essen.de) Heiderhoefen 116b,
 * D 46049 Oberhausen, Germany.
 */

#include <linux/autoconf.h>
#include <linux/module.h>

/*
 *
 * MICHAEL WELLER DISCLAIMS ALL WARRANTIES WITH REGARD
 * TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL MICHAEL WELLER BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/* Works only if Mach32 onboard VGA is enabled.                    */

/* This is a very simple and silly driver. It will use the first */
/* Mode defined in the EEPROM, or the highest VSYNC standard 1024x768 */
/* mode allowed by the EEPROM. We use the VGA aperture only as we want */
/* to learn about paging first b4 addressing high performance */

#define PIX_ALIGN 64		/* Align scanlinelength by this value for some modes */

#define LIN_MAPS 10
#define VGA_MAPS 5

#include <linux/types.h>        /* NULL */
#include <linux/kernel.h>       /* printk() */
#include <linux/errno.h>	/* put_fs* */
#include <linux/malloc.h>	/* kmalloc + friends */
#include <linux/mm.h>		/* GFP_KERNEL */
#include <asm/io.h>		/* io inlines */
#include <asm/segment.h>	/* PAGESIZE */
#include "fb.h"
#include "chipset.h"
#include "mach32.h"

/* #define DEBUG */

#ifdef DEBUG
#define DEB(__a) __a
#else
#define DEB(__a)
#endif

#include "fb_debug.h"

/* Exported variables: */
unsigned long chip_mode;

void chip_do_bank0r(int nr);
void chip_do_bank0w(int nr);
void chip_do_bank0rw(int nr);

/* We rely on these being inited with zeros  */
static struct mapping_desc maps_0r[VGA_MAPS], maps_0w[VGA_MAPS], maps_lin[LIN_MAPS];

struct chip_bank_desc chip_banks[3] = {
	/* VGA bank 0, read only, */
	{0xA0000UL, (64 * 1024) / PAGE_SIZE, 16, 0, chip_do_bank0r, chip_do_bank0rw,
		maps_0r, maps_0r + VGA_MAPS, maps_0r},
	/* VGA bank 0, write only, */
	{0xA0000UL, (64 * 1024) / PAGE_SIZE, 16, 0, chip_do_bank0w, NULL,
		maps_0w, maps_0w + VGA_MAPS, maps_0w},
	/* linear frame buffer (NOT YET) */
	{0, 0, 0, 0, NULL, NULL,
		maps_lin, maps_lin + LIN_MAPS, maps_lin},
	};

short chip_num_banks = 3;

/* Some are set dynamically */
fb_info_t chip_info = {
driver_name:    "mach32",
chipset_name:   "mach32",
copyright:      "(c) 1994 Michael Weller",
info:           "ATI Mach32",
rev_major:      0,
rev_minor:      1,
rev_tiny:       1,
totalmem:       0,
memtype:        0, /* Set later */  /* 0: DRAM, 1: VRAM */
number_modes:   0, /* Set later */
default_mode:   0,
};

/* A few tables of register to be saved.. */

/* ATI-EXT regs to save (ensure lock regs are set latest, that is they are listed first here!): */
#define MACH32_NUM_LOCK 4 /* 4 Lock registers */
static const unsigned char mach32_eregs[]={
        /* Lock regs: */
        0x38,0x34,0x2e,0x2b,
        /* All other extended regs. */
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,
        0x20,0x23,0x24,0x25,0x26,0x27,0x2c,
        0x2d,0x30,0x31,0x32,0x33,0x35,0x36,
        0x37,0x39,0x3a,0x3b,0x3c,0x3d,0x3e,
        0x3f };

/* Mach32 regs to save.. read, write address alternating: */
static const unsigned short mach32_acc_regs[]={
        0xB2EE, 0x06E8,         /* H_DISP(ALT H_TOTAL) */
        0xC2EE, 0x12E8,         /* V_TOTAL */
        0xC6EE, 0x16E8,         /* V_DISP */
        0xCAEE, 0x1AE8,         /* V_SYNC_STRT */
        0xD2EE, 0x1EE8,         /* V_SYNC_WID */
        0x4AEE, 0x4AEE,         /* CLOCK_SEL */
        0x96EE, 0x96EE,         /* BRES_COUNT */
        0x86E8, 0x86E8,         /* CUR_X */
        0x82E8, 0x82E8,         /* CUR_Y */
        0x22EE, 0x22EE,         /* DAC_CONT(PCI) */
        0xF2EE, 0xF2EE,         /* DEST_COLOR_CMP_MASK */
        0x92E8, 0x92E8,         /* ERR_TERM */
        0xA2EE, 0xA2EE,         /* LINEDRAW_OPT */
        0x32EE, 0x32EE,         /* LOCAL_CNTL */
        0x6AEE, 0x6AEE,         /* MAX_WAITSTATES / MISC_CONT(PCI) */
        0x36EE, 0x36EE,         /* MISC_OPTIONS */
        0x82EE, 0x82EE,         /* PATT_DATA_INDEX */
        0x8EEE, 0x7AEE,         /* EXT_GE_CONFIG */
        0xB6EE, 0x0AE8,         /* H_SYNC_STRT */
        0xBAEE, 0x0EE8,         /* H_SYNC_WID */
        0x92EE, 0x7EEE,         /* MISC_CNTL */
        0xDAEE, 0x8EE8,         /* SRC_X */
        0xDEEE, 0x8AE8,         /* SRC_Y */
        0x52EE, 0x52EE,         /* SCRATCH0 */
        0x56EE, 0x56EE,         /* SCRATCH1 */
        0x42EE, 0x42EE,         /* MEM_BNDRY */
        0x5EEE, 0x5EEE,         /* MEM_CFG */
        };

static const mode_entry predef_modes[]={
/* 640x 480*/   {0x9, 0xe, 0x4f, 0x63, 0x2c, 0x52, 0x418, 0x3bf, 0x3d6, 0x23, 0x22, 0x50,
			0, 0x0000, 0},
                {0x9, 0xe, 0x4f, 0x69, 0x25, 0x52, 0x40b, 0x3bf, 0x3d0, 0x23, 0x23, 0x24,
			0, 0x0001, 7},
/* 800x 600*/   {0x9, 0xe, 0x63, 0x84, 0x10, 0x6e, 0x580, 0x4ab, 0x4c2, 0x33, 0x2c, 0x7c,
			0, 0x003f, 8},
                {0x9, 0xe, 0x63, 0x84, 0x10, 0x6d, 0x580, 0x4ab, 0x4c2, 0x33, 0x0c, 0x0c,
			0, 0x003d, 8},
                {0x9, 0xe, 0x63, 0x7f, 0x09, 0x66, 0x4e0, 0x4ab, 0x4b0, 0x23, 0x02, 0x0c,
			0, 0x003c, 8},
                {0x9, 0xe, 0x63, 0x83, 0x10, 0x68, 0x4e3, 0x4ab, 0x4b3, 0x23, 0x04, 0x30,
			0, 0x0038, 8},
                {0x9, 0xe, 0x63, 0x7d, 0x12, 0x64, 0x4f3, 0x4ab, 0x4c0, 0x23, 0x2c, 0x1c,
			0, 0x0030, 8},
                {0x9, 0xe, 0x63, 0x82, 0x0f, 0x6a, 0x531, 0x4ab, 0x4f8, 0x23, 0x06, 0x10,
			0, 0x0020, 8},
/*1024x 768*/   {0xd, 0xe, 0x7f, 0x9d, 0x16, 0x81, 0x668, 0x5ff, 0x600, 0x33, 0x08, 0x1c,
			0, 0x0001, 9},
                {0xd, 0xe, 0x7f, 0xa7, 0x31, 0x82, 0x649, 0x5ff, 0x602, 0x23, 0x26, 0x3c,
			0, 0x0003, 9},
                {0xd, 0xe, 0x7f, 0xad, 0x16, 0x85, 0x65b, 0x5ff, 0x60b, 0x23, 0x04, 0x38,
			0, 0x0013, 9},
                {0xd, 0xe, 0x7f, 0xa5, 0x31, 0x83, 0x649, 0x5ff, 0x602, 0x23, 0x26, 0x38,
			0, 0x0017, 9},
                {0xd, 0xe, 0x7f, 0xa0, 0x31, 0x82, 0x649, 0x5ff, 0x602, 0x23, 0x26, 0x38,
			0, 0x001f, 9},
/*1280x1024*/   {0xe, 0xe, 0x9f, 0xc7, 0x0a, 0xa9, 0x8f8, 0x7ff, 0x861, 0x33, 0x0a, 0x2c,
			0, 0x0001, 10},
                {0xe, 0xe, 0x9f, 0xc7, 0x0a, 0xa9, 0x838, 0x7ff, 0x811, 0x33, 0x0a, 0x2c,
			0, 0x0003, 10},
                };

/* global state: */
static status_t mach32_state;

/* real globals, for all open fb's */
static unsigned short *mach32_eeprom = NULL;
static const mode_entry **mode_timings = NULL;
fb_mode_t *chip_modesinfo;


static void mach32_unlock(void)
{
unsigned short oldval;

ENTERING;

OUTB(ATIPORT,ATISEL(0x2e));
oldval=INB(ATIPORT+1)&~0x10;  /* Unlock CPUCLK Select */
OUTW(ATIPORT,ATISEL(0x2e)|(oldval<<8));

OUTB(ATIPORT,ATISEL(0x2b));
oldval=INB(ATIPORT+1)&~0x18;  /* Unlock DAC, Dbl Scan. */
OUTW(ATIPORT,ATISEL(0x2b)|(oldval<<8));

OUTB(ATIPORT,ATISEL(0x34));
oldval=INB(ATIPORT+1)&~0xfc; /* Unlock Crt9[0:4,7], Vtiming, Cursr start/end,
                                CRT0-7,8[0-6],CRT14[0-4]. but disable ignore of CRT11[7] */
OUTW(ATIPORT,ATISEL(0x34)|(oldval<<8));

OUTB(ATIPORT,ATISEL(0x38)); /* Unlock ATTR00-0f, ATTR11, whole vga, port 3c2 */
oldval=INB(ATIPORT+1)&~0x0f;
OUTW(ATIPORT,ATISEL(0x38)|(oldval<<8));

/* Unlock vga-memory too:*/

OUTW(MEM_BNDRY,0);

/* Unlock Mach32 CRT Shadowregisters... this one made me crazy...
   Thanx to the Xfree sources I finally figured it out....*/
OUTW(SHADOW_SET, 1);
OUTW(SHADOW_CTL, 0);
OUTW(SHADOW_SET, 2);
OUTW(SHADOW_CTL, 0);
OUTW(SHADOW_SET, 0);

LEAVING;
}

static int chip_saveregs(struct chip_regs_s *regs)
{
int i, retval;

ENTERING;
/*mach32_bltwait();       Ensure noone draws in the screen */

if ((retval = vga_saveregs(&regs->vga_regs)) < 0)
	return retval;

for(i = 0; i < sizeof(mach32_eregs); i++)
        {
        OUTB(ATIPORT, ATISEL(mach32_eregs[i]));
        regs->ati_ext_regs[i] = INB(ATIPORT + 1);
        }
mach32_state.mach32_ati2e = ATIIN(0x2e) & 0xf0;
regs->driverstat = mach32_state;
retval += i + 1;
for(i = 0; i < (sizeof(mach32_acc_regs) / (2 * sizeof(unsigned short))); i++)
        {
        regs->mach32_regs[i] = INW(mach32_acc_regs[i << 1]);
        }
retval += (i << 1);
/* NO DAC SUPPORT YET */
/*retval = mach32_sav_dac(retval, regs);*/
LEAVING;
return retval;
}

static int chip_restoreregs(struct chip_regs_s *regs)
{
int i,retval;

ENTERING;
mach32_unlock(); /* Actually not needed */

/*mach32_bltwait();       Ensure noone draws in the screen */
/*mach32_accelstate = R_UNKNOWN;    Accel registers need to be reset */
/* complex DAC handling removed */

for(i = (sizeof(mach32_acc_regs) / (2 * sizeof(unsigned short))) - 1; i >= 0; i--)
        {
        /* restore only appage in MEM_CFG... otherwise badly interaction
           with X that may change MEM_CFG and does not only not restore it's
           original value but also insist on it not being changed on VC
           change... =:-o */
        if(mach32_acc_regs[i >> 1] == MEM_CFG)
                OUTW(MEM_CFG, (INW(MEM_CFG) & ~0xc) | ((regs->mach32_regs[i]) & 0xc));
	else	OUTW(mach32_acc_regs[i >> 1], regs->mach32_regs[i]);
        }
mach32_state = regs->driverstat;
OUTB(DISP_CNTL, mach32_state.mach32_disp_shadow | 0x40); /* Mach32 CRT reset */
if(INW(CLOCK_SEL) & 1)    /* If in non VGA mode */
        OUTW(DISP_CNTL, mach32_state.mach32_disp_shadow | 0x20); /* Mach32 CRT enabled */

/* restore all non lock regs */
for(i = sizeof(mach32_eregs) - 1; i >= MACH32_NUM_LOCK; i--)
        {
        OUTB(ATIPORT, ATISEL(mach32_eregs[i]));
        OUTB(ATIPORT + 1, regs->ati_ext_regs[i]);
        }

retval = vga_restoreregs(&regs->vga_regs);

/* restore the lock registers (we can't do this b4 as we could lock VGA regs */

for(i = MACH32_NUM_LOCK - 1; i >= 0; i--)
        {
        OUTB(ATIPORT, ATISEL(mach32_eregs[i]));
        OUTB(ATIPORT + 1, regs->ati_ext_regs[i]);
        }
mach32_state.mach32_ati2e = ATIIN(0x2e) & 0xf0;

LEAVING;
return retval;
}

int chip_blank(void)
{
DOING;
vga_screenoff();
/* Signal that we do not fully support this function! */
return -ENXIO;
}

int chip_unblank(void)
{
DOING;
vga_screenon();
return 0;
}

int chip_setcolormap(fb_cmap_t *cmap)
{
        DOING;
        return -ENXIO;
}

int chip_getcolormap(fb_cmap_t *cmap)
{
        DOING;
        return -ENXIO;
}

void chip_savestate(void *state, void *font)
{
	ENTERING;
	chip_saveregs((struct chip_regs_s *)state);
	if (font) {
		mach32_unlock();
		vga_savetextfontdata(font);
		/* We restore all regs to restore the correct lock state. */
		chip_restoreregs((struct chip_regs_s *)state);
	}
	LEAVING;
}

void chip_restorestate(void *state, void *font)
{
	ENTERING;
	chip_restoreregs((struct chip_regs_s *)state);
	if (font) {
		mach32_unlock();
		vga_restoretextfontdata(font);
		/* We restore all regs to restore the correct lock state. */
		chip_restoreregs((struct chip_regs_s *)state);
	}
	LEAVING;
}

unsigned long chip_savestate_getsize(void)
{
	return sizeof(struct chip_regs_s);
}

/* Note! Scanline is in pixels!! */
static void int_setlinelength(unsigned int scanline)
{
	ENTERING;
	/* No feedback if scanline is out of range */
	scanline = (scanline >> 3) & 0xff;

	/* Setting the VGA is not needed here... */

	OUTW(CRT_PITCH, (unsigned short)scanline);
	OUTW(GE_PITCH, (unsigned short)scanline);
	LEAVING;
}

static void int_setbase(unsigned long offset)
{
	unsigned char tmp;

        ENTERING;

	mach32_state.mach32_origin=offset;

	offset+=mach32_state.mach32_baseptr;

	CRTOUT(0x0c, (unsigned char)offset);
	CRTOUT(0x0d, (unsigned char)(offset>>8));
	tmp=ATIIN(0x30)& ~0x40;
	ATIOUT(0x30, (offset&0x10000) ? (tmp|0x40) : tmp);
	tmp=ATIIN(0x23)& ~0x10;
	ATIOUT(0x23, (offset&0x20000) ? (tmp|0x10) : tmp);

	offset>>=2;

	OUTB(CRT_OFFSET_LO, (unsigned char)offset);
	OUTB(CRT_OFFSET_HI, (unsigned char)(offset>>8));
	OUTB(GE_OFFSET_LO, (unsigned char)offset);
	OUTB(GE_OFFSET_HI, (unsigned char)(offset>>8));

	LEAVING;
}

int chip_setorigin(fb_origin_t *origin)
{
	unsigned long tmp;

        ENTERING;
	if(get_fs_long(&(origin->x)) + mach32_state.mach32_mode->xsize >
						mach32_state.mach32_mode->xvirt)
		return -EINVAL;
	if(get_fs_long(&(origin->y)) + mach32_state.mach32_mode->ysize >
						mach32_state.mach32_mode->yvirt)
		return -EINVAL;
	tmp=get_fs_long(&(origin->x)) + mach32_state.mach32_mode->scanlinelength *
								 get_fs_long(&(origin->y));

	/* We assume that the maximum boundary was checked above */

	if(!mach32_state.is_vga)
		if(tmp&3)
			return -EINVAL;
	int_setbase(tmp);
	
	LEAVING;
        return 0;
}

void chip_do_bank0rw(int nr)
{
	DEB(int nr_in=nr;);

	chip_banks[0].value = chip_banks[1].value = nr;
	mach32_state.page_rh = (mach32_state.page_rh & 0xf3) | ((nr & 0x30) >> 2);
	mach32_state.page_wh = (mach32_state.page_wh & 0xfe) | (nr >> 4);
	ATIOUT(0x2e, mach32_state.mach32_ati2e | mach32_state.page_rh | mach32_state.page_wh);
	nr &= 15;
	mach32_state.page_r = rolb5(nr);
	mach32_state.page_w = nr<<1;
	ATIOUT(0x32, mach32_state.page_r | mach32_state.page_w );
	DEB(printk(KERN_DEBUG "chip_do_bank0rw(%i): ATI(2E)=%02x, ATI(32)=%02x\n",nr_in,
		mach32_state.mach32_ati2e | mach32_state.page_rh | mach32_state.page_wh,
		mach32_state.page_r | mach32_state.page_w);)
}

void chip_do_bank0r(int nr)
{
	DEB(int nr_in=nr;);

	chip_banks[0].value = nr;
	mach32_state.page_rh = (mach32_state.page_rh & 0xf3) | ((nr & 0x30) >> 2);
	ATIOUT(0x2e, mach32_state.mach32_ati2e | mach32_state.page_rh | mach32_state.page_wh);
	nr &= 15;
	mach32_state.page_r = rolb5(nr);
	ATIOUT(0x32, mach32_state.page_r | mach32_state.page_w );
	DEB(printk(KERN_DEBUG "chip_do_bank0r(%i): ATI(2E)=%02x, ATI(32)=%02x\n",nr_in,
		mach32_state.mach32_ati2e | mach32_state.page_rh | mach32_state.page_wh,
		mach32_state.page_r | mach32_state.page_w);)
}

void chip_do_bank0w(int nr)
{
	DEB(int nr_in=nr;);

	chip_banks[1].value = nr;
	mach32_state.page_wh = (mach32_state.page_wh & 0xfe) | (nr >> 4);
	ATIOUT(0x2e, mach32_state.mach32_ati2e | mach32_state.page_rh | mach32_state.page_wh);
	nr &= 15;
	mach32_state.page_w = nr<<1;
	ATIOUT(0x32, mach32_state.page_r | mach32_state.page_w );
	DEB(printk(KERN_DEBUG "chip_do_bank0w(%i): ATI(2E)=%02x, ATI(32)=%02x\n",nr_in,
		mach32_state.mach32_ati2e | mach32_state.page_rh | mach32_state.page_wh,
		mach32_state.page_r | mach32_state.page_w);)
}

#ifndef NDEBUG /* if no production kernel */
/* Auxiliary func for chip_test_test */
static void hexchar(char *buffer, char *data, int count)
{
char *ptr=buffer+3;
int items=0;

	buffer[0]='<';
	buffer[1]='6';
	buffer[2]='<';

	while(count--)
		{
		if(items == 8)
			{
			*ptr++ = ' ';
			*ptr++ = '-';
			}
		sprintf(ptr, " %02x", (unsigned char)(*data++));
		ptr+=3;
		items++;
		if(items++>=16)
			{
			*ptr++ = '\n';
			printk(buffer);
			items=0;
			ptr=buffer+3;
			}
		}
	if(items)
		{
		*ptr++ = '\n';
		printk(buffer);
		}
}
#endif /* NDEBUG */

void chip_test_test(void)
{
#ifndef NDEBUG /* if no production kernel */
	struct chip_regs_s *regs;

	ENTERING;
	regs = kmalloc(sizeof(struct chip_regs_s)+100, GFP_KERNEL);

	if(!regs){
		printk(KERN_WARNING
			"fb_mach32: No memory to store VGA state, operation cancelled.\n");
		goto leave;
		}

	chip_saveregs(regs);

	printk(KERN_INFO "fb_mach32: Listing current VGA state\n");
	printk(KERN_INFO "CRT REGS:\n");
	hexchar( ((char *)regs)+sizeof(struct chip_regs_s),
			regs->vga_regs.crt, CRT_C);
	printk(KERN_INFO "ATT REGS:\n");
	hexchar( ((char *)regs)+sizeof(struct chip_regs_s),
			regs->vga_regs.att, ATT_C);
	printk(KERN_INFO "GRA REGS:\n");
	hexchar( ((char *)regs)+sizeof(struct chip_regs_s),
			regs->vga_regs.gra, GRA_C);
	printk(KERN_INFO "SEQ REGS:\n");
	hexchar( ((char *)regs)+sizeof(struct chip_regs_s),
			regs->vga_regs.seq, SEQ_C);
	printk(KERN_INFO "MISC REG: %02x\n", regs->vga_regs.misc);
	printk(KERN_INFO "EXT REGS:\n");
	hexchar( ((char *)regs)+sizeof(struct chip_regs_s),
			regs->ati_ext_regs, sizeof(regs->ati_ext_regs));
	printk(KERN_INFO "MACH32 REGS:\n");
	hexchar( ((char *)regs)+sizeof(struct chip_regs_s),
			(char *)(regs->mach32_regs), sizeof(regs->mach32_regs));
	printk(KERN_INFO "Base: %06lx, Origin: %06lx, IsVga: %d, Shadow2E: %02x\n",
		regs->driverstat.mach32_baseptr,
		regs->driverstat.mach32_origin,
		regs->driverstat.is_vga,
		regs->driverstat.mach32_ati2e);
	printk(KERN_INFO "DispCntl: %02x, PageR: %02x, PageRh: %02x, PageW: %02x, PageWh: %02x\n",
		regs->driverstat.mach32_disp_shadow,
		regs->driverstat.page_r,
		regs->driverstat.page_rh,
		regs->driverstat.page_w,
		regs->driverstat.page_wh);
	
	kfree(regs);

	leave:
	LEAVING;
#else
	printk(KERN_INFO "fb_mach32: No test function in this production kernel.");
#endif /* No production kernel */
}

static void mach32_wait(void)
{
	int i;
	for(i=0;i<16;i++)
        	__asm__ __volatile__ ("nop\n"); /*Ensure that this loop is not optimized out. */
}


/* EEPROM Utility funcs. Maybe they'll go into an external setup util later */
static int mach32_eeclock(register int ati33)
{
	OUTW(ATIPORT,ati33|=0x200);   /* clock on */
	mach32_wait();
	OUTW(ATIPORT,ati33&= ~0x200); /* clock off */
	mach32_wait();
	return ati33;
}

static void mach32_eekeyout(register int ati33, register int offset, register int mask)
{
	do      {
        	if(mask&offset)
                	ati33|= 0x100;
        	else    ati33&=~0x100;
        	OUTW(ATIPORT,ati33);
        	mach32_eeclock(ati33);
        	}
	while(mask>>=1);
}

static int mach32_eeget(int offset)
{
register int ati33;
register int result,i;

	/* get current ATI33 */
	OUTB(ATIPORT,ATISEL(0x33));
	ati33=((int)INW(ATIPORT+1))<<8;
	ati33|=ATISEL(0x33);
	/* prepare offset.. cut and add header and trailer */
	offset=(0x600|(offset&0x7f))<<1;
	
	/* enable eeprom sequence */
	ati33=mach32_eeclock(ati33);
	/*input to zero..*/
	OUTW(ATIPORT,ati33&=~0x100);
	/*enable to one*/
	OUTW(ATIPORT,ati33|= 0x400);
	mach32_eeclock(ati33);
	/*select to one*/
	OUTW(ATIPORT,ati33|= 0x800);
	mach32_eeclock(ati33);
	mach32_eekeyout(ati33,offset,0x800);
	for(i=0,result=0;i<16;i++)
        	{
        	result<<=1;
        	OUTB(ATIPORT,ATISEL(0x37));
        	if(INB(ATIPORT+1)&0x8)
                	result|=1;
        	mach32_eeclock(ati33);
        	}
	/*deselect...*/
	OUTW(ATIPORT,ati33&=~0x800);
	mach32_eeclock(ati33);
	/*disable...*/
	OUTW(ATIPORT,ati33&=~0x400);
	mach32_eeclock(ati33);
	return result;
}

/* Long blitwait with timeout, busy loop.. should be ok, as it is only called from
   the insmod process.. */

static void mach32_i_bltwait(void)
{
int i;

	for (i=0; i < 100000; i++)
        	if(!(INW(GE_STAT) & (GE_BUSY | 1)))
                	break;
}

unsigned char *chip_test_chipset(void)
{
char result=0;
short tmp;

	ENTERING;
	tmp = INW(SCRATCH_PAD_0);
	OUTW(SCRATCH_PAD_0, 0x5555);
	mach32_i_bltwait();
	if(INW(SCRATCH_PAD_0) == 0x5555)
        	{
        	OUTW(SCRATCH_PAD_0, 0x2a2a);
        	mach32_i_bltwait();
        	if(INW(SCRATCH_PAD_0) == 0x2a2a)
                	{
                	/* Aha.. 8514/a detected.. */
                	result=1;
                	}
        	}
	OUTW(SCRATCH_PAD_0,tmp);
	if(!result)
        	goto quit;
	/* Now ensure it is not a plain 8514/a: */
	result=0;
	OUTW(DESTX_DIASTP, 0xaaaa);
	mach32_i_bltwait();
	if(INW(R_SRC_X)==0x02aa)
        	{
        	OUTW(DESTX_DIASTP, 0x5555);
        	mach32_i_bltwait();
        	if(INW(R_SRC_X)==0x0555)
        	OUTW(DESTX_DIASTP, 0x5555);
        	mach32_i_bltwait();
        	if(INW(R_SRC_X)==0x0555)
                	result=1;
        	}
	if(!result)
        	goto quit;
	/* Ok, now we have a Mach32. Unfortunately we need also the VGA to be enabled: */
	if(INW(CONF_STAT1)&ONLY_8514)
        	{
		result=0;
		printk(KERN_INFO "fb_mach32: Mach32 detected, but unusable with VGA disabled.\n");
        	goto quit;      /*VGA circuitry disabled.*/
        	}
	result=1;
	quit:
	LEAVING;
return result ? "Mach32" : NULL ;
}

/* return != 0 if offset points to valid 8514/A parameters,
   use bit in mask to see if the params are valid */
static int check_val_parm(unsigned short entry, unsigned short mask)
{
	if(!(entry&mask))
		return 0;	/* No table at all */
	entry >>= 8;	/* isolate table offset */
	if((entry < 0xd) || (entry > 0x67))
		return 0;	/* well, at least a non standard offset */
	if((mach32_eeprom[entry + 8] & 6) != 2)
		return 0;	/* We require skip 2 y values (others should not occur) */
	if((mach32_eeprom[entry + 10]) & 0x4000)
		return 0;	/* MUX not yet implemented */
	/* require 8514 table with full timings, no MUX mode */
	return (mach32_eeprom[entry] & 0x1140) == 0x140;
}

/* Setup modeinfo. If possible use custom mode, use offset and mask for validate */
static void setup_mode(int num, int offset, int mask)
{
	char *ptr;
	const mode_entry *entry;
	fb_mode_t *info;
	int i;

	if(check_val_parm(mach32_eeprom[offset], mask))
		{
		/* Use custom mode */
		entry = (mode_entry *)(mach32_eeprom + (mach32_eeprom[offset] >> 8) + 2);
		}
	else	{
		/* First is always ok (if we are searching for an 640x480 mode), if not
		   it will be overriden.. (we won't be called if no mode exists) */
		entry = predef_modes;

		/* Use best (latest) of predefined modes */
		for(i = 1; i < (sizeof(predef_modes) / sizeof(mode_entry)); i++)
			{
			if((offset == predef_modes[i].offset) &&
					(predef_modes[i].mask & mach32_eeprom[offset]))
				entry = predef_modes + i;
			}
		}

	mode_timings[num] = (mode_entry *)entry;
	info = chip_modesinfo + num;

	/* entry points to selected entry. Now setup the modetable from that info. */
	/* 1st zero unused entries, yes, I know.. silly.. but this should be enough at the
	   moment. */
	ptr = (void *)info;
	i = sizeof(fb_mode_t);
	while(i--)
		*ptr++ = 0; 

	info->num = num;
	info->xsize = ((entry->h_disp + 1) << 3);
	/* We may assured that this is in skip2 */
	i = entry->v_disp;
	i = (i & 3) | ((i >> 1) & ~3);
	info->ysize = i + 1;
	
	info->vidmem = info->vidmem2mmap = chip_info.totalmem;

	/* Ensure multiple of 64 for scanline */
	info->xdim = info->xvirt = info->scanlinelength = (info->xsize + 0x3f) & ~0x3f;
	info->ydim = info->yvirt = info->vidmem / info->scanlinelength;
	
	info->mmapmodel = 1;
	info->bitordering = 76543210;
	info->byteordering = 1234;
	info->numwindows = 1;
	info->windowsize = (1<<16);
	info->windowshift = 16;
	info->numbanks = 16;
	info->bpp = info->bpp_used = 8;
	info->bitsRGB = 6;
	info->white = 255;
}

int chip_init(void)
{
	int i;

	ENTERING;

	init_mapdesc(maps_0r, VGA_MAPS);
	init_mapdesc(maps_0w, VGA_MAPS);
	init_mapdesc(maps_lin, LIN_MAPS);

	{
	struct chip_regs_s *dummy;

	/* Check assertions: */
	if(!assert(sizeof(mach32_eregs) <= sizeof(dummy->ati_ext_regs),
			"chip_regs_s.ati_ext_regs too small"))
		return -1;
	if(!assert((sizeof(mach32_acc_regs)/2) <= sizeof(dummy->mach32_regs),
			"chip_regs_s.mach32_regs too small"))
		return -1;
	}

	/* Phase 1, getting eeprom.... */
	mach32_eeprom = kmalloc(256, GFP_KERNEL);
	if(!mach32_eeprom)
		{
		printk(KERN_ERR "fb_mach32: Unable to allocate EEPROM buffer.");
		return -1;
		}
	for(i=0; i<128; i++)
		mach32_eeprom[i] = mach32_eeget(i); 
	/* Phase 2, counting possible modes in modeinfo table, no checks for memory, etc.
	   we do believe what we find in the EEPROM */

	i = 1;	/* There is always the 640x480x8. */
	chip_info.default_mode = 0;

	if((mach32_eeprom[8] & 0x3f) || check_val_parm(mach32_eeprom[8], 0x80))
		{
		/* There is a 800x600 mode known: */
		chip_info.default_mode = i;
		i++;
		}
	if((mach32_eeprom[9] & 0x1f) || check_val_parm(mach32_eeprom[9], 0x80))
		{
		/* There is a 1024x768 mode known: */
		chip_info.default_mode = i;
		i++;
		}
	if((mach32_eeprom[10] & 0x3) || check_val_parm(mach32_eeprom[10], 0x80))
		{
		/* There is a 1280x1024 mode known: */
		/* It is never used as default mode */
		i++;
		}

	chip_mode = chip_info.default_mode;

	/* Phase 3, set_up chip_info, alloc space for modeinfo and modenum-> info ptr. */
	chip_info.number_modes = i;
	
	i= (1 << ((INW(CONF_STAT2) >> 4) & 7));
	chip_info.memtype = (i & 0x66) ? 1 : 0;
	if(i & 0x80)
		printk(KERN_INFO "fb_mach32: Strange memconfig, assuming DRAM.");

	/* Rumors are that sometimes wrong memsizes are reported.., actually this field
	   is r/w. We trust the bios to know what it puts there. */
	switch((INW(MISC_OPTIONS) >> 2) & 3)
		{
		case 3:
			chip_info.totalmem = 4096 * 1024;
			break;
		case 2:
			chip_info.totalmem = 2048 * 1024;
			break;
		case 1:
			chip_info.totalmem = 1024 * 1024;
			break;
		default:
			chip_info.totalmem = 512 * 1024;
		}
	
	/* Currently we support only the VGA aperture: */
	mode_timings = kmalloc(chip_info.number_modes *
				 	(sizeof(fb_mode_t) + sizeof(mode_entry *)) ,
				GFP_KERNEL);
	if(!mode_timings)
		{
		printk(KERN_ERR "fb_mach32: Unable to allocate mode_tables.");
		/* return already allocated eeprom buffer */
		kfree_s(mach32_eeprom, 256);
		mach32_eeprom = NULL;
		return -1;
		}

	chip_modesinfo = (fb_mode_t *)(((char *)mode_timings) +
			chip_info.number_modes * sizeof(mode_entry *));

	/* Phase 4: Setup mode_timings table and chip_modesinfo */
	i = 0;
	setup_mode(i++, 7, 0x02);
	if((mach32_eeprom[8] & 0x3f) || check_val_parm(mach32_eeprom[8], 0x80))
		setup_mode(i++, 8, 0x80);
	if((mach32_eeprom[9] & 0x1f) || check_val_parm(mach32_eeprom[9], 0x80))
		setup_mode(i++, 9, 0x80);
	if((mach32_eeprom[10] & 0x3) || check_val_parm(mach32_eeprom[10], 0x80))
		setup_mode(i++, 10, 0x80);

	vga_init();

        LEAVING;
        return 0;
}

void chip_cleanup(void)
{
	if(mach32_eeprom)
		{
		kfree_s(mach32_eeprom, 256);
		mach32_eeprom = NULL;
		}
	if(mode_timings)
		{
		kfree(mode_timings);
		mode_timings = NULL;
		}
}

int chip_graphics(void)
{
	unsigned short clock_intended;
	unsigned short act_ge_conf;

	const mode_entry *timings = mode_timings[chip_mode];
	const fb_mode_t *info = chip_modesinfo + chip_mode;

        ENTERING;
	mach32_unlock();
	vga_screenoff();

	/* Phase 1: Setup vga for memory access: */
	OUTW(SEQ_I, 4 | (0x0a << 8));
	OUTW(SEQ_I, 2 | (0x0f << 8));
	OUTW(ATIPORT, ATISEL(0x36) | (0x05 << 8));
	OUTW(GRA_I, 6 | (0x05 << 8));
	OUTW(GRA_I, 1 | (0x00 << 8));
	OUTW(GRA_I, 3 | (0x00 << 8));
	OUTW(GRA_I, 5 | (0x00 << 8));
	OUTW(GRA_I, 7 | (0x00 << 8));
	OUTW(GRA_I, 8 | (0xff << 8));

	mach32_state.mach32_ati2e = ATIIN(0x2e) & 0xf0;

	/* Phase 2: Setup Mach32 */
	mach32_state.mach32_disp_shadow = timings->disp_cntl & 0x7f;
	OUTW(DISP_CNTL, mach32_state.mach32_disp_shadow | 0x60); /* Mach32 CRT reset */

	/* No dac setup (only 256 color modes, non multiplex!!!) */
	act_ge_conf = INW(R_EXT_GE_CONF) & 0xf80f;
	act_ge_conf |= 0x0010; /* 8bpp, no multiplex! */

	OUTW(EXT_GE_CONF, act_ge_conf);

	clock_intended = timings->clock_sel;
	/* Enforce VFIFO 6 (good default) */
	clock_intended &= 0xf0ff;
	clock_intended |= 0x0601; /* The 1 is to disable vga */

	/* Set timings: */
	OUTB(H_TOTAL, timings->h_total);
	OUTW(H_DISP, timings->h_disp);
	OUTB(H_SYNC_STRT, timings->h_sync_strt);
	OUTB(H_SYNC_WID, timings->h_sync_wid);
	OUTW(V_TOTAL, timings->v_total);
	OUTW(V_DISP, timings->v_disp);
	OUTW(V_SYNC_STRT, timings->v_sync_strt);
	OUTW(V_SYNC_WID, timings->v_sync_wid);
	OUTW(CLOCK_SEL, clock_intended);

	/* init page selectors, setup memory layout */	
	int_setlinelength(info->xdim);
	int_setbase(0);
	chip_do_bank0rw(0);

	/* Fire up screen... */
	OUTW(DISP_CNTL, mach32_state.mach32_disp_shadow | 0x20); /* Mach32 CRT enable */
	
#if 0
printk(KERN_INFO "GE_CONF:%hx\n",act_ge_conf);
printk(KERN_INFO "Setting mode:h_disp: %x h_total: %x h_sync_wid: %x h_sync_strt: %x\n",
	timings->h_disp,timings->h_total,timings->h_sync_wid,timings->h_sync_strt);
printk(KERN_INFO "             v_disp: %x v_total: %x v_sync_wid: %x v_sync_strt: %x\n",
	timings->v_disp,timings->v_total,timings->v_sync_wid,timings->v_sync_strt);
printk(KERN_INFO "             clock_sel: %x disp_ctl: %x\n",
	clock_intended,mach32_state.mach32_disp_shadow);
#endif

        LEAVING;
        return 0;
}

int chip_choose_bank(int mode, int address) {
	if(mode & VM_WRITE) {
		if(mode & VM_READ)
			return 0 | NEEDS_NEXT_BANK;
		else	return 1;
	} else {
		return 0;
	}
}
