/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: Print_support.c
 * handle the actual output to the Printer
 ***************************************************************************
 * void Print_open(): opens the Printer
 * static int Print_of_fd(): makes an 'of' filter if needed
 * int Print_close(): closes the Printer
 * int Print_ready(): combines Print_open() and Print_of_fd()
 * static int of_stop(): stops the 'of' filter
 * int Print_string (str) : prints a string through 'of' or to Printer
 * int Print_copy (fd) : copies a file through 'of' or to Printer
 * int Print_filter (file, cmd) : makes a filter and copies file
 * int Print_banner(): prints a banner through 'of' or to Printer
 * Note: the above procedures which return values return JSUCC on success,
 *   and JFAIL or JABORT on failure
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "LPD/startprinter.h"
#include "LPD/setproctitle.h"
#include "common/setstatus.h"

static void Print_of_fd (), LockDevicefile ();

/***************************************************************************
 * Print_open()
 * 	Open the Printer, and set the Print_fd variables
 * 	If the RW printcap flag is set, output is opened RW, otherwise
 * 	opened writeonly in append mode.
 *
 * 	If the Printer is a tty (i.e.- isatty returns non-zero),
 * 	then the baud rate is set.  The baud rate table is stolen from
 * 	the original LPD code, which stole it from STTY, etc.
 * Side Effect:
 * 	sets the Print_fd variable to a non-zero value
 *  terminates if Printer is unable to be opened
 ****************************************************************************/

static pid_t lppipe_pid;	/* LP-pipe pid */
static int lppipe_running;	/* lp-pipe running */
static int is_lp_pipe;		/* is this LP a subprocess? */

static pid_t of_pid;		/* OF process pid */
static int of_running;		/* of filter running */
static int of_fd;		/* pipe to OF process */
static char filter_stop[] = "\031\001";	/* what to send filter */

static int Open_lp_pipe (), rw_pipe ();

#ifndef O_NOCTTY	
	/* O_NOCTTY = don't make this tty our controlling terminal */
#define O_NOCTTY	0	/* just fake it. */
#endif

void
Print_open (void) {
    int err;

    /*
     * check to see if it is open
     */
    if (Print_fd) {
	return;
    }
    if (LP == 0 || *LP == 0) {
	fatal (XLOG_INFO, "Missing LP value for local Printer");
    }
    setstatus ("Waiting for '%s' to become ready since %s (offline?).",
	       Printer, Time_str ());
    setproctitle ("waiting for '%s' to become ready.", Printer);

    if (*LP == '|') {
	is_lp_pipe = 1;
	Print_fd = Open_lp_pipe ();

    } else {
	is_lp_pipe = 0;
	daemon_to_root ();	/* setuid to open & lock the LP device */
	Print_fd = open (LP,
		(RW ? O_RDWR : O_WRONLY) | O_NOCTTY | O_APPEND, 0644);

	if (!strsame (LP, "/dev/null")) {
	    LockDevicefile (Print_fd, LP);
	}
	root_to_daemon ();	/* end setuid (open,lock the LP device) */
    }

    if (Print_fd < 0) {
	err = errno;
	setstatus ("Cannot open %s! (%s)", LP, Errormsg (err));
	errno = err;
	logerr_die (XLOG_INFO, "Print_open: cannot open %s", LP);

    } else if (Print_fd == 0) {
	fatal (XLOG_INFO, "Print_open: cannot happen - Print_fd == 0!");
    }
    /*
     * if it is a tty, set the baud rates and control information; also detach from it if
     * TIOCNOTTY permits.
     */
    if (!is_lp_pipe && isatty (Print_fd)) {
	Do_stty (Print_fd);

#ifdef TIOCNOTTY
	if ((ttyfd = open ("/dev/tty", O_WRONLY | O_APPEND,
			   0644)) != -1) {
	    if (ioctl (ttyfd, TIOCNOTTY, 0) < 0) {
		log (XLOG_INFO, "cannot detach from controlling terminal");
	    }
	    (void) close (ttyfd);
	}
#endif
    }
    if (DbgLocal > 7)
	log (XLOG_DEBUG, "Print_open: %s is fd %d", LP, Print_fd);
}

/*
 * Start up an output filter, if needed.
 */
static void
Print_of_fd (void) {
    int p[2];
    char *cmd;

    if (OF == 0 || *OF == 0 || of_pid) {
	return;
    }
    /*
     * set OF command line
     */
    cmd = Setup_filter ('o', OF);

    if (DbgLocal > 2)
	log (XLOG_DEBUG, "starting OF, '%s'", cmd);
    /*
     * create pipe and fork
     */
    if (pipe (p) < 0) {
	logerr_die (XLOG_NOTICE, "pipe failed");
    }
    assert ((p[0] >= 3) && (p[1] >= 3));
    if ((of_pid = fork ()) == 0) {		/* child */
	if (dup2 (p[0], 0) < 0			/* pipe is stdin */
	    || dup2 (Print_fd, 1) < 0) {	/* Printer is stdout */
	    logerr_die (XLOG_NOTICE, "Print_of_fd: dup2 failed");
	}
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "Print_of_fd: forked %d OF: %s", getpid (), cmd);

	setup_close_on_exec ();	/* this'll close the unused pipe fds */
	full_daemon_perms ();
	secure_exec (cmd);	/* run as "daemon" */

    } else if (of_pid < 0) {
	logerr_die (XLOG_NOTICE, "Print_of_fd: fork failed");
    }
    (void) close (p[0]);	/* close input side */
    of_fd = p[1];
    of_running = of_pid;
    if (DbgLocal > 3)
	log (XLOG_DEBUG, "started OF process %d, fd %d", of_pid, of_fd);
}

static int wait_for_lppipe (pid_t);

/***************************************************************************
 * Print_close()
 * 1. Close the pipes and outputs.
 * 2. Signal the of process that it is to do something.
 * 3. Signal the lp-pipe process that it is to do something.
 * 4. Wait for the lp-pipe process.
 ***************************************************************************/

int
Print_close (void) {
    int ret;

    if (DbgLocal > 2) {
	log (XLOG_DEBUG, "closing printer");
    }
    ret = JSUCC;
    (void) plp_signal (SIGCHLD, (plp_sigfunc_t)SIG_IGN);   /* don't reap this one */

    if (of_fd) {
	(void) close (of_fd);
	of_fd = 0;
    }
    if (Print_fd) {
	(void) close (Print_fd);	/* close Printer */
	Print_fd = 0;
    }
    /*
     * when you kill the printer, make sure that you do it neatly
     */
    if (of_pid) {
	if (kill (of_pid, SIGCONT))
	    logerr (XLOG_NOTICE, "Failed to send SIGCONT to of filter");
    }
    if (lppipe_pid) {
	(void) kill (lppipe_pid, SIGCONT);
	ret = wait_for_lppipe (lppipe_pid);
    }
    lppipe_pid = of_pid = 0;
    return (ret);
}

/***************************************************************************
 * Print_ready()
 * 1. open the Printer if neccessary
 * 2. start up the output filter
 * 3. if the filter has been stopped, start it
 * Return: JSUCC if started, JFAIL if not
 ***************************************************************************/
int
Print_ready (void) {
    /*
     * open the Printer
     */
    Print_open ();
    Print_of_fd ();
    if (of_pid && of_running == 0) {
	if (kill (of_pid, SIGCONT) < 0) {
	    logerr (XLOG_INFO, "cannot restart output filter");
	    if (Print_close () == JABORT) { return (JABORT); }
	    return (JFAIL);
	}
	of_running = of_pid;
    }
    if (lppipe_pid && lppipe_running == 0) {
	if (kill (lppipe_pid, SIGCONT) < 0) {
	    logerr (XLOG_INFO, "cannot restart lp-pipe");
	    if (Print_close () == JABORT) { return (JABORT); }
	    return (JFAIL);
	}
	lppipe_running = lppipe_pid;
    }
    return (JSUCC);
}

/***************************************************************************
 * of_stop()
 * stop the output filter and lp-pipe if neccessary
 * 1. open and create the filter
 * 2. flush output
 * 3. kill(SIGSTOP) it
 * Return: JSUCC if stopped, JFAIL if not
 ***************************************************************************/

int
of_stop (void) {
    plp_status_t statb;
    int pid;

    Print_open ();
    Print_of_fd ();
    (void) plp_signal (SIGCHLD, (plp_sigfunc_t)SIG_IGN);   /* don't reap this one */

    /*
     * stop the OF filter
     */
    if (of_pid && of_running) {
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "stopping output filter");
	if (write (of_fd, filter_stop, strlen (filter_stop))
	    != strlen (filter_stop)) {
	    logerr (XLOG_NOTICE, "of_stop: cannot write to OF");
	    if (Print_close () == JABORT) { return (JABORT); }
	    return (JFAIL);
	}
	/*
	 * wait until OF blocks
	 */
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "of_stop: waiting for OF %d", of_pid);

	pid = plp_waitpid (of_pid, &statb, WUNTRACED);
	if (pid < 0) {
	    logerr (XLOG_INFO, "waitpid");
	}

	if (!PLP_WIFSTOPPED (statb)) {
	    logerr (XLOG_INFO, "of_stop: output filter died (%s)",
		    Decode_status (&statb));
	    if (Print_close () == JABORT) { return (JABORT); }
	    return (JFAIL);
	}
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "of_stop: output filter stopped");
	of_running = 0;
    }

    /* now the lp-pipe. */

    if (lppipe_pid && lppipe_running) {
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "stopping lp-pipe");

	(void) plp_signal (SIGCHLD, (plp_sigfunc_t)SIG_IGN);   /* don't reap this one */

	if (kill (lppipe_pid, SIGSTOP) < 0) {
	    logerr (XLOG_INFO, "cannot stop lp-pipe");
	    if (Print_close () == JABORT) { return (JABORT); }
	    return (JFAIL);
	}
	/*
	 * wait until OF blocks
	 */
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "of_stop: waiting for lp-pipe %d", lppipe_pid);

	pid = plp_waitpid (lppipe_pid, &statb, WUNTRACED);
	if (pid < 0) {
	    logerr (XLOG_INFO, "waitpid");
	}

	if (!PLP_WIFSTOPPED (statb)) {
	    log (XLOG_INFO, "of_stop: lp-pipe died (%s)",
		    Decode_status (&statb));
	    if (Print_close () == JABORT) { return (JABORT); }
	    return (JFAIL);
	}
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "of_stop: lp-pipe stopped");
	lppipe_running = 0;
    }
    return (JSUCC);
}

/***************************************************************************
 * Print_string (char *str)
 * print a string through a filter
 * 1. Enable the line Printer
 * 2. get the filter or device fd;
 * 3. put out the string
 * 4. if unsuccessful, close the Printer
 * Return: JSUCC if successful, JFAIL otherwise
 ***************************************************************************/
int
Print_string (char *str) {
    int f;
    int l;

    if (Print_ready () != JSUCC) {
	return (JFAIL);
    }
    if (of_fd) {
	f = of_fd;
    } else {
	f = Print_fd;
    }
    l = strlen (str);
    if (write (f, str, l) != l) {
	logerr (XLOG_INFO, "Print_string: write error");
	if (Print_close () == JABORT) { return (JABORT); }
	return (JFAIL);
    }
    return (JSUCC);
}


/***************************************************************************
 * Print_copy (int fd)
 * copy a file through a filter
 * 1. ready the Printer
 * 2. copy the file to the appropriate output device
 * 3. if an error, close the Printer
 ***************************************************************************/
int
Print_copy (int fd) {
    int f;
    long cnt;
    char buf[BUFSIZ];
    int in;			/* bytes read */
    int out;			/* bytes written out */

    cnt = 0;
    if (Print_ready () != JSUCC) {
	return (JFAIL);
    }
    if (of_fd) {
	f = of_fd;
    } else {
	f = Print_fd;
    }
    while ((in = read (fd, buf, sizeof (buf))) > 0) {
	out = write (f, buf, in);
	if (in != out) {
	    logerr (XLOG_INFO, "Print_copy: write error");
	    if (Print_close () == JABORT) { return (JABORT); }
	    return (JFAIL);
	}
	cnt = cnt + out;
    }
    /*
     * completed the reading
     */
    if (in < 0) {
	logerr (XLOG_INFO, "Print_copy: read error");
	if (Print_close () == JABORT) { return (JABORT); }
	return (JFAIL);
    }
    if (DbgLocal > 3)
	log (XLOG_DEBUG, "Print_copy: printed %d bytes", cnt);

    if (CL) {
        return (Print_close ());
    } else {
	return (JSUCC);
    }
}

/***************************************************************************
 *Print_filter (int fd, char *cmd, int lflag)
 *  spawn a subprocess to do the printing
 * 1. stop the Printer
 * 2. fork a process
 * 3. wait for process to complete
 * 4. restart the process
 * Return: JSUCC if successful, JFAIL otherwise
 ***************************************************************************/
int
Print_filter (int file, char *cmd, int lflag) {
    int succ;			/* success code */
    pid_t filter_pid;		/* filter process */
    pid_t pid;			/* daughter pid */
    plp_status_t status;		/* daughter status */
    int err;			/* saved value of errno */

    if (lflag) {
	/*
	 * stop the Printer
	 */
	succ = of_stop ();
	if (succ != JSUCC) {
	    return (succ);
	}
    }

    (void) plp_signal (SIGCHLD, (plp_sigfunc_t)SIG_IGN);   /* don't reap this one */

    /*
     * fork a process,  and connect file to fd 0, Printer to fd 1.
     */
    if ((filter_pid = fork ()) == 0) {	/* daughter */
	/*
	 * dup input file to standard in and Printer to stdout
	 */
	if (dup2 (file, 0) < 0 || dup2 (Print_fd, 1) < 0) {
	    logerr_die (XLOG_NOTICE, "Print_filter: dup2 failed filter %s", cmd);
	}
	full_daemon_perms ();
	secure_exec (cmd);

    } else if (filter_pid < 0) {
	logerr (XLOG_NOTICE, "Print_filter: fork failed");
	if (lflag) {
	    if (Print_close () == JABORT) { return (JABORT); }
	}
	return (JFAIL);
    }
    /*
     * Wait for filter to complete
     */
    if (DbgLocal > 2)
	log (XLOG_DEBUG, "Print_filter: waiting for pid %d", filter_pid);

    pid = plp_waitpid (filter_pid, &status, WUNTRACED);

    err = errno;
    if (DbgLocal > 2)
	log (XLOG_DEBUG, "filter finished (%s)", Decode_status (&status));
    errno = err;

    /*
     * Check to see how filter terminated
     */
    if (pid < 0) {
	if (!Job_removed ()) {
	    /* if the job is removed, the filter exit status doesn't matter. */
	    logerr (XLOG_INFO, "waitpid");
	    return (JABORT);
	} else {
	    return (JSUCC);
	}

    } else if ((PLP_WIFEXITED (status)) && (PLP_WEXITSTATUS (status) > 1)) {
	/* exited with a status > 1, don't run this again */
	log (XLOG_INFO, "Filter malfunctioned (%s): %s",
	    Decode_status (&status), cmd);
	if (lflag) {
	    if (Print_close () == JABORT) { return (JABORT); }
	}
	return (JABORT);

    } else if ((!PLP_WIFEXITED (status)) || (PLP_WEXITSTATUS (status) != 0)) {
	/*
	 * either died from a signal, coredumped, or exited
	 * with an LP_RETRY status, so try again.
	 * 
	 * RT == 0 implies many retries expected - eg padprinters. No point logging such
	 * retries.
	 */
	if (RT != 0)
	    log (XLOG_INFO, "Filter wants retry: %s", cmd);

	if (lflag) {
	    if (Print_close () == JABORT) { return (JABORT); }
	}
	return (JFAIL);
    }
    return (JSUCC);
}

/*
 * Print_banner() 1. get the Printer ready 2. call the banner() routine with the correct
 * parameter
 */

int
Print_banner (void) {
    int f;

    if (DbgLocal > 3)
	log (XLOG_DEBUG, "Print_banner: printing banner");
    if (Print_ready () != JSUCC) {
	return (JFAIL);
    }
    if (of_fd) {
	f = of_fd;
    } else {
	f = Print_fd;
    }
    if (banner (f) != JSUCC) {
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "Print_banner: banner failed");
	if (Print_close () == JABORT) { return (JABORT); }
	return (JFAIL);
    }
    return (JSUCC);
}

/* called when the active job is suddenly lprm'ed. */
void
Print_intr (void) {
    if (Print_fd <= 0) {
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "Print_intr: printer is not open");
    }
    /* send a ^C in case it's a postscript printer. */
    Print_string ("\003\004");
    Print_close ();
}

/***************************************************************************
 * LockDevice(fd, char *devname)
 * Tries to lock the device file so that two or more queues can work on
 * the same print device. First does a non-blocking lock, if this fails,
 * puts a nice message in the status file and blocks in a second lock.
 * (contributed by Michael Joosten <joost@cadlab.de>)
 ***************************************************************************/

static void 
LockDevicefile (int fd, char *devname) {
    int lock;

    if ((lock = do_devlock (fd, devname, 1)) < 0) {
	logerr_die (XLOG_CRIT, "LockDevice: test lock '%s' failed", devname);
    } else if (lock == 0) {
	if (DbgLocal > 3)
	    log (XLOG_DEBUG, "LockDevice: device '%s' already locked", devname);
	setstatus ("Printer busy, waiting for lock on %s.", devname);
        setproctitle ("Waiting for '%s': blocking lock on device %s.", Printer, devname);

	/* must wait for other lpd incarnation to finish the job */
	lock = do_devlock (fd, devname, 0);	/* blocking lock */
	/* waiting here... */
	if (lock < 0) {
	    logerr_die (XLOG_CRIT, "LockDevice: lock '%s' failed", devname);
	}
    }
    if (DbgLocal > 3)
	log (XLOG_DEBUG, "LockDevice: device '%s' - got lock", devname);

    /* got the lock, proceed with printing */
}

/* Given an LP string as follows:
 * LP = "| /usr/local/etc/lp-pipe port=x host=y whatever",
 * set up an fd which pipes into this command (which gets the usual
 * filter args as well). The command's fds are as follows:
 * 0 = the input, 1 = closed, 2 = the logfile.
 */

static int
Open_lp_pipe (void) {
    char *lppipe, *pipecmd, *oAF;
    int fds[2];

    lppipe = LP + 1;

    if (DbgLocal > 1) {
	log (XLOG_DEBUG, "opening lp-pipe: '%s'", lppipe);
    }

    /* set it up with the std filter arguments. */

    oAF = AF;
    AF = ST;
    pipecmd = Setup_filter ('f', lppipe);
    AF = oAF;
    if (pipecmd == 0 || *pipecmd == 0) {
        logerr_die (XLOG_INFO, "cannot open lp-pipe '%s'", lppipe);
    }

    if (RW) {
	if (rw_pipe (fds) < 0) {
	    logerr_die (XLOG_INFO, "can't create r/w pipe for lp-pipe");
	}
    } else {
	if (pipe (fds) < 0) {
	    logerr_die (XLOG_INFO, "can't create pipe for lp-pipe");
	}
    }

    if ((lppipe_pid = fork ()) == 0) {		/* daughter */
	if (dup2 (fds[0], 0) < 0) {
	    logerr_die (XLOG_INFO, "can't dup lp-pipe input to stdin");
	}
	setup_close_on_exec ();
	full_daemon_perms ();
        secure_exec (pipecmd);

    } else if (lppipe_pid < 0) {
        logerr_die (XLOG_NOTICE, "lp-pipe: fork failed");
    }

    if (Debug > 3) {
	log (XLOG_DEBUG, "lp-pipe started, pid %d, filedes %d",
	    lppipe_pid, fds[1]);
    }
    
    close (fds[0]);
    lppipe_running = lppipe_pid;
    return (fds[1]);
}

static int wait_for_lppipe (pid_t lppid) {
    pid_t i;
    plp_status_t stat;          /* LP-pipe status */
    int err;

    i = plp_waitpid (lppid, &stat, WUNTRACED);
    err = errno;
    if (DbgLocal > 3) {
	log (XLOG_DEBUG, "wait for lp-pipe: finished (%s)",
	     Decode_status (&stat));
    }
    errno = err;

    if (i < 0) {
	logerr (XLOG_INFO, "waitpid");
	return (JABORT);
    }

    if (PLP_WEXITSTATUS(stat) > 1) {
	log (XLOG_INFO, "lp-pipe died seriously -- kill queue (%s)",
	    Decode_status (&stat));
	return (JABORT);

    } else if (PLP_WEXITSTATUS(stat) != 0) {
	if (RT < 0) {
	    if (DbgLocal) {
		log (XLOG_DEBUG, "lp-pipe died -- retry (%s)",
		    Decode_status (&stat));
	    }
	} else {
	    log (XLOG_INFO, "lp-pipe died (%s)",
		Decode_status (&stat));
	}
	return (JFAIL);
    }

    /* else, success */
    return (JSUCC);
}

/*
 * socketpair is not supported on some OSes, including SVR4 (according
 * to "Advanced Programming in the UNIX Environment", Stevens); however,
 * SVR4 uses pipe() to do the same thing, so we can support that here.
 * if we're on anything else, log a warning message but try pipe() anyway.
 */
static int
rw_pipe (int fds[]) {
#ifdef HAVE_SOCKETPAIR
    return (socketpair (AF_UNIX, SOCK_STREAM, 0, fds));
#else

#ifndef SVR4
    /* log a warning message if we're not on SVR4; pipe() may not
     * do the job.
     */
    log (XLOG_INFO, "warning: 'rw' LP-pipes may not be supported for this OS");
#endif

    return (pipe (fds));
#endif
}
