/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: remoteprinter.c
 * send a job to a remote Printer
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "LPD/setproctitle.h"
#include "LPD/startprinter.h"
#include "LPD/localprinter.h"
#include "common/setstatus.h"

static int sendfile (FILE *, char *, int);
static int sendname (FILE *, char *, int);
static int senddatafile (char *, int);

static void
log_spool_full (void) {
    setstatus ("Spool area full on %s@%s at %s, sleeping.", RP, RM, Time_str ());
    log (XLOG_INFO, "no space left on %s@%s", RP, RM);
    setproctitle ("%s: spool area full for %s@%s", RP, RM);
}

/***************************************************************************
 * int Setup_xfer()
 * sends the "\02Printer" message to the remote Host
 * Returns: successful JSUCC, otherwise JFAIL
 ***************************************************************************/
int
Setup_xfer (void) {
    char buf[BUFSIZ];
    int sleeptime, retry;

    if (RP == 0 || *RP == 0) {
	fatal (XLOG_INFO, "no RP specified for RM (%s)", RM);
    }
    (void) sprintf (buf, "%c%s\n", REQ_RECV, RP);
    retry = 1;
    sleeptime = 15;

    /* Do retries: WSLPD, JetDirect are single-connection
     * LPD implementations, and they need the capacity for configurable
     * retries, as if they were printer devices.
     */
    while ((RT > 0) ? retry < RT : 1) {
	setstatus ("Attempt %d to open link to %s at %s.", retry, RM, Time_str ());
	if (JSUCC == Link_line (1, buf)) {
	    if (DbgRem > 3) {
		log (XLOG_DEBUG, "connect to %s: waiting for confirm", RM);
	    }
	    if (JSUCC == Link_confirm ()) {
		if (DbgRem > 3) {
		    log (XLOG_DEBUG, "connect to %s: confirmed RECV request", RM);
		}
		return (JSUCC);
	    }

	    if (DbgRem > 3) {
		log (XLOG_DEBUG,
		     "connect to %s: RECV request not confirmed", RM);
	    }

	} else {
	    if (DbgRem > 3) {
		log (XLOG_DEBUG, "connect to %s: remote daemon busy", RM);
	    }
	}
	setstatus ("Remote daemon on %s busy, retry %d.", RM, retry);
	setproctitle ("%s: remote daemon on %s busy, retry %d", Printer,
		RM, retry);

	Link_close ();
	sleep (sleeptime);
	retry++;
    }
    return (JFAIL);
}

/***************************************************************************
 * Sendjob (FILE *cfp; struct plp_queue *q)
 *   1.  First scan extracts information which controls printing
 *			and finds the names of the files to send.
 *	 2.	 The data files are then sent.
 *	 3.	 Control file is then sent.
 *   4.  Job files are removed
 *   5.  Control file is removed
 * Returns:  JBUSY, JFAIL, JSUCC, JABORT
 * NOTE: this code was based on a description of the LPD file transfer
 *    protocol.
 *
 * File transfer protocol:
 * Sender: sends line of form 'flag' length filename
 * Receiver: acknowledges with 0 (single byte)
 * Sender: sends file (length bytes of information), then a teminating 0
 * Receiver: acknowledges with 0 (single byte)
 *
 * Tne end is signalled by closing the stream, which results in a read error.
 * Side effects:  sets information vector  CFparm[].
 *************************************************************************/
int
Sendjob (FILE *cfp, struct plp_queue *q) {
    char parm[MAXPATHLEN];	/* holds a line read in */
    char *arg;			/* control line argument */
    char opt;			/* control line option */
    int jstatus;		/* job status */
    int i;			/* ACME Integers, Inc. */

    setstatus ("Sending job to %s@%s.", RP, RM);
    setproctitle ("%s: sending job to %s@%s", Printer, RP, RM);

    if (Ctrlfile_logdir && *Ctrlfile_logdir) {
	char buff[128];
	(void) sprintf (buff, ".lpd-out.%s", ShortHost);
	Log_cf (q->q_name, buff, fileno (cfp));
    }

    /*
     * set job status
     */
    jstatus = JFAIL;		/* default */

    /*
     * initialize the CFparm array, which holds value read from file
     */
    for (i = 0; i < 26; ++i) {
        CFparm[i][0] = 0;
    }

    /*
     * read the control file and extract user information
     */
    Parmcount = 0;
    while (fgets (parm, sizeof (parm), cfp)) {
	if ((arg = (char *) strchr (parm, '\n')) == 0) {
	    log (XLOG_INFO, "Sendjob: bad control file format (%s), no endline",
		 q->q_name);
	    jstatus = JABORT;
	    goto error;
	}
	*arg = 0;
	opt = parm[0];
	arg = parm + 1;
	if (!isascii (opt) || !isalnum (opt) || (int) strlen (arg) > MAXPARMLEN) {
	    log (XLOG_INFO, "Sendjob: bad control file (%s), line(%s)",
		 q->q_name, parm);
	    jstatus = JABORT;
	    goto error;
	}

        /*
         * copy it into the appropriate place if needed
	 * (we need this for filters).
         */
        if (isupper (opt)) {
            switch (opt) {
            case 'U':
            case 'N':
                break;
            case 'X':
                opt = 'T';      /* old PLP compat -- stick with rfc */

            default:
                if (CFparm[opt - 'A'][0]) {
                    log (XLOG_INFO,
                         "send to remote: duplicate %c parm '%s', previous: '%s'",
                         opt, arg, CFparm[opt - 'A']);
                }
                if ((int) strlen (arg) >= MAXPARMLEN) {
                    log (XLOG_INFO,
                         "Printjob: control file %s line too long: '%s'",
                         q->q_name, arg);
                }
                (void) strcpy (CFparm[opt - 'A'], arg);
		if (opt == 'P') {
		    Person = LOGNAME;
		}
                break;
            }
	    continue;
	}

	if (Job_match (q->q_name, arg) == 0) {
	    log (XLOG_INFO, "Sendjob: bad file name format (%s)", arg);
	    jstatus = JABORT;
	    goto error;
	}

	if (islower (opt)) {
	    if (stat (arg, &LO_statb) < 0) {
		logerr (XLOG_INFO, "Sendjob: couldn't stat datafile '%s'");
		jstatus = JSUCC;
		goto error;
	    }

            if (is_race_symlink (arg, &LO_statb)) {
                log (XLOG_CRIT,
                    "CRACK ATTEMPT: 'lpr -s' symlink has been rerouted: %s@%s",
                    LOGNAME, FROMHOST);

                /* return a "successfully printed" status so the queue
                 * won't be stopped. */
                jstatus = JSUCC;
		goto error;
            }
	    (void) Add_name (arg);	/* cannot fail -- dynamic */

	} else {
	    if ((i = Find_name (arg)) < 0) {
		log (XLOG_INFO, "Sendjob: job(%s)U specified for '%s' not in job",
		     q->q_name, parm);
		jstatus = JABORT;
		goto error;
	    }
	}
    }
    /*
     * We send the control file first if we are using the PLP protocol
     */
    if (FJ) {
	jstatus = sendfile (cfp, q->q_name, CNAME);
	/* no need to check for WSLPD -- it won't be using FJ. */

	if (jstatus == JFULL) {
	    log_spool_full ();
	    goto error;

	} else if (jstatus != JSUCC) {
	    setstatus ("Control file transfer to %s@%s failed at %s.", RP, RM, Time_str ());
	    /* don't log anything here -- it's already logged */
	    setproctitle ("%s: transfer %s to %s failed.", 
			  Printer, q->q_name, RM);
	    goto error;
	}
    }
    /*
     * send the jobs in the file
     */
    for (i = 0; i < Parmcount; ++i) {
	arg = Parms[i].filename;
	/*
	 * send the file
	 */
	if (DbgRem > 3) {
	    log (XLOG_DEBUG, "Sendjob: STARTING DATA FILE TRANSFER: %s", arg);
	}
	jstatus = senddatafile (arg, DFILE);
	if (jstatus == JNOCONF) {
	    /* one of the data files failed to get through -- need to retry */
	    jstatus = JFAIL;
	}

	if (jstatus == JFULL) {
	    log_spool_full ();
	    if (DbgRem > 3) {
		log (XLOG_DEBUG, "Sendjob: FAILED (spool full) DATA FILE TRANSFER: %s", arg);
	    }
	    goto error;

	} else if (jstatus != JSUCC) {
	    setstatus ("Data file transfer to %s@%s failed at %s.", RP, RM, Time_str ());
	    setproctitle ("%s: transfer to %s@%s failed", Printer, RP, RM);
	    if (DbgRem > 3) {
		log (XLOG_DEBUG, "Sendjob: FAILED DATA FILE TRANSFER: %s", arg);
	    }
	    goto error;
	}
	if (DbgRem > 3) {
	    log (XLOG_DEBUG, "Sendjob: SUCCESSFUL DATA FILE TRANSFER: %s", arg);
	}
    }
    /*
     * send the control file
     */
    if (DbgRem > 3) {
	log (XLOG_DEBUG, "Sendjob: STARTING CONTROL FILE TRANSFER: %s", q->q_name);
    }
    if (FJ) {
	jstatus = sendname (cfp, q->q_name, CEND);
    } else {
	jstatus = sendfile (cfp, q->q_name, CFILE);
    }
    if (jstatus == JNOCONF) {
	/* WSLPD does this; it's in violation of the RFC, but we may as well
 	 * support it. "be liberal in what you accept", as the saying goes.
	 * Basically, WSLPD starts printing the file straightaway, instead of
	 * confirming the transfer; the Link_confirm() call times out, but
	 * the file _is_ printed, so don't retry it. (ugh)
	 */
	if (DbgRem) {
	    log (XLOG_DEBUG, "no confirm received, but job is probably printed");
	    if (DbgRem > 3) {
		log (XLOG_DEBUG,
		    "Sendjob: SUCCESSFUL CONTROL FILE TRANSFER: %s", q->q_name);
	    }
	}

    } else if (jstatus == JFULL) {
	log_spool_full ();
	if (DbgRem > 3) {
	    log (XLOG_DEBUG,
		"Sendjob: FAILED (spool full) CONTROL FILE TRANSFER: %s", q->q_name);
	}
	goto error;

    } else if (jstatus != JSUCC) {
	setstatus ("Control file transfer to %s@%s failed at %s.", RP, RM, Time_str ());
	if (DbgRem > 3) {
	    log (XLOG_DEBUG, "Sendjob: FAILED CONTROL FILE TRANSFER: %s", q->q_name);
	}
	goto error;

    } else {
	if (DbgRem > 3) {
	    log (XLOG_DEBUG, "Sendjob: SUCCESSFUL CONTROL FILE TRANSFER: %s", q->q_name);
	}
	/*
	 * finished!
	 */
	jstatus = JSUCC;
    }

    /*
     * Edinburgh University remote transfer logging addition:
     */
    if (AR && AF && *AF) {	/* remote logging */
	time_t t;
	FILE *f;

	t = time ((time_t *) 0);
	if ((f = fopen (AF, "a")) != NULL) {

	    fprintf (f, "%s\t%s\t%s\t%7d\t%s\t%s",
		     (Person && *Person) ? Person : "NULL",
		     (Host && *Host) ? Host : "NULL",
		     (Printer && *Printer) ? Printer : "NULL",
		     1 /* npages */ ,
		     "remote" /* format */ ,
		     ctime (&t));
	    (void) fclose (f);

	} else {
	    logerr (XLOG_INFO,
		    "can't write acct file '%s' for remote printer %s",
			AF, Printer);
	}
    }

error:
    if (jstatus != JSUCC) {
        Link_close ();
    }
    return (jstatus);
}

/***************************************************************************
 * sendfile (FILE* fp; char *name; int flag)
 *   sendfile protocol:
 * 1. line containing the flag, number of bytes, and file name is sent.
 *     This has the format:
 *        "%c%d %s", flag, filesize, name
 *    DFILE (data file) flag is 03,  a CFILE (control file) flag is 02
 * 2. the remote end responds by sending back a single 0 character;
 *    anything else is an error condition and terminates transfer
 * 3. the file is then copied to the remote end and terminated with 0
 * 4. if an error is detected, we send a non-zero character
 *    and close the link.
 ***************************************************************************/

static int
sendfile (FILE *fp, char *name, int flag) {
    int succ;			/* ACME Integer, Inc. */
    struct stat statb;		/* status buffer */

    if (Job_removed ()) {
	/* the current job has been removed; fake out the xfer */
	succ = Link_ack (2);	/* send "skip job" code */
	Link_close ();
	return (JSUCC);
    }
    /*
     * stat the file
     */
    if (fstat (fileno (fp), &statb) < 0) {
	logerr (XLOG_INFO, "sendfile: could not stat %s", name);
	return (JFAIL);
    }
    /*
     * rewind the file
     */
    if (fseek (fp, 0L, 0) < 0) {
	logerr_die (XLOG_INFO, "sendfile: fseek failed '%s'", name);
    }
    succ = sendname (fp, name, flag);
    if (succ == JSUCC) {
	succ = Link_copy (fp, (long) statb.st_size, name);
    }
    if (succ == JSUCC) {
	succ = Link_ack (0);
    }
    if (succ == JSUCC) {
	succ = Link_confirm ();
    }
    if (succ != JSUCC) {
	Link_close ();		/* in case it isn't already closed */
    }
    return (succ);
}

int 
Open_tmpfile (char *buf) {
    int fd;

    fd = Exlockcf (buf);
    if (DbgRem > 3) {
	log (XLOG_DEBUG, "Open_tmpfile: file %s, fd %d", buf, fd);
    }
    return (fd);
}

static int
senddatafile (char *name, int flag) {
    int succ;			/* ACME Integer, Inc. */
    FILE *raw;
    char *filter, tmpfile[BUFSIZ];

    if (Job_removed ()) {
	/* the current job has been removed; fake out the xfer */
	succ = Link_ack (2);	/* send "skip job" code */
	Link_close ();
	return (JSUCC);
    }
    if ((raw = fopen (name, "r")) == NULL) {
	logerr (XLOG_INFO, "senddatafile: cannot open data file %s", name);
	setstatus ("Cannot open data file %s!", name);
	return (JABORT);
    }
    if (IF == 0 || *IF == 0) {	/* non-filtered -- very boring */
	succ = sendfile (raw, name, flag);
	fclose (raw);
	return (succ);
    }
    /* this remote printer has a text filter... thrills 'n' spills! */

    if (DbgRem > 1) {
	log (XLOG_DEBUG, "senddatafile: IF: starting filter");
    }

    filter = Setup_filter ('f', IF);
    (void) strcpy (tmpfile, name);
    tmpfile[0] = 't';		/* indicate tmp file */

    /*
     * set up an fd, supposedly pointing to the printer, but really pointing to a
     * temporary output file.
     */
    if ((Print_fd = Open_tmpfile (tmpfile)) < 0) {
	return (JFAIL);		/* already logged, just return */
    }
    if (DbgRem) {
	log (XLOG_DEBUG, "senddatafile: starting filter %s", filter);
    }

    succ = Print_filter (fileno (raw), filter, 0);	/* non-local printer mode */
    close (Print_fd);		/* close the output file */
    fclose (raw);		/* close the input file */

    /* if the filter failed, give up here. */
    if (succ != JSUCC) {
	unlink (tmpfile);
	setstatus ("IF filter failed!");
	setproctitle ("%s: if filter failed.", Printer);
	return (succ);
    }
    /* otherwise, send the filtered output to the remote end */

    if (DbgRem > 1) {
	log (XLOG_DEBUG, "senddatafile: sending file %s, name %s, flag %d",
	     tmpfile, name, flag);
    }
    raw = fopen (tmpfile, "r");
    succ = sendfile (raw, name, flag);
    unlink (tmpfile);
    return (succ);
}

/***************************************************************************
 * Called when you have retried a couple of times
 ***************************************************************************/
void
Send_error (void) {
    log (XLOG_CRIT, "job cannot be sent from %s to %s", Host, RM);
    Link_close ();
}

/***************************************************************************
 * sendname (FILE *fp; char *name; int flag)
 *   sendname protocol:
 * 1. line containing the flag and file name is sent
 *     This has the format:
 *        "%c%d %s", flag, filesize, name
 *    DFILE (data file) flag is 03,  a CFILE (control file) flag is 02
 * 2. the remote end responds by sending back a single 0 character;
 *    anything else is an error condition and terminates transfer
 ***************************************************************************/
static int
sendname (FILE *fp, char *name, int flag) {
    struct stat statb;		/* status buffer */
    int succ;			/* ACME Integer, Inc. */
    char buf[BUFSIZ];

    if (Job_removed ()) {
	/* the current job has been removed; fake out the xfer */
	succ = Link_ack (1);	/* send "skip job" code */
	Link_close ();
	return (JSUCC);
    }

    /*
     * stat the file
     */
    if (fstat (fileno (fp), &statb) < 0) {
	logerr (XLOG_INFO, "sendname: could not stat %s", name);
	return (JFAIL);
    }

    (void) sprintf (buf, "%c%ld %s\n", flag, statb.st_size, name);
    succ = Link_send (buf);
    if (succ == JSUCC) {
	succ = Link_confirm ();
    }
    return (succ);
}

/***************************************************************************
 *    all files have been sent; send a closing 0 and shut down link
 ***************************************************************************/
void
Allsent (void) {
    (void) Link_ack (0);
    Link_close ();
}

