/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: displayq.c
 * Display the status of a Printer queue
 ***************************************************************************/

#include "lp.h"
#include "common/displayq.h"
#include "library/errormsg.h"
#include "library/utils.h"

static int header;
static int server;

static char *short_time(time_t);
static void showentry(int, struct plp_queue *);
static void showstatus(int);

extern void chdir_SD (void);

/***************************************************************************
 * Print either a file or the comment field itself.
 ***************************************************************************/
static void Print_comment(char *str)
{
    FILE *f;			/* a file to print */
    int pc, c;

    assert(str!=(char*)0);
    if (!strchr (str, ' ')) {
	if ((f = fopen (str, "r")) != NULL) {
	    putc (' ', stdout);
	    if ((pc = getc (f)) != EOF) {
		while ((c = getc (f)) != EOF) {
		    putc (pc, stdout);
		    pc = c;
		}
		if (pc != '\n') {
		    putc (pc, stdout);
		}
	    }
	}
    } else {
	(void) fputs (str, stdout);
    }
}

/***************************************************************************
 *  1. get the printcap entries
 *  2. get the queue entries
 *  3. get the active server files and see if they are in queue
 *  4. print the queue status to stdout
 ***************************************************************************/

void Displayq(int zeige)
{
    int i;			/* ACME Integers, Inc. */
    struct plp_queue *q, *top_job;	/* job entries */
    char buf[BUFSIZ];		/* status message buffer */
    FILE *afp;			/* attach file pointer */
    char afname[MAXPARMLEN];	/* attach file name */
    char unattached_printer[MAXPARMLEN]; /* Name of the unattached printer */
    short att_mark = 0;         /* Flag if a printer is attached to another printer */
    int seq;
    int k;                      /* Flag, used in showstatus; if jobs are held, but
				 * printing is enabled k has the value of 0, else 1 */

    header = 0;
    k = 0;

reattach:			/* restart from here if printer is attached to another */
    /*
     * get the printcap entry
     */
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	(void) printf ("Printer or queue '%s' does not exist!\n", Printer);
	(void) fflush (stdout);
	return;
    }
    if (SD == 0 || *SD == 0) {
	if (Debug > 0)
	    log (XLOG_DEBUG, "Printer or Queue '%s' does not have a spool queue.\n",
		 Printer);
	return;
    }
    if (Debug > 5)
	log (XLOG_DEBUG, "Displayq: doing %s on host %s", Printer, Host);
    /*
     * If you have an NFS spool dir, use the remote lpd.
     */
    if (RM && *RM && NW) {
	Remote_status ();
	return;
    }
    /* chdir to spool directory */
    chdir_SD ();

    /*
     * check to see if attached to another printerq, alter printer if attached
     */
    if (Attach_file && *Attach_file && ((afp = fopen (Attach_file, "r")))) {
	if (fscanf (afp, "%s", afname) == 1) {
	    if (strsame (afname, Printer)) {
		fatal (XLOG_INFO, "Printer '%s' attached to itself", Printer);
	    }
	    (void)strcpy(unattached_printer, Printer);
	    Printer = afname;
	    att_mark = 1;
	    fclose (afp);
	} else {
	     fatal (XLOG_INFO, 
		    "Fatal: attach file for printer '%s' corrupted!\nYou have to remove it by hand!",
		    Printer);
	}

	if (Debug > 2)
	    log (XLOG_INFO, "Displayq: printer %s reattached to %s", unattached_printer, Printer);
	fclose (afp);
	goto reattach;
    }
    /*
     * get the numbers of jobs here
     */
    Jobcount = Getq ();
    /*
     * Checkactive checks for the active jobs
     */
    server = Checkactive ();
    buf[0] = 0;
    if (LO_statb.st_mode & DISABLE_QUEUE) {
	(void) strcat (buf, "no spooling");
	k = 1;
    }
    if (LO_statb.st_mode & DISABLE_PRINT) {
	if (buf[0]) {
	    (void) strcat (buf, ", ");
	}
	k = 1;
	(void) strcat (buf, "printing disabled");
    }
    if (Show_empty_queues ? 1 :
	((Jobcount > 0) || (LO_statb.st_mode & (DISABLE_QUEUE | DISABLE_PRINT))
	|| (RM == NULL) || (*RM == '\0')))
    {
	if (Short_format) {
	    if (FQDN == Host) {
		if (att_mark) {
		    (void) printf ("%s=>%s: %d jobs", unattached_printer, Printer, Jobcount);
		} else {
		(void) printf ("%s: %d jobs", Printer, Jobcount);
		}
	    } else {
		if (att_mark) {
		(void) printf ("%s@%s=>%s@%s: %d jobs",
			       unattached_printer, Host, Printer, Host, Jobcount);
		} else {
		(void) printf ("%s@%s: %d jobs",
			       Printer, Host, Jobcount);
		}
	    }
	    if (buf[0]) {
		(void) printf (" (%s)", buf);
	    }
	    (void) printf ("\n");
	} else {
	    if (FQDN != Host) {
		(void) fputs ("Remote ", stdout);
	    } else {
		(void) fputs ("Local  ", stdout);
            }
	    if (SV == 0 || *SV == 0 ) {
		if (RM == 0 || *RM == 0) {
		    (void) fputs ("Printer ", stdout);
		} else {
		    (void) fputs ("Queue   ", stdout);
		} 
            } else {
		(void) fputs ("Queue   ", stdout);
            }

	    if (att_mark) {
	    (void) printf ("'%s' [attached on '%s'] (%s", unattached_printer, Printer, Host);
	    } else {
	    (void) printf ("'%s' (%s", Printer, Host);
	    }
	    if (CM && *CM) {
		(void) fputs (", ", stdout);
		Print_comment (CM);
	    }
	    (void) fputs ("):", stdout);

	    if (buf[0]) {
		(void) printf (" (%s)", buf);
	    }
	    (void) fputs ("\n", stdout);
	    /*
	     * show status files
	     */
	    showstatus (k);
	}
    }
    (void) fflush (stdout);

    seq = 1;

    /* go thru list of jobs first to find the active job(s).
     * display it straightaway, irrespective of priority.
     */
    for (i = 0; i < Jobcount; ++i) {
	q = &Queue[i];
	if (q->q_daemon && (Parmcount <= 0 || Match_entry (q))) {
	    showentry (seq, q);
	    seq++;
	}
    }

    /* now, display the non-active jobs. */
    for (i = 0; i < Jobcount; ++i, ++seq) {
	q = &Queue[i];
	if (q->q_daemon == 0) {
	    if (Parmcount <= 0 || Match_entry (q)) {
		showentry (seq, q);
	    }
	}
    }

    if (zeige) {
	if (RM && *RM) {
	    if (Debug > 4)
		log (XLOG_DEBUG, "Displayq: getting Remote_status():");
	    Remote_status ();
	}
    }
    if (FQDN == Host) {
	putchar ('\n');
    }
}

/*****************************************************************************
 *  show status for entry
 * Rank   Owner      Pr Job  Host         Files                Size
 * active sutantha   Z  260  umn-cs       (stdin)              177744 bytes
 * 2nd    powell     Z    1  attila       junk, morejunk,...    32767 bytes
 *
 * if you have multiple servers, the first column will contain the name
 * of the server.
 * Rank   Owner      Pr Job  Host         Files                Size
 * sv1    sutantha   Z  260  umn-cs       (stdin)              177744 bytes
 * sv2    powell     Z    1  attila       junk, morejunk,...    32767 bytes
 *****************************************************************************/

#define RANK_COLUMN_WIDTH	8
static char hdr_format[] = "%8s %-11s %-2s %-4s %-3s %-11s %-15.15s %-8s %-s\n";
static char info_format[] = "%8s %-11s %-c  %-4s %-3d %-11s %-15.15s %-8d %s\n";

static char *short_time (time_t tvec) {
    static char s[40];

    /*
     * 012345678901234567890123456789
     * Mon May 30 20:53:27 CDT 1988
     */
    if (tvec == (time_t)0) return ("-");

    (void) strcpy (s, ctime (&tvec));
    s[16] = 0;
    return (&s[4]);
}

static void showentry(int seq, struct plp_queue *q) {
    char buf[10];
    char *suffix;
    char name[BUFSIZ];

    assert(q!=(struct plp_queue*)0);
    if (Short_format) {
	(void) printf ("%d %s: %s\n", seq, q->q_name, q->q_user);
	(void) fflush (stdout);
	return;
    }
    if (header == 0) {
	(void) printf (hdr_format, "Rank", "Owner", "Pr", "Opt",
		       "Job", "Host", "Files", "Size", "Date");
	header = 1;
    }
    if (q->q_daemon) {
	if (SV == 0 || *SV == 0) {
	    (void) strcpy (buf, "active");
	} else {
	    char *abbrev_name;
	    int namelen;

	    namelen = strlen (q->q_server);
	    if (namelen >= RANK_COLUMN_WIDTH) {
		(void) strcpy (buf, q->q_server + (namelen - RANK_COLUMN_WIDTH));
		buf[0] = '/';
	    } else {
		(void) strcpy (buf, q->q_server);
	    }
	}
    } else {
	switch (seq) {
	case 1:
	    suffix = "st";
	    break;
	case 2:
	    suffix = "nd";
	    break;
	case 3:
	    suffix = "rd";
	    break;
	default:
	    suffix = "th";
	    break;
	}
	(void) sprintf (buf, "%d%s", seq, suffix);
    }

    if ((suffix = strchr (q->q_user, '@')) == NULL) {
	/*
	 * print short name
	 */
	(void) strcpy (name, &q->q_from);
	if ((suffix = strchr (name, '.')) != NULL)
	    *suffix = 0;
	(void) printf (info_format, buf, q->q_user,
		       q->q_held ? '*' : q->q_priority,
		       q->q_options ? q->q_options : DEF_OPTIONS, q->q_num,
		       name, q->q_data, q->q_size, short_time (q->q_time));
    } else {
	(void) strcpy (name, q->q_user);
	name[suffix - q->q_user] = '\0';
	(void) printf (info_format, buf, name,
		       q->q_held ? '*' : q->q_priority,
		       q->q_options ? q->q_options : DEF_OPTIONS, q->q_num,
		       suffix + 1, q->q_data, q->q_size, short_time (q->q_time));
    }
    (void) fflush (stdout);
}

/***************************************************************************
 * show the status files for the server devices
 ***************************************************************************/
static void showstatus (int k) {
    static char server_name[BUFSIZ];	/* Name of server file */
    int active;
    pid_t pid;
    char buf[BUFSIZ];
    char *sp, *ep;		/* ACME Pointer */
    char saved_Printer[PRNAMELEN];

    /*
     * If there are jobs in the queue, but no server running, print a warning message.
     */
    if (SV == 0 || *SV == 0) {
	if (Debug > 2) {
	    log (XLOG_DEBUG, "showstatus: value of k %d and sever %d", k, server);
	}
	if (Jobcount > 0 && server == 0 && k) {
	    (void) printf ("Warning: no server present.\n");
	/*
	 * For printers that support forms and/or size limits, suggest why nothing is
	 * moving.
	 */
	    if (!(LO_statb.st_mode & DISABLE_PRINT) && (
				   (MX > 0)
#ifdef EUCS_FORMS
				   || (FL && *FL)
#endif
				   )) {
		(void) printf ("This may be because");

		if (MX > 0) {
		    (void) printf (" all the jobs in the queue are too big\n");
		}
#ifdef EUCS_FORMS
		if (FL && *FL) {
		    if (MX > 0) {
			(void) printf ("or because");
		    }
		    (void) printf (" all the pending jobs need a different form type\n");
		    (void) printf ("from what's currently mounted\n");
		}
#endif				/* EUCS_FORMS */
	    }
	}
	printstatus ();
	return;
    }
    /*
     * check for each of the servers
     */
    (void) strcpy (server_name, SV);
    strcpy (saved_Printer, Printer);

    for (sp = server_name; sp; sp = ep) {
	int print_period;

	if ((ep = strchr (sp, ','))) {
	    *ep = 0;
	    ++ep;
	}
	strcpy (Printer, sp);
	/*
	 * get the lock file and the status from the server
	 */
	if (Set_pc_entry (sp, Server_pc_vars, Server_pc_len) == 0) {
	    fatal (XLOG_INFO, "no server entry for %s", sp);
	}
	buf[0] = 0;
	active = Checklockfile (sp, &pid, buf, sizeof (buf), &LO_statb);

	(void) printf (" Server '%s'", sp);

	/* display the comment field */
	if (CM && *CM) {
	    (void) fputs (" (", stdout);
	    Print_comment (CM);
	    (void) fputs (")", stdout);
	}
	printf (":");

	if (active) {
	    (void) printf (" (active)");
	}
	if (LO_statb.st_mode & DISABLE_PRINT) {
	    (void) printf (" (printing disabled)");
	}

	(void) printf ("\n");
	(void) fflush (stdout);
	printstatus ();
    }
    strcpy (Printer, saved_Printer);
}
