/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: rmjob.c
 * Remove jobs from a printer queue
 **************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/utils.h"

static void Print_comment_rm(char *);
static int remove_all;		/* remove all jobs */

/***********************************************************************
 *  a simple application of the removal decision table
 *
 *  user   local  job   from  remove_all    perms inlist    remove?
 *  ===============================================================
 *  root   yes    *     *     yes           *     *         yes
 *  root   yes    *     *     no            *     yes       yes
 *  root   no     *     mach  yes           *     *         yes
 *  root   no     *     mach  no            *     yes       yes
 *  user   -      user  mach  -             R     yes       yes
 *  user   -      *     *     -             C     yes       yes
 *
 * Returns: 1 if removal is indicated, 0 otherwise
 ***********************************************************************/
static int
shouldremove (struct plp_queue *q, int control_perms) {
    int i, same_host;

    assert(q!=(struct plp_queue*)0);
    if (Debug > 5)
	log (XLOG_DEBUG, "shouldremove: Person = %s, From = %s, job %s@%s",
	     Person, FQDN, q->q_user, &q->q_from);
    same_host = (hostcmp (FQDN, &q->q_from) == 0);
    i = Match_entry (q);
    if (Debug > 5)
	log (XLOG_DEBUG, "shouldremove: Match_entry = %d", i);
    if (Is_root && Is_local && remove_all)
	return (1);
    if (Is_root && Is_local && !remove_all && i)
	return (1);
    if (Is_root && !Is_local && same_host && remove_all)
	return (1);
    if (Is_root && !Is_local && same_host && !remove_all && i)
	return (1);
    if (strsame (Person, q->q_user) && i)
	return (1);
    if (i && control_perms)
	return (1);
    return (0);
}

/***************************************************************************
 * remove the job
 * 1. Lock the control file.
 * 2. If unsuccessful, find the server PID and kill it off.
 * 3. Use brute force and remove the files.
 ***************************************************************************/

static void
doremove (struct plp_queue *q) {
    FILE *cfp;
    int r;

    assert(q!=(struct plp_queue*)0);
    if ((cfp = Lockcf (q->q_name)) == NULL) {
	/* hmmm... looks like an active server */
	if ((cfp = fopen_daemon (q->q_name, "r")) == NULL) {
	    /* nope, the file has really gone */
	    logerr (XLOG_INFO, "control file %s not readable", q->q_name);
	    return;
	}
	/* well, we will just have to kill off this server */
	if (q->q_daemon == 0) {
	    /*
	     * Hmmm... we have this fellow running the file, and it is locked.  That
	     * means that it just started running this guy. Better check again.
	     */
	    (void) Checkactive ();
	}
	if (q->q_daemon) {
	    (void) fprintf (stdout, "Killing off %s server %d.\n",
			    q->q_server, q->q_daemon);

	    if (killpg (q->q_daemon, SIGINT) < 0) {
		if (Debug > 2)
		    log (XLOG_DEBUG,
			 "server %s (%d) was not alive",
			 q->q_server, q->q_daemon);
	    }
	}
    }
    /* use brute force;  we simply remove files */
    if (Debug > 3)
	log (XLOG_DEBUG, "removing files for job %s", q->q_name);
    Remove_job (cfp, q);
    r = fclose (cfp);
    assert(r==0);
}

/***************************************************************************
 * rmjob()
 *  1. get the printcap entries
 *  2. get the queue entries
 *  3. get the active server files and see if they are in queue
 *  4. scan the queue, checking each job for removal
 *
 *  user   local  job   from  remove_all    perms inlist    remove?
 *  ===============================================================
 *  root   yes    *     *     yes           *     *         yes
 *  root   yes    *     *     no            *     yes       yes
 *  root   no     *     mach  yes           *     *         yes
 *  root   no     *     mach  no            *     yes       yes
 *  user   -      user  mach  -             R     yes       yes
 *  user   -      *     *     -             C     yes       yes
 *
 *  5. remove the job;  this may necessitate stopping or killing a deamon
 ***************************************************************************/

void
rmjob (void) {
    int i;			/* ACME Integers, Inc. */
    struct plp_queue *q;	/* job entry */
    char buf[BUFSIZ];           /* status message buffer */
    int perms;			/* hold perms values */
    int control_perms;		/* has control perms */
    FILE *afp;			/* attach file pointer */
    char afname[MAXPARMLEN];	/* attach file name */
    char unattached_printer[MAXPARMLEN]; /* Name of the unattached printer */
    short att_mark = 0;         /* Flag if a printer is attached to another printer */

    /*
     * If explicitly asked for debug then echo messages to stdout otherwise (db flag in
     * printcap) just log messages in file.
     */
    Echo_on_stdout = Debug;

    /*
     * get the printcap entry
     */
reattach:			/* restart here if printer attached */
    if (Get_pc_entry (Printer, Status_pc_vars, Status_pc_len) == 0) {
	(void) fprintf (stdout, "Printer or Queue %s does not exist!\n", Printer);
	(void) fflush (stdout);
	return;
    }
    if (SD == 0 || *SD == 0) {
	if (Debug > 2)
	    log (XLOG_DEBUG, "not a Printer");
	return;
    }
    chdir_SD ();
    /*
     * check to see if attached to another printerq, alter printer if attached
     */
    if (*Attach_file && ((afp = fopen (Attach_file, "r")))) {
	if (fscanf (afp, "%s", afname) == 1) {
	    if (strsame (afname, Printer)) {
		fatal (XLOG_INFO, "Printer %s attached to itself", Printer);
	    }
	    (void)strcpy (unattached_printer, Printer);
	    Printer = afname;
	    att_mark = 1;
	}
	if (Debug > 2)
	    log (XLOG_INFO, "printer reattached to %s", Printer);
	fclose (afp);
	goto reattach;
    }
    /*
     * set the flags needed
     */
    Is_local = (hostcmp (FQDN, Host) == 0);
    Is_root = strsame (Person, "root");
    remove_all = (Parmcount > 0 && strsame (Parms[0].str, "-all"));
    /*
     * check to see that the user has RMJOB privs on this machine
     */
    perms = 'R';		/* must be able to at least use the Printer */
    if (!Is_root &&
	Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) <= 0) {
	(void) fprintf (stdout, 
		"Sorry user %s, but you do not have remove permission on '%s'!",
			Person, First_name);
	log_request (
		     "%s@%s: remove job '%s' from queue '%s' failed: no permission",
		     Person, From, (Parmcount > 0) ? Parms[0].str : "unspecified",
		     First_name);
	if (Debug > 4)
	    log (XLOG_DEBUG,
		 "rmjob: root %d, local %d, rm_all %d, ctrl_perms %d (failed)",
		 Is_root, Is_local, remove_all, control_perms);
	return;
    }
    perms = 'C';		/* check for control perms */
    control_perms = 1;
    if (!Is_root && Checkperm (FQDN, Person, First_name, &perms, (int *) 0, 0) <= 0)
    {
	control_perms = 0;
    }
    if (Debug > 4)
	log (XLOG_DEBUG,
	     "rmjob: Is_root %d, Is_local %d, remove_all %d, control_perms %d",
	     Is_root, Is_local, remove_all, control_perms);
    /*
     * check for remote machine and networked file system
     */
    if (RM && NW) {
	Remote_remove ();
	return;
    }
    /*
     * get the job queue
     */
    Jobcount = Getq ();
    /*
     * Checkactive checks for the active jobs
     */
    (void) Checkactive ();
/*    (void) fprintf (stdout, "Printer '%s' (%s):\n", Printer, Host); */
    buf[0] = 0;
    if (LO_statb.st_mode & DISABLE_QUEUE) {
        (void) strcat (buf, "no spooling");
    }
    if (LO_statb.st_mode & DISABLE_PRINT) {
        if (buf[0]) {
            (void) strcat (buf, ", ");
        }
	(void) strcat (buf, "no printing");
    }
    
    if (FQDN != Host) {
	(void) fputs ("Remote ", stdout);
    } else {
	(void) fputs ("Local  ", stdout);
    }
    if (SV == 0 || *SV == 0 ) {
	if (RM == 0 || *RM == 0) {
	    (void) fputs ("Printer ", stdout);
	} else {
	    (void) fputs (" Queue  ", stdout);
	}
    } else {
	(void) fputs (" Queue  ", stdout);
    }
    
    if (att_mark) {
	(void) printf ("'%s' [attached on '%s'] (%s", unattached_printer, Printer, Host);
    } else {
	(void) printf ("'%s' (%s", Printer, Host);
    }
    if (CM && *CM) {
	(void) fputs (", ", stdout);
	Print_comment_rm (CM);
    }
    (void) fputs ("):", stdout);
    
    if (buf[0]) {
	(void) printf (" (%s)", buf);
    }
    (void) fputs ("\n", stdout);

    (void) fflush (stdout);

    /*
     * run down list
     */
    for (i = 0; i < Jobcount; ++i) {
	q = &Queue[i];
	if (Debug > 5)
	    log (XLOG_DEBUG, "rmjob: checking %s, %d, %s",
		 q->q_name, q->q_num, q->q_user);
	if (shouldremove (q, control_perms)) {
	    (void) fprintf (stdout, "Removing %s, job %d owner %s from printer '%s'.\n",
			    q->q_name, q->q_num, q->q_user, Printer);

	    if (Log_LPRMs) {
		if (Log_no_filenames) {
		    log_request ("lprm job %d from queue %s for %s@%s",
				 q->q_num, Printer, Person, FQDN);
		} else {
		    log_request ("lprm %s(%d) from queue %s for %s@%s",
				 q->q_data, q->q_num, Printer, Person, FQDN);
		}
	    }
	    (void) fflush (stdout);
	    doremove (q);
	}
    }
    /*
     * check for remote machine
     */
    if (RM) {
	Remote_remove ();
    }
    /*
     * give the server a kick
     */
    (void) Startserver ();
}

/***************************************************************************
 * Print either a file or the comment field itself.
 ***************************************************************************/
static void Print_comment_rm (char *str)
{
    FILE *f;                    /* a file to print */
    int pc, c;

    assert(str != (char*)0);
    if (!strchr (str, ' ')) {
        if ((f = fopen (str, "r")) != NULL) {
            putc (' ', stdout);
            if ((pc = getc (f)) != EOF) {
                while ((c = getc (f)) != EOF) {
                    putc (pc, stdout);
                    pc = c;
                }
                if (pc != '\n') {
                    putc (pc, stdout);
                }
            }
        }
    } else {
        (void) fputs (str, stdout);
    }
}
