/*******************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 *******************************************************************
 * Some stuff for Solaris and other SVR4 impls; emulate BSD sm'tics.
 * feel free to delete the UCB stuff if we need to tighten up the
 * copyright -- it's just there to help porting.
 ******************************************************************/

#include "lp.h"
#include "library/errormsg.h"

/**************************************************************/

/* signal handling: SIGALRM should be the only signal
 * that interrupts system calls; all other signals should
 * NOT interrupt them.
 * here's a signal() emulation function to do just that.
 * (nicked from Advanced Programming in the UNIX Environment, Stevens)
 */
#ifdef HAVE_SIGACTION

/* solaris 2.3 note: don't compile this with "gcc -ansi -pedantic";
 * due to a bug in the header file, struct sigaction doesn't
 * get declared. :(
 */

plp_sigfunc_t plp_signal (int signo, plp_sigfunc_t func)
{
    struct sigaction act, oact;

    act.sa_handler = func;
    (void) sigemptyset (&act.sa_mask);
    act.sa_flags = 0;
    if (signo == SIGALRM) {
#ifdef SA_INTERRUPT
	act.sa_flags |= SA_INTERRUPT;		/* SunOS */
#endif
    } else {
#ifdef SA_RESTART
	act.sa_flags |= SA_RESTART;		/* SVR4, 4.3+BSD */
#endif
    }
    if (sigaction (signo, &act, &oact) < 0) {
	return (SIG_ERR);
    }
    return (plp_sigfunc_t) oact.sa_handler;
}

#else /* HAVE_SIGACTION */

plp_sigfunc_t plp_signal (int signo, plp_sigfunc_t func)
{
    /* sigaction is not supported. Just set the signals. */
    return (plp_sigfunc_t)signal (signo, func); 
}
#endif

/**************************************************************/

#ifdef HAVE_SIGPROCMASK
static sigset_t oblock;

void 
plp_block_signals () {
    sigset_t block;

    (void) sigemptyset (&block);
    (void) sigaddset (&block, SIGCHLD);
    (void) sigaddset (&block, SIGHUP);
    (void) sigaddset (&block, SIGINT);
    (void) sigaddset (&block, SIGQUIT);
    (void) sigaddset (&block, SIGTERM);
    if (sigprocmask (SIG_BLOCK, &block, &oblock) < 0)
	perror ("sigprocmask");
}

void 
plp_unblock_signals () {
    (void) sigprocmask (SIG_SETMASK, &oblock, (sigset_t *) NULL);
}

#else /* HAVE_SIGPROCMASK */

static int omask;

void 
plp_block_signals () {
    omask = sigblock (sigmask (SIGCHLD) | sigmask (SIGHUP)
		      | sigmask (SIGINT) | sigmask (SIGQUIT) | sigmask (SIGTERM));
}

void 
plp_unblock_signals () {
    (void) sigsetmask (omask);
}
#endif

/**************************************************************/

#ifndef HAVE_GETWD
char *
getwd (char *path) {
    /* MAXPATHLEN -- sys/param.h is included by lp.h */
    if (getcwd (path, MAXPATHLEN)) {
	return path;
    } else {
	char *x;

	x = Errormsg (errno);
	strcpy (path, x);
	return (char *) NULL;
    }
}
#endif

#ifndef HAVE_STRCASECMP
/* case insensitive compare for OS without it */
int
strcasecmp (char *s1, char *s2) {
    char c1, c2;
    for (;;) {
	c1 = *s1;
	c2 = *s2;
	if (isalpha (c1) && isalpha (c2)) {
	    c1 = tolower (c1);
	    c2 = tolower (c2);
	}
	if (c1 < c2)
	    return (-1);
	else if (c1 > c2)
	    return (1);
	if (c1 && c2) {
	    s1++;
	    s2++;
	} else
	    return (0);
    }
}
#endif				/* HAVE_STRCASECMP */

#ifndef HAVE_STRSTR
/*-
 * Copyright (c) 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Chris Torek.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that: (1) source distributions retain this entire copyright
 * notice and comment, and (2) distributions including binaries display
 * the following acknowledgement:  ``This product includes software
 * developed by the University of California, Berkeley and its contributors''
 * in the documentation or other materials provided with the distribution
 * and in all advertising materials mentioning features or use of this
 * software. Neither the name of the University nor the names of its
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * Find the first occurrence of find in s.
 */
char *
strstr (char *s, char *find) {
    register char c, sc;
    register int len;

    if ((c = *find++) != 0) {
	len = strlen (find);
	do {
	    do {
		if ((sc = *s++) == 0)
		    return (NULL);
	    } while (sc != c);
	} while (strncmp (s, find, len) != 0);
	s--;
    }
    return ((char *) s);
}
#endif				/* HAVE_STRSTR */

#if 0 /* #ifndef HAVE_WAITPID */

/* a waitpid function that caches already-caught dead kids,
 * so it's guaranteed to work without hassle. You may not
 * need this at all, but if plp_waitpid() (the macro) starts
 * giving you errors, you do.
 *
 * this doesn't work right now. Hack on it if you need it.
 */

/* a buffer for kids we've already caught (should be dynamic). */
#define KID_BUFFERSZ	50
static int already_sz = 0;
static pid_t already_pids[KID_BUFFERSZ];
static int already_stats[KID_BUFFERSZ];

static void
add_to_already (pid_t thispid, int statp) {
    already_pids[already_sz] = thispid;
    already_stats[already_sz] = statp;
    if (already_sz++ >= KID_BUFFERSZ) {
	logerr (XLOG_INFO, "overspilled wait() buffer");
    }
}

pid_t
plp_waitpid (pid_t wantpid, int *statp, int opts) {
    int i;
    pid_t thispid;

    if (wantpid > 0) {
	/* check to see if we've already caught this proc before. */
	if (already_sz) {
	    for (i = 0; i < already_sz; i++) {
		if (wantpid == already_pids[i]) {
		    /* found the desired pid in the already-caught buffer. */
		    *statp = already_stats[i];
		    for (; i < already_sz - 1; i++) {
			already_pids[i] = already_pids[i+1];
			already_stats[i] = already_stats[i+1];
		    }
		    already_sz--;
		    return (wantpid);
		}
	    }
	}

	/* nope, we haven't; wait for it here. */
	while ((thispid = wait (statp)) > 0 && thispid != wantpid) {
	    if (Debug > 3) {
		log (XLOG_DEBUG, "plp_waitpid: caught pid=%d (%s)",
			thispid, Decode_status (statp));
	    }
	    add_to_already (thispid, *statp);
	}
	return (thispid);

    } else if (wantpid == -1) {
	if (already_sz) {
	    thispid = already_pids[0];
	    *statp = already_stats[0];
	    for (i = 0; i < already_sz - 1; i++) {
		already_pids[i] = already_pids[i+1];
		already_stats[i] = already_stats[i+1];
	    }
	    already_sz--;
	    return (thispid);
	}
	return (wait (statp));

    } else if (wantpid == 0) {
	/* proc grp == this proc grp, any kid */
	logerr_die (XLOG_INFO, "plp_waitpid: waitpid(0, ...) unsupported");

    } else if (wantpid < -1) {
	/* proc grp == -wantpid, any kid */
	logerr_die (XLOG_INFO, "plp_waitpid: waitpid(< -1, ...) unsupported");
    }
}
#endif

#ifndef HAVE_VPRINTF
#ifndef HAVE_DOPRNT
error ! need at least _doprnt to implement vprintf!!
#else
/* souped together from BSD 4.3 sources and trial&error
 * experiments on BSD 4.2 (Pyramid OSx 4.1, that is)
 * joost@cadlab.de
 */
extern int _doprnt(char*, va_list, FILE*);

vprintf(fmt, args)
char *fmt;
va_list args;
{
	_doprnt(fmt, args, stdout);
	return(ferror(stdout)? EOF: 0);
}

vfprintf(iop, fmt, args)
register FILE *iop;
char *fmt;
va_list args;
{
	char localbuf[BUFSIZ];

	if (iop->_flag & _IONBF) {
		iop->_flag &= ~_IONBF;
		iop->_ptr = iop->_base = localbuf;
		iop->_bufsiz = BUFSIZ;
		_doprnt(fmt, args, iop);
		fflush(iop);
		iop->_flag |= _IONBF;
		iop->_base = NULL;
		iop->_bufsiz = NULL;
		iop->_cnt = 0;
	} else
		_doprnt(fmt, args, iop);
	return(ferror(iop)? EOF: 0);
}

char *vsprintf(str, fmt, args)
char *str, *fmt;
va_list args;
{
	FILE _strbuf;

	_strbuf._flag = _IOWRT+_IOSTRG;
	_strbuf._ptr = str;
	_strbuf._base= str;
#ifdef _NFILE
	_strbuf._file= _NFILE;
#endif
	_strbuf._cnt = 32767;
	_doprnt(fmt, args, &_strbuf);
	putc('\0', &_strbuf);
	return(str);
}

#endif /* HAVE_DOPRNT */
#endif /* ! HAVE_VPRINTF */


#ifndef HAVE_STRDUP
char *
strdup (char *p) {
    char *new;

    malloc_or_die (new, strlen (p) + 1);
    return strcpy (new, p);
}
#endif

#ifndef HAVE_STRTOUL
/* 
 *	Source code for the "strtoul" library procedure.
 * taken from TCL 7.1 by John Ousterhout (ouster@cs.berkeley.edu).
 *
 * Copyright (c) 1988 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <ctype.h>

/*
 * The table below is used to convert from ASCII digits to a
 * numerical equivalent.  It maps from '0' through 'z' to integers
 * (100 for non-digit characters).
 */

static char cvtIn[] = {
    0, 1, 2, 3, 4, 5, 6, 7, 8, 9,		/* '0' - '9' */
    100, 100, 100, 100, 100, 100, 100,		/* punctuation */
    10, 11, 12, 13, 14, 15, 16, 17, 18, 19,	/* 'A' - 'Z' */
    20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
    30, 31, 32, 33, 34, 35,
    100, 100, 100, 100, 100, 100,		/* punctuation */
    10, 11, 12, 13, 14, 15, 16, 17, 18, 19,	/* 'a' - 'z' */
    20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
    30, 31, 32, 33, 34, 35};

/*
 *----------------------------------------------------------------------
 *
 * strtoul --
 *
 *	Convert an ASCII string into an integer.
 *
 * Results:
 *	The return value is the integer equivalent of string.  If endPtr
 *	is non-NULL, then *endPtr is filled in with the character
 *	after the last one that was part of the integer.  If string
 *	doesn't contain a valid integer value, then zero is returned
 *	and *endPtr is set to string.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

unsigned long int
strtoul(const char *string, char **endPtr, int base)
    /* char *string;		 * String of ASCII digits, possibly
				 * preceded by white space.  For bases
				 * greater than 10, either lower- or
				 * upper-case digits may be used.
				 */
    /* char **endPtr;		 * Where to store address of terminating
				 * character, or NULL. */
    /* int base;		 * Base for conversion.  Must be less
				 * than 37.  If 0, then the base is chosen
				 * from the leading characters of string:
				 * "0x" means hex, "0" means octal, anything
				 * else means decimal.
				 */
{
    register char *p;
    register unsigned long int result = 0;
    register unsigned digit;
    int anyDigits = 0;

    /*
     * Skip any leading blanks.
     */

    p = string;
    while (isspace(*p)) {
	p += 1;
    }

    /*
     * If no base was provided, pick one from the leading characters
     * of the string.
     */
    
    if (base == 0)
    {
	if (*p == '0') {
	    p += 1;
	    if (*p == 'x') {
		p += 1;
		base = 16;
	    } else {

		/*
		 * Must set anyDigits here, otherwise "0" produces a
		 * "no digits" error.
		 */

		anyDigits = 1;
		base = 8;
	    }
	}
	else base = 10;
    } else if (base == 16) {

	/*
	 * Skip a leading "0x" from hex numbers.
	 */

	if ((p[0] == '0') && (p[1] == 'x')) {
	    p += 2;
	}
    }

    /*
     * Sorry this code is so messy, but speed seems important.  Do
     * different things for base 8, 10, 16, and other.
     */

    if (base == 8) {
	for ( ; ; p += 1) {
	    digit = *p - '0';
	    if (digit > 7) {
		break;
	    }
	    result = (result << 3) + digit;
	    anyDigits = 1;
	}
    } else if (base == 10) {
	for ( ; ; p += 1) {
	    digit = *p - '0';
	    if (digit > 9) {
		break;
	    }
	    result = (10*result) + digit;
	    anyDigits = 1;
	}
    } else if (base == 16) {
	for ( ; ; p += 1) {
	    digit = *p - '0';
	    if (digit > ('z' - '0')) {
		break;
	    }
	    digit = cvtIn[digit];
	    if (digit > 15) {
		break;
	    }
	    result = (result << 4) + digit;
	    anyDigits = 1;
	}
    } else {
	for ( ; ; p += 1) {
	    digit = *p - '0';
	    if (digit > ('z' - '0')) {
		break;
	    }
	    digit = cvtIn[digit];
	    if (digit >= base) {
		break;
	    }
	    result = result*base + digit;
	    anyDigits = 1;
	}
    }

    /*
     * See if there were any digits at all.
     */

    if (!anyDigits) {
	p = string;
    }

    if (endPtr != 0) {
	*endPtr = p;
    }

    return result;
}

#endif /* !HAVE_STRTOUL */

#ifndef HAVE_STRTOK
/*  $Revision: 1.2 $
**
**  This file has been modified to get it to compile more easily
**  on pre-4.4BSD systems.  Rich $alz, June 1991.
*/

/*
 * Copyright (c) 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that: (1) source distributions retain this entire copyright
 * notice and comment, and (2) distributions including binaries display
 * the following acknowledgement:  ``This product includes software
 * developed by the University of California, Berkeley and its contributors''
 * in the documentation or other materials provided with the distribution
 * and in all advertising materials mentioning features or use of this
 * software. Neither the name of the University nor the names of its
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

char *
strtok(s, delim)
	register char *s, *delim;
{
	register char *spanp;
	register int c, sc;
	char *tok;
	static char *last;


	if (s == NULL && (s = last) == NULL)
		return (NULL);

	/*
	 * Skip (span) leading delimiters (s += strspn(s, delim), sort of).
	 */
cont:
	c = *s++;
	for (spanp = delim; (sc = *spanp++) != 0;) {
		if (c == sc)
			goto cont;
	}

	if (c == 0) {		/* no non-delimiter characters */
		last = NULL;
		return (NULL);
	}
	tok = s - 1;

	/*
	 * Scan token (scan for delimiters: s += strcspn(s, delim), sort of).
	 * Note that delim must have one NUL; we stop if we see that, too.
	 */
	for (;;) {
		c = *s++;
		spanp = delim;
		do {
			if ((sc = *spanp++) == c) {
				if (c == 0)
					s = NULL;
				else
					s[-1] = 0;
				last = s;
				return (tok);
			}
		} while (sc != 0);
	}
	/* NOTREACHED */
}
#endif	/* !HAVE_STRTOK */

#ifndef HAVE_STRSPN
/*  $Revision: 1.2 $
**
**  This file has been modified to get it to compile more easily
**  on pre-4.4BSD systems.  Rich $alz, June 1991.
*/

/*
 * Copyright (c) 1989 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that: (1) source distributions retain this entire copyright
 * notice and comment, and (2) distributions including binaries display
 * the following acknowledgement:  ``This product includes software
 * developed by the University of California, Berkeley and its contributors''
 * in the documentation or other materials provided with the distribution
 * and in all advertising materials mentioning features or use of this
 * software. Neither the name of the University nor the names of its
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

/*
 * Span the string s2 (skip characters that are in s2).
 */
size_t
strspn(s1, s2)
	const char *s1;
	register const char *s2;
{
	register const char *p = s1, *spanp;
	register char c, sc;

	/*
	 * Skip any characters in s2, excluding the terminating \0.
	 */
cont:
	c = *p++;
	for (spanp = s2; (sc = *spanp++) != 0;)
		if (sc == c)
			goto cont;
	return (p - 1 - s1);
}
#endif	/* !HAVE_STRSPN */

#ifndef HAVE_STRCSPN
/*-
 * Copyright (c) 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Chris Torek.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Span the complement of string s2.
 */
size_t
strcspn(s1, s2)
	const char *s1;
	register const char *s2;
{
	register const char *p, *spanp;
	register char c, sc;

	/*
	 * Stop as soon as we find any character from s2.  Note that there
	 * must be a NUL in s2; it suffices to stop when we find that, too.
	 */
	for (p = s1;;) {
		c = *p++;
		spanp = s2;
		do {
			if ((sc = *spanp++) == c)
				return (p - 1 - s1);
		} while (sc != 0);
	}
	/* NOTREACHED */
}

#endif
