/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1994, Bjarne Steinsbo
 ***************************************************************************
 * MODULE: Filestack.c
 * functions to manipulate a stack of open files
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/filestack.h"
#include "library/readconf.h"

#define MAX_DEPTH	10		/* Max recursion depth on includes */

/*
 * NOTE!  Only one stack maintained.  This can be shared for different
 * purposes, but not at the same time.
 */

static FILE *file_stack[MAX_DEPTH];	/* A stack of files */
static int file_tos = 0;		/* Top of this stack */

/*
 * File_push (&file, new_filename, path)
 *
 * Push file on stack, fopen new filename, if successfull, set file to recently
 * fopened file.
 *
 * Returns: 1 = fopen was successful, 0 = fopen failed
 */

int
File_push (FILE **file, char *fname, char *path) {
    FILE *new = NULL;
    char *step;
    static char full_name[MAXPATHLEN];

    if (DbgPcap > 8)
	log (XLOG_DEBUG, "File_push: file name '%s'", fname);
    if (fname == 0 || *fname == 0)
	return 0;
    if (file_tos >= MAX_DEPTH)
	logerr_die (XLOG_INFO, "File_push: stack is full.  Recursion?");
    fname = reallocstrcpy (0, fname, 0);
    path = reallocstrcpy (0, path, 0);
    if (fname[0] == '/') {
	/* No substitutions on absolute path names */
	new = fopen (fname, "r");
    } else {
	fname = Expand_str (fname);
	step = (char *) strtok (path, ":");
	do {
	    strcpy (full_name, step);
	    strcat (full_name, "/");
	    strcat (full_name, fname);
	    if (DbgPcap > 8)
		log (XLOG_DEBUG, "File_push: try filename '%s'", full_name);
	    new = fopen (full_name, "r");
	    if (new) {
		break;
	    }
	} while ((step = (char *) strtok (NULL, ":")));
    }
    if (new == NULL) {
	log (XLOG_INFO, "include: could not open '%s'", fname);
    } else {
	file_stack[file_tos++] = *file;
	*file = new;
    }
    free (fname); free (path);
    return (new != NULL);
}

/*
 * File_pop ()
 *
 * Pop file from stack.
 *
 * Returns: The last pushed file, or NULL if stack is empty.
 */

FILE *
File_pop (void) {
    if (DbgPcap > 8)
	log (XLOG_DEBUG, "File_pop TOS = %d", file_tos);
    if (file_tos <= 0)
	return NULL;
    return file_stack[--file_tos];
}

/*
 * Special version of fgets where "include"ing other files is invisibly
 * taken care of.
 */
char *
Gets (char *buf, int n, FILE **file, char *path) {
    char *s, *t;
    FILE *f;

    s = fgets (buf, n, *file);
    while (s != NULL) {
	/* Check for "include" at beginning of line */
	if ((char *) strstr (s, "include") == s) {
	    s += strlen ("include");
	    /* Skip leading whitespace */
	    while (*s && (*s == ' ' || *s == '\t'))
		s++;
	    /* Allow whitespace at end of line */
	    for (t = s; *t && !strchr(" \t\n", *t); t++)
		;
	    *t = 0;
	    File_push (file, s, path);
	    s = fgets (buf, n, *file);
	} else {
	    break;
	}
    }
    /* Check for files on stack at EOF */
    while (!s) {
	f = File_pop ();
	if (!f) {
	    break;
	} else {
	    fclose (*file);
	    *file = f;
	    return (Gets (buf, n, file, path));		/* redo from start */
	}
    }
    return s;
}
