/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: Getopt.c
 * Getopt(3) implementation; modified so that the first time it is
 * called it sets "Name" to argv[0];
 ***************************************************************************
 * int Getopt ( argc, argv, optstring)
 *     int argc;
 *    	char **argv, *opstring;
 * int Optind, Opterr;
 * char *Optarg;
 * extern char *Name;
 * Returns: EOF if no more options left;
 *    '?' if option not in optstring;
 *    option character if in optstr.
 *      if option char is followed by : in opstring, an argument is required,
 *        and Optarg will point to the option argument
 *      if option char is followed by ? in opstring, an argument is optional
 *        and Optarg will point to the argument if it immediately follows
 *        the option character
 *	If there are a special set of characters (such as "+" ) that should also
 *	be allowed to indicate flags, these are specified in the Opt_flag,
 *  which indicates that the character is allowed to act as an option flag
 *
 *
 * Getopt places the argv index of the next argument to be processed in
 * Optind. Because Optind is external, it is automatically set to zero
 * before the first call to Getopt.  When all the options have been
 * processed (i.e., up to the first non-option argument), Getopt returns
 * EOF.  The special option -- may be used to delimit the end of the
 * options; EOF will be returned, and -- will be skipped.
 *
 * Getopt prints an error message on stderr and returns the offending
 * character when it encounters an option letter that is not included in
 * optstring.  This error message may be disabled by setting Opterr to a
 * non-zero value.
 *
 * Side Effect:  when Getopt is called and Optind is 0, Name is set to
 * argv[0].  This allows pulling the program Name from the file.
 * Errors: if an argument is specified and none is there, then Optarg is
 * set to NULL
 *
 ***************************************************************************/

#include <stdio.h>
#include "lp.h"
#include "library/errormsg.h"

int Optind = 0;			/* next argv to process */
int Opterr = 0;			/* Non-zero disables errors msgs */
char *Optarg = NULL;		/* Pointer to option argument */
static char *next_opt = NULL;	/* pointer to next option char */
extern char *Name;		/* the program Name */
char *Opt_flag;			/* first character is a flag */

int
Getopt (int argc, char *argv[], char *optstring)
{
    char  option;		/* current option found */
    char *match;		/* matched option in optstring */

    if (Optind == 0) {
	/*
	 * set up the Name variable for error messages setproctitle will change this, so
	 * make a copy.
	 */
	char *basei, *basename;

	for (basename = basei = argv[0]; *basei; basei++) {
	    if (*basei == '/') {
		basename = basei + 1;
	    }
	}
	Name = (char *) strdup (basename);
	Optind = 1;
    }
    if (next_opt == NULL || *next_opt == '\0') {
	/* No more arguments left in current or initial string */
	if (Optind >= argc) {
	    return (EOF);
	}
	next_opt = argv[Optind];
	if (next_opt[0] != '-'
	    && (Opt_flag == 0 || strchr (Opt_flag, next_opt[0]) == 0)) {
	    /* we hit the last argument */
	    return (EOF);
	}
	/*
	 * we have a '-' or a flag as first character
	 */
	if (next_opt[0] == '-') {
	    ++next_opt;
	}
	/* Single '-', end of opts */
	if (*next_opt == '\0') {
	    return (EOF);
	}
	Optind++;
    }
    option = *next_opt++;
    /*
     * Case of '--',  Force end of options
     */
    if (option == '-') {
	return (EOF);
    }
    /*
     * See if option is in optstring
     */
    if ((match = (char *) strchr (optstring, option)) == NULL) {
	if (Opterr == 0) {
	    (void) fprintf (stderr, "%s: Illegal option '%c'\n", Name, option);
	}
	return ('?');
    }
    /*
     * Argument always follows this option
     */
    if (match[1] == ':') {
	/*
	 * Set Optarg to proper value
	 */
	if (*next_opt != '\0') {
	    Optarg = next_opt;
	} else if (Optind < argc) {
	    Optarg = argv[Optind];
	    Optind++;
	    if (Optarg != NULL && *Optarg == '-') {
		if (Opterr == 0) {
		    (void) fprintf (stderr,
				    "%s: missing argument for '%c'\n",
				    Name, option);
		    Optarg = NULL;
		}
	    }
	} else {
	    if (Opterr == 0) {
		(void) fprintf (stderr,
				"%s: missing argument for '%c'\n", Name, option);
		Optarg = NULL;
	    }
	}
	next_opt = NULL;
    } else if (match[1] == '?') {
	/*
	 * Set Optarg to proper value
	 */
	if (*next_opt != '\0') {
	    Optarg = next_opt;
	} else {
	    Optarg = NULL;
	}
	next_opt = NULL;
    }
    return (option);
}

/***************************************************************************
 * Push_opt()
 * Some flags may have an optional argument.  The Push_opt is used to
 * push the argument that would normally be supplied back.  The assumption
 * is that the argument will start with a - sign, i.e.- be a flag,
 * and be the current value of Optind
 ***************************************************************************/
void
Push_opt (void)
{
    next_opt = 0;
    Optind = Optind - 1;
}

typedef struct debug_keys {
    char *keyword;
    int len;
    char *value;
} DEBUG_KEYS;

DEBUG_KEYS debugs[] = {
    "pcap",	4,	&DbgPcap,
    "remote",	6,	&DbgRem,
    "local",	5,	&DbgLocal,
    "conf",	4,	&DbgConf,
    "perms",	5,	&DbgPerms,
    NULL,	0,	NULL
};

void
Parse_debug (char *dbgstr)
{
    char *s, buf[BUFSIZ];
    int i, found;

    if (dbgstr == NULL) {
	Diemsg ("Debug usage: -D debugspec\n");
    }
    Debug = DbgPcap = DbgRem = DbgLocal = DbgConf = DbgPerms = 0;

    if ((int) strlen(dbgstr) < BUFSIZ-1) {
	strcpy(buf, dbgstr);
    } else {
	(void) strncpy (buf, dbgstr, BUFSIZ - 1);
	buf[BUFSIZ - 1] = '\0';
    }
    s = (char *) strtok (buf, ",");

    do {
	if (*s >= '0' && *s <= '9') {
	    Debug = atoi(s);
            continue;
        }
        found = 0;
        for (i = 0; debugs[i].len; i++) {
            if ((*s == *(debugs[i].keyword)) &&
                (!strncmp (s, debugs[i].keyword, debugs[i].len - 1)) &&
                s[debugs[i].len] == '=')
            {
                *debugs[i].value = (char) atoi (&(s[debugs[i].len + 1]));
                found = 1;
                break;
            }
        }
        if (found == 0) {
            fprintf (stderr, "debug usage: -D [ generallevel | keyword=level [, ...] ]\n");
            fprintf (stderr, "      keywords:");
            for (i = 0; debugs[i].len; i++) {
                fprintf (stderr, " %s", debugs[i].keyword);
            }
	    fprintf (stderr, "\n");
            exit (1);
        }
    } while ((s = (char *) strtok (NULL, ",")));

    /* if they're unset, set them to the default */
    if (DbgPcap == 0)  { DbgPcap = Debug; }
    if (DbgRem == 0)   { DbgRem = Debug; }
    if (DbgLocal == 0) { DbgLocal = Debug; }
    if (DbgConf == 0)  { DbgConf = Debug; }
    if (DbgPerms == 0) { DbgPerms = Debug; }
}
