/*
 * Following define controls conversion of key strings to upper case. CMU-TEK helpfully
 * converts all strings to lower case. Paul Haldane, EUCS, 16/07/91
 */
#define CMU_KLUDGE 1

#include "lp.h"
#include "library/errormsg.h"

#if defined(EUCS_BANNER) || defined (EUCS_ZOPTIONS)
#include "LPD/banner.h"
#include "library/parse.h"

char *
strcpy_nospace (char *s1, char *s2) {
    /* Like strcpy() but strips off leading and trailing spaces */
    /*
     * Actually only copies up to first space found after non-space characters
     */

    while (*s2 == ' ')
	s2++;
    while (*s2 && (*s2 != ' '))
	*s1++ = *s2++;
    *s1 = '\0';
    return (s1);
}

char *
strncpy_nospace (char *s1, char *s2, int l) {
    /* Like strncpy() but strips off leading and trailing spaces */
    /*
     * Actually only copies up to first space found after non-space characters
     */

    while (*s2 == ' ')
	s2++;
    while (*s2 && (*s2 != ' ') && (l-- > 0))
	*s1++ = *s2++;
    if (l > 0)
	*s1 = '\0';
    return (s1);
}

/*
 * Syntax of strings is as follows key=value[,key=value]... values containing ','s must
 * be enclosed in '"'s '"'s inside quoted strings must be doubled
 */
struct pair *
parse (char *s, int *pair_count) {
    struct pair *pair_ptr = NULL;
    char *ptr;
    char *in_str, *str;		/* make local copy of job string so that */
    /* we don't trample on the caller's copy */
    char *value;

    *pair_count = 0;

    if ((str = malloc (strlen (s) + 1)) == NULL) {
	logerr_die (XLOG_INFO, Malloc_failed_msg);
    }
    (void) strcpy (str, s);
    in_str = str;

    while ((in_str != NULL) && ((ptr = strchr (in_str, '=')) != NULL)) {
	if (pair_ptr == NULL) {
	    pair_ptr = (struct pair *) malloc ((unsigned) sizeof (struct pair));
	} else {
	    pair_ptr = (struct pair *) realloc ((char *) pair_ptr,
				 (unsigned) (sizeof (struct pair) * (*pair_count + 1)));
	}
	*ptr++ = '\0';
	(void) strcpy (pair_ptr[*pair_count].key, in_str);
#ifdef CMU_KLUDGE
	{
	    char *p;
	    p = pair_ptr[*pair_count].key;
	    while (*p) {
		if (isascii (*p) && islower (*p))
		    *p = toupper (*p);
		p++;
	    }
	}
#endif
	in_str = ptr;
	value = get_value (&in_str);
	(void) strcpy (pair_ptr[*pair_count].value, value);
	(*pair_count)++;
    }
    return (pair_ptr);
}


char *
get_value (char **start) {
    char *p = NULL;
    char result[BUFSIZ];
    int res_len = 0;
    int seen_one_quote = 0;
    int done = 0;

    if (start != NULL)
	p = *start;

    if (p == NULL)
	return (NULL);

    if (*p == '"') {
	p++;
	seen_one_quote = 0;
	while (!done) {
	    if (seen_one_quote) {
		if (*p == '"') {
		    result[res_len++] = '"';
		    p++;
		    seen_one_quote = 0;
		} else {
		    result[res_len] = '\0';
		    done = 1;
		}
	    } else {
		if (*p == '"') {
		    seen_one_quote = 1;
		    p++;
		} else if (*p == '\0') {
		    /*
		     * This is an error - we should have got a closing quote.
		     */
		    result[res_len] = '\0';
		    done = 1;
		    p = NULL;
		} else {
		    result[res_len++] = *p++;
		}
	    }
	}
	/*
	 * We have now seen either a closing quote or the end of string we now need to
	 * move past the comma if there is one
	 */
	if ((p != NULL) && (*p)) {
	    if (*p != ',')
		log (XLOG_INFO, "parse: no comma");
	    else
		p++;
	}
    } else {
	while ((*p != '\0') && (*p != ',')) {
	    result[res_len++] = *p++;
	}
	result[res_len] = '\0';
	if (*p)
	    p++;
	else
	    p = NULL;
    }
    *start = p;
    return (strcpy (malloc ((unsigned) (res_len + 1)), result));
}

int
interpret_job (char *j_line, char *j_files, char *j_gecos,
	char *j_delivery, char *j_options, char *erccreg, char *j_account)
{
    int fields = 0;
    struct pair *pairs;
    int i;
    char *gid_str = NULL;
    char *group_name = NULL;
    char *quality = NULL;
    char *source_queue = NULL;
    char *uid_str = NULL;
    char *version = NULL;
    char j_format[80];

    (void) sprintf (j_format, "%%[^@]%s:%%[^:]:%%[^\n]\\n", FILES_DELIM);


    if (strstr (j_line, FILES_DELIM) != NULL) {
	char fbuf[80];		/* temp buffer for sscanf */
	char tbuf[80];		/* dummy buffer for sscanf */
	char *colon_ptr;

	/* EUCS Magic format */
	/*
	 * These pointers may be null - we need them to point somewhere sensible for
	 * sscanf.
	 */
	if (j_gecos == NULL)
	    j_gecos = tbuf;
	if (j_delivery == NULL)
	    j_delivery = tbuf;
	if (j_files == NULL)
	    j_files = tbuf;
	fields = sscanf (j_line, j_format, fbuf, j_gecos, j_delivery);
	(void) strncpy (j_files, fbuf, FILES_LENGTH);
	j_files[FILES_LENGTH] = '\0';
	/*
	 * j_delivery now contains delivery info and (optionally) options.  If options
	 * are there it will be the stuff after the last colon in this string.
	 */
	if ((colon_ptr = strrchr (j_delivery, ':')) != NULL) {
	    /*
	     * delivery string contains ':'.  This could be that the delivery string had
	     * a ':' in it. or it could be options specification.
	     * 
	     * Try to recognise valid options - if it doesn't look like an option then
	     * assume that it's part of the delivery info.
	     */
	    colon_ptr++;	/* point to start of suspect string */
	    /* I can't see the options being longer than "ca4u" */
	    if (strlen (colon_ptr) < 5) {
		(void) strcpy_nospace (j_options, colon_ptr);
		*(colon_ptr - 1) = '\0';
	    }
	}
    } else {
	/* assume new EUCS format - comma delimited */
	pairs = parse (j_line, &fields);

	for (i = 0; i < fields; i++) {
	    if (strsame (pairs[i].key, "A")) {
		if (j_account == NULL)
		    continue;
		(void) strncpy (j_account, pairs[i].value, ACCOUNT_LENGTH);
		j_account[ACCOUNT_LENGTH] = '\0';
	    } else if (strsame (pairs[i].key, "D")) {
		if (j_delivery == NULL)
		    continue;
		(void) strncpy (j_delivery, pairs[i].value, DELIVERY_LENGTH);
		j_delivery[DELIVERY_LENGTH - 1] = '\0';
	    } else if (strsame (pairs[i].key, "E")) {
		if (erccreg == NULL)
		    continue;
		(void) strncpy_nospace (erccreg, pairs[i].value, ERCCREG_LENGTH);
		erccreg[ERCCREG_LENGTH] = '\0';
	    } else if (strsame (pairs[i].key, "F")) {
		char *cp;
		(void) strncpy_nospace (j_options, pairs[i].value, OPTIONS_LENGTH);
		j_options[OPTIONS_LENGTH - 1] = '\0';
		/*
		 * convert options to lower case - ie allow A4 as well as a4.
		 */
		for (cp = j_options; *cp != '\0'; cp++)
		    if (isascii (*cp) && isupper (*cp))
			*cp = tolower (*cp);
	    } else if (strsame (pairs[i].key, "G")) {
		gid_str = pairs[i].value;
	    } else if (strsame (pairs[i].key, "GN")) {
		group_name = pairs[i].value;
	    } else if (strsame (pairs[i].key, "N")) {
		if (j_gecos == NULL)
		    continue;
		(void) strncpy (j_gecos, pairs[i].value, USER_INFO_LENGTH);
		j_gecos[USER_INFO_LENGTH - 1] = '\0';
	    } else if (strsame (pairs[i].key, "Q")) {
		quality = pairs[i].value;
	    } else if (strsame (pairs[i].key, "SF")) {
		if (j_files == NULL)
		    continue;
		(void) strncpy (j_files, pairs[i].value, FILES_LENGTH);
		j_files[FILES_LENGTH - 1] = '\0';
	    } else if (strsame (pairs[i].key, "SQ")) {
		source_queue = pairs[i].value;
	    } else if (strsame (pairs[i].key, "U")) {
		uid_str = pairs[i].value;
	    } else if (strsame (pairs[i].key, "V")) {
		version = pairs[i].value;
	    }
	}

	if (version == NULL) {
	    /* Extended format MUST specify version */
	    return (0);
	}
    }

    return (fields);
}
#endif /* defined(EUCS_BANNER) || defined (EUCS_ZOPTIONS) */
