/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 ***************************************************************************
 * MODULE: printcap.c
 * Super slick printcap information extraction routines.
 * Note: these routines were designed after studying the CURSES routines,
 * which are HORRIBLE...  These may not be any better, but at least
 * they are a lot faster.
 * The design is based on the assumption that we will know all the printcap
 * entries that we want to know ahead of time,  and get them "en mass".
 * Further more,  these can be placed in sorted order.
 * Thus,  given a set of these,  we scan the printcap entry, extract the
 * field name,  determine if in the list,  and then update the information
 * only if in list.  Result:
 * 1.  1 scan of printcap file per lookup of printer name, finds K entries
 * 2.  Log N order search of list,  for a K log N search of printcap
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/printcap.h"

/*
 * char **All_printers() returns a pointer to an array of strings which contains the
 * names of the printers in the printcap file.  The list is terminated with a 0 (NIL)
 * entry.
 * 
 * char *First_printer() returns a pointer to the first printer in the printcap file.
 * 
 * init_pc_entry( PC_ENTRY pc[]; int len) set the default values for variables listed in
 * pc[]
 * 
 * The following routines return 1 if printcap entry found, 0 otherwise. of the printcap
 * entry.
 * 
 * int Set_pc_entry(char *printer; PC_ENTRY pc[]; int len) set the values of the variables
 * listed in pc from the printcap 1. scan the printcap file and extract the printcap
 * entry for printer.  Make a copy of the entry. 2. find the name of the printer (first
 * entry) and set the First_name variable to point to it. 3. Initialize the printcap
 * variables according to the defaults in the "pc" array. 4. Scan the printcap entry, and
 * for each tagged field in the entry: a. search the "pc" array for the tag using a
 * binary search. b. if the entry is found, determine the type; c. if an integer, convert
 * from string to integer and update variable. d. if a flag, set or clear it e. if a
 * string, copy into a buffer and set the string variable to point to it. int
 * Get_pc_entry(char *printer; PC_ENTRY pc[]; int len) call init_pc_entry() and then
 * Set_pc_entry.
 */

/*
 * The strbuf holds strings
 */
static char strbuf[BUFSIZ];	/* buffer to hold strings */
static char *strloc;		/* next string location */

#define max_conts 20		/* we will look at 20 continuations */
static int num_conts = 0;	/* number of continuations so far seen */

static char *pr_names;		/* buffer to hold them */
static int max_names = 0;	/* max number of names in printcap */
static int max_list;		/* size of list */
static char *el, *nl;		/* pointers into the buffers */

char **pr_list = NULL;		/* printer names in printcap */
int num_pcs;			/* number of printers in printcap */

static char *last_capbuf;
static char *last_continuation;

#if (defined(NIS) || defined(HESIOD)) && defined(CROSS_DOMAIN)
char *out_domain = NULL;	/* lookup was in the local domain */
int change_domain = 1;		/* allow changing of out_domain */

/*
 * set_domain(char *where, *domain) Check if match is in local domain and set out_domain
 * if not.
 */
void
set_domain (char *where, char *domain) {
    if (change_domain) {
	if (out_domain)
	    free (out_domain);

	if (strcasecmp (where, domain) == 0) {
	    out_domain = NULL;
	} else {
	    out_domain = (char *) malloc ((unsigned) strlen (where) + 1);
	    strcpy (out_domain, where);
	}
    }
}

#endif

/*
 * char *fix_str( char *str ) makes a copy of str in strbuf and returns the start Side
 * Effect: modifies strloc
 */
static char *
fix_str (char *str) {
    char *cp;			/* ACME Pointers, Inc. */
    int l;

    l = strlen (str);
    cp = strloc;
    if (!((strloc + l) < (strbuf + sizeof (strbuf)))) {
	fatal (XLOG_INFO, "fix_str: string table overflow");
    }
    (void) strcpy (strloc, str);
    strloc += (l + 1);
    return (cp);
}

/*
 * init_pc_entry(  PC_ENTRY pc_vars[]; int pc_len) initialize the variables in pc_vars
 * with their default values. sets the strloc to start of string buffer.
 */
void
init_pc_entry (PC_ENTRY pc_vars[], int pc_len) {
    int i;			/* ACME Integer, Inc. */
    /*
     * intialize with the defaults
     */
    First_name = 0;
    All_names = 0;
    for (i = 0; i < pc_len; ++i) {
	switch (pc_vars[i].kind) {
	case PC_NUM:
	    *(int *) pc_vars[i].var = pc_vars[i].idefault;
	    break;
	case PC_FLAG:
	    *(int *) pc_vars[i].var = pc_vars[i].idefault;
	    break;
	case PC_STRING:
	    *(char **) pc_vars[i].var = pc_vars[i].sdefault;
	    break;
	default:
	    fatal (XLOG_INFO, "getpcvars; bad kind %d in table entry %d",
		   pc_vars[i].kind, i);
	}
    }
    strloc = strbuf;
    getpcvars (pc_vars, pc_len, DefaultPrintcap, NO_CONT);
    if (DbgPcap > 7)
	log (XLOG_DEBUG, "init_pc_entry: done");
}

/*
 * Get_pc_entry( char *name; PC_ENTRY pc_vars[]; int pc_len) 1. calls init_pc_entry() to
 * initialize variables 2. calls Set_pc_entry() to set them
 */
int
Get_pc_entry (char *name, PC_ENTRY pc_vars[], int pc_len) {
    if (DbgPcap > 2)
	log (XLOG_DEBUG, "Get_pc_entry: %s", name);
    init_pc_entry (pc_vars, pc_len);
    return (Set_pc_entry (name, pc_vars, pc_len));
}

/*
 * Chk_name deals with name matching.  The first line of the printcap entry is a sequence
 * of names separated by |'s, so we compare against each such name.  The normal :
 * terminator after the last name (before the first field) stops us.
 */
int
Chk_name (char *buf, char *name) {
    char *end_of_names;
    int l;

    l = strlen (name);

    end_of_names = strchr (buf, ':');
    if (end_of_names == NULL) {
	fatal (XLOG_CRIT, "bad printcap: '%s'", buf);
    }
    *end_of_names = '\0';

    while (buf && *buf) {
	if (strncmp (name, buf, l) == 0 && (buf[l] == '\0' || buf[l] == '|')) {
	    /* match */
	    *end_of_names = ':';
	    return (1);
	}
	if ((buf = strchr (buf, '|'))) {
	    buf++;
	}
    }

    /* no match */
    *end_of_names = ':';
    return (0);
}

/*
 * Return the (numeric) option id. Numeric options look like li#80 i.e. the option string
 * is separated from the numeric value by a # character.  If the option is not found we
 * return -1. Note that we handle octal numbers beginning with 0.
 */
static inline int
getnum (char *bp) {
    int i, base;

    if (*bp != '#') {
	return (-1);
    }
    bp = bp + 1;
    i = 0;
    base = 10;
    if (*bp == '0') {
	base = 8;
    }
    while (isdigit (*bp)) {
	i = i * base + (*bp - '0');
	++bp;
    }
    return (i);
}

/*
 * Handle a flag option. Flag options are given "naked", i.e. followed by a :, @ or end
 * of the buffer.  Return 1 if we find the option, or 0 if it is not given.
 */
static inline int
getflag (char *bp) {
    if (*bp == 0 || *bp == ':') {
	return (1);
    }
    return (0);
}

/*
 * Get a string valued option. These are given as xx=string There is a limited amount of
 * decoding done. \n -> '\n', \nnn -> '\nnn' ^<CHAR> -> CTRL-CHAR
 */
static inline char *
getstr (char *str) {
    char buf[BUFSIZ];
    char *bp, *cp, *op;		/* ACME Chain and Pointers, Inc */
    int c, i;			/* ACME Integers, Inc */
    static char norm[] = "E^\\:nrtbf";
    static char esc[] = "\033^\\:\n\r\t\b\f";

    bp = str;
    op = buf;			/* output area */
    if (*bp != '=')
	return ((char *) 0);
    bp++;

    while ((c = *bp++) && c != ':') {
	switch (c) {
	case '^':
	    c = *bp++;
	    if (c == 0) {
		fatal (XLOG_INFO, "getstr: bad escape string in printcap");
	    }
	    c = c & 037;
	    break;
	case '\\':
	    c = *bp++;
	    if (c == 0) {
		fatal (XLOG_INFO, "getstr: bad escape string in printcap");
	    }
	    cp = strchr (norm, c);
	    if (cp) {
		c = esc[cp - norm];
	    } else if (isdigit (c)) {
		c = c - '0';
		for (i = 0; i < 3 && isdigit (*bp); ++i) {
		    c = 8 * c + *bp - '0';
		    ++bp;
		}
	    }
	    break;
	}
	*op++ = c;
    }
    *op++ = 0;
    if ((int) strlen (buf) > 0) {
	return (fix_str (buf));
    } else {
	return ((char *) 0);
    }
}

/*
 * fixentry: fix up the value for the entry in the printcap
 */
static void
fixentry (PC_ENTRY *pc_vars, char *cp) {
    if (DbgPcap > 7)
	log (XLOG_DEBUG, "found %c%c", cp[-2], cp[-1]);
    switch (pc_vars->kind) {
    case PC_NUM:
	*(int *) pc_vars->var = getnum (cp);
	if (DbgPcap > 7)
	    log (XLOG_DEBUG, "value # %d", *(int *) pc_vars->var);
	break;
    case PC_FLAG:
	*(int *) pc_vars->var = getflag (cp);
	if (DbgPcap > 7)
	    log (XLOG_DEBUG, "value @ %d", *(int *) pc_vars->var);
	break;
    case PC_STRING:
	*(char **) pc_vars->var = getstr (cp);
	if (DbgPcap > 7)
	    log (XLOG_DEBUG, "value = %s",
		 *(char **) pc_vars->var ? *(char **) pc_vars->var
		 : "(NULL)");
	break;
    default:
	fatal (XLOG_INFO, "fixentry: bad kind in the pc_vars");
	break;
    }
}

/*
 * Set_pc_entry( char *name; PC_ENTRY pc_vars[]; int pc_len) 1. gets the "name" printcap
 * entry; capbuf[] holds it; strbuf is used to hold string constants. 2. calls getpcvar
 * routine to find fields in the printcap entry and set variables. Returns: 1 if
 * successful, and printcap entry found; 0 otherwise. Side Effect: sets the First_name to
 * the first name in the printcap entry and All_names to the list of printcap entry
 * names.
 *
 * use a wrapper so we can recurse from getpcvars().
 */
int
_set_pc_entry (char *name, PC_ENTRY pc_vars[], int pc_len) {
    int retval, i;
    char ep, *bp;
    char capbuf[BUFSIZ];
#ifdef HESIOD
    int hes_flag = 0;
#endif
#ifdef NIS
    int nis_flag = 0;
#endif

    if (DbgPcap > 1)
	log (XLOG_DEBUG, "Set_pc_entry: %s", name);
    if (num_conts <= max_conts) {
	num_conts++;
    }
    if (num_conts > max_conts) {
	log (XLOG_INFO, "printcap 'tc': infinite loop");
	--num_conts;
	return (0);
    }
    if (name == NULL || *name == '\0') {
	--num_conts;
	return (0);
    }
    retval = 0;
    if (DbgPcap > 2)
	log (XLOG_DEBUG, "printcap-order: %d conf entries", num_confs);
    for (i = 0; i < num_confs; i++) {
	if (DbgPcap > 2)
	    log (XLOG_DEBUG, "printcap-order: %d", conf[i].type);

	switch (conf[i].type) {
#ifdef LOCAL
	case LOCAL_TYPE:
	    if (!retval) {
		retval = Set_pc_entry_file (name, capbuf);
		if (DbgPcap > 5)
		    log (XLOG_DEBUG, "Set_pc_entry_wrapper: found=%d", retval);
	    }
	    break;
#endif
#ifdef NIS
	case NIS_TYPE:
	    if (!retval) {
		if (DbgPcap > 2)
		    log (XLOG_DEBUG, "Looking in NIS, domain %s",
			     conf[i].domain ? conf[i].domain : "(null)");
		nis_flag = retval = (int) Set_pc_entry_nis (conf[i].domain, name, capbuf);
	    }
	    break;
#endif
#ifdef HESIOD
	case HESIOD_TYPE:
	    if (!retval) {
		if (DbgPcap > 2)
		    log (XLOG_DEBUG, "Looking in hesiod, domain %s",
			     conf[i].domain ? conf[i].domain : "(null)");
		hes_flag = retval = (int) Set_pc_entry_hes (conf[i].domain, name, capbuf);
	    }
	    break;
#endif
	default:
	    log (XLOG_INFO,
		     "Cannot do a lookup in printcap of type %d for domain %s.\n",
		     conf[i].type, (conf[i].domain ? conf[i].domain : "local domain"));
	}
    }
    if (retval) {
	char *end_of_names;

	end_of_names = strchr (capbuf, ':');
	if (end_of_names == NULL) {
	    fatal (XLOG_CRIT, "bad printcap: '%s'", capbuf);
	}
	*end_of_names = '\0';

	if (First_name == NULL) {
 	    bp = strchr (capbuf, '|');
	    if (bp) {
		ep = *bp;
		*bp = '\0'; First_name = fix_str (capbuf);
		*bp = ep;
	    } else {
		First_name = fix_str (capbuf);
	    }
	}
	if (All_names == NULL) {
	    All_names = fix_str (capbuf);
	}
	*end_of_names = ':';

	if (DbgPcap > 4) {
	    log (XLOG_DEBUG, "printer %s, %s, len %d printcap ",
			    name, First_name, pc_len);
	    log (XLOG_DEBUG, capbuf);
	}
	/*
	 * get the printcap variable values
	 */
	getpcvars (pc_vars, pc_len, capbuf, DO_CONT);

#if (defined(NIS) || defined(HESIOD)) && defined(CROSS_DOMAIN)
	if (out_domain) {
	    change_domain = 0;	/* printrelay doesn't affect out_domain */
	    if (DbgPcap > 5)
		log (XLOG_DEBUG, "printer %s in remote domain %s",
		     name, out_domain);

	    if (
#ifdef HESIOD
		(hes_flag && Set_pc_entry_hes (conf[i].domain, "printrelay", capbuf)) ||
#endif
#ifdef NIS
		(nis_flag && Set_pc_entry_nis (conf[i].domain, "printrelay", capbuf)) ||
#endif
		0) {		/* override printcap variables */

		getpcvars (pc_vars, pc_len, capbuf, DO_CONT);
		if (RP && !strchr (RP, '@') &&
		    Printer && strchr (Printer, '@')) {
		    char buf[BUFSIZ];	/* add domain to RP */
		    strcpy (buf, RP);
		    strcat (buf, strchr (Printer, '@'));
		    RP = fix_str (buf);
		}
	    }
	    change_domain = 1;
	}
#endif
	if (DbgPcap > 7)
	    show_pc (pc_vars, pc_len);
    }
    --num_conts;
    return (retval);
}

/* avoid infinite looping, when looking up a 'tc' continuation entry
 * causes the printcap-reading interface (file, nis, whatever) to
 * rewind to the start.
 */
static void
continuation (char *capbuf, char *name, PC_ENTRY pc_vars[], int pc_len)
{
    if (last_continuation == NULL) {
	return;
    }
    if (strsame (last_continuation, name) && strsame (last_capbuf, capbuf)) {
	if (DbgPcap > 3) {
	    log (XLOG_DEBUG, "continuation: deja vu, skipping '%s'", name);
	}
    } else {
	if (DbgPcap > 3) {
	    log (XLOG_DEBUG, "continuation: jamais vu, including '%s'", name);
	}
        last_continuation = name;
	last_capbuf = strdup (capbuf);
	(void) _set_pc_entry (name, pc_vars, pc_len);
    }
}

/*
 * getpcvars- passed a table of entries, tries to find them all
 */
void
getpcvars (PC_ENTRY *pc_vars, int pc_len, char capbuf[], int nocont) {
    int h, l, m;		/* high, low, middle */
    char *cp, *cp_start;	/* string value */
    int dir;			/* direction */
    char *cont;			/* continuation */

    /*
     * now scan the printcap entry, getting the variables
     */
    cp_start = cp = capbuf;
    while (cp && (cp = strchr (cp, ':'))) {
	/*
	 * we are now positioned at the ":"
	 */
	cp = cp + 1;
	if ((cp - cp_start > 1) && (cp[-2] == '\\')) {
	    continue;
	}
	/* note: islower is a macro! do not combine the above statement! */
	if (!islower (cp[0]) || !islower (cp[1])) {
	    continue;
	}
	if (DbgPcap > 10)
	    log (XLOG_DEBUG, "getpcvars: entry %c%c", cp[0], cp[1]);
	/*
	 * check for continuations and recurse if there is one
	 */
	if ((nocont) && (cp[0] == 't') && (cp[1] == 'c')) {
	    cont = getstr (cp + 2);
	    continuation (capbuf, cont, pc_vars, pc_len);

	} else {
	    /*
	     * binary search the table
	     * (set m here to shut up gcc -Wall)
	     */
	    m = l = 0;
	    h = pc_len - 1;
	    dir = 1;
	    while (dir && l <= h) {
		m = (l + h) / 2;
		dir = strncmp (cp, pc_vars[m].pc_name, 2);
		if (dir < 0) {
		    h = m - 1;	/* too high */
		} else {
		    l = m + 1;	/* too low */
		}
	    }
	    if (dir == 0) {
		/* bingo! we found it */
		if (DbgPcap > 10)
		    log (XLOG_DEBUG, "getpcvars: found %c%c%c%c%c",
			 cp[0], cp[1], cp[2], cp[3], cp[4]);
		(void) fixentry (&pc_vars[m], cp + 2);
	    }
	}
	/*
	 * we finished, on to next
	 */
    }
}

/*
 * Debugging and display information
 */
void
show_pc (PC_ENTRY *pc_vars, int pc_len) {
    int i;

    (void) log (XLOG_DEBUG, "printcap entry %s:", First_name ?
		    First_name : "(no first name)");

    for (i = 0; i < pc_len; ++i) {
	switch (pc_vars[i].kind) {
	case PC_NUM:
	    log (XLOG_DEBUG, "     %s: PC_NUM %d", pc_vars[i].pc_name,
			    *(int *) pc_vars[i].var);
	    break;
	case PC_FLAG:
	    log (XLOG_DEBUG, "     %s: PC_FLAG %d", pc_vars[i].pc_name,
	   		    *(int *) pc_vars[i].var);
	    break;
	case PC_STRING:
	    log (XLOG_DEBUG, "     %s: PC_STRING %s", pc_vars[i].pc_name,
			    *(char **) pc_vars[i].var ?
			    *(char **) pc_vars[i].var :
			    "(NULL)");
	    break;
	}
    }
}

/****************************************************************************
 * init_printer_list initialises the list of all printers
 ****************************************************************************/
void
init_printer_list (void) {
    if (DbgPcap > 5)
	log (XLOG_DEBUG, "init_printer_list: max_names %d, max_list %d",
	     max_names, max_list);
	
    if (max_names == 0) {
	max_list = BUFSIZ;
	max_names = MAXPCNAMES;
	pr_names = (char *) malloc ((unsigned) (max_list));
	if (pr_names == NULL) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
	pr_list = (char **) malloc ((unsigned) (max_names * sizeof (char *)));
	if (pr_list == NULL) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
	el = pr_names + max_list;
    }

    nl = pr_names;
    num_pcs = 0;
    pr_list[0] = NULL;
    last_continuation = NULL;
}

/****************************************************************************
 * add_printer_list adds a printer name to the printer list
 ****************************************************************************/
void
add_printer_list (char *n) {
    int offset, i;
    char *name = (char*)malloc(strlen(n)+1);

    strcpy(name, n);
    if (strchr (name, PNAME_SEPCH))
	*(strchr (name, PNAME_SEPCH)) = '\0';

    if (DbgPcap > 4)
	log (XLOG_DEBUG, "add_printer_list: adding %s", name);

    for (i = 0; i < num_pcs; i++)	/* check if in list already */
	if (strsame (name, pr_list[i])) {
	    if (DbgPcap > 4)
		log (XLOG_DEBUG, "add_printer_list: got %s already", name);
	    free(name);
	    return;
	}
tryagain:
    pr_list[num_pcs] = nl;
    nl = estrcp (nl, name, el);
    if (nl) {
	++nl;
	++num_pcs;
    }
    if (num_pcs >= max_names) {
	if (DbgPcap > 1)
	    log (XLOG_DEBUG, "add_printer_list: need more names");
	max_names += MAXPCNAMES;
	pr_list = (char **)
		realloc ((char *) pr_list, (unsigned) max_names * sizeof (char *));
	if (pr_list == NULL) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
    }
    if (nl == NULL || nl >= el) {
	if (DbgPcap > 1)
	    log (XLOG_DEBUG, "add_printer_list: need more space");
	max_list += BUFSIZ;
	pr_names = (char *) realloc (pr_names, (unsigned) (max_list));
	if (pr_names == NULL) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
	offset = (int) pr_names - (int) pr_list[0];
	for (i = 0; i < num_pcs; i++) {
	    pr_list[i] += offset;
	}
	el = pr_names + max_list;
	if (nl == NULL) {
	    nl = pr_list[num_pcs];
	    goto tryagain;
	} else {
	    nl += offset;
	}
    }
    pr_list[num_pcs] = NULL;
    free(name);
}

/****************************************************************************
 * extract_pc_entry takes the string held in capbuf and extracts the pc vars
 * and puts the names of the printers into a list
 ****************************************************************************/
void
extract_pc_entry (char capbuf[], int cont) {
    char *cp;			/* ACME Pointer, Inc. */

    init_pc_entry (Status_pc_vars, Status_pc_len);
    getpcvars (Status_pc_vars, Status_pc_len, capbuf, cont);

    if (DbgPcap > 7)
	show_pc (Status_pc_vars, Status_pc_len);

    if (SD != NULL && *SD != '\0') {
	char *end_of_names;

	end_of_names = strchr (capbuf, ':');
	if (end_of_names == NULL) {
	    fatal (XLOG_CRIT, "bad printcap: '%s'", capbuf);
	}
	*end_of_names = '\0';

	if ((cp = strchr (capbuf, '|')) != NULL) {
	    *cp = '\0';
	}
	add_printer_list (capbuf);
    }
}

char **
All_printers (void) {
    int i;

    init_printer_list ();
    if (DbgPcap > 2)
	log (XLOG_DEBUG, "printcap-order: %d conf entries", num_confs);
    for (i = 0; i < num_confs; i++) {
	if (DbgPcap > 2)
	    log (XLOG_DEBUG, "printcap-order: %d", conf[i].type);

	switch (conf[i].type) {
#ifdef LOCAL
	case LOCAL_TYPE:
	    (void) All_printers_file ();
	    break;
#endif
#ifdef NIS
	case NIS_TYPE:
	    (void) All_printers_nis (conf[i].domain);
	    break;
#endif
#ifdef HESIOD
	case HESIOD_TYPE:
	    (void) All_printers_hes (conf[i].domain);
	    break;
#endif
	default:
	    log (XLOG_INFO, "Cannot do a lookup in printcap of type %d for domain %s.",
		     conf[i].type, (conf[i].domain ? conf[i].domain : "local domain"));
	}
    }
    if (DbgPcap > 3) {
	int j;
	log (XLOG_DEBUG, "All_printers: %d,", num_pcs);
	for (j = 0; j < num_pcs; ++j) {
	    log (XLOG_DEBUG, " %s", pr_list[j]);
	}
    }
    return (pr_list);
}

char *
First_printer () {
    char *retval = NULL;
    int i;

    if (DbgPcap > 2)
	log (XLOG_DEBUG, "printcap-order: %d conf entries", num_confs);
    for (i = 0; i < num_confs; i++) {
	if (DbgPcap > 2)
	    log (XLOG_DEBUG, "printcap-order: %d", conf[i].type);

	switch (conf[i].type) {
#ifdef LOCAL
	case LOCAL_TYPE:
	    if (!retval)
		retval = (char *) First_printer_file ();
	    break;
#endif
#ifdef NIS
	case NIS_TYPE:
	    if (!retval)
		retval = (char *) First_printer_nis (conf[i].domain);
	    break;
#endif
#ifdef HESIOD
	case HESIOD_TYPE:
	    if (!retval)
		retval = (char *) First_printer_hes (conf[i].domain);
	    break;
#endif
	default:
	    log (XLOG_INFO,
		     "Cannot do a lookup in printcap of type %d for domain %s.",
		     conf[i].type, (conf[i].domain ? conf[i].domain : "local domain"));
	}
    }
    if (retval == NULL) {
	fatal (XLOG_INFO, "First printer: bad printcap");
	return (NULL);
    } else {
	if (DbgPcap > 3)
	    log (XLOG_DEBUG, "First_printer: %s", retval);
	return (retval);
    }
}

int
Set_pc_entry (char *name, PC_ENTRY pc_vars[], int pc_len) {
    num_conts = 0;
    return (_set_pc_entry(name, pc_vars, pc_len));
}
