/***************************************************************************
 * MODULE: printcap_hesiod.c
 * Hesiod support routines
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/printcap.h"

#ifdef HESIOD
#include <hesiod.h>

static char **hes_list;		/* printer list for hesiod lookup */

extern int change_domain, num_pcs;
extern char **pr_list;

/*
 * hesiod_match(char *name, char *type, char *domain) Wrapper to hes_resolve which
 * follows continuation lines marked by "+" as first character. Returns same as
 * hes_resolve; i.e. an array of char*, terminated by NULL. hr array is re-used every at
 * every call to hes_match.
 */
char **
hesiod_match (char *name, char *type, char *domain) {
    char **hp;
    int len, hi, done;		/* hes_resolve index */
    char *kend;			/* end of name */
    char key[HES_MAXKEYLEN];	/* name buffer */
    static char **hr = 0;	/* hes_resolve array */
    static int hrsize = 0;	/* size of hr array */

    strcpy (key, name);		/* copy to name buffer */
    kend = key + strlen (key);
    hi = 0;			/* return array index */
    do {
	done = 1;
	if (domain && *domain) {
	    *kend = '@';
	    strcpy (kend + 1, domain);
	}
	if (DbgPcap > 5) {
	    log (XLOG_DEBUG, "hesiod_match: querying key %s type %s", key, type);
	}
	if ((hp = hes_resolve (key, type)) == NULL)
	    return NULL;
	for (len = 0; hp[len]; len++);	/* get length of resolved array */
	if (len + hi > hrsize) {
	    if (hr) {
		if ((hr = (char **) realloc ((char *) hr,
				   (unsigned) (len + hi + 1) * sizeof (char *))) == 0) {
		    logerr_die (XLOG_INFO, Malloc_failed_msg);
		}
	    } else if ((hr = (char **) malloc ((unsigned) (len + hi + 1) * sizeof (char *))) == 0) {
		logerr_die (XLOG_INFO, Malloc_failed_msg);
	    }
	}
	for (; *hp; hp++) {	/* copy to hr array */
	    if (**hp == '+') {	/* there's more to come */
		strcpy (key, *hp + 1);	/* get next record name */
		kend = key + strlen (key);
#ifdef HESIOD_FREE
		(void) free (*hp);
#endif
		done = 0;
	    } else
		hr[hi++] = *hp;
	}
    } while (!done);
    hr[hi] = NULL;

    return (hr);
}

/*
 * hesiod_lookup(char *name, char *type, char *domain) Searches for printcap entry name
 * of type "type" in domain, using hesiod_match to return complete matches. Search order
 * depends on whether name domain is specified (contains @), and whether lookup domain is
 * default (NULL): name, default domain:        name/host, name name, domain given:
 * name@domain* name@where, default domain:  name@where.local* name@where, domain given:
 * name@where.domain* We use the environment variable HES_DOMAIN to get the current
 * domain (if available), or try to work it out from the host name otherwise. Note *:
 * domain is progressively truncated from left until match found or no more. Note:
 * hostname is used without domain for local lookup.
 */
char **
hesiod_lookup (char *name, char *type, char *domain) {
    char **hp;			/* Hesiod return pointer */
    char *np, *ip, *kp;		/* name, index, key, ptrs */
    char *stddomain;		/* standard domain */
    char key[HES_MAXKEYLEN];	/* lookup key */
    char where[HES_MAXKEYLEN];	/* domain to search */
    extern char *getenv ();

    if (DbgPcap > 5) {
	log (XLOG_DEBUG, "hesiod_lookup: name %s, type %s, domain %s", name, type,
	     domain ? domain : "(null)");
    }
    stddomain = getenv ("HES_DOMAIN");	/* current domain pointer */

    if (stddomain == NULL && (ip = strchr (Host, '.'))) {
	stddomain = ip + 1;	/* fallback for current domain pointer */
    }
    strcpy (key, name);		/* put name in key */
    if ((kp = strchr (key, '@'))) {
	*kp = '\0';		/* truncate printer name at '@' */
	strcpy (where, kp + 1);	/* put domain in where */
	np = where + strlen (where);	/* next domain ptr */
    } else {
	kp = key + strlen (key);
	*where = '\0';
	np = where;		/* next domain ptr */
    }

    if (strchr (key, PNAME_SEPCH)) {
	logerr_die (XLOG_INFO, "printer name contains '%c' %s", PNAME_SEPCH,
		    name);
    }
    if (*where || domain) {	/* domain search required */
	char *up;		/* truncated domain pointer (ancestors) */

	if (*where) {		/* try name as given first (printer@where) */
	    if (DbgPcap > 5) {
		log (XLOG_DEBUG, "hesiod_lookup: domain supplied is %s", where);
		log (XLOG_DEBUG, "hesiod_lookup: lookup domain is %s",
		     stddomain ? stddomain : "local");
	    }
	    if ((hp = hesiod_match (key, type, where))) {	/* lookup name@where */
		set_domain (where, stddomain);
		return (hp);
	    }
	    *np++ = '.';
	}
	/* lookup printer@domain for all ancestor domains */
	for (up = (domain ? domain : stddomain);
	     up && *up; up = strchr (up, '.') + 1) {
	    strcpy (np, up);
	    if (DbgPcap > 5) {
		log (XLOG_DEBUG, "hesiod_lookup: domain supplied is %s", where);
		log (XLOG_DEBUG, "hesiod_lookup: lookup domain is %s", stddomain);
	    }
	    if ((hp = hesiod_match (key, type, where))) {	/* lookup name@ancestor */
		set_domain (where, stddomain);
		return (hp);
	    }
	}			/* failed to match */
    } else {
	if (change_domain)
	    out_domain = NULL;	/* local lookup */
	*kp = PNAME_SEPCH;
	strcpy (kp + 1, ShortHost);	/* tag on host name */
	if ((hp = hesiod_match (key, type, where)))	/* lookup name.host */
	    return (hp);

	*kp = '\0';		/* try without domain */
	if ((hp = hesiod_match (key, type, where)))	/* lookup name */
	    return (hp);
    }

    return (NULL);
}

void
All_printers_hes (char *HESDomain) {
    int i;			/* ACME Integer, Inc. */
    char **hp;			/* Pointer to Hesiod return value */

    if ((hp = hesiod_lookup ("printers", Hesiod_printcap_key, HESDomain))) {
	for (i = 0; hp[i]; i++) {
	    add_printer_list (hp[i]);
#ifdef HESIOD_FREE
	    (void) free ((char *) hp[i]);
#endif
	}
    }
    return;
}

char **
All_printers_zone (char *HESDomain) {
    int i;			/* ACME Integer, Inc. */
    char **hp;			/* Pointer to Hesiod return value */
    char buf[HES_MAXKEYLEN];

    strcpy (buf, "printers");
    if (HESDomain) {
	strcat (buf, "@");
	strcat (buf, HESDomain);
    }
    init_printer_list ();
    if ((hp = hesiod_lookup (buf, Hesiod_printcap_key, NULL))) {
	for (i = 0; hp[i]; i++) {
	    add_printer_list (hp[i]);
#ifdef HESIOD_FREE
	    (void) free ((char *) hp[i]);
#endif
	}
    }
    if (DbgPcap > 5) {
	int i;
	(void) fprintf (stdout, "All_printers_zone: %d,", num_pcs);
	for (i = 0; i < num_pcs; ++i) {
	    (void) fprintf (stdout, " %s", pr_list[i]);
	}
	(void) fprintf (stdout, "\n");
    }
    return (pr_list);
}

/*
 * pc_entry_hes(char *HESDomain; char *Pname; int size) Tries to find a named Hesiod
 * printcap entry. If Pname is null, uses static hes_list to find next Hesiod entry.
 * Returns 0 for fail, 1 for exact hit, 2 for iterated hit.
 */
static int
pc_entry_hes (char *HESDomain, char *tb, int size, char *Pname) {
    char *bp, *ep;		/* next read position */
    int l;			/* line length */
    char **hp;			/* hesiod pointer */
    int match = 1;		/* match type=exact */

    if (Pname == NULL) {	/* get next name from local printer list */
	if (!hes_list) {	/* initialize list */
	    init_printer_list ();
	    All_printers_hes (HESDomain);
	    hes_list = pr_list;
	}
	if (hes_list) {		/* get first element of list */
	    if (*hes_list) {
		Pname = *hes_list++;
		match = 2;	/* not exact match */
	    } else
		return (0);
	}
    }
    if (Pname && (hp = hesiod_lookup (Pname, Hesiod_printcap_key, HESDomain))) {
	ep = tb + size;
	bp = tb;
	for (l = 0; hp[l]; l++)	/* put in first line */
	    if (*hp[l] != ':' && *hp[l] != '#') {
		bp = estrcp (bp, hp[l], ep);
		break;
	    }
	if (bp == 0) {
	    fatal (XLOG_INFO, "printcap entry too long %s", tb);
	}
	for (l = 0; hp[l]; l++) {	/* merge lines into buffer */
	    if (*hp[l] == ':') {
		bp = estrcp (bp, hp[l], ep);
		if (bp == 0) {
		    fatal (XLOG_INFO, "printcap entry too long %s", tb);
		}
	    }
#ifdef HESIOD_FREE
	    (void) free ((char *) hp[l]);
#endif
	}
	if (DbgPcap > 8)
	    log (XLOG_DEBUG, "pc_entry: %s", tb);
	return (match);
    }
    return (0);
}

char *
First_printer_hes (char *HESDomain) {
    char capbuf[BUFSIZ];
    char *bp;
    static char first[PRNAMELEN + 1];

    hes_list = NULL;
    while (pc_entry_hes (HESDomain, capbuf, BUFSIZ, NULL)) {
	init_pc_entry (Status_pc_vars, Status_pc_len);
	getpcvars (Status_pc_vars, Status_pc_len, DefaultPrintcap, DO_CONT);
	getpcvars (Status_pc_vars, Status_pc_len, capbuf, DO_CONT);
	/* get the printcap variable values */

	if (DbgPcap > 6)
	    show_pc (Status_pc_vars, Status_pc_len);
	if (SD != NULL && *SD != NULL) {
	    if ((bp = strchr (capbuf, '|')) == NULL) {
		bp = strchr (capbuf, ':');
	    }
	    *bp = '\0';
	    (void) strcpy (first, capbuf);
	    hes_list = NULL;
	    return (first);
	}
    }
    return (NULL);
}

int
Set_pc_entry_hes (char *HESDomain, char *name, char capbuf[]) {
    char *cp;
    int match;			/* match type */
    static char buf[PRNAMELEN + 1];	/* buffer for printer name */

    strncpy (buf, name, PRNAMELEN);
    buf[PRNAMELEN] = '\0';

    if ((cp = strchr (buf, '@')))/* restrict to printer name */
	*cp = '\0';

    if (DbgPcap > 7)
	log (XLOG_DEBUG, "Set_pc_entry: %s %s", name, buf);
    if (name == 0 || *name == 0) {
	return (0);
    }
    hes_list = NULL;
    while ((match = pc_entry_hes (HESDomain, capbuf, BUFSIZ, name)) > 1) {
	if (DbgPcap > 7)
	    log (XLOG_DEBUG, "Set_pc_entry: pc_entry %s", capbuf);
	if (Chk_name (capbuf, buf)) {
	    return (1);
	}
	name = NULL;
    }
    return (match);
}
#endif				/* HESIOD */
