/***************************************************************************
 * U. Minnesota LPD Software * Copyright 1987, 1988, Patrick Powell
 * version 3.3.0 Justin Mason July 1994
 ***************************************************************************
 * MODULE: utils.c
 * Utility routines used by several programs
 ***************************************************************************/

#include "lp.h"
#include "library/errormsg.h"
#include "library/utils.h"

static int compar ();		/* for sorting files */
static void pr_stat_file ();	/* print status file */
extern time_t time ();

/*
 * Time_str: return "cleaned up" ctime() string...
 * 
 * Thu Aug 4 12:34:17 BST 1994 -> Thu Aug  4 12:34:17
 */
char *Time_str (void) {
    time_t tvec;
    static char s[99];

    (void) time (&tvec);
    (void) strcpy (s, (char *) ctime (&tvec));
    s[19] = 0;
    return (&(s[4]));
}

/*
 * Open a file as daemon.
 */
inline int open_daemon (char *name, int flag, int perms) {
    int fd;
    user_to_daemon ();
    fd = open (name, flag, perms);
    daemon_to_user ();
    return (fd);
}

/*
 * fopen a file; the file has been created, and we want to do this as daemon for
 * permissions reasons.
 */
inline FILE *fopen_daemon (char *name, char *how) {
    FILE *fp;
    user_to_daemon ();
    fp = fopen (name, how);
    daemon_to_user ();
    return (fp);
}

/*
 * UNLINK a file as daemon
 */
inline int unlink_daemon (char *name) {
    int f;
    user_to_daemon ();
    f = unlink (name);
    daemon_to_user ();
    return (f);
}


/*
 * CHMOD a file as daemon
 */
inline int chmod_daemon (char *name, int perms) {
    int f;
    user_to_daemon ();
    f = chmod (name, perms);
    daemon_to_user ();
    return (f);
}

/*
 * CHOWN a file to daemon
 */
inline int chown_daemon (char *name) {
    int f;

    user_to_root ();
    f = chown (name, DaemonUID, DaemonGID);
    root_to_user ();
    return (f);
}

/*
 * Make a dir as daemon, owned by daemon. Make it as root, then chown it, otherwise ..
 * must be writable by daemon, which isn't acceptable.
 */
inline int mkdir_daemon (char *name, int mode) {
    int f;

    user_to_root ();
    f = mkdir (name, mode);
    chown (name, DaemonUID, DaemonGID);
    root_to_user ();
    return (f);
}

/***************************************************************************/

void fix_SD (void) {
    struct stat sdstat;
    int f;
    int last;

    if (!SD || !*SD || strsame (SD, "/dev/null"))
	return;

    last = 0;
    while (1) {
	f = stat (SD, &sdstat);

	if (f < 0) {
	    if (last == errno) {/* didn't fix the last error */
		logerr_die (XLOG_NOTICE, "fix_SD: can't fix \"%s\"!!", SD);

	    } else if (errno == ENOENT) {	/* dir doesn't exist */
		logerr (XLOG_NOTICE, "creating the spool dir \"%s\"", SD);
		if (mkdir_daemon (SD, 0777)) {
		    logerr_die (XLOG_NOTICE, "fix_SD: couldn't mkdir \"%s\"", SD);
		}
		chmod_daemon (SD, 0755);

	    } else {
		logerr_die (XLOG_NOTICE, "fix_SD: can't fix \"%s\"!!", SD);
	    }
	    last = errno;

	} else {		/* it exists, anyway */
	    if ((sdstat.st_uid != DaemonUID) || (sdstat.st_gid != DaemonGID)) {
		log (XLOG_NOTICE, "fixing owner/group of \"%s\"", SD);
		chown_daemon (SD);

	    } else if (sdstat.st_mode & S_IFDIR == 0) {
		log (XLOG_NOTICE, "\"%s\": not a directory, fixing", SD);
		if (!unlink (SD)) {
		    logerr_die (XLOG_NOTICE, "fix_SD: couldn't unlink \"%s\"", SD);
		}
		if (mkdir (SD, 0777)) {
		    logerr_die (XLOG_NOTICE, "fix_SD: couldn't mkdir \"%s\"", SD);
		}
		chmod_daemon (SD, 0755);
		chown_daemon (SD);

	    } else if ((sdstat.st_mode & 0777) != 0755) {
		log (XLOG_NOTICE, "fixing permissions on \"%s\": (was mode %o)",
		     SD, sdstat.st_mode & 0777);
		chmod_daemon (SD, 0755);

	    } else {
		/* wahey! it's fixed */
		break;
	    }
	}
    }
}

inline void chdir_SD (void) {
    if (chdir (SD) == 0) {
	return;			/* c'est bon */
    }
    fix_SD ();
    if (chdir (SD) < 0) {
	logerr_die (XLOG_NOTICE, "fix_SD: can't fix \"%s\"!!", SD);
    }
}

inline DIR *opendir_SD (void) {
    DIR *dir;

    if ((dir = opendir (SD)) != NULL) {
	return (dir);		/* c'est bon */
    }
    fix_SD ();
    if ((dir = opendir (SD)) == NULL) {
	logerr_die (XLOG_NOTICE, "fix_SD: can't fix \"%s\"!!", SD);
    }
    return (dir);
}

/***************************************************************************
 * Remove_job (FILE *fp; struct plp_queue *q)
 * Remove a job and all associated files; if success == JSUCC, remove
 * U links as well
 * ACTIONS:
 * 1. Read the control file
 * 2. get data file names and unlink them if they match the job
 * 3. Unlink the control file
 ***************************************************************************/

void
Remove_job (FILE *fp, struct plp_queue *q) {
    char parm[BUFSIZ];		/* line buffer */
    char *arg;			/* pointer to the argument */
    int c;			/* ACME Integers, Inc. */

    if (Debug > 5)
	log (XLOG_DEBUG, "removing job %s", q->q_name);
    clearerr (fp);
    if (fseek (fp, 0L, 0) < 0) {
	logerr_die (XLOG_INFO,
		    "Remove_job: fseek failed (%s)", q->q_name);
    } else {
	/*
	 * read through the file, looking for things to remove
	 */
	while (fgets (parm, sizeof (parm), fp)) {
	    if ((arg = (char *) strchr (parm, '\n')) == 0) {
		log (XLOG_INFO, "%s: bad control file, no endline",
		     q->q_name);
		continue;
	    }
	    *arg = 0;
	    arg = parm + 1;
	    c = parm[0];
	    /*
	     * is it a format line?
	     */
	    if (!isascii (c) || !islower (c)) {
		continue;
	    }
	    if (Job_match (q->q_name, arg) == 0) {
		if (Debug > 4)
		    log (XLOG_DEBUG, "Remove_job: bad file name '%s'",
			 arg);
		continue;
	    }
	    (void) unlink_daemon (arg);
	}
    }
    if (unlink_daemon (q->q_name) < 0) {
	logerr (XLOG_INFO, "remove job %s: cannot unlink data file", q->q_name);
    }
}

/*
 * Compare queue entries by 1. Priority 2. Time 3. millisecond timing (if there)
 */
static int compar (const void *v1, const void *v2) {
    struct plp_queue *c1, *c2;
    int x;

    c1 = (struct plp_queue *) v1;
    c2 = (struct plp_queue *) v2;

    if ((x = ((c1)->q_priority - (c2)->q_priority)))	/* priority */
	return (x);
    if ((x = ((c1)->q_time - (c2)->q_time)))		/* time spooled by lpr */
	return (x);
    if ((x = ((c1)->q_time_hires - (c2)->q_time_hires))) /* millisecs of q_time */
	return (x);
    return (0);
}

/***************************************************************************
 * int Getq();
 * Scan the spool directory and make a list of server files sorted by
 * priority, creation time, etc.
 * Returns: the number of entries
 * Side Effects: sets struct plp_queue *Queue to point to an array of pointers
 *   to queue entries
 ***************************************************************************/

#ifdef EUCS_ZOPTIONS
#include "LPD/banner.h"
#endif

int Getq (void) {
    static int arraysz;		/* number of entries in array */
    plp_dir_t *d;		/* directory entry */
    int nitems;			/* ACME Integers, Inc. */
    struct stat stbuf;		/* for statting files */
    DIR *dirp;			/* used by directory routines */
    char l[BUFSIZ];		/* for reading the control file */
    char fullname[BUFSIZ];	/* for the full pathname */
    char *filepart;		/* end of the pathname */
    FILE *cfp;			/* control file FP */
    struct plp_queue *q;	/* pointer to queue entry */
    char *bp, *ep;		/* ACME Pointers, Inc. */

#ifdef EUCS_ZOPTIONS
    char j_files[FILES_LENGTH + 1];
    char j_gecos[USER_INFO_LENGTH + 1];
    char j_delivery[DELIVERY_LENGTH + 1];
    char j_options[OPTIONS_LENGTH + 1];
    /* char j_erccreg[ERCCREG_LENGTH + 1]; */
    int j_fields;
#endif

    if (SD == 0 || *SD == 0) {
	fatal (XLOG_INFO, "spool directory is not set!");
    }
    if ((dirp = opendir_SD ()) == NULL) {
	logerr_die (XLOG_INFO, "cannot open spool directory %s", SD);
    }
    /*
     * get the directory name and copy to buffer
     */
    if (strlen (SD) + sizeof (q->q_name) + 2 > sizeof (fullname)) {
	logerr_die (XLOG_NOTICE, "Getq: file name too big");
    }
    (void) strcpy (fullname, SD);
    filepart = &fullname[strlen (fullname)];
    *filepart++ = '/';
    /*
     * assume you have 100 entries to start with
     */
    if (arraysz == 0) {
	arraysz = 100;
	Queue = (struct plp_queue *)
		malloc ((unsigned) (arraysz * sizeof (struct plp_queue)));
	if (Queue == NULL) {
	    logerr_die (XLOG_INFO, Malloc_failed_msg);
	}
    }
    /*
     * set the item count to 0 and scan the directory
     */
    nitems = 0;
    user_to_daemon ();
    while ((d = readdir (dirp)) != NULL) {

	if (!(((d->d_name[0] == 'h') ||
	       (d->d_name[0] == 'c')) && (d->d_name[1] == 'f')))
	{
	    continue;		/* server control files only */
	}
	/*
	 * Check to make sure the array has space left and realloc the maximum size.
	 */
	if (nitems >= arraysz) {
	    /* try to get another buffer */
	    arraysz = arraysz + 20;
	    Queue = (struct plp_queue *) realloc ((char *) Queue,
		    (unsigned) ((arraysz) * sizeof (struct plp_queue)));
	    if (Queue == NULL) {
		logerr_die (XLOG_NOTICE, Malloc_failed_msg);
	    }
	}
	/*
	 * get next queue entry and clear it
	 */
	q = &Queue[nitems];
	(void) bzero (q, sizeof (struct plp_queue));

#ifdef EUCS_ZOPTIONS
	*j_options = '\0';
	*j_gecos = '\0';
	*j_files = '\0';
	*j_delivery = '\0';
#endif

	/* copy the name of the file */
	if (strlen (d->d_name) >= CFNAMELEN) {
	    logerr_die (XLOG_NOTICE, "control file name %s too long", d->d_name);
	}

	(void) strcpy (q->q_name, d->d_name);
	(void) strcpy (filepart, d->d_name);

	if (d->d_name[0] == 'h') 
	    {
		q->q_held = 1;
	    }
	else
	    {
		q->q_held = 0;
	    }

	/*
	 * open the control file and read the information
	 * (we are daemon throughout this loop!)
	 */
	if ((cfp = fopen (fullname, "r")) == NULL) {
	    if (Debug > 4)
		log (XLOG_DEBUG, "%s cannot be opened - skipping", fullname);
	    continue;		/* it disappeared - skip it */
	}

	/* set a priority */
	q->q_priority = d->d_name[2];
	/* get the job number */
	q->q_num = atoi (&d->d_name[3]);

	bp = q->q_data;
	ep = bp + sizeof (q->q_data);
	while (fgets (l, sizeof (l), cfp)) {
	    l[strlen (l) - 1] = 0;		/* clobber the \n */

	    if (islower (l[0])) {
		/* get size of file to be printed */
		(void) strcpy (filepart, &l[1]);
		if (stat (fullname, &stbuf) < 0) {
		    (void) fclose (cfp);
		    if (Debug > 5)
			log (XLOG_DEBUG,
			     "Getq: data file %s does not exist", l);
		} else {
		    q->q_size += stbuf.st_size;
		}

	    } else
		switch (l[0]) {
		case 'E':	/* when to be unspooled */
		    q->q_unsp = (time_t) atol (&l[1]);
		    break;
		case 'N':	/* name of file */
		    bp = estrcp (bp, &l[1], ep);
		    bp = estrcp (bp, " ", ep);
		    break;
		case 'P':	/* name of user */
		    (void) strncpy (q->q_user, &l[1], sizeof (q->q_user) - 1);
		    q->q_user[sizeof (q->q_user) - 1] = '\0';
		    break;

		case 'D':	/* when spooled (non-RFC) */
		    {
			char *dot;
			if ((dot = strchr (&l[1], '.'))) {
			    *dot = '\0'; dot++;
			    q->q_time_hires = (unsigned long) atol (dot);
			} else {
			    q->q_time_hires = 0;
			}
			q->q_time = (time_t) atol (&l[1]);
		    }
		    break;

#ifdef EUCS_ZOPTIONS
		case 'J':	/* job name with extra info */
		    if (EU) {
			j_fields = interpret_job (l + 1, j_files, j_gecos,
						  NULL, j_options, NULL, NULL);
			if (*j_options != '\0' &&
			    (!RM || *RM == '\0') &&
			    strcmp (j_options, DEF_OPTIONS) != 0)
			{
			    q->q_options = strdup (j_options);
			} else {
			    q->q_options = NULL;
			}
		    }
		    break;
#endif				/* EUCS_ZOPTIONS */

		default:
		    break;
		}
	}
	if (Debug > 5)
	    log (XLOG_DEBUG, "Getq: control file %s is ok", d->d_name);

	if (Debug > 4)
	    log (XLOG_DEBUG, "Getq: '%s' %d,%s,%c,%s", q->q_name,
		q->q_num, q->q_user, q->q_priority, q->q_data);

	(void) fclose (cfp);

	/* BSD compatibility: if spool-time isn't set, get it from the
	 * spoolfile's modification date.
	 */
        if (q->q_time == 0) {
	    if (stat (q->q_name, &stbuf) < 0) {
		q->q_time = time ((time_t *) 0);
	    } else {
		q->q_time = stbuf.st_mtime;
	    }
	}
	++nitems;
    }
    closedir (dirp);
    daemon_to_user ();
    *((Queue[nitems]).q_name) = '\0';

    /*
     * sort the queue
     */
    if (nitems) {
	qsort ((char *) Queue, nitems, sizeof (struct plp_queue), compar);
    }
    return (nitems);
}

/***************************************************************************
 * Match_entry (struct plp_queue *q)
 * check the user name and job number against the parmlist starting at start;
 * Return: 1 if match found, 0 otherwise
 ***************************************************************************/
int Match_entry (struct plp_queue *q) {
    int i;			/* ACME Integer, Inc. */

    if (Debug > 6)
	log (XLOG_DEBUG, "Match_entry on %s, %d", q->q_user, q->q_num);
    for (i = 0; i < Parmcount; ++i) {
	if ((Parms[i].num >= 0 && Parms[i].num == q->q_num)
	    || strsame (Parms[i].str, q->q_user)) {
	    if (Debug > 4)
		log (XLOG_DEBUG, "Match_entry on %s, %d succeeded",
		     q->q_user, q->q_num);

	    return (1);
	}
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "Match_entry on %s, %d failed", q->q_user, q->q_num);
    return (0);
}

/***************************************************************************
 * bpread (int fd ; char *buf; int size)
 * Almost bombproof way to read command line from socket
 * Returns: -1 if unable to read a line terminated with \n
 *     0 if 0 length line or first character is null (i.e.- no input)
 *     otherwise: length of line
 ***************************************************************************/
int bpread (int fd, char *buf, int size) {
    int n;			/* ACME Integer, Inc. */
    int i;

    /*
     * Bombproof line reading
     */
    for (i = 0; i < size; ++i) {
	n = read (fd, &buf[i], 1);
	if (n < 0) {
	    logerr (XLOG_INFO, "bpread: lost connection");
	    return (-1);
	} else if (n == 0) {
	    if (i == 0) {
		if (Debug > 3)
		    log (XLOG_DEBUG, "bpread: EOF");
	    } else {
		if (Debug > 3)
		    log (XLOG_INFO,
			 "bpread: bad format '%d' %s len = %d",
			 buf[0], buf + 1, i);
	    }
	    return (0);
	}
	if (buf[i] == '\n') {
	    buf[i] = 0;
	    if (Debug > 3)
		log (XLOG_DEBUG, "bpread: '%d' %s", buf[0], &buf[1]);
	    return (i);
	} else if (buf[i] == 0 && i == 0) {
	    if (Debug > 3)
		log (XLOG_DEBUG, "bpread: zero byte first");
	    return (i);
	}
    }
    log (XLOG_INFO, "bpread: bad line %s", buf);
    return (-1);
}

/***************************************************************************
 * estrcp(char *s, *t, *e)
 * copies t to s, and returns the end position
 * Returns: end of string if the result is shorter than e, NULL otherwise
 * Note: this is a very useful routine.  Think about adding it to string
 * functions.
 ***************************************************************************/
char *estrcp (char *s, char *t, char *e) {
    if (s) {
	while ((*s++ = *t++) && (s < e));
	if (s < e) {
	    return (s - 1);
	} else {
	    return ((char *) 0);
	}
    }
    return (s);
}

/* some Parms dynamic-allocation routines:
 *
 * allocParmsIfNeeded  -- allocate the Parms array, if it isn't already;
 *                        also, shrink the Parms array if it's too large
 *                        (garbage collection)
 *
 * growParmsIfNeeded   -- increase size of the Parms array if necessary
 */

#define PARMS_BUFSIZ	20		/* arbitrary */

void allocParmsIfNeeded (void) {
    if (ParmSz == 0) {
	ParmSz = PARMS_BUFSIZ;
	if(Debug>6) log(XLOG_DEBUG, "allocating Parms to %d", ParmSz);

	Parms = (struct parm *)
		malloc ((unsigned) (ParmSz * sizeof (struct parm)));

    } else if (Parmcount + PARMS_BUFSIZ < ParmSz) {
	/* shrink the array to slightly larger than it needs to be */
	ParmSz = Parmcount + PARMS_BUFSIZ;
	if(Debug>6) log(XLOG_DEBUG, "shrinking Parms to %d", ParmSz);

	Parms = (struct parm *) realloc ((char *) Parms,
				(unsigned) (ParmSz * sizeof (struct parm)));
    }
    if (Parms == NULL) {
	logerr_die (XLOG_NOTICE, Malloc_failed_msg);
    }
}

void growParmsIfNeeded (void) {
    if (Parmcount >= ParmSz) {
	/* try to get another buffer */
	ParmSz = ParmSz + PARMS_BUFSIZ;

        if(Debug>6) log(XLOG_DEBUG, "growing Parms to %d", ParmSz);
	Parms = (struct parm *) realloc ((char *) Parms,
				(unsigned) (ParmSz * sizeof (struct parm)));
	if (Parms == NULL) {
	    logerr_die (XLOG_NOTICE, Malloc_failed_msg);
	}
    }
}

/***************************************************************************
 * splitline (char *line)
 *    splits a line into tokens, and places them in the Parms[] array.
 *    if firstarg is present then it is set to the first argument
 * Side Effects: modifies Parms, Parmcount
 ***************************************************************************/

void splitline (char *bp) {
    allocParmsIfNeeded ();
    /* find the start */
    while (*bp) {
	while (*bp && isspace (*bp))
	    bp++;
	if (*bp) {
	    growParmsIfNeeded ();
	    Parms[Parmcount].str = bp;
	    if (isdigit (*bp)) {
		Parms[Parmcount].num = atoi (bp);
	    } else {
		Parms[Parmcount].num = -1;
	    }
	    Parmcount++;
	}
	while (*bp && !isspace (*bp))
	    bp++;
	if (*bp)
	    *bp++ = 0;
    }
}

/**********************************************************************
 * Shift_parms (int n)
 * left shift the entries in the Parms[] array by n
 * Side Effects: modifies Parms and Parmcount
 **********************************************************************/

void
Shift_parms (int n) {
    int i;			/* ACME Chain and Integer, Inc. */
    while (Parmcount > 0 && n > 0) {
	for (i = 1; i < Parmcount; ++i) {
	    Parms[i - 1] = Parms[i];
	}
	--Parmcount;
	--n;
    }
    allocParmsIfNeeded ();
}

char *possible_options (void) {
#ifdef EUCS_FORMS
    if ((!Forms_file) || (!FL || *FL == '\0')) {
	return NULL;
    }
    return (strdup (FL));
#else
    return NULL;
#endif
}

char *current_options (void) {
#ifdef EUCS_FORMS
    FILE *opt_fp;
    char *ip, *ret_str;
    char opt_str[80];
    char opt_file_name[MAXPATHLEN];

    if (!Forms_file) {
	return DEF_OPTIONS;
    }
    if (!FL || *FL == '\0')
	return NULL;
    strcpy (opt_file_name, Forms_file);
    if ((ip = strchr (Printer, '.')))
	strcat (opt_file_name, ip);

    if ((opt_fp = fopen_daemon (opt_file_name, "r")) == NULL) {
	log (XLOG_INFO, "Can't open options file for %s", Printer);
	return DEF_OPTIONS;
    } else {
	if (fscanf (opt_fp, "%s", opt_str) != 1) {
	    log (XLOG_INFO, "Couldn't get line from options file for %s", Printer
		    );
	    fclose (opt_fp);
	    return DEF_OPTIONS;
	}
	fclose (opt_fp);
	ret_str = strdup (opt_str);

	return (ret_str);
    }
#else
    return DEF_OPTIONS;
#endif
}

/* forms option to try
 * comma delimited list of possible forms
 */
int valid_forms (char *forms, char *choices) {
#ifdef EUCS_FORMS
    char buffer[256];
    char *ptr = buffer;
    char *this;

    if (!Forms_file) {
	return (1);
    }
    strcpy (buffer, choices);

    while ((this = strtok (ptr, ",")) != NULL) {
	if (strsame (this, forms)) {
	    return (1);
	}
	ptr = NULL;
    }
    return (0);
#else
    return (1);			/* allow anything */
#endif
}

/*
 * current_limit() returns the current limit on the size of print jobs (in 1k blocks).
 * If there is no limit then it returns -1. Note that this is different from the mx value
 * in the printcap entry which limits the size of jobs which can be queued.
 * 
 * We need to be able to have different limits on printers attached to one queue.  This
 * allows us to have a generic queue but still give good service for small jobs.  In
 * order to do this we distinguish between limits files by using the "li" printcap
 * parameter, which can either accept a numeric value (the actual limit for that queue)
 * or a filename (a file to be opened to get the limit).
 */

long current_limit (void) {
    static long limit = -1L;
    static struct stat last_buf, buf;
    static char cached_Printer[PRNAMELEN];
    static char cached_filename[MAXPATHLEN];

    if (!LI || !*LI) {
	/* no limit on this printer */
	return (-1L);
    }

    if (*LI >= '0' && *LI <= '9') {
	/* limits value is numeric */
	return (atol (LI));
    }

    /* otherwise, limits value is a filename to open.
     * We cache the last limits-file lookup to save
     * repeated opens and closes, and will get a new
     * value if the Printer is different, the filename
     * changes or the modtime of the file changes.
     */
    if (strcmp (Printer, cached_Printer) || strcmp (LI, cached_filename))
    {
	/* it's changed; force a reload */
	last_buf.st_mtime = 0;
	(void) strcpy (cached_Printer, Printer);
	(void) strcpy (cached_filename, LI);
    }

    if (stat (LI, &buf) != 0) {
	/* the limits file isn't there */
	limit = -1L;
	if (Debug > 2) {
	    log (XLOG_DEBUG, "limits file is missing, assuming no limit");
	}
	return (-1L);
    }

    if (buf.st_mtime > last_buf.st_mtime) {
	/* file has changed - need to read it */
	int ret;
	FILE *fp;

	if ((fp = fopen (LI, "r")) == NULL) {
	    /* the file exists (the stat worked), but we can't read it */
	    log (XLOG_INFO, "current_limit: can't read limits file");
	    return (-1L);
	}
	ret = fscanf (fp, "%ld", &limit);
	(void) fclose (fp);
	if (ret != 1) {
	    log (XLOG_INFO,
		 "current_limit: can't read value from limits file");
	    return (-1L);
	}
    }

    /* else the limit value from the last lookup is still cached;
     * return it.
     */
    return (limit);
}

inline void printstatus (void) {
    if (ST && *ST) {
	pr_stat_file (ST);
    }
    if (PS && *PS) {
	pr_stat_file (PS);
    }
}

static void pr_stat_file (char *file) {
    FILE *fp;
    char buf[BUFSIZ];

    if ((fp = fopen_daemon (file, "r")) != NULL) {
	while (fgets (buf, sizeof (buf), fp) != NULL) {
	    (void) fprintf (stdout, "  %s", buf);
	}
	(void) fflush (stdout);
	(void) fclose (fp);
    }
    /* Only print forms and limit information for local printers */
    if ((RM == NULL) || (*RM == '\0')) {
	long limit;
	int output = 0;

#ifdef EUCS_ZOPTIONS
	if (FL && *FL) {
	    fprintf (stdout, "  Current forms = %s (%s)",
		     current_options (), possible_options ());
	    output = 1;
	}
#endif
	if ((limit = current_limit ()) != -1L) {
	    fprintf (stdout, "  Current size limit = %ldk",
		     limit);
	    output = 1;
	}
	if (output)
	    putchar ('\n');
    }
}

/***************************************************************************
 * char *Sigstr(n)
 * Return a printable form the the signal
 ***************************************************************************/

static struct signame {
    char *str;
    int value;
} signals[] = {

    {
	"SIGHUP", SIGHUP
    }, {
	"SIGINT", SIGINT
    },
    {
	"SIGQUIT", SIGQUIT
    }, {
	"SIGILL", SIGILL
    },
    {
	"SIGTRAP", SIGTRAP
    }, {
	"SIGFPE", SIGFPE
    },
    {
	"SIGKILL", SIGKILL
    }, {
	"SIGSEGV", SIGSEGV
    },
    {
	"SIGPIPE", SIGPIPE
    }, {
	"SIGALRM", SIGALRM
    },
    {
	"SIGTERM", SIGTERM
    }, {
	"SIGSTOP", SIGSTOP
    },
    {
	"SIGTSTP", SIGTSTP
    }, {
	"SIGCONT", SIGCONT
    },
    {
	"SIGCHLD", SIGCHLD
    },
#ifdef SIGIOT
    { "SIGIOT", SIGIOT },
#endif
#ifdef SIGEMT
    { "SIGEMT", SIGEMT },
#endif
#ifdef SIGBUS
    { "SIGBUS", SIGBUS },
#endif
#ifdef SIGSYS
    { "SIGSYS", SIGSYS },
#endif
#ifdef SIGIO
    { "SIGIO", SIGIO },
#endif
#ifdef SIGXCPU
    { "SIGXCPU", SIGXCPU },
#endif
#ifdef SIGXFSZ
    { "SIGXFSZ", SIGXFSZ },
#endif
    { "SIGPROF", SIGPROF }
    /* that's all */
};

static int nsignals = sizeof (signals) / sizeof (struct signame);

char *Sigstr (int n) {
    int i;
    static char buf[40];

    for (i = 0; i < nsignals; ++i) {
	if (signals[i].value == n) {
	    return (signals[i].str);
	}
    }
    (void) sprintf (buf, "unknown signal");
    return (buf);
}

/***************************************************************************
 * Decode_status (plp_status_t *status)
 * returns a printable string encoding return status
 ***************************************************************************/

char *Decode_status (plp_status_t *status) {
    static char msg[BUFSIZ];

    *msg = '\0';		/* just in case! */
    if (PLP_WIFEXITED (*status)) {
	(void) sprintf (msg, "exited with status %d", PLP_WEXITSTATUS (*status));

    } else {
	(void) sprintf (msg, "died from signal %d%s", PLP_WTERMSIG (*status),
			PLP_WCOREDUMP (*status) ? " and dumped core" : "");

	if (PLP_WTERMSIG (*status)) {
	    (void) sprintf (msg + strlen (msg), " (%s)",
			    Sigstr ((int) PLP_WTERMSIG (*status)));
	}
    }
    return (msg);
}

/***************************************************************************
 * pid_t Checkactive() 
 * find the currently active files in the spool queue. 
 * 1.  check for the single server possibility first. 
 * 2.  find the Names of the servers 
 * 3.  find the active files for each of the servers
 ***************************************************************************/
  
pid_t Checkactive (void) {
    int i;			/* ACME Integers, Inc. */
    char buf[BUFSIZ];		/* Name of active file */
    static char server[BUFSIZ];	/* Name of server file */
    char *sp, *ep;		/* ACME Pointer */
    pid_t pid;

    pid = 0;
    buf[0] = 0;
    if (Checklockfile (LO, &pid, buf, sizeof (buf), &LO_statb) && buf[0]) {
	for (i = 0; i < Jobcount; ++i) {
	    if (strsame (buf, Queue[i].q_name)) {
		Queue[i].q_daemon = pid;
		Queue[i].q_server = Printer;
	    } else {
		Queue[i].q_daemon = 0;
		Queue[i].q_server = NULL;
	    }
	}
    }
    /*
     * check for each of the servers
     */
    if (SV && *SV) {
	for (i = 0; i < Jobcount; ++i) {
	    Queue[i].q_daemon = 0;
	    Queue[i].q_server = NULL;
	}
	(void) strcpy (server, SV);

	for (sp = server; sp; sp = ep) {
	    if ((ep = (char *) strchr (sp, ','))) {
		*ep = 0;
		++ep;
	    }
	    /*
	     * get the lock file and the status from the server
	     */
	    pid = 0;
	    buf[0] = 0;
	    if (Checklockfile (sp, &pid, buf, sizeof (buf),
	   	 (struct stat *) NULL) && buf[0])
	    {
		for (i = 0; i < Jobcount; ++i) {
		    if (strsame (buf, Queue[i].q_name)) {
			Queue[i].q_daemon = pid;
			Queue[i].q_server = sp;
			break;
		    }
		}
	    }
	}
    }
    return (pid);
}


/***************************************************************************
 * Get_Daemon()
 * Get the daemon uid and gid
 * Assume password entry has both values set for the user daemon
 ***************************************************************************/

void Get_Daemon (void) {
    struct passwd *passwd;
    if (Daemon_user == NULL) {
	logerr_die (XLOG_INFO, "haven't read config file yet");
    }
    if ( (DaemonUID == 0) || (DaemonGID == 0) ){
	if ((passwd = getpwnam (Daemon_user)) == 0) {
	    logerr_die (XLOG_INFO, "Get_Daemon: getpwnam(%s) failed", Daemon_user);
	}
	DaemonUID = passwd->pw_uid;
	DaemonGID = passwd->pw_gid;
    }
}

/***************************************************************************
 * Job_match (char *control, *data)
 * Check to see if the control and data file names are matching
 * 1. data file must start with "df"
 * 2. next must be letter
 * 3. next must be 3 digit sequence number
 * 4. control and data file sequence number must match
 * return 1 if OK, 0 if not
 ***************************************************************************/

int Job_match (char *control, char *data) {
    int c, i, j;		/* ACME Chain and Integers, Inc. */
#ifdef EUCS_ZOPTIONS
    char *dot_ptr;
#endif

    i = strlen (data);
    for (j = 0; j < i; ++j) {
	c = data[j];
	if (!isascii (c) || !isprint (c)) {
	    log (XLOG_INFO, "Job_match: bad char in '%s'", data);
	    return (0);
	}
    }
    if (strchr (data, '/')) {
	/*
	 * tried to embed a / in the file name
	 */
	log (XLOG_INFO, "Job_match: '/' present in filename \"%s\"", data);
	return (0);
    }
#ifdef EUCS_ZOPTIONS
    if ((dot_ptr = strchr (data, '.')) == NULL) {
	dot_ptr = data + strlen (data);
    }
#endif
    if (i > CFNAMELEN
	|| i <= STARTFR
	|| strchr ("tcd", data[0]) == 0
	|| (data[1] != 'f')
	|| !isalpha (data[STARTPR])
	|| i != strlen (control)
	|| hostcmp (control + STARTID, data + STARTID)) {
	log (XLOG_INFO, "Job_match: bad match control '%s', data '%s'",
	     control, data);
	return (0);
    }
    for (j = 0; j < IDLEN; ++j) {
	if (!isdigit (data[j + STARTID])) {
	    log (XLOG_INFO, "Job_match: bad sequence control '%s', data '%s'",
		 control, data);
	    return (0);
	}
    }
    return (1);
}

/* get the fully-qualified domain name for a host.
 * usage: strcpy (fqhostname, get_fqdn (hostname));
 * NOTE: this returns a pointer to static data, so copy the result!!
 */

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 256
#endif

char *get_fqdn (char *shorthost) {
    static char *returnspc = NULL;	/* dataspace to return ptr to (if needed) */
    struct hostent *host_ent;		/* host entry from data base */
    char **ali, pattern[HOSTLEN + 1];
    int patlen;
    char shorthost_cpy[SHORTHOSTLEN];

    if (strchr (shorthost, '.') != NULL) {
	/* it's already fully-qualified, just return itself */
	return (shorthost);
    }

    /* this is needed in case gethostbyname is destructive */
    shorthost_cpy[0] = '\0';
    (void)strcpy (shorthost_cpy, shorthost);

    DomainNameUnset = 0;
    host_ent = gethostbyname (shorthost_cpy);
    if (host_ent == 0) {
	if (Debug > 4) log (XLOG_DEBUG, "Hostname: can't resolve '%s'", ShortHost);
	if (Host_domain && *Host_domain) {
	    if (Debug > 4) log (XLOG_DEBUG, "Hostname: appending domain name");
	    if (returnspc == NULL) { malloc_or_die (returnspc, HOSTLEN); }
	    (void) sprintf (returnspc, "%s.%s", shorthost, Host_domain);
	    return (returnspc);
	} else {

	    DomainNameUnset = 1;
	    return (shorthost);
	}
    }
    
    if (strchr (host_ent->h_name, '.') == NULL) {
        if (Debug > 4) log (XLOG_DEBUG, "Hostname: h_name is not qualified");
	/* we'll assume that if one of the aliases matches /^Shorthost\./,
	 * then it's the fully-qualified domain name. (ugh)
	 */
	(void) strcpy (pattern, shorthost);
	patlen = strlen (pattern);
	pattern[patlen++] = '.';
	pattern[patlen] = '\0';
	if (returnspc == NULL) { malloc_or_die (returnspc, HOSTLEN); }

	*returnspc = 0;
	if (host_ent->h_aliases) {
	    for (ali = host_ent->h_aliases; *ali; ali++) {
		if (Debug > 5) log (XLOG_DEBUG,
			    "checking '%s' against pattern '%s' (len=%d)",
			    *ali, pattern, patlen);
		if (!strncmp (*ali, pattern, patlen)) {
		    if (Debug > 4) log (XLOG_DEBUG, "Hostname: h_aliases is qualified");
		    return (*ali);
		}
	    }
	}

	if (!*returnspc) {
            if (Debug > 4) log (XLOG_DEBUG,
			"Hostname: h_aliases is not qualified - trying gethostbyaddr");
	    
	    {
		/* all this copying is ridiculous, but necessary, due to the
		 * resolver's use of a single static hostent structure.
		 */
		char h_first_addr[8];
		int h_length_cpy;
		int h_addrtype_cpy;

		h_addrtype_cpy = host_ent->h_addrtype;
		h_length_cpy = host_ent->h_length;
		h_first_addr[0] = host_ent->h_addr_list[0][0];
		h_first_addr[1] = host_ent->h_addr_list[0][1];
		h_first_addr[2] = host_ent->h_addr_list[0][2];
		h_first_addr[3] = host_ent->h_addr_list[0][3];
		host_ent = gethostbyaddr (h_first_addr, h_length_cpy, h_addrtype_cpy);
	    }
	    
	    if (host_ent && strchr (host_ent->h_name, '.') != NULL) {
		if (Debug > 4) log (XLOG_DEBUG, "Hostname: ptr lookup is qualified");
		return (host_ent->h_name);

	    } else {
		if (Debug > 4) log (XLOG_DEBUG, "Hostname: ptr lookup is non-qualified");
		if (Host_domain && *Host_domain) {
		    if (Debug > 4) log (XLOG_DEBUG, "Hostname: appending domain name");
		    (void) sprintf (returnspc, "%s.%s", shorthost, Host_domain);
		    return (returnspc);
		} else {

		    DomainNameUnset = 1;
		    return (shorthost);
		}
	    }
	}
    }
    if (Debug > 4) log (XLOG_DEBUG, "Hostname: h_name is qualified");
    return (host_ent->h_name);
}

/***************************************************************************
 * Get_Local_fqdn()
 * Get the hostname and fully-qualified hostname of the local host.
 * Tricky this; needs to be run after the config file has been read;
 * also, it depends a lot on the local DNS/NIS/hosts-file/etc. setup.
 ***************************************************************************/

void Get_Local_fqdn (void) {
    char *c;
    const char *fqdn;

     /*
      * get the Host computer Name
      */
#ifdef HAVE_GETHOSTNAME
    if (gethostname (ShortHost, sizeof (ShortHost)) < 0) {
	logerr_die (XLOG_INFO, "Get_Local_fqdn: gethostname failed");
    }
#else
    {
	struct utsname unamestuff;	/* structure to catch data from uname */
	if (uname (&unamestuff) < 0) {
	    logerr_die (XLOG_INFO, "Get_Local_fqdn: uname failed");
	}
	(void) strncpy (ShortHost, unamestuff.nodename, sizeof (ShortHost) - 1);
    }
#endif
    ShortHost[sizeof (ShortHost) - 1] = '\0';	/* trim to size (just in case) */
    
    fqdn = get_fqdn (ShortHost);
    if (strlen (fqdn) > sizeof (Host)) {
	log (XLOG_INFO,
	     "fully-qualified host name (%s) is too long, using short version",
	     fqdn);
	(void) strcpy (Host, ShortHost);

    } else {
        (void) strcpy (Host, fqdn);
    }

    /* check in case someone's setup is very sick:
     * ShortHost is fully-qualified and Host isn't.
     */
    if ((strchr (ShortHost, '.') != NULL) && (strchr (Host, '.') == NULL))
	{
	    (void) strcpy (Host, ShortHost);
	}
    
    /* Strip out the domain name from ShortHost (if present) */
    while ((c = (char *) strchr (ShortHost, '.')) != NULL) {
	*c = '\0';
    }
    
    if (Debug > 3)
	log (XLOG_DEBUG, "Hostname: short=%s, fqdn=%s", ShortHost, Host);
}

/***************************************************************************
 * Std_environ()
 * Make sure that fd 0, 1, 2 exist by opening /dev/null for them if they
 * have not been provided; set up the host information; also, sanitize
 * potentially insecure environment variables.
 ***************************************************************************/

void Std_environ (int argc, char **argv, char **envp) {
    char *c, *p;
    int i, j;
#ifdef SETPROCTITLE
    extern char **Argv;
    extern char *LastArgv;
#endif

    /*
     * from sendmail 8.6.4: copy the environment out of the way for use in the
     * secure_system() code (setuid.c).
     */
    for (i = j = 0; j < MAXUSERENVIRON && (p = envp[i]) != NULL; i++) {
	/*
	 * skip EVERYTHING except the timezone; we'll regenerate the essentials
	 * ourselves. Add other exceptions as needed.
	 */
	if (strncmp (p, "TZ=", 3))
	    continue;

	c = strdup (p);
	SafeEnviron[j++] = c;
    }
    SafeEnviron[j] = NULL;

#ifdef SETPROCTITLE
    /* Save start and extent of argv for setproctitle. */
    Argv = argv;
    if (i > 0) {
	LastArgv = envp[i - 1] + strlen (envp[i - 1]);
    } else {
	LastArgv = argv[argc - 1] + strlen (argv[argc - 1]);
    }
#endif				/* SETPROCTITLE */

    setup_clean_fds ();
}

/***************************************************************************
 * Rename_cf (char *tf)
 * rename the temporary file to a control file
 ***************************************************************************/

void Rename_cf (char *tf) {
    char from[MAXPATHLEN + 1];
    char to[MAXPATHLEN + 1];
    int i;

    (void) sprintf (from, "%s/%s", SD, tf);
    (void) sprintf (to, "%s/c%s", SD, tf + 1);
    if (Debug > 4)
	log (XLOG_DEBUG, "renaming %s to %s", from, to);

    user_to_daemon ();
    i = rename (from, to);
    daemon_to_user ();

    if (i < 0) {
	logerr_die (XLOG_INFO, "Rename_cf: rename %s to %s failed", from, to);
    }
}

/***************************************************************************
 * Log_cf (char *tf, char *logstr, controlfile_fd)
 * copy the temporary file to the controlfile logging directory
 * for later examination (debugging aid). Should be run BEFORE
 * Rename_cf(). logstr is appended to the log's filename.
 * If controlfile_fd is >= 0, it's assumed to be an open file descriptor
 * with the file on it. The seek position will be altered.
 ***************************************************************************/

/* copy a file: returns -1 if copy failed */
int copy_file (char *from, char *to, int in_fd) {
    int out_fd, n, in_fd_was_passed;
    char buf[BUFSIZ];

    if (in_fd < 0) {
	in_fd = open (from, O_RDONLY, 0);
	in_fd_was_passed = 0;
    } else {
	(void) lseek (in_fd, 0L, SEEK_SET);
	in_fd_was_passed = 1;
    }
    out_fd = open (to, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
    if (out_fd <= 0) {
	logerr (XLOG_INFO, "copy_file: cannot open file %s.", to);
	return (-1);
    }

    while ((n = read (in_fd, buf, sizeof (buf))) > 0) {
	if (write (out_fd, buf, n) != n) {
	    logerr (XLOG_INFO, "copy_file: cannot write to file %s.", to);
	    return (-1);
	}
    }
    if (n < 0) {
	logerr (XLOG_INFO, "copy_file: error reading from %s", from);
	return (-1);
    }

    if (!in_fd_was_passed)
	(void) close (in_fd);
    (void) close (out_fd);
    return (0);
}

void Log_cf (char *tf, char *logstr, int ctrl_fd) {
    char from[MAXPATHLEN + 1];
    char to[MAXPATHLEN + 1];

    if (!Ctrlfile_logdir || !*Ctrlfile_logdir) {
	return;			/* control-file logging is disabled */
    }
    (void) sprintf (from, "%s/%s", SD, tf);
    (void) sprintf (to, "%s/c%s.%s", Ctrlfile_logdir, tf + 1, logstr);
    if (Debug > 4)
	log (XLOG_DEBUG, "making log copy of %s", to);
    user_to_daemon ();
    /* ignore the return status; we don't really mind if this fails */
    (void) copy_file (from, to, ctrl_fd);
    daemon_to_user ();
}

/* copy a list of h_addr unions. Returns a copy of the arg. */

char **copy_haddr_list (char **from) {
    char **i, **j;
    int jindex, arysize;

    arysize = 4; /* make room for this many addresses before resize */
    j = (char **) malloc ((unsigned) sizeof (i) * arysize);
    if (j == NULL) {
	logerr_die (XLOG_NOTICE, Malloc_failed_msg);
    }
    jindex = 0;
    for (i = from; *i; i++) {
	int k;
	chkandrealloc_list (j, jindex, arysize, IP_ADDRESS_LEN);
	malloc_or_die (j[jindex], sizeof (**i) * IP_ADDRESS_LEN);
	for (k = 0; k < IP_ADDRESS_LEN; k++) {
	  j[jindex][k] = (*i)[k];
	}
	jindex++;
    }
    j[jindex] = NULL;
    return (j);
}

void free_haddr_list (char **list) {
    char **del;

    if (list == NULL) {
	return;
    }
    del = list;
    while (*del != NULL) {
	free (*del);
	del++;
    }
    free (list);
}

/*
 * host comparison function: allows hostnames to be fully-qualified or
 * not, and will work with multi-homed hosts.
 */
int hostcmp (char *s1, char *s2) {
    struct hostent *tmp;
    char **to_free, **i1, **i2;
    int strcmp_result;

    /* try strcmp'ing them, in case they're both fully-qualified,
     * or both non-qualified, or numeric addresses.
     */
    if (*s1 == *s2) {
	strcmp_result = strcasecmp (s1, s2);
	if (!strcmp_result)
	    return (0);

    } else {
	strcmp_result = 1;
    }

    /* If either of the names don't resolve, return a non-match;
     * we've already checked to see if the names match using
     * strcmp().
     */
    if ((tmp = gethostbyname (s1)) == NULL)
	return (1);

    /* we need to copy this -- it's in tmp storage space;
     * keep a pointer to the start of the copy so we can
     * free it all later.
     */
    to_free = i1 = copy_haddr_list (tmp->h_addr_list);

    if ((tmp = gethostbyname (s2)) == NULL) {
        free_haddr_list (to_free);
	return (1);
    }

    i2 = tmp->h_addr_list;	/* no need to copy here */

    for (; *i1; i1++) {
	for (; *i2; i2++) {
	    if (((*i1)[0] == (*i2)[0]) && ((*i1)[1] == (*i2)[1]) &&
		((*i1)[2] == (*i2)[2]) && ((*i1)[3] == (*i2)[3]))
	    {
		free_haddr_list (to_free);
		return (0);
	    }
	}
    }
    free_haddr_list (to_free);
    return (1);			/* none of the hosts' addresses matched! */
}

/*
 * fopen_path will open the first available file from a list of files, separated by :s.
 * Note that it does NOT do this as the daemon user; it's too slow, and fopen_path()
 * is only ever used to open publicly-accessible config files (such as printcaps,
 * perms files, etc.)
 */
FILE *fopen_path (char *path, char *flags) {
    char *tmppath, *step;
    FILE *filep;

    /*
     * we need this to work around a HP/UX feature; (can't free tmppath after using
     * fopen(tmppath, flags) on it)
     */
    static char path_filename[MAXPATHLEN];

    if (path == NULL || !*path) {
	logerr_die (XLOG_INFO, "path is unset!");
    }
    if (Debug > 4)
	log (XLOG_DEBUG, "path to open: %s", path);
    filep = NULL;

    tmppath = strdup (path);
    /* we can now rip tmppath to shreds with impunity */

    step = (char *) strtok (tmppath, ":");
    do {
	strcpy (path_filename, step);	/* argh */

	if ((filep = fopen (path_filename, flags)) != NULL) {
	    if (Debug > 4)
		log (XLOG_DEBUG, "opened %s", step);
	    daemon_to_user ();
	    free (tmppath);
	    return (filep);
	}
    } while ((step = (char *) strtok (NULL, ":")));

    if (Debug > 4)
	log (XLOG_DEBUG, "no files could be opened.");
    free (tmppath);
    return (NULL);
}

/***************************************************************************
 * char *reallocstrcpy (to, from, fromsize)
 * strcpy with dynamic allocation. The "to" buffer is returned.
 * if you know the size of from's buffer, pass it in as the fourth arg,
 * otherwise pass in 0 (the strlen will be used).
 * use as follows: to = reallocstrcpy(to, "hello", 0);
 **************************************************************************/

char *reallocstrcpy (char *to, char *from, int fromspace) {
    int tospace;

    if (fromspace <= 0) {	/* is it set? */
	fromspace = strlen (from);
    }
    fromspace++;		/* copy the trailing null */

    if (!to) {
	tospace = fromspace;
	malloc_or_die (to, fromspace);
	*to = 0;
    } else {
	tospace = strlen (to);
    }

    /* if insufficient room, dynamically allocate buffer */
    if (tospace < fromspace) {
	if (Debug > 8)
	    log (XLOG_DEBUG, "reallocing buffer to %d chars", fromspace);
	to = (char *) realloc (to, (unsigned) fromspace);
	if (to == NULL) {
	    logerr_die (XLOG_NOTICE, Malloc_failed_msg);
	}
    }
    (void) bcopy (from, to, fromspace);
    return (to);
}

/**************************************************************************
 * open /dev/null on all non-open std fds (stdin, stdout, stderr).
 *************************************************************************/

void setup_std_fds (void) {
    int fd;
    while ((fd = open ("/dev/null", O_RDWR, 0)) >= 0) {
	if (fd >= 3) {
	    break;	/* just 0, 1 and 2. */
	} else if (Debug > 9) {
	    log (XLOG_DEBUG, "setup_std_fds: opened /dev/null on fd %d", fd);
	}
    }
    if (fd < 0) {
	/* well, this is pretty serious */
	logerr_die (XLOG_CRIT, "setup_std_fds: cannot open /dev/null");
    } else {
	(void) close (fd);
    }
}

/*
 * close all non-std file descriptors up to getdtablesize (), and
 * call setup_std_fds().
 */

void setup_clean_fds (void) {
    int fd;

    for (fd = 3; fd < getdtablesize (); fd++) {
	(void) close (fd);
    }
    setup_std_fds ();
}

/*
 * setup all non-std file descriptors up to getdtablesize ()
 * so they'll close before the next exec () call, and call
 * setup_std_fds ().
 * We can't simply use setup_clean_fds () because that breaks stuff
 * on IRIX 5.1.1 (connection to NIS? getpwuid() fails, anyway).
 */

void setup_close_on_exec (void) {
    int fd;

#ifdef HAVE_FCNTL
    int val;

    for (fd = 3; fd < getdtablesize (); fd++) {
	/* if this fails, the fd is probably not open, so skip it... */
	if ((val = fcntl (fd, F_GETFD, 0)) >= 0) {
#ifdef FD_CLOEXEC
	    val |= FD_CLOEXEC;	/* Advanced UNIX Programming */
#else
	    val |= 1;		/* SVR4 and others */
#endif
	    if (fcntl (fd, F_SETFD, val) == -1) {
		if (errno != EBADF) {
		    logerr (XLOG_INFO, "could not set close-on-exec on fd %d");
		}
	    }
	}
    }
    setup_std_fds ();

#else
    /* if we don't have fcntl(), just use setup_clean_fds (). */
    setup_clean_fds ();
#endif
}

/* provide us with an fd map */
void debug_fds (void) {
    int fd;

    for (fd = 0; fd < getdtablesize (); fd++) {
	if (fcntl (fd, F_GETFL, 0) < 0) {
	    log (XLOG_DEBUG, "debug_fds: fd %d is closed", fd);
	} else {
	    log (XLOG_DEBUG, "debug_fds: fd %d is open", fd);
	}
    }
}

/* Check to see if the nonqualified hostname "host" is valid
 * when the domain "domain" is appended to it. Returns the
 * "host.domain" string if it is, NULL otherwise.
 */
char *host_in_domain (char *host, char *domain) {
    int hlen;
    char *fq_hostname;

    hlen = strlen (host);
    if (*domain == '.') {
	domain++;		/* skip a leading . if it's there */
    }
    malloc_or_die (fq_hostname, hlen + strlen (domain) + 2);
    (void) strcpy (fq_hostname, host);
    fq_hostname [hlen] = '.';
    (void) strcpy (fq_hostname + hlen + 1, domain);

    /* DON'T use res_options here! PLP will usually be used in an
     * internal setting as opposed to accross the internet, and
     * relative domains, such as eg. "tst.com.iona.ie", are actually
     * more likely than domains such as "tst.com".
     */
    if (gethostbyname (fq_hostname) != NULL) {
	if (Debug > 2) {
	    log (XLOG_DEBUG,
		"host_in_domain: host %s exists in domain %s", host, domain);
	}
	return fq_hostname;

    } else {
	if (Debug > 2) {
	    log (XLOG_DEBUG,
		"host_in_domain: host %s does not exist in domain %s", host, domain);
	}
	free (fq_hostname);
	return NULL;
    }
}
